define("utils", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module utils */

  const IS_MAC = /Mac/.test(navigator.userAgent);

  /**
   * Utility functions.
   */
  Espo.Utils = {
    /**
     * Handle a click event action.
     *
     * @param {module:view} view A view.
     * @param {MouseEvent} event An event.
     * @param {HTMLElement} element An  element.
     * @param {{
     *     action?: string,
     *     handler?: string,
     *     actionFunction?: string,
     *     actionItems?: Array<{
     *         onClick?: function(),
     *         name?: string,
     *         handler?: string,
     *         actionFunction?: string,
     *     }>,
     *     className?: string,
     * }} [actionData] Data. If an action is not specified, it will be fetched from a target element.
     * @return {boolean} True if handled.
     */
    handleAction: function (view, event, element, actionData) {
      actionData = actionData || {};
      const $target = $(element);
      const action = actionData.action || $target.data('action');
      const name = $target.data('name') || action;
      let method;
      let handler;
      if (name && actionData.actionItems && (!actionData.className || element.classList.contains(actionData.className))) {
        const data = actionData.actionItems.find(item => {
          return item.name === name || item.action === name;
        });
        if (data && data.onClick) {
          data.onClick();
          return true;
        }
        if (data) {
          handler = data.handler;
          method = data.actionFunction;
        }
      }
      if (!action && !actionData.actionFunction && !method) {
        return false;
      }
      if (event.ctrlKey || event.metaKey || event.shiftKey) {
        const href = $target.attr('href');
        if (href && href !== 'javascript:') {
          return false;
        }
      }
      const data = $target.data();
      method = actionData.actionFunction || method || 'action' + Espo.Utils.upperCaseFirst(action);
      handler = actionData.handler || handler || data.handler;
      let fired = false;
      if (handler) {
        event.preventDefault();
        event.stopPropagation();
        fired = true;
        Espo.loader.require(handler, Handler => {
          const handler = new Handler(view);
          handler[method].call(handler, data, event);
        });
      } else if (typeof view[method] === 'function') {
        view[method].call(view, data, event);
        event.preventDefault();
        event.stopPropagation();
        fired = true;
      }
      if (!fired) {
        return false;
      }
      this._processAfterActionDropdown($target);
      return true;
    },
    /**
     * @private
     * @param {JQuery} $target
     */
    _processAfterActionDropdown: function ($target) {
      const $dropdown = $target.closest('.dropdown-menu');
      if (!$dropdown.length) {
        return;
      }
      const $dropdownToggle = $dropdown.parent().find('[data-toggle="dropdown"]');
      if (!$dropdownToggle.length) {
        return;
      }
      let isDisabled = false;
      if ($dropdownToggle.attr('disabled')) {
        isDisabled = true;
        $dropdownToggle.removeAttr('disabled').removeClass('disabled');
      }

      // noinspection JSUnresolvedReference
      $dropdownToggle.dropdown('toggle');
      $dropdownToggle.focus();
      if (isDisabled) {
        $dropdownToggle.attr('disabled', 'disabled').addClass('disabled');
      }
    },
    /**
     * @typedef {Object} Espo.Utils~ActionAvailabilityDefs
     *
     * @property {string|null} [configCheck] A config path to check. Path items are separated
     *   by the dot. If a config value is not empty, then the action is allowed.
     *   The `!` prefix reverses the check.
     */

    /**
     * Check action availability.
     *
     * @param {module:view-helper} helper A view helper.
     * @param {Espo.Utils~ActionAvailabilityDefs} item Definitions.
     * @returns {boolean}
     */
    checkActionAvailability: function (helper, item) {
      const config = helper.config;
      if (item.configCheck) {
        let configCheck = item.configCheck;
        let opposite = false;
        if (configCheck.substring(0, 1) === '!') {
          opposite = true;
          configCheck = configCheck.substring(1);
        }
        let configCheckResult = config.getByPath(configCheck.split('.'));
        if (opposite) {
          configCheckResult = !configCheckResult;
        }
        if (!configCheckResult) {
          return false;
        }
      }
      return true;
    },
    /**
     * @typedef {Object} Espo.Utils~ActionAccessDefs
     *
     * @property {'create'|'read'|'edit'|'stream'|'delete'|null} acl An ACL action to check.
     * @property {string|null} [aclScope] A scope to check.
     * @property {string|null} [scope] Deprecated. Use `aclScope`.
     */

    /**
     * Check access to an action.
     *
     * @param {module:acl-manager} acl An ACL manager.
     * @param {string|module:model|null} [obj] A scope or a model.
     * @param {Espo.Utils~ActionAccessDefs} item Definitions.
     * @param {boolean} [isPrecise=false] To return `null` if not enough data is set in a model.
     *   E.g. the `teams` field is not yet loaded.
     * @returns {boolean|null}
     */
    checkActionAccess: function (acl, obj, item, isPrecise) {
      let hasAccess = true;
      if (item.acl) {
        if (!item.aclScope) {
          if (obj) {
            if (typeof obj === 'string' || obj instanceof String) {
              hasAccess = acl.check(obj, item.acl);
            } else {
              hasAccess = acl.checkModel(obj, item.acl, isPrecise);
            }
          } else {
            hasAccess = acl.check(item.scope, item.acl);
          }
        } else {
          hasAccess = acl.check(item.aclScope, item.acl);
        }
      } else if (item.aclScope) {
        hasAccess = acl.checkScope(item.aclScope);
      }
      return hasAccess;
    },
    /**
     * @typedef {Object} Espo.Utils~AccessDefs
     *
     * @property {'create'|'read'|'edit'|'stream'|'delete'|null} action An ACL action to check.
     * @property {string|null} [scope] A scope to check.
     * @property {string[]} [portalIdList] A portal ID list. To check whether a user in one of portals.
     * @property {string[]} [teamIdList] A team ID list. To check whether a user in one of teams.
     * @property {boolean} [isPortalOnly=false] Allow for portal users only.
     * @property {boolean} [inPortalDisabled=false] Disable for portal users.
     * @property {boolean} [isAdminOnly=false] Allow for admin users only.
     */

    /**
     * Check access to an action.
     *
     * @param {module:utils~AccessDefs[]} dataList List of definitions.
     * @param {module:acl-manager} acl An ACL manager.
     * @param {module:models/user} user A user.
     * @param {module:model|null} [entity] A model.
     * @param {boolean} [allowAllForAdmin=false] Allow all for an admin.
     * @returns {boolean}
     */
    checkAccessDataList: function (dataList, acl, user, entity, allowAllForAdmin) {
      if (!dataList || !dataList.length) {
        return true;
      }
      for (const i in dataList) {
        const item = dataList[i];
        if (item.scope) {
          if (item.action) {
            if (!acl.check(item.scope, item.action)) {
              return false;
            }
          } else {
            if (!acl.checkScope(item.scope)) {
              return false;
            }
          }
        } else if (item.action) {
          if (entity) {
            if (!acl.check(entity, item.action)) {
              return false;
            }
          }
        }
        if (item.teamIdList) {
          if (user && !(allowAllForAdmin && user.isAdmin())) {
            let inTeam = false;
            user.getLinkMultipleIdList('teams').forEach(teamId => {
              if (~item.teamIdList.indexOf(teamId)) {
                inTeam = true;
              }
            });
            if (!inTeam) {
              return false;
            }
          }
        }
        if (item.portalIdList) {
          if (user && !(allowAllForAdmin && user.isAdmin())) {
            let inPortal = false;
            user.getLinkMultipleIdList('portals').forEach(portalId => {
              if (~item.portalIdList.indexOf(portalId)) {
                inPortal = true;
              }
            });
            if (!inPortal) {
              return false;
            }
          }
        }
        if (item.isPortalOnly) {
          if (user && !(allowAllForAdmin && user.isAdmin())) {
            if (!user.isPortal()) {
              return false;
            }
          }
        } else if (item.inPortalDisabled) {
          if (user && !(allowAllForAdmin && user.isAdmin())) {
            if (user.isPortal()) {
              return false;
            }
          }
        }
        if (item.isAdminOnly) {
          if (user) {
            if (!user.isAdmin()) {
              return false;
            }
          }
        }
      }
      return true;
    },
    /**
     * @private
     * @param {string} string
     * @param {string} p
     * @returns {string}
     */
    convert: function (string, p) {
      if (string === null) {
        return string;
      }
      let result = string;
      switch (p) {
        case 'c-h':
        case 'C-h':
          result = Espo.Utils.camelCaseToHyphen(string);
          break;
        case 'h-c':
          result = Espo.Utils.hyphenToCamelCase(string);
          break;
        case 'h-C':
          result = Espo.Utils.hyphenToUpperCamelCase(string);
          break;
      }
      return result;
    },
    /**
     * Is object.
     *
     * @param {*} obj What to check.
     * @returns {boolean}
     */
    isObject: function (obj) {
      if (obj === null) {
        return false;
      }
      return typeof obj === 'object';
    },
    /**
     * A shallow clone.
     *
     * @template {*} TObject
     * @param {TObject} obj An object.
     * @returns {TObject}
     */
    clone: function (obj) {
      if (!Espo.Utils.isObject(obj)) {
        return obj;
      }
      return _.isArray(obj) ? obj.slice() : _.extend({}, obj);
    },
    /**
     * A deep clone.
     *
     * @template {*} TObject
     * @param {TObject} data An object.
     * @returns {TObject}
     */
    cloneDeep: function (data) {
      data = Espo.Utils.clone(data);
      if (Espo.Utils.isObject(data) || _.isArray(data)) {
        for (const i in data) {
          data[i] = this.cloneDeep(data[i]);
        }
      }
      return data;
    },
    /**
     * Deep comparison.
     *
     * @param {Object} a1 An argument 1.
     * @param {Object} a2 An argument 2.
     * @return {boolean}
     */
    areEqual: function (a1, a2) {
      return _.isEqual(a1, a2);
    },
    /**
     * Compose a class name.
     *
     * @param {string} module A module.
     * @param {string} name A name.
     * @param {string} [location=''] A location.
     * @return {string}
     */
    composeClassName: function (module, name, location) {
      if (module) {
        module = this.camelCaseToHyphen(module);
        name = this.camelCaseToHyphen(name).split('.').join('/');
        location = this.camelCaseToHyphen(location || '');
        return module + ':' + location + '/' + name;
      } else {
        name = this.camelCaseToHyphen(name).split('.').join('/');
        return location + '/' + name;
      }
    },
    /**
     * Compose a view class name.
     *
     * @param {string} name A name.
     * @returns {string}
     */
    composeViewClassName: function (name) {
      if (name && name[0] === name[0].toLowerCase()) {
        return name;
      }
      if (name.indexOf(':') !== -1) {
        const arr = name.split(':');
        let modPart = arr[0];
        let namePart = arr[1];
        modPart = this.camelCaseToHyphen(modPart);
        namePart = this.camelCaseToHyphen(namePart).split('.').join('/');
        return modPart + ':' + 'views' + '/' + namePart;
      } else {
        name = this.camelCaseToHyphen(name).split('.').join('/');
        return 'views' + '/' + name;
      }
    },
    /**
     * Convert a string from camelCase to hyphen and replace dots with hyphens.
     * Useful for setting to DOM attributes.
     *
     * @param {string} string A string.
     * @returns {string}
     */
    toDom: function (string) {
      return Espo.Utils.convert(string, 'c-h').split('.').join('-');
    },
    /**
     * Lower-case a first character.
     *
     * @param  {string} string A string.
     * @returns {string}
     */
    lowerCaseFirst: function (string) {
      if (string === null) {
        return string;
      }
      return string.charAt(0).toLowerCase() + string.slice(1);
    },
    /**
     * Upper-case a first character.
     *
     * @param  {string} string A string.
     * @returns {string}
     */
    upperCaseFirst: function (string) {
      if (string === null) {
        return string;
      }
      return string.charAt(0).toUpperCase() + string.slice(1);
    },
    /**
     * Hyphen to UpperCamelCase.
     *
     * @param {string} string A string.
     * @returns {string}
     */
    hyphenToUpperCamelCase: function (string) {
      if (string === null) {
        return string;
      }
      return this.upperCaseFirst(string.replace(/-([a-z])/g, function (g) {
        return g[1].toUpperCase();
      }));
    },
    /**
     * Hyphen to camelCase.
     *
     * @param {string} string A string.
     * @returns {string}
     */
    hyphenToCamelCase: function (string) {
      if (string === null) {
        return string;
      }
      return string.replace(/-([a-z])/g, function (g) {
        return g[1].toUpperCase();
      });
    },
    /**
     * CamelCase to hyphen.
     *
     * @param {string} string A string.
     * @returns {string}
     */
    camelCaseToHyphen: function (string) {
      if (string === null) {
        return string;
      }
      return string.replace(/([a-z])([A-Z])/g, '$1-$2').toLowerCase();
    },
    /**
     * Trim an ending slash.
     *
     * @param {String} str A string.
     * @returns {string}
     */
    trimSlash: function (str) {
      if (str.slice(-1) === '/') {
        return str.slice(0, -1);
      }
      return str;
    },
    /**
     * Parse params in string URL options.
     *
     * @param {string} string An URL part.
     * @returns {Object.<string,string>}
     */
    parseUrlOptionsParam: function (string) {
      if (!string) {
        return {};
      }
      if (string.indexOf('&') === -1 && string.indexOf('=') === -1) {
        return {};
      }
      const options = {};
      if (typeof string !== 'undefined') {
        string.split('&').forEach(item => {
          const p = item.split('=');
          options[p[0]] = true;
          if (p.length > 1) {
            options[p[0]] = p[1];
          }
        });
      }
      return options;
    },
    /**
     * Key a key from a key-event.
     *
     * @param {JQueryKeyEventObject|KeyboardEvent} e A key event.
     * @return {string}
     */
    getKeyFromKeyEvent: function (e) {
      let key = e.code;
      key = keyMap[key] || key;
      if (e.shiftKey) {
        key = 'Shift+' + key;
      }
      if (e.altKey) {
        key = 'Alt+' + key;
      }
      if (IS_MAC ? e.metaKey : e.ctrlKey) {
        key = 'Control+' + key;
      }
      return key;
    },
    /**
     * Generate an ID. Not to be used by 3rd party code.
     *
     * @internal
     * @return {string}
     */
    generateId: function () {
      return Math.floor(Math.random() * 10000001).toString();
    },
    /**
     * Not to be used in custom code. Can be removed in future versions.
     * @internal
     * @return {string}
     */
    obtainBaseUrl: function () {
      let baseUrl = window.location.origin + window.location.pathname;
      if (baseUrl.slice(-1) !== '/') {
        baseUrl = window.location.pathname.includes('.') ? baseUrl.slice(0, baseUrl.lastIndexOf('/')) + '/' : baseUrl + '/';
      }
      return baseUrl;
    }
  };
  const keyMap = {
    'NumpadEnter': 'Enter'
  };

  /**
   * @deprecated Use `Espo.Utils`.
   */
  Espo.utils = Espo.Utils;
  var _default = _exports.default = Espo.Utils;
});

define("model", ["exports", "bullbone", "underscore"], function (_exports, _bullbone, _underscore) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _underscore = _interopRequireDefault(_underscore);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module model */

  /**
   * When attributes have changed.
   *
   * @event Model#change
   * @param {Model} model A model.
   * @param {Object.<string, *>} o Options.
   */

  /**
   * On sync with backend.
   *
   * @event Model#sync
   * @param {Model} model A model.
   * @param {Object} response Response from backend.
   * @param {Object.<string, *>} o Options.
   */

  /**
   * Definitions.
   *
   * @typedef module:model~defs
   * @type {Object}
   * @property {Object.<string, module:model~fieldDefs>} [fields] Fields.
   * @property {Object.<string, Object.<string, *>>} [links] Links.
   */

  /**
   * Field definitions.
   *
   * @typedef module:model~fieldDefs
   * @type {Object & Record}
   * @property {string} type A type.
   */

  /** @typedef {import('bullbone')} Bull */

  /**
   * A model.
   *
   * @mixes Bull.Events
   */
  class Model {
    /**
     * A root URL. An ID will be appended. Used for syncing with backend.
     *
     * @type {string|null}
     */
    urlRoot = null;

    /**
     * A URL. If not empty, then will be used for syncing instead of `urlRoot`.
     *
     * @type {string|null}
     */
    url = null;

    /**
     * A name.
     *
     * @type {string|null}
     */
    name = null;

    /**
     * An entity type.
     *
     * @type {string|null}
     */
    entityType = null;

    /**
     * A last request promise.
     *
     * @type {module:ajax.AjaxPromise|null}
     */
    lastSyncPromise = null;

    /** @private */
    _pending;
    /** @private */
    _changing;

    /**
     * @param {Object.<string, *>|Model} [attributes]
     * @param {{
     *     collection?: module:collection,
     *     entityType?: string,
     *     urlRoot?: string,
     *     url?: string,
     *     defs?: module:model~defs,
     *     user?: module:models/user,
     *     dateTime?: module:date-time,
     * }} [options]
     */
    constructor(attributes, options) {
      options = options || {};

      /**
       * An ID attribute.
       * @type {string}
       */
      this.idAttribute = 'id';

      /**
       * A record ID.
       * @type {string|null}
       */
      this.id = null;

      /**
       * An instance ID.
       * @type {string}
       */
      this.cid = _underscore.default.uniqueId('c');

      /**
       * Attribute values.
       * @type {Object.<string, *>}
       */
      this.attributes = {};
      if (options.collection) {
        this.collection = options.collection;
      }
      this.set(attributes || {});

      /**
       * Definitions.
       */
      this.defs = options.defs || {};
      if (!this.defs.fields) {
        this.defs.fields = {};
      }
      if (options.entityType) {
        this.entityType = options.entityType;
        this.name = options.entityType;
        this.urlRoot = options.entityType;
      }
      this.urlRoot = options.urlRoot || this.urlRoot;
      this.url = options.url || this.url;

      /** @private */
      this.dateTime = options.dateTime || null;

      /** @private */
      this.changed = {};
      /** @private */
      this._previousAttributes = null;
    }

    /**
     * @protected
     * @param {string} [method] HTTP method.
     * @param {Model} model
     * @param {Object.<string, *>} [options]
     * @returns {module:ajax.AjaxPromise|Promise}
     */
    sync(method, model, options) {
      const methodMap = {
        'create': 'POST',
        'update': 'PUT',
        'patch': 'PUT',
        'delete': 'DELETE',
        'read': 'GET'
      };
      const httpMethod = methodMap[method];
      if (!httpMethod) {
        throw new Error(`Bad request method '${method}'.`);
      }
      options = options || {};
      const url = this.composeSyncUrl();
      if (!url) {
        throw new Error(`No 'url'.`);
      }
      const data = model && ['create', 'update', 'patch'].includes(method) ? options.attributes || model.getClonedAttributes() : null;
      const error = options.error;
      options.error = (xhr, textStatus, errorThrown) => {
        options.textStatus = textStatus;
        options.errorThrown = errorThrown;
        if (error) {
          error.call(options.context, xhr, textStatus, errorThrown);
        }
      };
      const stringData = data ? JSON.stringify(data) : null;
      const ajaxPromise = !options.bypassRequest ? Espo.Ajax.request(url, httpMethod, stringData, options) : Promise.resolve();
      options.xhr = ajaxPromise.xhr;
      model.trigger('request', url, httpMethod, data, ajaxPromise, options);
      return ajaxPromise;
    }

    /**
     * Set an attribute value.
     *
     * @param {(string|Object)} attribute An attribute name or a {key => value} object.
     * @param {*} [value] A value or options if the first argument is an object.
     * @param {{silent?: boolean} & Object.<string, *>} [options] Options. `silent` won't trigger a `change` event.
     * @returns {this}
     * @fires Model#change Unless `{silent: true}`.
     */
    set(attribute, value, options) {
      if (attribute == null) {
        return this;
      }
      let attributes;
      if (typeof attribute === 'object') {
        return this.setMultiple(attribute, value);
      }
      attributes = {};
      attributes[attribute] = value;
      return this.setMultiple(attributes, options);
    }

    /**
     * Set attributes values.
     *
     * @param {Object.<string, *>} attributes
     * @param {{
     *     silent?: boolean,
     *     unset?: boolean,
     * } & Object.<string, *>} [options] Options. `silent` won't trigger a `change` event.
     * @return {this}
     * @fires Model#change Unless `{silent: true}`.
     * @copyright Credits to Backbone.js.
     */
    setMultiple(attributes, options) {
      if (this.idAttribute in attributes) {
        this.id = attributes[this.idAttribute];
      }
      options = options || {};
      const changes = [];
      const changing = this._changing;
      this._changing = true;
      if (!changing) {
        this._previousAttributes = _underscore.default.clone(this.attributes);
        this.changed = {};
      }
      const current = this.attributes;
      const changed = this.changed;
      const previous = this._previousAttributes;
      for (const attribute in attributes) {
        const value = attributes[attribute];
        if (!_underscore.default.isEqual(current[attribute], value)) {
          changes.push(attribute);
        }
        if (!_underscore.default.isEqual(previous[attribute], value)) {
          changed[attribute] = value;
        } else {
          delete changed[attribute];
        }
        options.unset ? delete current[attribute] : current[attribute] = value;
      }
      if (!options.silent) {
        if (changes.length) {
          this._pending = options;
        }
        for (let i = 0; i < changes.length; i++) {
          this.trigger('change:' + changes[i], this, current[changes[i]], options);
        }
      }
      if (changing) {
        return this;
      }
      if (!options.silent) {
        // Changes can be recursively nested within `change` events.
        while (this._pending) {
          options = this._pending;
          this._pending = false;
          this.trigger('change', this, options);
        }
      }
      this._pending = false;
      this._changing = false;
      return this;
    }

    /**
     * Unset an attribute.
     *
     * @param {string} attribute An attribute.
     * @param {{silent?: boolean} & Object.<string, *>} [options] Options.
     * @return {Model}
     */
    unset(attribute, options) {
      options = {
        ...options,
        unset: true
      };
      const attributes = {};
      attributes[attribute] = null;
      return this.setMultiple(attributes, options);
    }

    /**
     * Get an attribute value.
     *
     * @param {string} attribute An attribute name.
     * @returns {*}
     */
    get(attribute) {
      if (attribute === this.idAttribute && this.id) {
        return this.id;
      }
      return this.attributes[attribute];
    }

    /**
     * Whether attribute is set.
     *
     * @param {string} attribute An attribute name.
     * @returns {boolean}
     */
    has(attribute) {
      const value = this.get(attribute);
      return typeof value !== 'undefined';
    }

    /**
     * Removes all attributes from the model.
     * Fires a `change` event unless `silent` is passed as an option.
     *
     * @param {{silent?: boolean} & Object.<string, *>} [options] Options.
     */
    clear(options) {
      const attributes = {};
      for (const key in this.attributes) {
        attributes[key] = void 0;
      }
      options = {
        ...options,
        unset: true
      };
      return this.set(attributes, options);
    }

    /**
     * Whether is new.
     *
     * @returns {boolean}
     */
    isNew() {
      return !this.id;
    }

    /**
     * Whether an attribute changed. To be called only within a 'change' event handler.
     *
     * @param {string} [attribute]
     * @return {boolean}
     */
    hasChanged(attribute) {
      if (!attribute) {
        return !_underscore.default.isEmpty(this.changed);
      }
      return _underscore.default.has(this.changed, attribute);
    }

    /**
     * Get changed attribute values. To be called only within a 'change' event handler.
     *
     * @return {Object.<string, *>}
     */
    changedAttributes() {
      return this.hasChanged() ? _underscore.default.clone(this.changed) : {};
    }

    /**
     * Get previous attributes. To be called only within a 'change' event handler.
     *
     * @return {Object.<string, *>}
     */
    previousAttributes() {
      return _underscore.default.clone(this._previousAttributes);
    }

    /**
     * Get a previous attribute value. To be called only within a 'change' event handler.
     *
     * @param attribute
     * @return {*}
     */
    previous(attribute) {
      if (!this._previousAttributes) {
        return null;
      }
      return this._previousAttributes[attribute];
    }

    /**
     * Fetch values from the backend.
     *
     * @param {Object.<string, *>} [options] Options.
     * @returns {Promise}
     * @fires Model#sync
     */
    fetch(options) {
      options = {
        ...options
      };
      const success = options.success;
      options.success = response => {
        const serverAttributes = this.prepareAttributes(response, options);
        this.set(serverAttributes, options);
        if (success) {
          success.call(options.context, this, response, options);
        }
        this.trigger('sync', this, response, options);
      };
      this.lastSyncPromise = this.sync('read', this, options);
      return this.lastSyncPromise;
    }

    /**
     * Save values to the backend.
     *
     * @param {Object.<string, *>} [attributes] Attribute values.
     * @param {{
     *     patch?: boolean,
     *     wait?: boolean,
     * } & Object.<string, *>} [options] Options.
     * @returns {Promise<Object.<string, *>>}
     * @fires Model#sync
     * @copyright Credits to Backbone.js.
     */
    save(attributes, options) {
      options = {
        ...options
      };
      if (attributes && !options.wait) {
        this.setMultiple(attributes, options);
      }
      const success = options.success;
      const setAttributes = this.attributes;
      options.success = response => {
        this.attributes = setAttributes;
        let responseAttributes = this.prepareAttributes(response, options);
        if (options.wait) {
          responseAttributes = {
            ...setAttributes,
            ...responseAttributes
          };
        }
        if (responseAttributes) {
          this.setMultiple(responseAttributes, options);
        }
        if (success) {
          success.call(options.context, this, response, options);
        }
        this.trigger('sync', this, response, options);
      };
      const error = options.error;
      options.error = response => {
        if (error) {
          error.call(options.context, this, response, options);
        }
        this.trigger('error', this, response, options);
      };
      if (attributes && options.wait) {
        // Set temporary attributes to properly find new IDs.
        this.attributes = {
          ...setAttributes,
          ...attributes
        };
      }
      const method = this.isNew() ? 'create' : options.patch ? 'patch' : 'update';
      if (method === 'patch') {
        options.attributes = attributes;
      }
      const result = this.sync(method, this, options);
      this.attributes = setAttributes;
      return result;
    }

    /**
     * Delete the record in the backend.
     *
     * @param {{wait: boolean} & Object.<string, *>} [options] Options.
     * @returns {Promise}
     * @fires Model#sync
     * @copyright Credits to Backbone.js.
     */
    destroy(options) {
      options = _underscore.default.clone(options || {});
      const success = options.success;
      const destroy = () => {
        this.stopListening();
        this.trigger('destroy', this, this.collection, options);
      };
      options.success = response => {
        if (options.wait) {
          destroy();
        }
        if (success) {
          success.call(options.context, this, response, options);
        }
        if (!this.isNew()) {
          this.trigger('sync', this, response, options);
        }
      };
      if (this.isNew()) {
        _underscore.default.defer(options.success);
        if (!options.wait) {
          destroy();
        }
        return Promise.resolve();
      }
      const error = options.error;
      options.error = response => {
        if (error) {
          error.call(options.context, this, response, options);
        }
        this.trigger('error', this, response, options);
      };
      const result = this.sync('delete', this, options);
      if (!options.wait) {
        destroy();
      }
      return result;
    }

    /**
     * Compose a URL for syncing.
     *
     * @protected
     * @return {string}
     */
    composeSyncUrl() {
      if (this.url) {
        return this.url;
      }
      let urlRoot = this.urlRoot;
      if (!urlRoot && this.collection) {
        urlRoot = this.collection.urlRoot;
      }
      if (!urlRoot) {
        throw new Error("No urlRoot.");
      }
      if (this.isNew()) {
        return urlRoot;
      }
      const id = this.get(this.idAttribute);
      return urlRoot.replace(/[^\/]$/, '$&/') + encodeURIComponent(id);
    }

    // noinspection JSUnusedLocalSymbols
    /**
     * Prepare attributes.
     *
     * @param {*} response A response from the backend.
     * @param {Object.<string, *>} options Options.
     * @return {*} Attributes.
     * @internal
     */
    prepareAttributes(response, options) {
      return response;
    }

    /**
     * Clone.
     *
     * @return {Model}
     */
    clone() {
      return new this.constructor(Espo.Utils.cloneDeep(this.attributes), {
        entityType: this.entityType,
        urlRoot: this.urlRoot,
        url: this.url,
        defs: this.defs,
        dateTime: this.dateTime
      });
    }

    /**
     * Set defs.
     *
     * @param {module:model~defs} defs
     */
    setDefs(defs) {
      this.defs = defs || {};
      if (!this.defs.fields) {
        this.defs.fields = {};
      }
    }

    /**
     * Get cloned attribute values.
     *
     * @returns {Object.<string, *>}
     */
    getClonedAttributes() {
      return Espo.Utils.cloneDeep(this.attributes);
    }

    /**
     * Populate default values.
     */
    populateDefaults() {
      let defaultHash = {};
      const fieldDefs = this.defs.fields;
      for (const field in fieldDefs) {
        if (this.hasFieldParam(field, 'default')) {
          try {
            defaultHash[field] = this.parseDefaultValue(this.getFieldParam(field, 'default'));
          } catch (e) {
            console.error(e);
          }
        }
        const defaultAttributes = this.getFieldParam(field, 'defaultAttributes');
        if (defaultAttributes) {
          for (const attribute in defaultAttributes) {
            defaultHash[attribute] = defaultAttributes[attribute];
          }
        }
      }
      defaultHash = Espo.Utils.cloneDeep(defaultHash);
      for (const attr in defaultHash) {
        if (this.has(attr)) {
          delete defaultHash[attr];
        }
      }
      this.set(defaultHash, {
        silent: true
      });
    }

    /**
     * @protected
     * @param {*} defaultValue
     * @returns {*}
     */
    parseDefaultValue(defaultValue) {
      if (typeof defaultValue === 'string' && defaultValue.indexOf('javascript:') === 0) {
        const code = defaultValue.substring(11);
        defaultValue = new Function("with(this) { " + code + "}").call(this);
      }
      return defaultValue;
    }

    /**
     * Get a link multiple column value.
     *
     * @param {string} field
     * @param {string} column
     * @param {string} id
     * @returns {*}
     */
    getLinkMultipleColumn(field, column, id) {
      return ((this.get(field + 'Columns') || {})[id] || {})[column];
    }

    /**
     * @typedef {Object} model:model~setRelateItem
     * @property {string} link A link.
     * @property {import('model').default} model A model.
     */

    /**
     * Set relate data (when creating a related record).
     *
     * @param {model:model~setRelateItem | model:model~setRelateItem[]} data
     */
    setRelate(data) {
      const setRelate = options => {
        const link = options.link;
        const model = /** @type {module:model} */options.model;
        if (!link || !model) {
          throw new Error('Bad related options');
        }
        const type = this.defs.links[link].type;
        switch (type) {
          case 'belongsToParent':
            this.set(link + 'Id', model.id);
            this.set(link + 'Type', model.entityType);
            this.set(link + 'Name', model.get('name'));
            break;
          case 'belongsTo':
            this.set(link + 'Id', model.id);
            this.set(link + 'Name', model.get('name'));
            break;
          case 'hasMany':
            const ids = [];
            ids.push(model.id);
            const names = {};
            names[model.id] = model.get('name');
            this.set(link + 'Ids', ids);
            this.set(link + 'Names', names);
            break;
        }
      };
      if (Object.prototype.toString.call(data) === '[object Array]') {
        data.forEach(options => {
          setRelate(options);
        });
        return;
      }
      setRelate(data);
    }

    /**
     * Get a field list.
     *
     * @return {string[]}
     */
    getFieldList() {
      if (!this.defs || !this.defs.fields) {
        return [];
      }
      return Object.keys(this.defs.fields);
    }

    /**
     * Get a field type.
     *
     * @param {string} field
     * @returns {string|null}
     */
    getFieldType(field) {
      if (!this.defs || !this.defs.fields) {
        return null;
      }
      if (field in this.defs.fields) {
        return this.defs.fields[field].type || null;
      }
      return null;
    }

    /**
     * Get a field param.
     *
     * @param {string} field
     * @param {string} param
     * @returns {*}
     */
    getFieldParam(field, param) {
      if (!this.defs || !this.defs.fields) {
        return null;
      }
      if (field in this.defs.fields) {
        if (param in this.defs.fields[field]) {
          return this.defs.fields[field][param];
        }
      }
      return null;
    }
    hasFieldParam(field, param) {
      if (!this.defs || !this.defs.fields) {
        return false;
      }
      if (field in this.defs.fields) {
        if (param in this.defs.fields[field]) {
          return true;
        }
      }
      return false;
    }

    /**
     * Get a link type.
     *
     * @param {string} link
     * @returns {string|null}
     */
    getLinkType(link) {
      if (!this.defs || !this.defs.links) {
        return null;
      }
      if (link in this.defs.links) {
        return this.defs.links[link].type || null;
      }
      return null;
    }

    /**
     * Get a link param.
     *
     * @param {string} link A link.
     * @param {string} param A param.
     * @returns {*}
     */
    getLinkParam(link, param) {
      if (!this.defs || !this.defs.links) {
        return null;
      }
      if (link in this.defs.links) {
        if (param in this.defs.links[link]) {
          return this.defs.links[link][param];
        }
      }
      return null;
    }

    /**
     * Is a field read-only.
     *
     * @param {string} field A field.
     * @returns {bool}
     */
    isFieldReadOnly(field) {
      return this.getFieldParam(field, 'readOnly') || false;
    }

    /**
     * If a field required.
     *
     * @param {string} field A field.
     * @returns {bool}
     */
    isRequired(field) {
      return this.getFieldParam(field, 'required') || false;
    }

    /**
     * Get IDs of a link-multiple field.
     *
     * @param {string} field A link-multiple field name.
     * @returns {string[]}
     */
    getLinkMultipleIdList(field) {
      return this.get(field + 'Ids') || [];
    }

    /**
     * Get team IDs.
     *
     * @returns {string[]}
     */
    getTeamIdList() {
      return this.get('teamsIds') || [];
    }

    /**
     * Whether it has a field.
     *
     * @param {string} field A field.
     * @returns {boolean}
     */
    hasField(field) {
      return 'defs' in this && 'fields' in this.defs && field in this.defs.fields;
    }

    /**
     * Has a link.
     *
     * @param {string} link A link.
     * @returns {boolean}
     */
    hasLink(link) {
      return 'defs' in this && 'links' in this.defs && link in this.defs.links;
    }

    /**
     * @returns {boolean}
     */
    isEditable() {
      return true;
    }

    /**
     * @returns {boolean}
     */
    isRemovable() {
      return true;
    }

    /**
     * Get an entity type.
     *
     * @returns {string}
     */
    getEntityType() {
      return this.name;
    }

    /**
     * Abort the last fetch.
     */
    abortLastFetch() {
      if (this.lastSyncPromise && this.lastSyncPromise.getReadyState() < 4) {
        this.lastSyncPromise.abort();
      }
    }
  }
  Object.assign(Model.prototype, _bullbone.Events);
  Model.extend = _bullbone.View.extend;
  var _default = _exports.default = Model;
});

define("acl", ["exports", "bullbone"], function (_exports, _bullbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module acl */

  /**
   * Internal class for access checking. Can be extended to customize access checking
   * for a specific scope.
   */
  class Acl {
    /**
     * @param {module:models/user} user A user.
     * @param {string} scope A scope.
     * @param {Object} params Parameters.
     * @param {import('acl-manager').default} aclManager
     */
    constructor(user, scope, params, aclManager) {
      /**
       * A user.
       *
       * @type {module:models/user|null}
       * @protected
       */
      this.user = user || null;
      this.scope = scope;
      params = params || {};
      this.aclAllowDeleteCreated = params.aclAllowDeleteCreated;
      this.teamsFieldIsForbidden = params.teamsFieldIsForbidden;

      /**
       * @type {string[]}
       */
      this.forbiddenFieldList = params.forbiddenFieldList || [];

      /**
       * @protected
       * @type {boolean}
       */
      this.collaboratorsFieldIsForbidden = this.forbiddenFieldList.includes('collaborators');

      /**
       * @type {import('acl-manager').default}
       * @private
       */
      this._aclManager = aclManager;
    }

    /**
     * Get a user.
     *
     * @returns {module:models/user}
     * @protected
     */
    getUser() {
      return this.user;
    }

    /**
     * Check access to a scope.
     *
     * @param {string|boolean|Object.<string, string>} data Access data.
     * @param {module:acl-manager~action|null} [action=null] An action.
     * @param {boolean} [precise=false] To return `null` if `inTeam == null`.
     * @param {Record.<string, boolean|null>|null} [entityAccessData=null] Entity access data. `inTeam`, `isOwner`.
     * @returns {boolean|null} True if access allowed.
     */
    checkScope(data, action, precise, entityAccessData) {
      entityAccessData = entityAccessData || {};
      const inTeam = entityAccessData.inTeam;
      const isOwner = entityAccessData.isOwner;
      const isShared = entityAccessData.isShared;
      if (this.getUser().isAdmin()) {
        if (data === false) {
          return false;
        }
        return true;
      }
      if (data === false) {
        return false;
      }
      if (data === true) {
        return true;
      }
      if (typeof data === 'string') {
        return true;
      }
      if (data === null) {
        return false;
      }
      action = action || null;
      if (action === null) {
        return true;
      }
      if (!(action in data)) {
        return false;
      }
      const value = data[action];
      if (value === 'all') {
        return true;
      }
      if (value === 'yes') {
        return true;
      }
      if (value === 'no') {
        return false;
      }
      if (isOwner === undefined) {
        return true;
      }
      if (isOwner) {
        if (value === 'own' || value === 'team') {
          return true;
        }
      }
      if (isShared) {
        return true;
      }
      if (inTeam) {
        if (value === 'team') {
          return true;
        }
      }
      let result = false;
      if (value === 'team') {
        if (inTeam === null && precise) {
          result = null;
        }
      }
      if (isOwner === null && precise) {
        result = null;
      }
      if (isShared === null) {
        result = null;
      }
      return result;
    }

    /**
     * Check access to model (entity).
     *
     * @param {module:model} model A model.
     * @param {Object.<string, string>|string|null} data Access data.
     * @param {module:acl-manager~action|null} [action=null] Action to check.
     * @param {boolean} [precise=false] To return `null` if not enough data is set in a model.
     *   E.g. the `teams` field is not yet loaded.
     * @returns {boolean|null} True if access allowed, null if not enough data to determine.
     */
    checkModel(model, data, action, precise) {
      if (this.getUser().isAdmin()) {
        return true;
      }
      let isShared = false;
      if (action === 'read' || action === 'stream') {
        isShared = this.checkIsShared(model);
      }
      const entityAccessData = {
        isOwner: this.checkIsOwner(model),
        inTeam: this.checkInTeam(model),
        isShared: isShared
      };
      return this.checkScope(data, action, precise, entityAccessData);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Check `delete` access to model.
     *
     * @param {module:model} model A model.
     * @param {Object.<string, string>|string|null} data Access data.
     * @param {boolean} [precise=false] To return `null` if not enough data is set in a model.
     *   E.g. the `teams` field is not yet loaded.
     * @returns {boolean} True if access allowed.
     */
    checkModelDelete(model, data, precise) {
      const result = this.checkModel(model, data, 'delete', precise);
      if (result) {
        return true;
      }
      if (data === false) {
        return false;
      }
      const d = data || {};
      if (d.read === 'no') {
        return false;
      }
      if (model.has('createdById') && model.get('createdById') === this.getUser().id && this.aclAllowDeleteCreated) {
        if (!model.has('assignedUserId')) {
          return true;
        }
        if (!model.get('assignedUserId')) {
          return true;
        }
        if (model.get('assignedUserId') === this.getUser().id) {
          return true;
        }
      }
      return result;
    }

    /**
     * Check if a user is owner to a model.
     *
     * @param {module:model} model A model.
     * @returns {boolean|null} True if owner. Null if not clear.
     */
    checkIsOwner(model) {
      let result = false;
      if (model.hasField('assignedUser')) {
        if (this.getUser().id === model.get('assignedUserId')) {
          return true;
        }
        if (!model.has('assignedUserId')) {
          result = null;
        }
      } else if (model.hasField('createdBy')) {
        if (this.getUser().id === model.get('createdById')) {
          return true;
        }
        if (!model.has('createdById')) {
          result = null;
        }
      }
      if (model.hasField('assignedUsers')) {
        if (!model.has('assignedUsersIds')) {
          return null;
        }
        if ((model.get('assignedUsersIds') || []).includes(this.getUser().id)) {
          return true;
        }
        result = false;
      }
      return result;
    }

    /**
     * Check if a user in a team of a model.
     *
     * @param {module:model} model A model.
     * @returns {boolean|null} True if in a team. Null if not enough data to determine.
     */
    checkInTeam(model) {
      const userTeamIdList = this.getUser().getTeamIdList();
      if (!model.has('teamsIds')) {
        if (this.teamsFieldIsForbidden) {
          return true;
        }
        if (!model.hasField('teams')) {
          return false;
        }
        return null;
      }
      const teamIdList = model.getTeamIdList();
      let inTeam = false;
      userTeamIdList.forEach(id => {
        if (teamIdList.includes(id)) {
          inTeam = true;
        }
      });
      return inTeam;
    }

    /**
     * Check if a record is shared with the user.
     *
     * @param {module:model} model A model.
     * @returns {boolean|null} True if shared. Null if not enough data to determine.
     */
    checkIsShared(model) {
      if (!model.has('collaboratorsIds')) {
        if (this.collaboratorsFieldIsForbidden) {
          return true;
        }
        if (!model.hasField('collaborators')) {
          return false;
        }
        return null;
      }
      const collaboratorsIds = model.getLinkMultipleIdList('collaborators');
      return collaboratorsIds.includes(this.user.id);
    }

    /**
     * Get a permission level.
     *
     * @protected
     * @param {string} permission A permission name.
     * @returns {'yes'|'all'|'team'|'no'}
     */
    getPermissionLevel(permission) {
      return this._aclManager.getPermissionLevel(permission);
    }
  }
  Acl.extend = _bullbone.View.extend;
  var _default = _exports.default = Acl;
});

define("view-helper", ["exports", "marked", "dompurify", "handlebars"], function (_exports, _marked, _dompurify, _handlebars) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _dompurify = _interopRequireDefault(_dompurify);
  _handlebars = _interopRequireDefault(_handlebars);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module view-helper */

  /**
   * A view helper.
   */
  class ViewHelper {
    constructor() {
      this._registerHandlebarsHelpers();

      /** @private */
      this.mdBeforeList = [
      /*{
          regex: /```\n?([\s\S]*?)```/g,
          value: (s, string) => {
              return '```\n' + string.replace(/\\\>/g, '>') + '```';
          },
      },*/
      {
        // Also covers triple-backtick blocks.
        regex: /`([\s\S]*?)`/g,
        value: (s, string) => {
          // noinspection RegExpRedundantEscape
          return '`' + string.replace(/\\\</g, '<') + '`';
        }
      }];
      _marked.marked.setOptions({
        breaks: true,
        tables: false,
        headerIds: false
      });
      _dompurify.default.addHook('beforeSanitizeAttributes', function (node) {
        if (node instanceof HTMLAnchorElement) {
          if (node.getAttribute('target')) {
            node.targetBlank = true;
          } else {
            node.targetBlank = false;
          }
        }
        if (node instanceof HTMLOListElement && node.start && node.start > 99) {
          node.removeAttribute('start');
        }
        if (node instanceof HTMLFormElement) {
          if (node.action) {
            node.removeAttribute('action');
          }
          if (node.hasAttribute('method')) {
            node.removeAttribute('method');
          }
        }
        if (node instanceof HTMLButtonElement) {
          if (node.type === 'submit') {
            node.type = 'button';
          }
        }
      });
      _dompurify.default.addHook('afterSanitizeAttributes', function (node) {
        if (node instanceof HTMLAnchorElement) {
          const href = node.getAttribute('href');
          if (href && !href.startsWith('#')) {
            node.setAttribute('rel', 'noopener noreferrer');
          }
          if (node.targetBlank) {
            node.setAttribute('target', '_blank');
            node.setAttribute('rel', 'noopener noreferrer');
          }
        }
      });
      _dompurify.default.addHook('uponSanitizeAttribute', (node, data) => {
        if (data.attrName === 'style') {
          const style = data.attrValue.split(';').map(s => s.trim()).filter(rule => {
            const [property, value] = rule.split(':').map(s => s.trim().toLowerCase());
            if (property === 'position' && ['absolute', 'fixed', 'sticky'].includes(value)) {
              return false;
            }
            return true;
          });
          data.attrValue = style.join('; ');
        }
      });
    }

    /**
     * A layout manager.
     *
     * @type {module:layout-manager}
     */
    layoutManager = null;

    /**
     * A config.
     *
     * @type {module:models/settings}
     */
    settings = null;

    /**
     * A config.
     *
     * @type {module:models/settings}
     */
    config = null;

    /**
     * A current user.
     *
     * @type {module:models/user}
     */
    user = null;

    /**
     * A preferences.
     *
     * @type {module:models/preferences}
     */
    preferences = null;

    /**
     * An ACL manager.
     *
     * @type {module:acl-manager}
     */
    acl = null;

    /**
     * A model factory.
     *
     * @type {module:model-factory}
     */
    modelFactory = null;

    /**
     * A collection factory.
     *
     * @type {module:collection-factory}
     */
    collectionFactory = null;

    /**
     * A router.
     *
     * @type {module:router}
     */
    router = null;

    /**
     * A storage.
     *
     * @type {module:storage}
     */
    storage = null;

    /**
     * A session storage.
     *
     * @type {module:session-storage}
     */
    sessionStorage = null;

    /**
     * A date-time util.
     *
     * @type {module:date-time}
     */
    dateTime = null;

    /**
     * A language.
     *
     * @type {module:language}
     */
    language = null;

    /**
     * A metadata.
     *
     * @type {module:metadata}
     */
    metadata = null;

    /**
     * A field-manager util.
     *
     * @type {module:field-manager}
     */
    fieldManager = null;

    /**
     * A cache.
     *
     * @type {module:cache}
     */
    cache = null;

    /**
     * A theme manager.
     *
     * @type {module:theme-manager}
     */
    themeManager = null;

    /**
     * A web-socket manager. Null if not enabled.
     *
     * @type {?module:web-socket-manager}
     */
    webSocketManager = null;

    /**
     * A number util.
     *
     * @type {module:num-util}
     */
    numberUtil = null;

    /**
     * A page-title util.
     *
     * @type {module:page-title}
     */
    pageTitle = null;

    /**
     * A broadcast channel.
     *
     * @type {?module:broadcast-channel}
     */
    broadcastChannel = null;

    /**
     * A base path.
     *
     * @type {string}
     */
    basePath = '';

    /**
     * Application parameters.
     *
     * @type {import('app-params').default|null}
     */
    appParams = null;

    /**
     * @private
     */
    _registerHandlebarsHelpers() {
      _handlebars.default.registerHelper('img', img => {
        return new _handlebars.default.SafeString(`<img src="img/${img}" alt="img">`);
      });
      _handlebars.default.registerHelper('prop', (object, name) => {
        if (object === undefined) {
          console.warn(`Undefined value passed to 'prop' helper.`);
          return undefined;
        }
        if (name in object) {
          return object[name];
        }
        return undefined;
      });
      _handlebars.default.registerHelper('var', (name, context, options) => {
        if (typeof context === 'undefined') {
          return null;
        }
        let contents = context[name];
        if (options.hash.trim) {
          contents = contents.trim();
        }
        return new _handlebars.default.SafeString(contents);
      });
      _handlebars.default.registerHelper('concat', function (left, right) {
        return left + right;
      });
      _handlebars.default.registerHelper('ifEqual', function (left, right, options) {
        // noinspection EqualityComparisonWithCoercionJS
        if (left == right) {
          return options.fn(this);
        }
        return options.inverse(this);
      });
      _handlebars.default.registerHelper('ifNotEqual', function (left, right, options) {
        // noinspection EqualityComparisonWithCoercionJS
        if (left != right) {
          return options.fn(this);
        }
        return options.inverse(this);
      });
      _handlebars.default.registerHelper('ifPropEquals', function (object, property, value, options) {
        // noinspection EqualityComparisonWithCoercionJS
        if (object[property] == value) {
          return options.fn(this);
        }
        return options.inverse(this);
      });
      _handlebars.default.registerHelper('ifAttrEquals', function (model, attr, value, options) {
        // noinspection EqualityComparisonWithCoercionJS
        if (model.get(attr) == value) {
          return options.fn(this);
        }
        return options.inverse(this);
      });
      _handlebars.default.registerHelper('ifAttrNotEmpty', function (model, attr, options) {
        const value = model.get(attr);
        if (value !== null && typeof value !== 'undefined') {
          return options.fn(this);
        }
        return options.inverse(this);
      });
      _handlebars.default.registerHelper('get', (model, name) => model.get(name));
      _handlebars.default.registerHelper('length', arr => arr.length);
      _handlebars.default.registerHelper('translate', (name, options) => {
        const scope = options.hash.scope || null;
        const category = options.hash.category || null;
        if (name === 'null') {
          return '';
        }
        return this.language.translate(name, category, scope);
      });
      _handlebars.default.registerHelper('dropdownItem', (name, options) => {
        const scope = options.hash.scope || null;
        const label = options.hash.label;
        const labelTranslation = options.hash.labelTranslation;
        const data = options.hash.data;
        const hidden = options.hash.hidden;
        const disabled = options.hash.disabled;
        const title = options.hash.title;
        const link = options.hash.link;
        const action = options.hash.action || name;
        const iconHtml = options.hash.iconHtml;
        const iconClass = options.hash.iconClass;
        let html = options.hash.html || options.hash.text || (labelTranslation ? this.language.translatePath(labelTranslation) : this.language.translate(label, 'labels', scope));
        if (!options.hash.html) {
          html = this.escapeString(html);
        }
        if (iconHtml) {
          html = iconHtml + ' ' + html;
        } else if (iconClass) {
          const iconHtml = $('<span>').addClass(iconClass).get(0).outerHTML;
          html = iconHtml + ' ' + html;
        }
        const $li = $('<li>').addClass(hidden ? 'hidden' : '').addClass(disabled ? 'disabled' : '');
        const $a = $('<a>').attr('role', 'button').attr('tabindex', '0').attr('data-name', name).addClass(options.hash.className || '').addClass('action').html(html);
        if (action) {
          $a.attr('data-action', action);
        }
        $li.append($a);
        link ? $a.attr('href', link) : $a.attr('role', 'button');
        if (data) {
          for (const key in data) {
            $a.attr('data-' + Espo.Utils.camelCaseToHyphen(key), data[key]);
          }
        }
        if (disabled) {
          $li.attr('disabled', 'disabled');
        }
        if (title) {
          $a.attr('title', title);
        }
        return new _handlebars.default.SafeString($li.get(0).outerHTML);
      });
      _handlebars.default.registerHelper('button', (name, options) => {
        const style = options.hash.style || 'default';
        const scope = options.hash.scope || null;
        const label = options.hash.label || name;
        const labelTranslation = options.hash.labelTranslation;
        const link = options.hash.link;
        const iconHtml = options.hash.iconHtml;
        const iconClass = options.hash.iconClass;
        let html = options.hash.html || options.hash.text || (labelTranslation ? this.language.translatePath(labelTranslation) : this.language.translate(label, 'labels', scope));
        if (!options.hash.html) {
          html = this.escapeString(html);
        }
        if (iconHtml) {
          html = iconHtml + ' ' + '<span>' + html + '</span>';
        } else if (iconClass) {
          const iconHtml = $('<span>').addClass(iconClass).get(0).outerHTML;
          html = iconHtml + ' ' + '<span>' + html + '</span>';
        }
        const tag = link ? '<a>' : '<button>';
        const $button = $(tag).addClass('btn action').addClass(options.hash.className || '').addClass(options.hash.hidden ? 'hidden' : '').addClass(options.hash.disabled ? 'disabled' : '').attr('data-action', name).attr('data-name', name).addClass('btn-' + style).html(html);
        link ? $button.href(link) : $button.attr('type', 'button');
        if (options.hash.disabled) {
          $button.attr('disabled', 'disabled');
        }
        if (options.hash.title) {
          $button.attr('title', options.hash.title);
        }
        return new _handlebars.default.SafeString($button.get(0).outerHTML);
      });
      _handlebars.default.registerHelper('hyphen', string => {
        return Espo.Utils.convert(string, 'c-h');
      });
      _handlebars.default.registerHelper('toDom', string => {
        return Espo.Utils.toDom(string);
      });

      // noinspection SpellCheckingInspection
      _handlebars.default.registerHelper('breaklines', text => {
        text = _handlebars.default.Utils.escapeExpression(text || '');
        text = text.replace(/(\r\n|\n|\r)/gm, '<br>');
        return new _handlebars.default.SafeString(text);
      });
      _handlebars.default.registerHelper('complexText', (text, options) => {
        if (typeof text !== 'string' && !(text instanceof String)) {
          return '';
        }
        return this.transformMarkdownText(text, options.hash);
      });
      _handlebars.default.registerHelper('translateOption', (name, options) => {
        const scope = options.hash.scope || null;
        const field = options.hash.field || null;
        if (!field) {
          return '';
        }
        let translationHash = options.hash.translatedOptions || null;
        if (translationHash === null) {
          translationHash = this.language.translate(/** @type {string} */field, 'options', scope) || {};
          if (typeof translationHash !== 'object') {
            translationHash = {};
          }
        }
        if (name === null) {
          name = '';
        }
        return translationHash[name] || name;
      });
      _handlebars.default.registerHelper('options', (/** any[] */list, value, options) => {
        if (typeof value === 'undefined') {
          value = false;
        }
        list = list || [];
        let html = '';
        const multiple = Object.prototype.toString.call(value) === '[object Array]';
        const checkOption = name => {
          if (multiple) {
            return value.indexOf(name) !== -1;
          }
          return value === name || !value && !name && name !== 0;
        };
        options.hash = /** @type {Record} */options.hash || {};
        const scope = options.hash.scope || false;
        const category = options.hash.category || false;
        const field = options.hash.field || false;
        const styleMap = options.hash.styleMap || {};
        if (!multiple && options.hash.includeMissingOption && (value || value === '')) {
          if (!list.includes(value)) {
            list = Espo.Utils.clone(list);
            list.push(value);
          }
        }
        let translationHash = options.hash.translationHash || options.hash.translatedOptions || null;
        if (translationHash === null) {
          translationHash = {};
          if (!category && field) {
            translationHash = this.language.translate(/** @type {string} */field, 'options', /** @type {string} */scope) || {};
            if (typeof translationHash !== 'object') {
              translationHash = {};
            }
          }
        }
        const translate = name => {
          if (!category) {
            return translationHash[name] || name;
          }
          return this.language.translate(name, category, /** @type {string} */scope);
        };
        for (const key in list) {
          const value = list[key];
          const label = translate(value);
          const $option = $('<option>').attr('value', value).addClass(styleMap[value] ? 'text-' + styleMap[value] : '').text(label);
          if (checkOption(list[key])) {
            $option.attr('selected', 'selected');
          }
          html += $option.get(0).outerHTML;
        }
        return new _handlebars.default.SafeString(html);
      });
      _handlebars.default.registerHelper('basePath', () => {
        return this.basePath || '';
      });
    }

    /**
     * Get an application parameter.
     *
     * @param {string} name
     * @returns {*}
     */
    getAppParam(name) {
      if (!this.appParams) {
        return undefined;
      }
      return this.appParams.get(name);
    }

    /**
     * Escape a string.
     *
     * @param {string} text A string.
     * @returns {string}
     */
    escapeString(text) {
      return _handlebars.default.Utils.escapeExpression(text);
    }

    /**
     * Get a user avatar HTML.
     *
     * @param {string} id A user ID.
     * @param {'small'|'medium'|'large'} [size='small'] A size.
     * @param {int} [width=16]
     * @param {string} [additionalClassName]  An additional class-name.
     * @returns {string}
     */
    getAvatarHtml(id, size, width, additionalClassName) {
      if (this.config.get('avatarsDisabled')) {
        return '';
      }
      const t = this.cache ? this.cache.get('app', 'timestamp') : this.settings.get('cacheTimestamp');
      const basePath = this.basePath || '';
      size = size || 'small';
      width = width || 16;
      let className = 'avatar';
      if (additionalClassName) {
        className += ' ' + additionalClassName;
      }

      // noinspection RequiredAttributes,HtmlRequiredAltAttribute
      return $(`<img>`).attr('src', `${basePath}?entryPoint=avatar&size=${size}&id=${id}&t=${t}`).attr('alt', 'avatar').addClass(className).attr('data-width', width.toString()).css('width', `var(--${width.toString()}px)`).attr('draggable', 'false').get(0).outerHTML;
    }

    /**
     * A Markdown text to HTML (one-line).
     *
     * @param {string} text A text.
     * @returns {Handlebars.SafeString} HTML.
     */
    transformMarkdownInlineText(text) {
      return this.transformMarkdownText(text, {
        inline: true
      });
    }

    /**
     * A Markdown text to HTML.
     *
     * @param {string} text A text.
     * @param {{inline?: boolean, linksInNewTab?: boolean}} [options] Options.
     * @returns {Handlebars.SafeString} HTML.
     */
    transformMarkdownText(text, options) {
      text = text || '';

      // noinspection RegExpRedundantEscape
      text = text.replace(/\</g, '\\<');
      this.mdBeforeList.forEach(item => {
        text = text.replace(item.regex, item.value);
      });
      options = options || {};
      text = options.inline ? _marked.marked.parseInline(text) : _marked.marked.parse(text);
      text = _dompurify.default.sanitize(text, {}).toString();
      if (options.linksInNewTab) {
        text = text.replace(/<a href=/gm, '<a target="_blank" rel="noopener noreferrer" href=');
      }
      text = text.replace(/<a href="mailto:([^"]*)"/gm, '<a role="button" class="selectable" data-email-address="$1" data-action="mailTo"');
      return new _handlebars.default.SafeString(text);
    }

    /**
     * Get a color-icon HTML for a scope.
     *
     * @param {string} scope A scope.
     * @param {boolean} [noWhiteSpace=false] No white space.
     * @param {string} [additionalClassName] An additional class-name.
     * @returns {string}
     */
    getScopeColorIconHtml(scope, noWhiteSpace, additionalClassName) {
      if (this.config.get('scopeColorsDisabled') || this.preferences.get('scopeColorsDisabled')) {
        return '';
      }
      const color = this.metadata.get(['clientDefs', scope, 'color']);
      let html = '';
      if (color) {
        const $span = $('<span class="color-icon fas fa-square">');
        $span.css('color', color);
        if (additionalClassName) {
          $span.addClass(additionalClassName);
        }
        html = $span.get(0).outerHTML;
      }
      if (!noWhiteSpace) {
        if (html) {
          html += `<span style="user-select: none;">&nbsp;</span>`;
        }
      }
      return html;
    }

    /**
     * Sanitize HTML.
     *
     * @param {string} text HTML.
     * @param {Object} [options] Options.
     * @returns {string}
     */
    sanitizeHtml(text, options) {
      return _dompurify.default.sanitize(text, options);
    }

    /**
     * Moderately sanitize HTML.
     *
     * @param {string} value HTML.
     * @returns {string}
     */
    moderateSanitizeHtml(value) {
      value = value || '';
      value = value.replace(/<\/?(base)[^><]*>/gi, '');
      value = value.replace(/<\/?(object)[^><]*>/gi, '');
      value = value.replace(/<\/?(embed)[^><]*>/gi, '');
      value = value.replace(/<\/?(applet)[^><]*>/gi, '');
      value = value.replace(/<\/?(iframe)[^><]*>/gi, '');
      value = value.replace(/<\/?(script)[^><]*>/gi, '');
      value = value.replace(/<[^><]*([^a-z]on[a-z]+)=[^><]*>/gi, function (match) {
        return match.replace(/[^a-z]on[a-z]+=/gi, ' data-handler-stripped=');
      });
      value = this.stripEventHandlersInHtml(value);
      value = value.replace(/href=" *javascript:(.*?)"/gi, () => {
        return 'removed=""';
      });
      value = value.replace(/href=' *javascript:(.*?)'/gi, () => {
        return 'removed=""';
      });
      value = value.replace(/src=" *javascript:(.*?)"/gi, () => {
        return 'removed=""';
      });
      value = value.replace(/src=' *javascript:(.*?)'/gi, () => {
        return 'removed=""';
      });
      return value;
    }

    /**
     * Strip event handlers in HTML.
     *
     * @param {string} html HTML.
     * @returns {string}
     */
    stripEventHandlersInHtml(html) {
      let j; // @todo Revise.

      function stripHTML() {
        html = html.slice(0, strip) + html.slice(j);
        j = strip;
        strip = false;
      }
      function isValidTagChar(str) {
        return str.match(/[a-z?\\\/!]/i);
      }
      let strip = false;
      let lastQuote = false;
      for (let i = 0; i < html.length; i++) {
        if (html[i] === '<' && html[i + 1] && isValidTagChar(html[i + 1])) {
          i++;
          for (let j = i; j < html.length; j++) {
            if (!lastQuote && html[j] === '>') {
              if (strip) {
                stripHTML();
              }
              i = j;
              break;
            }

            // noinspection JSIncompatibleTypesComparison
            if (lastQuote === html[j]) {
              lastQuote = false;
              continue;
            }
            if (!lastQuote && html[j - 1] === "=" && (html[j] === "'" || html[j] === '"')) {
              lastQuote = html[j];
            }
            if (!lastQuote && html[j - 2] === " " && html[j - 1] === "o" && html[j] === "n") {
              strip = j - 2;
            }
            if (strip && html[j] === " " && !lastQuote) {
              stripHTML();
            }
          }
        }
      }
      return html;
    }

    /**
     * Calculate a content container height.
     *
     * @param {HTMLElement|JQuery} element An element.
     * @returns {number}
     */
    calculateContentContainerHeight(element) {
      const smallScreenWidth = this.themeManager.getParam('screenWidthXs');
      const $window = $(window);
      const footerHeight = $('#footer').height() || 26;
      let top = 0;
      element = $(element).get(0);
      if (element) {
        top = element.getBoundingClientRect().top;
        if ($window.width() < smallScreenWidth) {
          const $navbarCollapse = $('#navbar .navbar-body');
          if ($navbarCollapse.hasClass('in') || $navbarCollapse.hasClass('collapsing')) {
            top -= $navbarCollapse.height();
          }
        }
      }
      const spaceHeight = top + footerHeight;
      return $window.height() - spaceHeight - 20;
    }

    /**
     * Process view-setup-handlers.
     *
     * @param {module:view} view A view.
     * @param {string} type A view-setup-handler type.
     * @param {string} [scope] A scope.
     * @return Promise
     */
    processSetupHandlers(view, type, scope) {
      // noinspection JSUnresolvedReference
      scope = scope || view.scope || view.entityType;
      let handlerIdList = this.metadata.get(['clientDefs', 'Global', 'viewSetupHandlers', type]) || [];
      if (scope) {
        handlerIdList = handlerIdList.concat(this.metadata.get(['clientDefs', scope, 'viewSetupHandlers', type]) || []);
      }
      if (handlerIdList.length === 0) {
        return Promise.resolve();
      }

      /**
       * @interface
       * @name ViewHelper~Handler
       */

      /**
       * @function
       * @name ViewHelper~Handler#process
       * @param {module:view} [view] Deprecated.
       */
      const promiseList = [];
      for (const id of handlerIdList) {
        const promise = new Promise(resolve => {
          Espo.loader.require(id, /** typeof ViewHelper~Handler */Handler => {
            const result = new Handler(view).process(view);
            if (result && Object.prototype.toString.call(result) === '[object Promise]') {
              result.then(() => resolve());
              return;
            }
            resolve();
          });
        });
        promiseList.push(promise);
      }
      return Promise.all(promiseList);
    }

    /** @private */
    _isXsScreen;

    /**
     * Is xs screen width.
     *
     * @return {boolean}
     */
    isXsScreen() {
      if (this._isXsScreen == null) {
        this._isXsScreen = window.innerWidth < this.themeManager.getParam('screenWidthXs');
      }
      return this._isXsScreen;
    }
  }
  var _default = _exports.default = ViewHelper;
});

define("metadata", ["exports", "bullbone"], function (_exports, _bullbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module metadata */

  /**
   * Application metadata.
   *
   * @mixes Bull.Events
   */
  class Metadata {
    /**
     * Application metadata.
     *
     * @param {module:cache} [cache] A cache.
     */
    constructor(cache) {
      /**
       * @private
       * @type {module:cache|null}
       */
      this.cache = cache || null;

      /**
       * @private
       * @type {Object}
       */
      this.data = {};
    }

    /** @private */
    url = 'Metadata';

    /**
     * Load from cache or the backend (if not yet cached).
     *
     * @param {Function|null} [callback] Deprecated. Use a promise.
     * @param {boolean} [disableCache=false] Deprecated.
     * @returns {Promise}
     */
    load(callback, disableCache) {
      if (!disableCache) {
        if (this.loadFromCache()) {
          this.trigger('sync');
          if (callback) {
            callback();
          }
          return Promise.resolve();
        }
      }
      return this.fetch().then(() => {
        if (callback) {
          callback();
        }
      });
    }

    /**
     * Load from the server.
     *
     * @returns {Promise}
     */
    loadSkipCache() {
      return this.load(null, true);
    }

    /**
     * @private
     * @returns {Promise}
     */
    fetch() {
      return Espo.Ajax.getRequest(this.url).then(data => {
        this.data = data;
        this.storeToCache();
        this.trigger('sync');
      });
    }

    /**
     * Get a value.
     *
     * @param {string[]|string} path A key path.
     * @param {*} [defaultValue] A value to return if not set.
     * @returns {*} Null if not set.
     */
    get(path, defaultValue) {
      defaultValue = defaultValue || null;
      let arr;
      if (Array && Array.isArray && Array.isArray(path)) {
        arr = path;
      } else {
        arr = path.split('.');
      }
      let pointer = this.data;
      let result = defaultValue;
      for (let i = 0; i < arr.length; i++) {
        const key = arr[i];
        if (pointer == null || !(key in pointer)) {
          result = defaultValue;
          break;
        }
        if (arr.length - 1 === i) {
          result = pointer[key];
        }
        pointer = pointer[key];
      }
      return result;
    }

    /**
     * @private
     * @returns {boolean|null} True if success.
     */
    loadFromCache() {
      if (this.cache) {
        const cached = this.cache.get('app', 'metadata');
        if (cached) {
          this.data = cached;
          return true;
        }
      }
      return null;
    }

    /** @private */
    storeToCache() {
      if (this.cache) {
        this.cache.set('app', 'metadata', this.data);
      }
    }

    /**
     * Clear cache.
     */
    clearCache() {
      if (!this.cache) {
        return;
      }
      this.cache.clear('app', 'metadata');
    }

    /**
     * Get a scope list.
     *
     * @returns {string[]}
     */
    getScopeList() {
      const scopes = this.get('scopes') || {};
      const scopeList = [];
      for (const scope in scopes) {
        const d = scopes[scope];
        if (d.disabled) {
          continue;
        }
        scopeList.push(scope);
      }
      return scopeList;
    }

    /**
     * Get an object-scope list. An object-scope represents a business entity.
     *
     * @returns {string[]}
     */
    getScopeObjectList() {
      const scopes = this.get('scopes') || {};
      const scopeList = [];
      for (const scope in scopes) {
        const d = scopes[scope];
        if (d.disabled) {
          continue;
        }
        if (!d.object) {
          continue;
        }
        scopeList.push(scope);
      }
      return scopeList;
    }

    /**
     * Get an entity-scope list. Scopes that represents entities.
     *
     * @returns {string[]}
     */
    getScopeEntityList() {
      const scopes = this.get('scopes') || {};
      const scopeList = [];
      for (const scope in scopes) {
        const d = scopes[scope];
        if (d.disabled) {
          continue;
        }
        if (!d.entity) {
          continue;
        }
        scopeList.push(scope);
      }
      return scopeList;
    }

    /**
     * Do not use.
     *
     * @internal
     * @param {Record} data
     */
    setData(data) {
      this.data = data;
    }
  }
  Object.assign(Metadata.prototype, _bullbone.Events);
  var _default = _exports.default = Metadata;
});

define("di", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.container = void 0;
  _exports.inject = inject;
  _exports.register = register;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  const registry = new Map();
  const container = _exports.container = new Map();

  /**
   * A DI container.
   */

  /**
   * A 'register' decorator.
   *
   * @param {*[]} argumentList Arguments.
   * @return {function(typeof Object)}
   */
  function register(argumentList = []) {
    return function (classObject) {
      registry.set(classObject, argumentList);
    };
  }

  /**
   * An 'inject' decorator.
   *
   * @param {typeof Object} classObject A class.
   * @return {(function(*, Object): void)}
   */
  function inject(classObject) {
    /**
     * @param {{addInitializer: function(function())}} context
     */
    return function (value, context) {
      context.addInitializer(function () {
        let instance = container.get(classObject);
        if (!instance) {
          instance = Reflect.construct(classObject, registry.get(classObject));
          container.set(classObject, instance);
        }
        this[context.name] = instance;
      });
    };
  }
});

define("acl-manager", ["exports", "acl", "utils", "bullbone"], function (_exports, _acl, _utils, _bullbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _acl = _interopRequireDefault(_acl);
  _utils = _interopRequireDefault(_utils);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module acl-manager */

  /**
   * An action.
   *
   * @typedef {'create'|'read'|'edit'|'delete'|'stream'} module:acl-manager~action
   */

  /**
   * An access checking class for a specific scope.
   */
  class AclManager {
    /** @protected */
    data = null;
    fieldLevelList = ['yes', 'no'];

    /**
     * @param {module:models/user} user A user.
     * @param {Object} implementationClassMap `acl` implementations.
     * @param {boolean} aclAllowDeleteCreated Allow a user to delete records they created regardless a
     *   role access level.
     */
    constructor(user, implementationClassMap, aclAllowDeleteCreated) {
      this.setEmpty();

      /** @protected */
      this.user = user || null;
      this.implementationClassMap = implementationClassMap || {};
      this.aclAllowDeleteCreated = aclAllowDeleteCreated;
    }

    /**
     * @protected
     */
    setEmpty() {
      this.data = {
        table: {},
        fieldTable: {},
        fieldTableQuickAccess: {}
      };
      this.implementationHash = {};
      this.forbiddenFieldsCache = {};
      this.implementationClassMap = {};
      this.forbiddenAttributesCache = {};
    }

    /**
     * Get an `acl` implementation.
     *
     * @protected
     * @param {string} scope A scope.
     * @returns {module:acl}
     */
    getImplementation(scope) {
      if (!(scope in this.implementationHash)) {
        let implementationClass = _acl.default;
        if (scope in this.implementationClassMap) {
          implementationClass = this.implementationClassMap[scope];
        }
        const forbiddenFieldList = this.getScopeForbiddenFieldList(scope);
        const params = {
          aclAllowDeleteCreated: this.aclAllowDeleteCreated,
          teamsFieldIsForbidden: forbiddenFieldList.includes('teams'),
          forbiddenFieldList: forbiddenFieldList
        };
        this.implementationHash[scope] = new implementationClass(this.getUser(), scope, params, this);
      }
      return this.implementationHash[scope];
    }

    /**
     * @return {import('models/user').default}
     * @protected
     */
    getUser() {
      return this.user;
    }

    /**
     * @internal
     */
    set(data) {
      data = data || {};
      this.data = data;
      this.data.table = this.data.table || {};
      this.data.fieldTable = this.data.fieldTable || {};
      this.data.attributeTable = this.data.attributeTable || {};
    }

    /**
     * @deprecated Use `getPermissionLevel`.
     *
     * @returns {string|null}
     */
    get(name) {
      return this.data[name] || null;
    }

    /**
     * Get a permission level.
     *
     * @param {string} permission A permission name.
     * @returns {'yes'|'all'|'team'|'no'}
     */
    getPermissionLevel(permission) {
      let permissionKey = permission;
      if (permission.slice(-10) !== 'Permission') {
        permissionKey = permission + 'Permission';
      }
      return this.data[permissionKey] || 'no';
    }

    /**
     * Get access level to a scope action.
     *
     * @param {string} scope A scope.
     * @param {module:acl-manager~action} action An action.
     * @returns {'yes'|'all'|'team'|'own'|'no'|null}
     */
    getLevel(scope, action) {
      if (!(scope in this.data.table)) {
        return null;
      }
      const scopeItem = this.data.table[scope];
      if (typeof scopeItem !== 'object' || !(action in scopeItem)) {
        return null;
      }
      return scopeItem[action];
    }

    /**
     * Clear access data.
     *
     * @internal
     */
    clear() {
      this.setEmpty();
    }

    /**
     * Check whether a scope has ACL.
     *
     * @param {string} scope A scope.
     * @returns {boolean}
     */
    checkScopeHasAcl(scope) {
      const data = (this.data.table || {})[scope];
      if (typeof data === 'undefined') {
        return false;
      }
      return true;
    }

    /**
     * Check access to a scope.
     *
     * @param {string} scope A scope.
     * @param {module:acl-manager~action|null} [action=null] An action.
     * @param {boolean} [precise=false] Deprecated. Not used.
     * @returns {boolean} True if access allowed.
     */
    checkScope(scope, action, precise) {
      let data = (this.data.table || {})[scope];
      if (typeof data === 'undefined') {
        data = null;
      }
      return this.getImplementation(scope).checkScope(data, action, precise);
    }

    /**
     * Check access to a model.
     *
     * @param {module:model} model A model.
     * @param {module:acl-manager~action|null} [action=null] An action.
     * @param {boolean} [precise=false] To return `null` if not enough data is set in a model.
     *   E.g. the `teams` field is not yet loaded.
     * @returns {boolean|null} True if access allowed, null if not enough data to determine.
     */
    checkModel(model, action, precise) {
      const scope = model.entityType;

      // todo move this to custom acl
      if (action === 'edit') {
        if (!model.isEditable()) {
          return false;
        }
      }
      if (action === 'delete') {
        if (!model.isRemovable()) {
          return false;
        }
      }
      let data = (this.data.table || {})[scope];
      if (typeof data === 'undefined') {
        data = null;
      }
      const impl = this.getImplementation(scope);
      if (action) {
        const methodName = 'checkModel' + _utils.default.upperCaseFirst(action);
        if (methodName in impl) {
          return impl[methodName](model, data, precise);
        }
      }
      return impl.checkModel(model, data, action, precise);
    }

    /**
     * Check access to a scope or a model.
     *
     * @param {string|module:model} subject What to check. A scope or a model.
     * @param {module:acl-manager~action|null} [action=null] An action.
     * @param {boolean} [precise=false]  To return `null` if not enough data is set in a model.
     *   E.g. the `teams` field is not yet loaded.
     * @returns {boolean|null} True if access allowed, null if not enough data to determine.
     */
    check(subject, action, precise) {
      if (typeof subject === 'string') {
        return this.checkScope(subject, action, precise);
      }
      return this.checkModel(subject, action, precise);
    }

    /**
     * Check if a user is owner to a model.
     *
     * @param {module:model} model A model.
     * @returns {boolean|null} True if owner, null if not clear.
     */
    checkIsOwner(model) {
      return this.getImplementation(model.entityType).checkIsOwner(model);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Check if a user in a team of a model.
     *
     * @param {module:model} model A model.
     * @returns {boolean|null} True if in a team, null if not clear.
     */
    checkInTeam(model) {
      return this.getImplementation(model.entityType).checkInTeam(model);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Check if a record is shared with the user.
     *
     * @param {module:model} model A model.
     * @returns {boolean|null} True if shared, null if not clear.
     */
    checkIsShared(model) {
      return this.getImplementation(model.entityType).checkIsShared(model);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Check an assignment permission to a user.
     *
     * @param {module:models/user} user A user.
     * @returns {boolean} True if access allowed.
     */
    checkAssignmentPermission(user) {
      return this.checkPermission('assignmentPermission', user);
    }

    /**
     * Check a user permission to a user.
     *
     * @param {module:models/user} user A user.
     * @returns {boolean} True if access allowed.
     */
    checkUserPermission(user) {
      return this.checkPermission('userPermission', user);
    }

    /**
     * Check a specific permission to a user.
     *
     * @param {string} permission A permission name.
     * @param {module:models/user} user A user.
     * @returns {boolean|null} True if access allowed. Null if not enough data loaded to know for sure.
     */
    checkPermission(permission, user) {
      if (this.getUser().isAdmin()) {
        return true;
      }
      const level = this.getPermissionLevel(permission);
      if (level === 'no') {
        if (user.id === this.getUser().id) {
          return true;
        }
        return false;
      }
      if (level === 'team') {
        if (!user.has('teamsIds')) {
          return null;
        }
        let result = false;
        const teamsIds = user.get('teamsIds') || [];
        teamsIds.forEach(id => {
          if ((this.getUser().get('teamsIds') || []).includes(id)) {
            result = true;
          }
        });
        return result;
      }
      if (level === 'all') {
        return true;
      }
      if (level === 'yes') {
        return true;
      }
      return false;
    }

    /**
     * Get a list of forbidden fields for an entity type.
     *
     * @param {string} scope An entity type.
     * @param {'read'|'edit'} [action='read'] An action.
     * @param {'yes'|'no'} [thresholdLevel='no'] A threshold level.
     * @returns {string[]} A forbidden field list.
     */
    getScopeForbiddenFieldList(scope, action, thresholdLevel) {
      action = action || 'read';
      thresholdLevel = thresholdLevel || 'no';
      const key = scope + '_' + action + '_' + thresholdLevel;
      if (key in this.forbiddenFieldsCache) {
        return _utils.default.clone(this.forbiddenFieldsCache[key]);
      }
      const levelList = this.fieldLevelList.slice(this.fieldLevelList.indexOf(thresholdLevel));
      const fieldTableQuickAccess = this.data.fieldTableQuickAccess || {};
      const scopeData = fieldTableQuickAccess[scope] || {};
      const fieldsData = scopeData.fields || {};
      const actionData = fieldsData[action] || {};
      const fieldList = [];
      levelList.forEach(level => {
        const list = actionData[level] || [];
        list.forEach(field => {
          if (fieldList.includes(field)) {
            return;
          }
          fieldList.push(field);
        });
      });
      this.forbiddenFieldsCache[key] = fieldList;
      return _utils.default.clone(fieldList);
    }

    /**
     * Get a list of forbidden attributes for an entity type.
     *
     * @param {string} scope An entity type.
     * @param {'read'|'edit'} [action='read'] An action.
     * @param {'yes'|'no'} [thresholdLevel='no'] A threshold level.
     * @returns {string[]} A forbidden attribute list.
     */
    getScopeForbiddenAttributeList(scope, action, thresholdLevel) {
      action = action || 'read';
      thresholdLevel = thresholdLevel || 'no';
      const key = scope + '_' + action + '_' + thresholdLevel;
      if (key in this.forbiddenAttributesCache) {
        return _utils.default.clone(this.forbiddenAttributesCache[key]);
      }
      const levelList = this.fieldLevelList.slice(this.fieldLevelList.indexOf(thresholdLevel));
      const fieldTableQuickAccess = this.data.fieldTableQuickAccess || {};
      const scopeData = fieldTableQuickAccess[scope] || {};
      const attributesData = scopeData.attributes || {};
      const actionData = attributesData[action] || {};
      const attributeList = [];
      levelList.forEach(level => {
        const list = actionData[level] || [];
        list.forEach(attribute => {
          if (attributeList.includes(attribute)) {
            return;
          }
          attributeList.push(attribute);
        });
      });
      this.forbiddenAttributesCache[key] = attributeList;
      return _utils.default.clone(attributeList);
    }

    /**
     * Check an assignment permission to a team.
     *
     * @param {string} teamId A team ID.
     * @returns {boolean} True if access allowed.
     */
    checkTeamAssignmentPermission(teamId) {
      if (this.getPermissionLevel('assignmentPermission') === 'all') {
        return true;
      }
      return this.getUser().getLinkMultipleIdList('teams').includes(teamId);
    }

    /**
     * Check access to a field.
     * @param {string} scope An entity type.
     * @param {string} field A field.
     * @param {'read'|'edit'} [action='read'] An action.
     * @returns {boolean} True if access allowed.
     */
    checkField(scope, field, action) {
      return !this.getScopeForbiddenFieldList(scope, action).includes(field);
    }
  }
  AclManager.extend = _bullbone.View.extend;
  var _default = _exports.default = AclManager;
});

define("models/user", ["exports", "model"], function (_exports, _model) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _model = _interopRequireDefault(_model);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module models/user */

  /**
   * A user.
   */
  class User extends _model.default {
    name = 'User';
    entityType = 'User';
    urlRoot = 'User';

    /**
     * Is admin.
     *
     * @returns {boolean}
     */
    isAdmin() {
      return this.get('type') === 'admin' || this.isSuperAdmin();
    }

    /**
     * Is portal.
     *
     * @returns {boolean}
     */
    isPortal() {
      return this.get('type') === 'portal';
    }

    /**
     * Is API.
     *
     * @returns {boolean}
     */
    isApi() {
      return this.get('type') === 'api';
    }

    /**
     * Is regular.
     *
     * @returns {boolean}
     */
    isRegular() {
      return this.get('type') === 'regular';
    }

    /**
     * Is system.
     *
     * @returns {boolean}
     */
    isSystem() {
      return this.get('type') === 'system';
    }

    /**
     * Is super-admin.
     *
     * @returns {boolean}
     */
    isSuperAdmin() {
      return this.get('type') === 'super-admin';
    }
  }
  var _default = _exports.default = User;
});

define("models/settings", ["exports", "model"], function (_exports, _model) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _model = _interopRequireDefault(_model);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module models/settings */

  /**
   * A config.
   */
  class Settings extends _model.default {
    name = 'Settings';
    entityType = 'Settings';
    urlRoot = 'Settings';

    /**
     * Load.
     *
     * @returns {Promise}
     */
    load() {
      return new Promise(resolve => {
        this.fetch().then(() => resolve());
      });
    }

    /**
     * Get a value by a path.
     *
     * @param {string[]} path A path.
     * @returns {*} Null if not set.
     */
    getByPath(path) {
      if (!path.length) {
        return null;
      }
      let p;
      for (let i = 0; i < path.length; i++) {
        const item = path[i];
        if (i === 0) {
          p = this.get(item);
        } else {
          if (item in p) {
            p = p[item];
          } else {
            return null;
          }
        }
        if (i === path.length - 1) {
          return p;
        }
        if (p === null || typeof p !== 'object') {
          return null;
        }
      }
    }
  }
  var _default = _exports.default = Settings;
});

define("models/preferences", ["exports", "model"], function (_exports, _model) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _model = _interopRequireDefault(_model);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module models/preferences */

  /**
   * User preferences.
   */
  class Preferences extends _model.default {
    name = 'Preferences';
    entityType = 'Preferences';
    urlRoot = 'Preferences';

    /**
     * Get dashlet options.
     *
     * @param {string} id A dashlet ID.
     * @returns {Object|null}
     */
    getDashletOptions(id) {
      const value = this.get('dashletsOptions') || {};
      return value[id] || null;
    }

    /**
     * Whether a user is portal.
     *
     * @returns {boolean}
     */
    isPortal() {
      return this.get('isPortalUser');
    }
  }
  var _default = _exports.default = Preferences;
});

define("view", ["exports", "bullbone"], function (_exports, _bullbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module view */

  /**
   * A base view. All views should extend this class.
   *
   * @see https://docs.espocrm.com/development/view/
   * @mixes Bull.Events
   */
  class View extends _bullbone.View {
    /**
     * @callback module:view~actionHandlerCallback
     * @param {MouseEvent} event A DOM event.
     * @param {HTMLElement} element A target element.
     */

    /**
     * A helper.
     *
     * @name _helper
     * @type {module:view-helper}
     * @memberOf View.prototype
     * @private
     */

    /**
     * A model.
     *
     * @type {import('model').default}
     */
    model;

    /**
     * A collection.
     *
     * @type {import('collection').default}
     */
    collection;

    /**
     * @param {Record<string, *>} options
     */
    constructor(options = {}) {
      super(options);
      if (options.model) {
        this.model = options.model;
      }
      if (options.collection) {
        this.collection = options.collection;
      }
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * When the view is ready. Can be useful to prevent race condition when re-initialization is needed
     * in-between initialization and render.
     *
     * @return Promise
     * @todo Move to Bull.View.
     */
    whenReady() {
      if (this.isReady) {
        return Promise.resolve();
      }
      return new Promise(resolve => {
        this.once('ready', () => resolve());
      });
    }

    /**
     * Add a DOM click event handler for a target defined by `data-action="{name}"` attribute.
     *
     * @param {string} action An action name.
     * @param {module:view~actionHandlerCallback} handler A handler.
     */
    addActionHandler(action, handler) {
      const fullAction = `click [data-action="${action}"]`;
      this.events[fullAction] = e => {
        // noinspection JSUnresolvedReference
        handler.call(this, e.originalEvent, e.currentTarget);
      };
    }

    /**
     * Escape a string.
     *
     * @param {string} string
     * @returns {string}
     */
    escapeString(string) {
      return Handlebars.Utils.escapeExpression(string);
    }

    /**
     * Show a notify-message.
     *
     * @deprecated Use `Espo.Ui.notify`.
     * @param {string|false} label
     * @param {string} [type]
     * @param {number} [timeout]
     * @param {string} [scope]
     */
    notify(label, type, timeout, scope) {
      if (!label) {
        Espo.Ui.notify(false);
        return;
      }
      scope = scope || null;
      timeout = timeout || 2000;
      if (!type) {
        timeout = void 0;
      }
      const text = this.getLanguage().translate(label, 'labels', scope);
      Espo.Ui.notify(text, type, timeout);
    }

    /**
     * Get a view-helper.
     *
     * @returns {module:view-helper}
     */
    getHelper() {
      return this._helper;
    }

    /**
     * Get a current user.
     *
     * @returns {module:models/user}
     */
    getUser() {
      return this._helper.user;
    }

    /**
     * Get the preferences.
     *
     * @returns {module:models/preferences}
     */
    getPreferences() {
      return this._helper.preferences;
    }

    /**
     * Get the config.
     *
     * @returns {module:models/settings}
     */
    getConfig() {
      return this._helper.settings;
    }

    /**
     * Get the ACL.
     *
     * @returns {module:acl-manager}
     */
    getAcl() {
      return this._helper.acl;
    }

    /**
     * Get the model factory.
     *
     * @returns {module:model-factory}
     */
    getModelFactory() {
      return this._helper.modelFactory;
    }

    /**
     * Get the collection factory.
     *
     * @returns {module:collection-factory}
     */
    getCollectionFactory() {
      return this._helper.collectionFactory;
    }

    /**
     * Get the router.
     *
     * @returns {module:router}
     */
    getRouter() {
      return this._helper.router;
    }

    /**
     * Get the storage-util.
     *
     * @returns {module:storage}
     */
    getStorage() {
      return this._helper.storage;
    }

    /**
     * Get the session-storage-util.
     *
     * @returns {module:session-storage}
     */
    getSessionStorage() {
      return this._helper.sessionStorage;
    }

    /**
     * Get the language-util.
     *
     * @returns {module:language}
     */
    getLanguage() {
      return this._helper.language;
    }

    /**
     * Get metadata.
     *
     * @returns {module:metadata}
     */
    getMetadata() {
      return this._helper.metadata;
    }

    /**
     * Get the cache-util.
     *
     * @returns {module:cache}
     */
    getCache() {
      return this._helper.cache;
    }

    /**
     * Get the date-time util.
     *
     * @returns {module:date-time}
     */
    getDateTime() {
      return this._helper.dateTime;
    }

    /**
     * Get the number-util.
     *
     * @returns {module:num-util}
     */
    getNumberUtil() {
      return this._helper.numberUtil;
    }

    /**
     * Get the field manager.
     *
     * @returns {module:field-manager}
     */
    getFieldManager() {
      return this._helper.fieldManager;
    }

    /**
     * Get the base-controller.
     *
     * @returns {module:controllers/base}
     */
    getBaseController() {
      return this._helper.baseController;
    }

    /**
     * Get the theme manager.
     *
     * @returns {module:theme-manager}
     */
    getThemeManager() {
      return this._helper.themeManager;
    }

    /**
     * Update a page title. Supposed to be overridden if needed.
     */
    updatePageTitle() {
      const title = this.getConfig().get('applicationName') || 'EspoCRM';
      this.setPageTitle(title);
    }

    /**
     * Set a page title.
     *
     * @param {string} title A title.
     */
    setPageTitle(title) {
      this.getHelper().pageTitle.setTitle(title);
    }

    /**
     * Translate a label.
     *
     * @param {string} label Label.
     * @param {string|'messages'|'labels'|'fields'|'links'|'scopeNames'|'scopeNamesPlural'} [category='labels'] Category.
     * @param {string} [scope='Global'] Scope.
     * @returns {string}
     */
    translate(label, category, scope) {
      return this.getLanguage().translate(label, category, scope);
    }

    /**
     * Get a base path.
     *
     * @returns {string}
     */
    getBasePath() {
      return this._helper.basePath || '';
    }

    /**
     * @typedef {Object} module:view~ConfirmOptions
     *
     * @property {string} message A message.
     * @property {string} [confirmText] A confirm-button text.
     * @property {string} [cancelText] A cancel-button text.
     * @property {'danger'|'success'|'warning'|'default'} [confirmStyle='danger'] A confirm-button style.
     * @property {'static'|boolean} [backdrop=false] A backdrop.
     * @property {function():void} [cancelCallback] A cancel-callback.
     */

    /**
     * Show a confirmation dialog.
     *
     * @param {string|module:view~ConfirmOptions} o A message or options.
     * @param [callback] A callback. Deprecated, use a promise.
     * @param [context] A context. Deprecated.
     * @returns {Promise} To be resolved if confirmed.
     */
    confirm(o, callback, context) {
      let message;
      if (typeof o === 'string' || o instanceof String) {
        message = o;
        o = /** @type {module:view~ConfirmOptions} */{};
      } else {
        o = o || {};
        message = o.message;
      }
      if (message) {
        message = this.getHelper().transformMarkdownText(message, {
          linksInNewTab: true
        }).toString();
      }
      const confirmText = o.confirmText || this.translate('Yes');
      const confirmStyle = o.confirmStyle || null;
      const cancelText = o.cancelText || this.translate('Cancel');
      return Espo.Ui.confirm(message, {
        confirmText: confirmText,
        cancelText: cancelText,
        confirmStyle: confirmStyle,
        backdrop: 'backdrop' in o ? o.backdrop : true,
        isHtml: true
      }, callback, context);
    }
  }
  var _default = _exports.default = View;
});

define("view-record-helper", ["exports", "bullbone"], function (_exports, _bullbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module view-record-helper */

  /**
   * @mixes Bull.Events
   */
  class ViewRecordHelper {
    /**
     * @param {Object.<string, *>} [defaultFieldStates] Default field states.
     * @param {Object.<string, *>} [defaultPanelStates] Default panel states.
     */
    constructor(defaultFieldStates, defaultPanelStates) {
      /**
       * @private
       * @type {Object}
       */
      this.defaultFieldStates = defaultFieldStates || {};
      /**
       * @private
       * @type {Object}
       */
      this.defaultPanelStates = defaultPanelStates || {};
      /** @private */
      this.fieldStateMap = {};
      /** @private */
      this.panelStateMap = {};
      /** @private */
      this.hiddenFields = {};
      /** @private */
      this.hiddenPanels = {};
      /** @private */
      this.fieldOptionListMap = {};
    }

    /**
     * Get hidden fields.
     *
     * @returns {Object.<string, boolean>}
     */
    getHiddenFields() {
      return this.hiddenFields;
    }

    /**
     * Get hidden panels.
     *
     * @returns {Object.<string,boolean>}
     */
    getHiddenPanels() {
      return this.hiddenPanels;
    }

    /**
     * Set a field-state parameter.
     *
     * @param {string} field A field name.
     * @param {string|'hidden'} name A parameter.
     * @param {*} value A value.
     */
    setFieldStateParam(field, name, value) {
      switch (name) {
        case 'hidden':
          if (value) {
            this.hiddenFields[field] = true;
          } else {
            delete this.hiddenFields[field];
          }
          break;
      }
      this.fieldStateMap[field] = this.fieldStateMap[field] || {};
      this.fieldStateMap[field][name] = value;
      this.trigger('field-change');
    }

    /**
     * Get a field-state parameter.
     *
     * @param {string} field A field name.
     * @param {string} name A parameter.
     * @returns {*} A value.
     */
    getFieldStateParam(field, name) {
      if (field in this.fieldStateMap) {
        if (name in this.fieldStateMap[field]) {
          return this.fieldStateMap[field][name];
        }
      }
      if (name in this.defaultFieldStates) {
        return this.defaultFieldStates[name];
      }
      return null;
    }

    /**
     * Set a panel-state parameter.
     *
     * @param {string} panel A panel name.
     * @param {string|'hidden'} name A parameter.
     * @param {*} value A value.
     */
    setPanelStateParam(panel, name, value) {
      switch (name) {
        case 'hidden':
          if (value) {
            this.hiddenPanels[panel] = true;
          } else {
            delete this.hiddenPanels[panel];
          }
          break;
      }
      this.panelStateMap[panel] = this.panelStateMap[panel] || {};
      this.panelStateMap[panel][name] = value;
    }

    /**
     * Get a panel-state parameter.
     *
     * @param {string} panel A panel name.
     * @param {string|'hidden'} name A parameter.
     * @returns {*} A value.
     */
    getPanelStateParam(panel, name) {
      if (panel in this.panelStateMap) {
        if (name in this.panelStateMap[panel]) {
          return this.panelStateMap[panel][name];
        }
      }
      if (name in this.defaultPanelStates) {
        return this.defaultPanelStates[name];
      }
      return null;
    }

    /**
     * Set a field option list.
     *
     * @param {string} field A field name.
     * @param {string[]} list An option list.
     */
    setFieldOptionList(field, list) {
      this.fieldOptionListMap[field] = list;
    }

    /**
     * Clear a field option list.
     *
     * @param {string} field A field name.
     */
    clearFieldOptionList(field) {
      delete this.fieldOptionListMap[field];
    }

    /**
     * Get a field option list.
     *
     * @param {string} field A field name.
     * @returns {string[]|null} Null if not set.
     */
    getFieldOptionList(field) {
      return this.fieldOptionListMap[field] || null;
    }

    /**
     * Whether a field option list is set.
     *
     * @param {string} field A field name.
     * @returns {boolean}
     */
    hasFieldOptionList(field) {
      return field in this.fieldOptionListMap;
    }
  }
  Object.assign(ViewRecordHelper.prototype, _bullbone.Events);
  var _default = _exports.default = ViewRecordHelper;
});

define("language", ["exports", "bullbone"], function (_exports, _bullbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module language */

  /**
   * A language.
   *
   * @mixes Bull.Events
   */
  class Language {
    /** @private */
    url = 'I18n';

    /**
     * @class
     * @param {module:cache} [cache] A cache.
     */
    constructor(cache) {
      /**
       * @private
       * @type {module:cache|null}
       */
      this.cache = cache || null;

      /**
       * @private
       * @type {Object}
       */
      this.data = {};

      /**
       * A name.
       *
       * @type {string}
       */
      this.name = 'default';
    }

    /**
     * Whether an item is set in language data.
     *
     * @param {string} scope A scope.
     * @param {string} category A category.
     * @param {string} name An item name.
     * @returns {boolean}
     */
    has(name, category, scope) {
      if (scope in this.data) {
        if (category in this.data[scope]) {
          if (name in this.data[scope][category]) {
            return true;
          }
        }
      }
      return false;
    }

    /**
     * Get a value set in language data.
     *
     * @param {string} scope A scope.
     * @param {string} category A category.
     * @param {string} name An item name.
     * @returns {*}
     */
    get(scope, category, name) {
      if (scope in this.data) {
        if (category in this.data[scope]) {
          if (name in this.data[scope][category]) {
            return this.data[scope][category][name];
          }
        }
      }
      if (scope === 'Global') {
        return name;
      }
      return false;
    }

    /**
     * Translate a label.
     *
     * @param {string} name An item name.
     * @param {string|'messages'|'labels'|'fields'|'links'|'scopeNames'|'scopeNamesPlural'} [category='labels'] A category.
     * @param {string} [scope='Global'] A scope.
     * @returns {string}
     */
    translate(name, category, scope) {
      scope = scope || 'Global';
      category = category || 'labels';
      let res = this.get(scope, category, name);
      if (res === false && scope !== 'Global') {
        res = this.get('Global', category, name);
      }
      return res;
    }

    /**
     * Translation an option item value.
     *
     * @param {string} value An option value.
     * @param {string} field A field name.
     * @param {string} [scope='Global'] A scope.
     * @returns {string}
     */
    translateOption(value, field, scope) {
      let translation = this.translate(field, 'options', scope);
      if (typeof translation !== 'object') {
        translation = {};
      }
      return translation[value] || value;
    }

    /**
     * @private
     */
    loadFromCache(loadDefault) {
      let name = this.name;
      if (loadDefault) {
        name = 'default';
      }
      if (this.cache) {
        const cached = this.cache.get('app', 'language-' + name);
        if (cached) {
          this.data = cached;
          return true;
        }
      }
      return null;
    }

    /**
     * Clear a language cache.
     */
    clearCache() {
      if (this.cache) {
        this.cache.clear('app', 'language-' + this.name);
      }
    }

    /**
     * @private
     */
    storeToCache(loadDefault) {
      let name = this.name;
      if (loadDefault) {
        name = 'default';
      }
      if (this.cache) {
        this.cache.set('app', 'language-' + name, this.data);
      }
    }

    /**
     * Load data from cache or backend (if not yet cached).
     *
     * @returns {Promise}
     */
    load() {
      return this._loadInternal();
    }

    /**
     * @private
     * @param {boolean} [disableCache=false]
     * @param {boolean} [loadDefault=false].
     * @returns {Promise}
     */
    _loadInternal(disableCache, loadDefault) {
      if (!disableCache && this.loadFromCache(loadDefault)) {
        this.trigger('sync');
        return Promise.resolve();
      }
      return this.fetch(loadDefault);
    }

    /**
     * Load default-language data from the backend.
     *
     * @returns {Promise}
     */
    loadDefault() {
      return this._loadInternal(false, true);
    }

    /**
     * Load data from the backend.
     *
     * @returns {Promise}
     */
    loadSkipCache() {
      return this._loadInternal(true);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Load default-language data from the backend.
     *
     * @returns {Promise}
     */
    loadDefaultSkipCache() {
      return this._loadInternal(true, true);
    }

    /**
     * @private
     * @param {boolean} loadDefault
     * @returns {Promise}
     */
    fetch(loadDefault) {
      return Espo.Ajax.getRequest(this.url, {
        default: loadDefault
      }).then(data => {
        this.data = data;
        this.storeToCache(loadDefault);
        this.trigger('sync');
      });
    }

    /**
     * Sort a field list by a translated name.
     *
     * @param {string} scope An entity type.
     * @param {string[]} fieldList A field list.
     * @returns {string[]}
     */
    sortFieldList(scope, fieldList) {
      return fieldList.sort((v1, v2) => {
        return this.translate(v1, 'fields', scope).localeCompare(this.translate(v2, 'fields', scope));
      });
    }

    /**
     * Sort an entity type list by a translated name.
     *
     * @param {string[]} entityList An entity type list.
     * @param {boolean} [plural=false] Use a plural label.
     * @returns {string[]}
     */
    sortEntityList(entityList, plural) {
      let category = 'scopeNames';
      if (plural) {
        category += 'Plural';
      }
      return entityList.sort((v1, v2) => {
        return this.translate(v1, category).localeCompare(this.translate(v2, category));
      });
    }

    /**
     * Get a value by a path.
     *
     * @param {string[]|string} path A path.
     * @returns {*}
     */
    translatePath(path) {
      if (typeof path === 'string' || path instanceof String) {
        path = path.split('.');
      }
      let pointer = this.data;
      path.forEach(key => {
        if (key in pointer) {
          pointer = pointer[key];
        }
      });
      return pointer;
    }

    /**
     * Do not use.
     *
     * @param {string} [scope]
     * @param {Record} [data]
     * @internal
     */
    setScopeData(scope, data) {
      this.data[scope] = data;
    }
  }
  Object.assign(Language.prototype, _bullbone.Events);
  var _default = _exports.default = Language;
});

define("dynamic-logic", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module dynamic-logic */

  /**
   * Dynamic logic. Handles form appearance and behaviour depending on conditions.
   *
   * @internal Instantiated in advanced-pack.
   */
  class DynamicLogic {
    /**
     * @param {Object} defs Definitions.
     * @param {module:views/record/base} recordView A record view.
     */
    constructor(defs, recordView) {
      /**
       * @type {Object} Definitions.
       * @private
       */
      this.defs = defs || {};

      /**
       *
       * @type {module:views/record/base}
       * @private
       */
      this.recordView = recordView;

      /**
       * @type {string[]}
       * @private
       */
      this.fieldTypeList = ['visible', 'required', 'readOnly'];

      /**
       * @type {string[]}
       * @private
       */
      this.panelTypeList = ['visible', 'styled'];
    }

    /**
     * Process.
     */
    process() {
      const fields = this.defs.fields || {};
      Object.keys(fields).forEach(field => {
        const item = fields[field] || {};
        this.fieldTypeList.forEach(type => {
          if (!(type in item)) {
            return;
          }
          if (!item[type]) {
            return;
          }
          const typeItem = item[type] || {};
          if (!typeItem.conditionGroup) {
            return;
          }
          const result = this.checkConditionGroup(typeItem.conditionGroup);
          let methodName;
          methodName = result ? 'makeField' + Espo.Utils.upperCaseFirst(type) + 'True' : 'makeField' + Espo.Utils.upperCaseFirst(type) + 'False';
          this[methodName](field);
        });
      });
      const panels = this.defs.panels || {};
      Object.keys(panels).forEach(panel => {
        this.panelTypeList.forEach(type => {
          this.processPanel(panel, type);
        });
      });
      const options = this.defs.options || {};
      Object.keys(options).forEach(field => {
        const itemList = options[field];
        if (!options[field]) {
          return;
        }
        let isMet = false;
        for (const i in itemList) {
          const item = itemList[i];
          if (this.checkConditionGroup(item.conditionGroup)) {
            this.setOptionList(field, item.optionList || []);
            isMet = true;
            break;
          }
        }
        if (!isMet) {
          this.resetOptionList(field);
        }
      });
    }

    /**
     * @param {string} panel A panel name.
     * @param {string} type A type.
     * @private
     */
    processPanel(panel, type) {
      const panels = this.defs.panels || {};
      const item = panels[panel] || {};
      if (!(type in item)) {
        return;
      }
      const typeItem = item[type] || {};
      if (!typeItem.conditionGroup) {
        return;
      }
      const result = this.checkConditionGroup(typeItem.conditionGroup);
      let methodName;
      if (result) {
        methodName = 'makePanel' + Espo.Utils.upperCaseFirst(type) + 'True';
      } else {
        methodName = 'makePanel' + Espo.Utils.upperCaseFirst(type) + 'False';
      }
      this[methodName](panel);
    }

    /**
     * Check a condition group.
     * @param {Object} data A condition group.
     * @param {'and'|'or'|'not'} [type='and'] A type.
     * @returns {boolean}
     */
    checkConditionGroup(data, type) {
      type = type || 'and';
      let list;
      let result = false;
      if (type === 'and') {
        list = data || [];
        result = true;
        for (const i in list) {
          if (!this.checkCondition(list[i])) {
            result = false;
            break;
          }
        }
      } else if (type === 'or') {
        list = data || [];
        for (const i in list) {
          if (this.checkCondition(list[i])) {
            result = true;
            break;
          }
        }
      } else if (type === 'not') {
        if (data) {
          result = !this.checkCondition(data);
        }
      }
      return result;
    }

    /**
     * @private
     * @param {string} attribute
     * @return {*}
     */
    getAttributeValue(attribute) {
      if (attribute.startsWith('$')) {
        if (attribute === '$user.id') {
          return this.recordView.getUser().id;
        }
        if (attribute === '$user.teamsIds') {
          return this.recordView.getUser().getTeamIdList();
        }
      }
      if (!this.recordView.model.has(attribute)) {
        return undefined;
      }
      return this.recordView.model.get(attribute);
    }

    /**
     * Check a condition.
     *
     * @param {Object} defs Definitions.
     * @returns {boolean}
     */
    checkCondition(defs) {
      defs = defs || {};
      const type = defs.type || 'equals';
      if (['or', 'and', 'not'].includes(type)) {
        return this.checkConditionGroup(defs.value, /** @type {'or'|'and'|'not'} */type);
      }
      const attribute = defs.attribute;
      const value = defs.value;
      if (!attribute) {
        return false;
      }
      const setValue = this.getAttributeValue(attribute);
      if (type === 'equals') {
        return setValue === value;
      }
      if (type === 'notEquals') {
        return setValue !== value;
      }
      if (type === 'isEmpty') {
        if (Array.isArray(setValue)) {
          return !setValue.length;
        }
        return setValue === null || setValue === '' || typeof setValue === 'undefined';
      }
      if (type === 'isNotEmpty') {
        if (Array.isArray(setValue)) {
          return !!setValue.length;
        }
        return setValue !== null && setValue !== '' && typeof setValue !== 'undefined';
      }
      if (type === 'isTrue') {
        return !!setValue;
      }
      if (type === 'isFalse') {
        return !setValue;
      }
      if (type === 'contains' || type === 'has') {
        if (!setValue) {
          return false;
        }
        return !!~setValue.indexOf(value);
      }
      if (type === 'notContains' || type === 'notHas') {
        if (!setValue) {
          return true;
        }
        return !~setValue.indexOf(value);
      }
      if (type === 'startsWith') {
        if (!setValue) {
          return false;
        }
        return setValue.indexOf(value) === 0;
      }
      if (type === 'endsWith') {
        if (!setValue) {
          return false;
        }
        return setValue.indexOf(value) === setValue.length - value.length;
      }
      if (type === 'matches') {
        if (!setValue) {
          return false;
        }
        const match = /^\/(.*)\/([a-z]*)$/.exec(value);
        if (!match || match.length < 2) {
          return false;
        }
        return new RegExp(match[1], match[2]).test(setValue);
      }
      if (type === 'greaterThan') {
        return setValue > value;
      }
      if (type === 'lessThan') {
        return setValue < value;
      }
      if (type === 'greaterThanOrEquals') {
        return setValue >= value;
      }
      if (type === 'lessThanOrEquals') {
        return setValue <= value;
      }
      if (type === 'in') {
        return !!~value.indexOf(setValue);
      }
      if (type === 'notIn') {
        return !~value.indexOf(setValue);
      }
      if (type === 'isToday') {
        const dateTime = this.recordView.getDateTime();
        if (!setValue) {
          return false;
        }
        if (setValue.length > 10) {
          return dateTime.toMoment(setValue).isSame(dateTime.getNowMoment(), 'day');
        }
        return dateTime.toMomentDate(setValue).isSame(dateTime.getNowMoment(), 'day');
      }
      if (type === 'inFuture') {
        const dateTime = this.recordView.getDateTime();
        if (!setValue) {
          return false;
        }
        if (setValue.length > 10) {
          return dateTime.toMoment(setValue).isAfter(dateTime.getNowMoment(), 'second');
        }
        return dateTime.toMomentDate(setValue).isAfter(dateTime.getNowMoment(), 'day');
      }
      if (type === 'inPast') {
        const dateTime = this.recordView.getDateTime();
        if (!setValue) {
          return false;
        }
        if (setValue.length > 10) {
          return dateTime.toMoment(setValue).isBefore(dateTime.getNowMoment(), 'second');
        }
        return dateTime.toMomentDate(setValue).isBefore(dateTime.getNowMoment(), 'day');
      }
      return false;
    }

    /**
     * @param {string} field
     * @param {string[]} optionList
     * @private
     */
    setOptionList(field, optionList) {
      this.recordView.setFieldOptionList(field, optionList);
    }

    /**
     * @param {string} field
     * @private
     */
    resetOptionList(field) {
      this.recordView.resetFieldOptionList(field);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} field
     * @private
     */
    makeFieldVisibleTrue(field) {
      this.recordView.showField(field);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} field
     * @private
     */
    makeFieldVisibleFalse(field) {
      this.recordView.hideField(field);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} field
     * @private
     */
    makeFieldRequiredTrue(field) {
      this.recordView.setFieldRequired(field);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} field
     * @private
     */
    makeFieldRequiredFalse(field) {
      this.recordView.setFieldNotRequired(field);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} field
     * @private
     */
    makeFieldReadOnlyTrue(field) {
      this.recordView.setFieldReadOnly(field);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} field
     * @private
     */
    makeFieldReadOnlyFalse(field) {
      this.recordView.setFieldNotReadOnly(field);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} panel
     * @private
     */
    makePanelVisibleTrue(panel) {
      this.recordView.showPanel(panel, 'dynamicLogic');
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} panel
     * @private
     */
    makePanelVisibleFalse(panel) {
      this.recordView.hidePanel(panel, false, 'dynamicLogic');
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} panel
     * @private
     */
    makePanelStyledTrue(panel) {
      this.recordView.stylePanel(panel);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {string} panel
     * @private
     */
    makePanelStyledFalse(panel) {
      this.recordView.unstylePanel(panel);
    }

    /**
     * Add a panel-visible condition.
     *
     * @param {string} name A panel name.
     * @param {Object} item Condition definitions.
     */
    addPanelVisibleCondition(name, item) {
      this.defs.panels = this.defs.panels || {};
      this.defs.panels[name] = this.defs.panels[name] || {};
      this.defs.panels[name].visible = item;
      this.processPanel(name, 'visible');
    }

    /**
     * Add a panel-styled condition.
     *
     * @param {string} name A panel name.
     * @param {Object} item Condition definitions.
     */
    addPanelStyledCondition(name, item) {
      this.defs.panels = this.defs.panels || {};
      this.defs.panels[name] = this.defs.panels[name] || {};
      this.defs.panels[name].styled = item;
      this.processPanel(name, 'styled');
    }
  }
  var _default = _exports.default = DynamicLogic;
});

define("helpers/model/defaults-populator", ["exports", "di", "metadata", "view-helper", "models/settings", "models/user", "acl-manager", "models/preferences"], function (_exports, _di, _metadata, _viewHelper, _settings, _user, _aclManager, _preferences) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _metadata = _interopRequireDefault(_metadata);
  _viewHelper = _interopRequireDefault(_viewHelper);
  _settings = _interopRequireDefault(_settings);
  _user = _interopRequireDefault(_user);
  _aclManager = _interopRequireDefault(_aclManager);
  _preferences = _interopRequireDefault(_preferences);
  let _init_metadata, _init_extra_metadata, _init_viewHelper, _init_extra_viewHelper, _init_config, _init_extra_config, _init_user, _init_extra_user, _init_preferences, _init_extra_preferences, _init_acl, _init_extra_acl;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * Defaults populator.
   */
  class DefaultsPopulator {
    static {
      [_init_metadata, _init_extra_metadata, _init_viewHelper, _init_extra_viewHelper, _init_config, _init_extra_config, _init_user, _init_extra_user, _init_preferences, _init_extra_preferences, _init_acl, _init_extra_acl] = _applyDecs(this, [], [[(0, _di.inject)(_metadata.default), 0, "metadata"], [(0, _di.inject)(_viewHelper.default), 0, "viewHelper"], [(0, _di.inject)(_settings.default), 0, "config"], [(0, _di.inject)(_user.default), 0, "user"], [(0, _di.inject)(_preferences.default), 0, "preferences"], [(0, _di.inject)(_aclManager.default), 0, "acl"]]).e;
    }
    constructor() {
      _init_extra_acl(this);
    }
    /**
     * @private
     * @type {Metadata}
     */
    metadata = _init_metadata(this);

    /**
     * @private
     * @type {ViewHelper}
     */
    viewHelper = (_init_extra_metadata(this), _init_viewHelper(this));

    /**
     * @private
     * @type {Settings}
     */
    config = (_init_extra_viewHelper(this), _init_config(this));

    /**
     * @private
     * @type {User}
     */
    user = (_init_extra_config(this), _init_user(this));

    /**
     * @private
     * @type {Preferences}
     */
    preferences = (_init_extra_user(this), _init_preferences(this));

    /**
     * @private
     * @type {AclManager}
     */
    acl = (_init_extra_preferences(this), _init_acl(this));

    /**
     * Populate default values.
     *
     * @param {module:model} model A model.
     * @return {Promise|undefined}
     */
    populate(model) {
      model.populateDefaults();
      const defaultHash = {};
      if (!this.user.isPortal()) {
        this.prepare(model, defaultHash);
      }
      if (this.user.isPortal()) {
        this.prepareForPortal(model, defaultHash);
      }
      this.prepareFields(model, defaultHash);
      for (const attr in defaultHash) {
        if (model.has(attr)) {
          delete defaultHash[attr];
        }
      }
      model.set(defaultHash, {
        silent: true
      });
      const preparatorClass = this.metadata.get(`clientDefs.${model.entityType}.modelDefaultsPreparator`);
      if (!preparatorClass) {
        return undefined;
      }
      return Espo.loader.requirePromise(preparatorClass).then(Class => {
        /** @type {import('handlers/model/defaults-preparator').default} */
        const preparator = new Class(this.viewHelper);
        return preparator.prepare(model);
      }).then(attributes => {
        model.set(attributes, {
          silent: true
        });
      });
    }

    /**
     * @param {module:model} model
     * @param {Object.<string, *>} defaultHash
     * @private
     */
    prepare(model, defaultHash) {
      const hasAssignedUsers = model.hasField('assignedUsers') && model.getLinkParam('assignedUsers', 'entity') === 'User';
      if (model.hasField('assignedUser') || hasAssignedUsers) {
        let assignedUserField = 'assignedUser';
        if (hasAssignedUsers) {
          assignedUserField = 'assignedUsers';
        }
        if (this.toFillAssignedUser(model, assignedUserField)) {
          if (hasAssignedUsers) {
            defaultHash['assignedUsersIds'] = [this.user.id];
            defaultHash['assignedUsersNames'] = {};
            defaultHash['assignedUsersNames'][this.user.id] = this.user.get('name');
          } else {
            defaultHash['assignedUserId'] = this.user.id;
            defaultHash['assignedUserName'] = this.user.get('name');
          }
        }
      }
      const defaultTeamId = this.user.get('defaultTeamId');
      if (defaultTeamId) {
        if (model.hasField('teams') && !model.getFieldParam('teams', 'default') && Espo.Utils.lowerCaseFirst(model.getLinkParam('teams', 'relationName') || '') === 'entityTeam') {
          defaultHash['teamsIds'] = [defaultTeamId];
          defaultHash['teamsNames'] = {};
          defaultHash['teamsNames'][defaultTeamId] = this.user.get('defaultTeamName');
        }
      }
      const hasCollaborators = model.hasField('collaborators') && model.getLinkParam('collaborators', 'entity') === 'User' && this.metadata.get(`scopes.${model.entityType}.collaborators`);
      if (hasCollaborators) {
        defaultHash.collaboratorsIds = [this.user.id];
        defaultHash.collaboratorsNames = {
          [this.user.id]: this.user.attributes.name
        };
      }
    }

    /**
     *
     * @param {import('model').default} model
     * @param {string} assignedUserField
     */
    toFillAssignedUser(model, assignedUserField) {
      if (!this.preferences.get('doNotFillAssignedUserIfNotRequired')) {
        return true;
      }
      if (model.getFieldParam(assignedUserField, 'required')) {
        return true;
      }
      if (this.acl.getPermissionLevel('assignmentPermission') === 'no') {
        return true;
      }
      if (this.acl.getPermissionLevel('assignmentPermission') === 'team' && !this.user.get('defaultTeamId')) {
        return true;
      }
      if (this.acl.getLevel(model.entityType, 'read') === 'own') {
        return true;
      }
      if (!this.acl.checkField(model.entityType, assignedUserField, 'edit')) {
        return true;
      }
      return false;
    }

    /**
     * @param {module:model} model
     * @param {Object.<string, *>} defaultHash
     * @private
     */
    prepareForPortal(model, defaultHash) {
      if (model.hasField('account') && ['belongsTo', 'hasOne'].includes(model.getLinkType('account')) && model.getLinkParam('account', 'entity') === 'Account') {
        if (this.user.get('accountId')) {
          defaultHash['accountId'] = this.user.get('accountId');
          defaultHash['accountName'] = this.user.get('accountName');
        }
      }
      if (model.hasField('contact') && ['belongsTo', 'hasOne'].includes(model.getLinkType('contact')) && model.getLinkParam('contact', 'entity') === 'Contact') {
        if (this.user.get('contactId')) {
          defaultHash['contactId'] = this.user.get('contactId');
          defaultHash['contactName'] = this.user.get('contactName');
        }
      }
      if (model.hasField('parent') && model.getLinkType('parent') === 'belongsToParent') {
        if (!this.config.get('b2cMode')) {
          if (this.user.get('accountId')) {
            if ((model.getFieldParam('parent', 'entityList') || []).includes('Account')) {
              defaultHash['parentId'] = this.user.get('accountId');
              defaultHash['parentName'] = this.user.get('accountName');
              defaultHash['parentType'] = 'Account';
            }
          }
        } else {
          if (this.user.get('contactId')) {
            if ((model.getFieldParam('parent', 'entityList') || []).includes('Contact')) {
              defaultHash['contactId'] = this.user.get('contactId');
              defaultHash['parentName'] = this.user.get('contactName');
              defaultHash['parentType'] = 'Contact';
            }
          }
        }
      }
      if (model.hasField('accounts') && model.getLinkType('accounts') === 'hasMany' && model.getLinkParam('accounts', 'entity') === 'Account') {
        if (this.user.get('accountsIds')) {
          defaultHash['accountsIds'] = this.user.get('accountsIds');
          defaultHash['accountsNames'] = this.user.get('accountsNames');
        }
      }
      if (model.hasField('contacts') && model.getLinkType('contacts') === 'hasMany' && model.getLinkParam('contacts', 'entity') === 'Contact') {
        if (this.user.get('contactId')) {
          defaultHash['contactsIds'] = [this.user.get('contactId')];
          const names = {};
          names[this.user.get('contactId')] = this.user.get('contactName');
          defaultHash['contactsNames'] = names;
        }
      }
    }

    /**
     * @param {module:model} model
     * @param {Object.<string, *>} defaultHash
     * @private
     */
    prepareFields(model, defaultHash) {
      const set = (attribute, value) => {
        if (attribute in defaultHash || model.has(attribute)) {
          return;
        }
        defaultHash[attribute] = value;
      };
      model.getFieldList().forEach(field => {
        const type = model.getFieldType(field);
        if (!type) {
          return;
        }
        if (model.getFieldParam(field, 'disabled') || model.getFieldParam(field, 'utility')) {
          return;
        }
        if (type === 'enum') {
          /** @type {string[]} */
          const options = model.getFieldParam(field, 'options') || [];
          let value = options[0] || '';
          value = value !== '' ? value : null;
          if (value) {
            set(field, value);
          }
        }
      });
    }
  }
  var _default = _exports.default = DefaultsPopulator;
});

define("views/record/base", ["exports", "view", "view-record-helper", "dynamic-logic", "underscore", "jquery", "helpers/model/defaults-populator"], function (_exports, _view, _viewRecordHelper, _dynamicLogic, _underscore, _jquery, _defaultsPopulator) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  _viewRecordHelper = _interopRequireDefault(_viewRecordHelper);
  _dynamicLogic = _interopRequireDefault(_dynamicLogic);
  _underscore = _interopRequireDefault(_underscore);
  _jquery = _interopRequireDefault(_jquery);
  _defaultsPopulator = _interopRequireDefault(_defaultsPopulator);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/record/base */

  /**
   * A base record view. To be extended.
   */
  class BaseRecordView extends _view.default {
    /**
     * A type.
     */
    type = 'edit';

    /**
     * An entity type.
     *
     * @type {string|null}
     */
    entityType = null;

    /**
     * A scope.
     *
     * @type {string|null}
     */
    scope = null;

    /**
     * Is new. Is set automatically.
     */
    isNew = false;

    /**
     * @deprecated
     * @protected
     */
    dependencyDefs = {};

    /**
     * Dynamic logic.
     *
     * @protected
     * @type {Object}
     */
    dynamicLogicDefs = {};

    /**
     * A field list.
     *
     * @protected
     */
    fieldList = null;

    /**
     * A mode.
     *
     * @type {'detail'|'edit'|null}
     */
    mode = null;

    /**
     * A last save cancel reason.
     *
     * @protected
     * @type {string|null}
     */
    lastSaveCancelReason = null;

    /**
     * A record-helper.
     *
     * @type {module:view-record-helper}
     */
    recordHelper = null;

    /** @const */
    MODE_DETAIL = 'detail';
    /** @const */
    MODE_EDIT = 'edit';

    /** @const */
    TYPE_DETAIL = 'detail';
    // noinspection JSUnusedGlobalSymbols
    /** @const  */
    TYPE_EDIT = 'edit';

    /**
     * Hide a field.
     *
     * @param {string} name A field name.
     * @param {boolean } [locked] To lock. Won't be able to un-hide.
     */
    hideField(name, locked) {
      this.recordHelper.setFieldStateParam(name, 'hidden', true);
      if (locked) {
        this.recordHelper.setFieldStateParam(name, 'hiddenLocked', true);
      }
      const processHtml = () => {
        const fieldView = this.getFieldView(name);
        if (fieldView) {
          const $field = fieldView.$el;
          const $cell = $field.closest('.cell[data-name="' + name + '"]');
          const $label = $cell.find('label.control-label[data-name="' + name + '"]');
          $field.addClass('hidden');
          $label.addClass('hidden');
          $cell.addClass('hidden-cell');
        } else {
          this.$el.find('.cell[data-name="' + name + '"]').addClass('hidden-cell');
          this.$el.find('.field[data-name="' + name + '"]').addClass('hidden');
          this.$el.find('label.control-label[data-name="' + name + '"]').addClass('hidden');
        }
      };
      if (this.isRendered()) {
        processHtml();
      } else {
        this.once('after:render', () => {
          processHtml();
        });
      }
      const view = this.getFieldView(name);
      if (view) {
        view.setDisabled(locked);
      }
    }

    /**
     * Show a field.
     *
     * @param {string} name A field name.
     */
    showField(name) {
      if (this.recordHelper.getFieldStateParam(name, 'hiddenLocked')) {
        return;
      }
      this.recordHelper.setFieldStateParam(name, 'hidden', false);
      const processHtml = () => {
        const fieldView = this.getFieldView(name);
        if (fieldView) {
          const $field = fieldView.$el;
          const $cell = $field.closest('.cell[data-name="' + name + '"]');
          const $label = $cell.find('label.control-label[data-name="' + name + '"]');
          $field.removeClass('hidden');
          $label.removeClass('hidden');
          $cell.removeClass('hidden-cell');
          return;
        }
        this.$el.find('.cell[data-name="' + name + '"]').removeClass('hidden-cell');
        this.$el.find('.field[data-name="' + name + '"]').removeClass('hidden');
        this.$el.find('label.control-label[data-name="' + name + '"]').removeClass('hidden');
      };
      if (this.isRendered()) {
        processHtml();
      } else {
        this.once('after:render', () => {
          processHtml();
        });
      }
      const view = this.getFieldView(name);
      if (view) {
        if (!view.disabledLocked) {
          view.setNotDisabled();
        }
      }
    }

    /**
     * Set a field as read-only.
     *
     * @param {string} name A field name.
     * @param {boolean } [locked] To lock. Won't be able to un-set.
     */
    setFieldReadOnly(name, locked) {
      const previousValue = this.recordHelper.getFieldStateParam(name, 'readOnly');
      this.recordHelper.setFieldStateParam(name, 'readOnly', true);
      if (locked) {
        this.recordHelper.setFieldStateParam(name, 'readOnlyLocked', true);
      }
      const view = this.getFieldView(name);
      if (view) {
        view.setReadOnly(locked).catch(() => {});
      }
      if (!previousValue) {
        this.trigger('set-field-read-only', name);
      }

      /**
       * @todo
       *   Move to fields/base. Listen to recordHelper 'field-change' (if recordHelper is available).
       *   Same for set state methods.
       *   Issue is that sometimes state is changed in between view initialization (for bottom views with fields).
       */

      if (!view && !this.isReady) {
        this.once('ready', () => {
          const view = this.getFieldView(name);
          if (view && !view.readOnly && this.recordHelper.getFieldStateParam(name, 'readOnly')) {
            view.setReadOnly(locked);
          }
        });
      }
    }

    /**
     * Set a field as not read-only.
     *
     * @param {string} name A field name.
     */
    setFieldNotReadOnly(name) {
      const previousValue = this.recordHelper.getFieldStateParam(name, 'readOnly');
      this.recordHelper.setFieldStateParam(name, 'readOnly', false);
      if (this.readOnly) {
        return;
      }
      const view = this.getFieldView(name);
      if (view && view.readOnly) {
        view.setNotReadOnly();
        if (this.mode === this.MODE_EDIT) {
          if (!view.readOnlyLocked && view.isDetailMode()) {
            view.setEditMode().then(() => view.reRender());
          }
        }
      }
      if (previousValue) {
        this.trigger('set-field-not-read-only', name);
      }
      if (!view && !this.isReady) {
        this.once('ready', () => {
          const view = this.getFieldView(name);
          if (view && view.readOnly && !this.recordHelper.getFieldStateParam(name, 'readOnly')) {
            view.setNotReadOnly();
          }
        });
      }
    }

    /**
     * Set a field as required.
     *
     * @param {string} name A field name.
     */
    setFieldRequired(name) {
      const previousValue = this.recordHelper.getFieldStateParam(name, 'required');
      this.recordHelper.setFieldStateParam(name, 'required', true);
      const view = this.getFieldView(name);
      if (view) {
        view.setRequired();
      }
      if (!previousValue) {
        this.trigger('set-field-required', name);
      }
    }

    /**
     * Set a field as not required.
     *
     * @param {string} name A field name.
     */
    setFieldNotRequired(name) {
      const previousValue = this.recordHelper.getFieldStateParam(name, 'required');
      this.recordHelper.setFieldStateParam(name, 'required', false);
      const view = this.getFieldView(name);
      if (view) {
        view.setNotRequired();
      }
      if (previousValue) {
        this.trigger('set-field-not-required', name);
      }
    }

    /**
     * Set an option list for a field.
     *
     * @param {string} name A field name.
     * @param {string[]} list Options.
     */
    setFieldOptionList(name, list) {
      const had = this.recordHelper.hasFieldOptionList(name);
      const previousList = this.recordHelper.getFieldOptionList(name);
      this.recordHelper.setFieldOptionList(name, list);
      const view = this.getFieldView(name);
      if (view) {
        if ('setOptionList' in view) {
          view.setOptionList(list);
        }
      }
      if (!had || !(0, _underscore.default)(previousList).isEqual(list)) {
        this.trigger('set-field-option-list', name, list);
      }
    }

    /**
     * Reset field options (revert to default).
     *
     * @param {string} name A field name.
     */
    resetFieldOptionList(name) {
      const had = this.recordHelper.hasFieldOptionList(name);
      this.recordHelper.clearFieldOptionList(name);
      const view = this.getFieldView(name);
      if (view) {
        if ('resetOptionList' in view) {
          view.resetOptionList();
        }
      }
      if (had) {
        this.trigger('reset-field-option-list', name);
      }
    }

    /**
     * Show a panel.
     *
     * @param {string} name A panel name.
     * @param [softLockedType] Omitted.
     */
    showPanel(name, softLockedType) {
      this.recordHelper.setPanelStateParam(name, 'hidden', false);
      if (this.isRendered()) {
        this.$el.find('.panel[data-name="' + name + '"]').removeClass('hidden');
      }
    }

    /**
     * Hide a panel.
     *
     * @param {string} name A panel name.
     * @param {boolean} [locked=false] Won't be able to un-hide.
     * @param {module:views/record/detail~panelSoftLockedType} [softLockedType='default']
     */
    hidePanel(name, locked, softLockedType) {
      this.recordHelper.setPanelStateParam(name, 'hidden', true);
      if (this.isRendered()) {
        this.$el.find('.panel[data-name="' + name + '"]').addClass('hidden');
      }
    }

    /**
     * Style a panel. Style is set in the `data-style` DOM attribute.
     *
     * @param {string} name A panel name.
     */
    stylePanel(name) {
      this.recordHelper.setPanelStateParam(name, 'styled', true);
      const process = () => {
        const $panel = this.$el.find('.panel[data-name="' + name + '"]');
        const $btn = $panel.find('> .panel-heading .btn');
        const style = $panel.attr('data-style');
        if (!style) {
          return;
        }
        $panel.removeClass('panel-default');
        $panel.addClass('panel-' + style);
        $btn.removeClass('btn-default');
        $btn.addClass('btn-' + style);
      };
      if (this.isRendered()) {
        process();
        return;
      }
      this.once('after:render', () => {
        process();
      });
    }

    /**
     * Un-style a panel.
     *
     * @param {string} name A panel name.
     */
    unstylePanel(name) {
      this.recordHelper.setPanelStateParam(name, 'styled', false);
      const process = () => {
        const $panel = this.$el.find('.panel[data-name="' + name + '"]');
        const $btn = $panel.find('> .panel-heading .btn');
        const style = $panel.attr('data-style');
        if (!style) {
          return;
        }
        $panel.removeClass('panel-' + style);
        $panel.addClass('panel-default');
        $btn.removeClass('btn-' + style);
        $btn.addClass('btn-default');
      };
      if (this.isRendered()) {
        process();
        return;
      }
      this.once('after:render', () => {
        process();
      });
    }

    /**
     * Set/unset a confirmation upon leaving the form.
     *
     * @param {boolean} value True sets a required confirmation.
     */
    setConfirmLeaveOut(value) {
      if (!this.getRouter()) {
        return;
      }
      this.getRouter().confirmLeaveOut = value;
    }

    /**
     * Get field views.
     *
     * @param {boolean} [withHidden] With hidden.
     * @return {Object.<string, module:views/fields/base>}
     */
    getFieldViews(withHidden) {
      const fields = {};
      this.fieldList.forEach(item => {
        const view = this.getFieldView(item);
        if (view) {
          fields[item] = view;
        }
      });
      return fields;
    }

    /**
     * Get a field view.
     *
     * @param {string} name A field name.
     * @return {module:views/fields/base|null}
     */
    getFieldView(name) {
      /** @type {module:views/fields/base|null} */
      let view = this.getView(name + 'Field') || null;

      // @todo Remove.
      if (!view) {
        view = this.getView(name) || null;
      }
      return view;
    }

    /**
     * @deprecated Use `getFieldView`.
     * @return {module:views/fields/base|null}
     */
    getField(name) {
      return this.getFieldView(name);
    }

    /**
     * Get a field list.
     *
     * @return {string[]}
     */
    getFieldList() {
      return Object.keys(this.getFieldViews());
    }

    /**
     * Get a field view list.
     *
     * @return {module:views/fields/base[]}
     */
    getFieldViewList() {
      return this.getFieldList().map(field => this.getFieldView(field)).filter(view => view !== null);
    }

    /**
     * @inheritDoc
     */
    data() {
      return {
        scope: this.scope,
        entityType: this.entityType,
        hiddenPanels: this.recordHelper.getHiddenPanels(),
        hiddenFields: this.recordHelper.getHiddenFields()
      };
    }

    /**
     * @todo Remove.
     * @private
     */
    handleDataBeforeRender(data) {
      this.getFieldList().forEach(field => {
        const viewKey = field + 'Field';
        data[field] = data[viewKey];
      });
    }

    /**
     * @inheritDoc
     * Warning. Is not called by record/detail.
     */
    setup() {
      if (typeof this.model === 'undefined') {
        throw new Error('Model has not been injected into record view.');
      }

      /** @type {module:view-record-helper} */
      this.recordHelper = this.options.recordHelper || new _viewRecordHelper.default();
      this.dynamicLogicDefs = this.options.dynamicLogicDefs || this.dynamicLogicDefs;
      this.on('remove', () => {
        if (this.isChanged) {
          this.resetModelChanges();
        }
        this.setIsNotChanged();
      });
      this.entityType = this.model.entityType || this.model.name || 'Common';
      this.scope = this.options.scope || this.entityType;
      this.fieldList = this.options.fieldList || this.fieldList || [];
      this.numId = Math.floor(Math.random() * 10000 + 1);
      this.id = Espo.Utils.toDom(this.entityType) + '-' + Espo.Utils.toDom(this.type) + '-' + this.numId;
      if (this.model.isNew()) {
        this.isNew = true;
      }
      this.setupBeforeFinal();
    }

    /**
     * Set up before final.
     *
     * @protected
     */
    setupBeforeFinal() {
      this.attributes = this.model.getClonedAttributes();
      this.listenTo(this.model, 'change', (m, o) => {
        if (o.sync) {
          for (const attribute in m.attributes) {
            if (!m.hasChanged(attribute)) {
              continue;
            }
            this.attributes[attribute] = Espo.Utils.cloneDeep(m.get(attribute));
          }
          return;
        }
        if (this.mode === this.MODE_EDIT) {
          this.setIsChanged();
        }
      });
      if (this.options.attributes) {
        this.model.set(this.options.attributes);
      }
      this.listenTo(this.model, 'sync', () => {
        this.attributes = this.model.getClonedAttributes();
      });
      this.initDependency();
      this.initDynamicLogic();
    }

    /**
     * Set an initial attribute value.
     *
     * @protected
     * @param {string} attribute An attribute name.
     * @param {*} value
     */
    setInitialAttributeValue(attribute, value) {
      this.attributes[attribute] = value;
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Check whether a current attribute value differs from initial.
     *
     * @param {string} name An attribute name.
     * @return {boolean}
     */
    checkAttributeIsChanged(name) {
      return !_underscore.default.isEqual(this.attributes[name], this.model.get(name));
    }

    /**
     * Reset model changes.
     */
    resetModelChanges() {
      if (this.updatedAttributes) {
        this.attributes = this.updatedAttributes;
        this.updatedAttributes = null;
      }
      const attributes = this.model.attributes;
      for (const attr in attributes) {
        if (!(attr in this.attributes)) {
          this.model.unset(attr);
        }
      }
      this.model.set(this.attributes, {
        skipReRender: true
      });
    }

    /**
     * Init dynamic logic.
     *
     * @protected
     */
    initDynamicLogic() {
      this.dynamicLogicDefs = Espo.Utils.clone(this.dynamicLogicDefs || {});
      this.dynamicLogicDefs.fields = Espo.Utils.clone(this.dynamicLogicDefs.fields);
      this.dynamicLogicDefs.panels = Espo.Utils.clone(this.dynamicLogicDefs.panels);
      this.dynamicLogic = new _dynamicLogic.default(this.dynamicLogicDefs, this);
      this.listenTo(this.model, 'change', () => this.processDynamicLogic());
      this.processDynamicLogic();
    }

    /**
     * Process dynamic logic.
     *
     * @protected
     */
    processDynamicLogic() {
      this.dynamicLogic.process();
    }

    /**
     * @protected
     * @internal
     */
    initDependency() {
      // noinspection JSDeprecatedSymbols
      Object.keys(this.dependencyDefs || {}).forEach(attr => {
        this.listenTo(this.model, 'change:' + attr, () => {
          this._handleDependencyAttribute(attr);
        });
      });
      this._handleDependencyAttributes();
    }

    /**
     * Set up a field level security.
     *
     * @protected
     */
    setupFieldLevelSecurity() {
      const forbiddenFieldList = this.getAcl().getScopeForbiddenFieldList(this.entityType, 'read');
      forbiddenFieldList.forEach(field => {
        this.hideField(field, true);
      });
      const readOnlyFieldList = this.getAcl().getScopeForbiddenFieldList(this.entityType, 'edit');
      readOnlyFieldList.forEach(field => {
        this.setFieldReadOnly(field, true);
      });
    }

    /**
     * Set is changed.
     *
     * @protected
     */
    setIsChanged() {
      this.isChanged = true;
    }

    /**
     * Set is not changed.
     *
     * @protected
     */
    setIsNotChanged() {
      this.isChanged = false;
    }

    /**
     * Validate.
     *
     * @return {boolean} True if not valid.
     */
    validate() {
      const invalidFieldList = [];
      this.getFieldList().forEach(field => {
        const fieldIsInvalid = this.validateField(field);
        if (fieldIsInvalid) {
          invalidFieldList.push(field);
        }
      });
      if (!!invalidFieldList.length) {
        this.onInvalid(invalidFieldList);
      }
      return !!invalidFieldList.length;
    }

    /**
     * @protected
     * @param {string[]} invalidFieldList Invalid fields.
     */
    onInvalid(invalidFieldList) {}

    /**
     * Validate a specific field.
     *
     * @param {string} field A field name.
     * @return {boolean} True if not valid.
     */
    validateField(field) {
      const msg = this.translate('fieldInvalid', 'messages').replace('{field}', this.translate(field, 'fields', this.entityType));
      const fieldView = this.getFieldView(field);
      if (!fieldView) {
        return false;
      }
      let notValid = false;
      if (fieldView.isEditMode() && !fieldView.disabled && !fieldView.readOnly) {
        notValid = fieldView.validate() || notValid;
      }
      if (notValid) {
        if (fieldView.$el) {
          const rect = fieldView.$el.get(0).getBoundingClientRect();
          if (rect.top === 0 && rect.bottom === 0 && rect.left === 0 && fieldView.$el.closest('.panel.hidden').length) {
            setTimeout(() => {
              const msg = this.translate('Not valid') + ': ' + (fieldView.lastValidationMessage || this.translate(field, 'fields', this.entityType));
              Espo.Ui.error(msg, true);
            }, 10);
          }
        }
        return true;
      }
      if (this.dynamicLogic && this.dynamicLogicDefs && this.dynamicLogicDefs.fields && this.dynamicLogicDefs.fields[field] && this.dynamicLogicDefs.fields[field].invalid && this.dynamicLogicDefs.fields[field].invalid.conditionGroup) {
        const invalidConditionGroup = this.dynamicLogicDefs.fields[field].invalid.conditionGroup;
        const fieldInvalid = this.dynamicLogic.checkConditionGroup(invalidConditionGroup);
        notValid = fieldInvalid || notValid;
        if (fieldInvalid) {
          fieldView.showValidationMessage(msg);
          fieldView.trigger('invalid');
        }
      }
      return notValid;
    }

    /**
     * Processed after save.
     *
     * @protected
     */
    afterSave() {
      if (this.isNew) {
        Espo.Ui.success(this.translate('Created'));
      } else {
        Espo.Ui.success(this.translate('Saved'));
      }
      this.setIsNotChanged();
    }

    /**
     * Processed before before-save.
     *
     * @protected
     */
    beforeBeforeSave() {}

    /**
     * Processed before save.
     *
     * @protected
     */
    beforeSave() {
      Espo.Ui.notify(this.translate('saving', 'messages'));
    }

    /**
     * Processed after save error.
     *
     * @protected
     */
    afterSaveError() {}

    /**
     * Processed after save a not modified record.
     *
     * @protected
     */
    afterNotModified() {
      Espo.Ui.warning(this.translate('notModified', 'messages'));
      this.setIsNotChanged();
    }

    /**
     * Processed after save not valid.
     *
     * @protected
     */
    afterNotValid() {
      Espo.Ui.error(this.translate('Not valid'));
    }

    /**
     * Get changed attribute values. For new record, returns all attributes.
     *
     * @protected
     * @param {string[]} [attributeList]
     * @return {Record}
     */
    getChangedAttributes(attributeList = null) {
      const attributes = this.model.getClonedAttributes();
      if (this.model.isNew()) {
        return attributes;
      }
      const setAttributes = {};
      for (const attr in attributes) {
        if (Espo.Utils.areEqual(this.attributes[attr], attributes[attr])) {
          continue;
        }
        setAttributes[attr] = attributes[attr];
      }

      /** @type {Record.<string, string[]>} */
      const map = this.forcePatchAttributeDependencyMap || {};
      for (const attr in map) {
        if (attr in setAttributes) {
          continue;
        }
        if (attributeList && !attributeList.includes(attr)) {
          continue;
        }
        const depAttributes = map[attr];
        const treatAsChanged = !!depAttributes.find(attr => attr in setAttributes);
        if (treatAsChanged) {
          setAttributes[attr] = attributes[attr];
        }
      }
      return setAttributes;
    }

    /**
     * Save options.
     *
     * @typedef {Object} module:views/record/base~saveOptions
     *
     * @property {Object.<string,string>} [headers] HTTP headers.
     * @property {boolean} [skipNotModifiedWarning] Don't show a not-modified warning.
     * @property {function():void} [afterValidate] A callback called after validate.
     * @property {boolean} [bypassClose] Bypass closing. Only for inline-edit.
     */

    /**
     * Save.
     *
     * @param {module:views/record/base~saveOptions} [options] Options.
     * @return {Promise}
     */
    save(options) {
      options = options || {};
      const headers = options.headers || {};
      const model = this.model;
      this.lastSaveCancelReason = null;
      this.beforeBeforeSave();

      // A model is supposed to always contain actual values.
      // Fetch may not be needed, but some field views may not have data sync implemented.
      // We resort to fetching the entire form.

      const fetchedAttributes = this.fetch();
      this.model.set(fetchedAttributes, {
        silent: true
      });
      const setAttributes = this.getChangedAttributes(Object.keys(fetchedAttributes));
      if (Object.keys(setAttributes).length === 0) {
        if (!options.skipNotModifiedWarning) {
          this.afterNotModified();
        }
        this.lastSaveCancelReason = 'notModified';
        this.trigger('cancel:save', {
          reason: 'notModified'
        });
        return Promise.reject('notModified');
      }
      if (this.validate()) {
        this.afterNotValid();
        this.lastSaveCancelReason = 'invalid';
        this.trigger('cancel:save', {
          reason: 'invalid'
        });
        return Promise.reject('invalid');
      }
      if (options.afterValidate) {
        options.afterValidate();
      }
      const optimisticConcurrencyControl = this.getMetadata().get(['entityDefs', this.entityType, 'optimisticConcurrencyControl']);
      if (optimisticConcurrencyControl && this.model.get('versionNumber') !== null) {
        headers['X-Version-Number'] = this.model.get('versionNumber');
      }
      if (this.model.isNew() && this.options.duplicateSourceId) {
        headers['X-Duplicate-Source-Id'] = this.options.duplicateSourceId;
      }
      this.beforeSave();
      this.trigger('before:save');
      model.trigger('before:save');
      const initialAttributes = this.attributes;
      return new Promise((resolve, reject) => {
        model.save(setAttributes, {
          patch: !model.isNew(),
          headers: headers
        }).then(() => {
          this.trigger('save', initialAttributes, Object.keys(setAttributes));
          this.afterSave();
          if (this.isNew) {
            this.isNew = false;
          }
          this.trigger('after:save');
          model.trigger('after:save');
          resolve();
        }).catch(xhr => {
          this.handleSaveError(xhr, options, resolve, reject).then(skipReject => {
            if (skipReject) {
              return;
            }
            reject('error');
          });
          this.afterSaveError();
          this.lastSaveCancelReason = 'error';
          this.trigger('error:save');
          this.trigger('cancel:save', {
            reason: 'error'
          });
        });
      });
    }

    /**
     * Handle a save error.
     *
     * @param {module:ajax.Xhr} xhr XHR.
     * @param {module:views/record/base~saveOptions} [options] Options.
     * @param {function} saveResolve Resolve the save promise.
     * @param {function} saveReject Reject the same promise.
     * @return {Promise<boolean>}
     *
     * @protected
     */
    handleSaveError(xhr, options, saveResolve, saveReject) {
      let handlerData = null;
      if (~[409, 500].indexOf(xhr.status)) {
        const statusReason = xhr.getResponseHeader('X-Status-Reason');
        if (!statusReason) {
          return Promise.resolve(false);
        }
        try {
          handlerData = JSON.parse(statusReason);
        } catch (e) {}
        if (!handlerData) {
          handlerData = {
            reason: statusReason.toString()
          };
          if (xhr.responseText) {
            let data;
            try {
              data = JSON.parse(xhr.responseText);
            } catch (e) {
              console.error('Could not parse error response body.');
              return Promise.resolve(false);
            }
            handlerData.data = data;
          }
        }
      }
      if (!handlerData || !handlerData.reason) {
        return Promise.resolve(false);
      }
      const reason = handlerData.reason;
      const handlerName = this.getMetadata().get(['clientDefs', this.scope, 'saveErrorHandlers', reason]) || this.getMetadata().get(['clientDefs', 'Global', 'saveErrorHandlers', reason]);
      return new Promise(resolve => {
        if (handlerName) {
          Espo.loader.require(handlerName, Handler => {
            const handler = new Handler(this);
            handler.process(handlerData.data, options);
            resolve(false);
          });
          xhr.errorIsHandled = true;
          return;
        }
        const methodName = 'errorHandler' + Espo.Utils.upperCaseFirst(reason);
        if (methodName in this) {
          xhr.errorIsHandled = true;
          const skipReject = this[methodName](handlerData.data, options, saveResolve, saveReject);
          resolve(skipReject || false);
          return;
        }
        resolve(false);
      });
    }

    /**
     * Fetch data from the form.
     *
     * @return {Object.<string, *>}
     */
    fetch() {
      let data = {};
      const fieldViews = this.getFieldViews();
      for (const i in fieldViews) {
        const view = fieldViews[i];
        if (!view.isEditMode()) {
          continue;
        }
        if (!view.disabled && !view.readOnly && view.isFullyRendered()) {
          data = {
            ...data,
            ...view.fetch()
          };
        }
      }
      return data;
    }

    /**
     * Process fetch.
     *
     * @return {Object<string,*>|null}
     */
    processFetch() {
      const data = this.fetch();
      this.model.set(data);
      if (this.validate()) {
        return null;
      }
      return data;
    }

    /**
     * Populate defaults.
     *
     * @return {Promise|undefined}
     */
    populateDefaults() {
      const populator = new _defaultsPopulator.default();
      return populator.populate(this.model);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @protected
     * @param duplicates
     */
    errorHandlerDuplicate(duplicates) {}

    /**
     * @private
     */
    _handleDependencyAttributes() {
      // noinspection JSDeprecatedSymbols
      Object.keys(this.dependencyDefs || {}).forEach(attr => {
        this._handleDependencyAttribute(attr);
      });
    }

    /**
     * @private
     */
    _handleDependencyAttribute(attr) {
      // noinspection JSDeprecatedSymbols
      const data = this.dependencyDefs[attr];
      const value = this.model.get(attr);
      if (value in (data.map || {})) {
        (data.map[value] || []).forEach(item => {
          this._doDependencyAction(item);
        });
        return;
      }
      if ('default' in data) {
        (data.default || []).forEach(item => {
          this._doDependencyAction(item);
        });
      }
    }

    /**
     * @private
     */
    _doDependencyAction(data) {
      const action = data.action;
      const methodName = 'dependencyAction' + Espo.Utils.upperCaseFirst(action);
      if (methodName in this && typeof this.methodName === 'function') {
        this.methodName(data);
        return;
      }
      const fieldList = data.fieldList || data.fields || [];
      const panelList = data.panelList || data.panels || [];
      switch (action) {
        case 'hide':
          panelList.forEach(item => {
            this.hidePanel(item);
          });
          fieldList.forEach(item => {
            this.hideField(item);
          });
          break;
        case 'show':
          panelList.forEach(item => {
            this.showPanel(item);
          });
          fieldList.forEach(item => {
            this.showField(item);
          });
          break;
        case 'setRequired':
          fieldList.forEach(field => {
            this.setFieldRequired(field);
          });
          break;
        case 'setNotRequired':
          fieldList.forEach(field => {
            this.setFieldNotRequired(field);
          });
          break;
        case 'setReadOnly':
          fieldList.forEach(field => {
            this.setFieldReadOnly(field);
          });
          break;
        case 'setNotReadOnly':
          fieldList.forEach(field => {
            this.setFieldNotReadOnly(field);
          });
          break;
      }
    }

    /**
     * Create a field view.
     *
     * @protected
     * @param {string} name A field name.
     * @param {string|null} [view] A view name/path.
     * @param {Object<string,*>} [params] Field params.
     * @param {'detail'|'edit'} [mode='edit'] A mode.
     * @param {boolean} [readOnly] Read-only.
     * @param {Object<string,*>} [options] View options.
     */
    createField(name, view, params, mode, readOnly, options) {
      const o = {
        model: this.model,
        mode: mode || 'edit',
        selector: '.field[data-name="' + name + '"]',
        defs: {
          name: name,
          params: params || {}
        }
      };
      if (readOnly) {
        o.readOnly = true;
      }
      view = view || this.model.getFieldParam(name, 'view');
      if (!view) {
        const type = this.model.getFieldType(name) || 'base';
        view = this.getFieldManager().getViewName(type);
      }
      if (options) {
        for (const param in options) {
          o[param] = options[param];
        }
      }
      if (this.recordHelper.getFieldStateParam(name, 'hidden')) {
        o.disabled = true;
      }
      if (this.recordHelper.getFieldStateParam(name, 'readOnly')) {
        o.readOnly = true;
      }
      if (this.recordHelper.getFieldStateParam(name, 'required') !== null) {
        o.defs.params.required = this.recordHelper.getFieldStateParam(name, 'required');
      }
      if (this.recordHelper.hasFieldOptionList(name)) {
        o.customOptionList = this.recordHelper.getFieldOptionList(name);
      }
      const viewKey = name + 'Field';
      this.createView(viewKey, view, o);
      if (!~this.fieldList.indexOf(name)) {
        this.fieldList.push(name);
      }
    }

    /**
     * Get a currently focused field view.
     *
     * @return {module:views/fields/base|null}
     */
    getFocusedFieldView() {
      const $active = (0, _jquery.default)(window.document.activeElement);
      if (!$active.length) {
        return null;
      }
      const $field = $active.closest('.field');
      if (!$field.length) {
        return null;
      }
      const name = $field.attr('data-name');
      if (!name) {
        return null;
      }
      return this.getFieldView(name);
    }

    /**
     * Process exit.
     *
     * @param {string} [after] An exit parameter.
     */
    exit(after) {}
  }
  var _default = _exports.default = BaseRecordView;
});

define("ui/select", ["exports", "lib!selectize"], function (_exports, _libSelectize) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _libSelectize = _interopRequireDefault(_libSelectize);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module module:ui/select */

  /**
   * @typedef module:ui/select~Options
   * @type {Object}
   * @property {boolean} [selectOnTab=false] To select on tab.
   * @property {boolean} [matchAnyWord=false] To match any word when searching.
   * @property {function(string, module:ui/select~OptionItemsCallback): void} [load] Loads additional items
   *   when typing in search.
   * @property {function(string, module:ui/select~OptionItemFull): Number} [score] A score function scoring
   *   searched items.
   * @property {'value'|'text'|'$order'|'$score'} [sortBy='$order'] Item sorting.
   * @property {'asc'|'desc'} [sortDirection='asc'] Sort direction.
   * @property {function()} [onFocus] On-focus callback.
   */

  /**
   * @callback  module:ui/select~OptionItemsCallback
   * @param {module:ui/select~OptionItem[]} list An option item list.
   */

  /**
   * @typedef module:ui/select~OptionItem
   * @type {Object}
   * @property {string} value A value.
   * @property {string} text A label.
   */

  /**
   * @typedef module:ui/select~OptionItemFull
   * @type {Object}
   * @property {string} value A value.
   * @property {string} text A label.
   * @property {Number} $order An order index.
   */

  /**
   * @module ui/select
   *
   * Important. The Selectize library is heavily customized to fix multitude of UIX issues.
   * Upgrading is not advisable. Consider forking.
   */
  const Select = {
    /**
     * @param {Element|JQuery} element An element.
     * @param {module:ui/select~Options} [options] Options.
     */
    init: function (element, options = {}) {
      const score = options.score;
      const $el = $(element);
      options = Select.applyDefaultOptions(options || {});
      const plugins = [];
      Select.loadEspoSelectPlugin();
      plugins.push('espo_select');
      const itemClasses = {};
      const allowedValues = $el.children().toArray().map(item => {
        const value = item.getAttributeNode('value').value;
        if (item.classList) {
          itemClasses[value] = item.classList.toString();
        }
        return value;
      });
      let $relativeParent = null;
      const $modalBody = $el.closest('.modal-body');
      if ($modalBody.length) {
        $relativeParent = $modalBody;
      }

      // noinspection JSUnusedGlobalSymbols
      const selectizeOptions = {
        sortField: [{
          field: options.sortBy,
          direction: options.sortDirection
        }],
        load: options.load,
        loadThrottle: 1,
        plugins: plugins,
        highlight: false,
        selectOnTab: options.selectOnTab,
        copyClassesToDropdown: false,
        allowEmptyOption: allowedValues.includes(''),
        showEmptyOptionInDropdown: true,
        $relativeParent: $relativeParent,
        render: {
          item: function (data) {
            return $('<div>').addClass('item').addClass(itemClasses[data.value] || '').text(data.text).get(0).outerHTML;
          },
          option: function (data) {
            const $div = $('<div>').addClass('option').addClass(data.value === '' ? 'selectize-dropdown-emptyoptionlabel' : '').addClass(itemClasses[data.value] || '').val(data.value).text(data.text);
            if (data.text === '') {
              $div.html('&nbsp;');
            }
            return $div.get(0).outerHTML;
          }
        },
        onDelete: function (values) {
          while (values.length) {
            this.removeItem(values.pop(), true);
          }

          // noinspection JSUnresolvedReference
          this.showInput();
          this.positionDropdown();
          this.refreshOptions(true);
        }
      };
      if (options.onFocus) {
        selectizeOptions.onFocus = function () {
          options.onFocus();
        };
      }
      if (!options.matchAnyWord) {
        /** @this Selectize */
        selectizeOptions.score = function (search) {
          // noinspection JSUnresolvedReference
          const score = this.getScoreFunction(search);
          search = search.toLowerCase();
          return function (item) {
            if (item.text.toLowerCase().indexOf(search) === 0) {
              return score(item);
            }
            return 0;
          };
        };
      }
      if (options.matchAnyWord) {
        /** @this Selectize */
        selectizeOptions.score = function (search) {
          // noinspection JSUnresolvedReference
          const score = this.getScoreFunction(search);
          search = search.toLowerCase();
          return function (item) {
            const text = item.text.toLowerCase();
            if (!text.split(' ').find(item => item.startsWith(search)) && !text.startsWith(search)) {
              return 0;
            }
            return score(item);
          };
        };
      }
      if (options.score) {
        selectizeOptions.score = function (search) {
          return function (item) {
            return score(search, item);
          };
        };
      }
      $el.selectize(selectizeOptions);
    },
    /**
     * Focus.
     *
     * @param {Element|JQuery} element An element.
     * @param {{noTrigger?: boolean}} [options] Options.
     */
    focus: function (element, options) {
      const $el = $(element);
      options = options || {};
      if (!$el[0] || !$el[0].selectize) {
        return;
      }
      const selectize = $el[0].selectize;
      if (options.noTrigger) {
        selectize.focusNoTrigger = true;
      }
      selectize.focus();
      if (options.noTrigger) {
        setTimeout(() => selectize.focusNoTrigger = false, 100);
      }
    },
    /**
     * Set options.
     * @todo Side effects may occur if called multiple times. Workaround is to clone and re-initialize. To be fixed.
     *
     * @param {Element|JQuery} element An element.
     * @param {{value: string, text: string}[]} options Options.
     */
    setOptions: function (element, options) {
      const $el = $(element);
      const selectize = $el.get(0).selectize;
      selectize.clearOptions(true);
      selectize.load(callback => {
        callback(options.map(item => {
          return {
            value: item.value,
            text: item.text || item.label
          };
        }));
      });
    },
    /**
     * Set value.
     *
     * @param {HTMLElement|JQuery} element An element.
     * @param {string} value A value.
     */
    setValue: function (element, value) {
      if (!(element instanceof HTMLElement)) {
        element = $(element).get(0);
      }
      const selectize = element.selectize;
      selectize.setValue(value, true);
    },
    /**
     * Destroy.
     *
     * @param {HTMLElement|JQuery} element An element.
     */
    destroy: function (element) {
      if (!element) {
        return;
      }
      if (!(element instanceof HTMLElement)) {
        element = $(element).get(0);
      }
      if (!element || !element.selectize) {
        return;
      }
      element.selectize.destroy();
    },
    /**
     * @private
     * @param {module:ui/select~Options} options
     * @return {module:ui/select~Options}
     */
    applyDefaultOptions: function (options) {
      options = Espo.Utils.clone(options);
      const defaults = {
        selectOnTab: false,
        matchAnyWord: false,
        sortBy: '$order',
        sortDirection: 'asc'
      };
      for (const key in defaults) {
        if (key in options) {
          continue;
        }
        options[key] = defaults[key];
      }
      return options;
    },
    /**
     * @private
     */
    loadEspoSelectPlugin: function () {
      if ('espo_select' in _libSelectize.default.plugins) {
        return;
      }
      const IS_MAC = /Mac/.test(navigator.userAgent);
      const KEY_BACKSPACE = 8;
      _libSelectize.default.define('espo_select', function () {
        const self = this;
        this.setup = function () {
          const original = self.setup;
          return function () {
            original.apply(this, arguments);
            self.selectedValue = self.items[0];
            self.$dropdown.on('mouseup', '[data-selectable]', function () {
              $(document).off('mouseup.select');
              return self.onOptionSelect.apply(self, arguments);
            });
            self.$dropdown.on('mousedown', '[data-selectable]', function () {
              // Prevent issue when down inside, up outside.
              $(document).one('mouseup.select', function () {
                self.focusOnControlSilently();
              });
            });
            self.$control_input.css({
              'width': '4px'
            });
          };
        }();
        this.focusOnControlSilently = function () {
          self.preventReOpenOnFocus = true;
          self.$control_input[0].focus();
          self.preventReOpenOnFocus = false;
        };

        /*this.positionDropdown = (function () {
            let original = self.positionDropdown;
              return function () {
                original.apply(this, arguments);
                  this.$dropdown.css({margin: 'unset'});
            };
        })();*/

        this.refreshOptions = function () {
          const original = self.refreshOptions;
          return function () {
            if (self.focusNoTrigger) {
              original.apply(this, [false]);
              return;
            }
            original.apply(this, arguments);
          };
        }();
        this.blur = function () {
          const original = self.blur;
          return function () {
            // Prevent closing on mouse down.
            if (self.preventClose) {
              return;
            }
            original.apply(this, arguments);
          };
        }();
        this.close = function () {
          const original = self.close;
          return function () {
            if (self.preventClose) {
              return;
            }
            original.apply(this, arguments);
          };
        }();
        this.onOptionSelect = function () {
          const original = self.onOptionSelect;
          return function (e) {
            if (e.type === 'mousedown' || e.type === 'click') {
              self.preventClose = true;
              setTimeout(() => self.preventClose = false, 100);
              return;
            }
            self.preventClose = false;
            if (e.type === 'mouseup') {
              setTimeout(() => self.focusOnControlSilently(), 50);
            }
            original.apply(this, arguments);
            self.selectedValue = $(e.currentTarget).attr('data-value');
          };
        }();
        this.open = function () {
          const original = self.open;
          return function () {
            const toProcess = !(self.isLocked || self.isOpen);
            original.apply(this, arguments);
            if (!toProcess) {
              return;
            }
            const $dropdownContent = self.$dropdown.children().first();
            const $selected = $dropdownContent.find('.selected');
            if (!$selected.length) {
              return;
            }
            let scrollTo = $selected.get(0).offsetTop - $dropdownContent.get(0).clientHeight;
            scrollTo = scrollTo >= 0 ? scrollTo : 0;
            $dropdownContent.find('.selectize-dropdown-content').scrollTop(scrollTo);
          };
        }();
        this.onMouseDown = function () {
          const original = self.onMouseDown;
          return function (e) {
            // Prevent flicking when clicking on input.
            if (!self.isOpen && !self.isInputHidden && self.$control_input.val()) {
              return;
            }
            if (self.isOpen) {
              self.closedByMouseDown = true;
            }
            return original.apply(this, arguments);
          };
        }();
        this.onFocus = function () {
          const original = self.onFocus;
          return function (e) {
            if (self.preventReOpenOnFocus) {
              return;
            }
            if (self.closedByMouseDown) {
              self.closedByMouseDown = false;
              return;
            }
            self.closedByMouseDown = false;
            return original.apply(this, arguments);
          };
        }();
        this.restoreSelectedValue = function () {
          if (this.preventRevertLoop) {
            return;
          }
          this.preventRevertLoop = true;
          setTimeout(() => this.preventRevertLoop = false, 10);
          this.setValue(this.selectedValue, true);
        };
        this.onBlur = function () {
          const original = self.onBlur;
          return function () {
            // Prevent closing on mouse down.
            if (self.preventClose) {
              return;
            }
            self.restoreSelectedValue();
            self.$control_input.css({
              width: '4px'
            });
            return original.apply(this, arguments);
          };
        }();
        this.onKeyDown = function () {
          const original = self.onKeyDown;
          return function (e) {
            if (IS_MAC ? e.metaKey : e.ctrlKey) {
              if (!self.items.length) {
                self.restoreSelectedValue();
                self.focus();
              }
              return;
            }
            if (e.code === 'Escape') {
              if (self.isOpen || !self.isInputHidden) {
                e.stopPropagation();
              }
              if (self.isOpen) {
                self.close();
              }
              if (!self.isInputHidden) {
                self.hideInput();
              }
              self.addItem(this.selectedValue, true);
            }
            if (self.isFull() || self.isInputHidden) {
              if (e.key.length === 1 && (e.code.match(/Key[A-Z]/i) || e.key.match(/[0-9]/) || RegExp(/^\p{L}/, 'u').test(e.key) // is letter
              )) {
                const keyCode = e.keyCode;
                e.keyCode = KEY_BACKSPACE;
                self.deleteSelection(e);
                e.keyCode = keyCode;
                self.$control_input.width(15);
              }
            }
            return original.apply(this, arguments);
          };
        }();
        this.positionDropdown = function () {
          const POSITION = {
            top: 'top',
            bottom: 'bottom'
          };
          return function () {
            const $control = self.$control;
            const offset = this.settings.dropdownParent === 'body' ? $control.offset() : $control.position();
            offset.top += $control.outerHeight(true);
            const dropdownHeight = self.$dropdown.prop('scrollHeight') + 5;
            const controlPosTop = self.$control.get(0).getBoundingClientRect().top;
            const wrapperHeight = self.$wrapper.height();
            const controlPosBottom = self.$control.get(0).getBoundingClientRect().bottom;
            const boundaryTop = !this.settings.$relativeParent ? 0 : this.settings.$relativeParent.get(0).getBoundingClientRect().top;
            const position = controlPosTop + dropdownHeight + wrapperHeight > window.innerHeight && controlPosBottom - dropdownHeight - wrapperHeight >= boundaryTop ? POSITION.top : POSITION.bottom;
            const styles = {
              width: $control.outerWidth(),
              left: offset.left
            };
            if (position === POSITION.top) {
              Object.assign(styles, {
                bottom: offset.top,
                top: 'unset',
                margin: '0 0 0 0'
              });
              self.$dropdown.addClass('selectize-position-top');
            } else {
              Object.assign(styles, {
                top: offset.top,
                bottom: 'unset',
                margin: '0 0 0 0'
              });
              self.$dropdown.removeClass('selectize-position-top');
            }
            self.$dropdown.css(styles);
          };
        }();
      });
    }
  };
  var _default = _exports.default = Select;
});

define("helpers/action-item-setup", ["exports", "di", "metadata", "view-helper", "acl-manager", "language"], function (_exports, _di, _metadata, _viewHelper, _aclManager, _language) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _metadata = _interopRequireDefault(_metadata);
  _viewHelper = _interopRequireDefault(_viewHelper);
  _aclManager = _interopRequireDefault(_aclManager);
  _language = _interopRequireDefault(_language);
  let _init_metadata, _init_extra_metadata, _init_viewHelper, _init_extra_viewHelper, _init_acl, _init_extra_acl, _init_language, _init_extra_language;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /** @module helpers/action-item-setup */

  class ActionItemSetupHelper {
    static {
      [_init_metadata, _init_extra_metadata, _init_viewHelper, _init_extra_viewHelper, _init_acl, _init_extra_acl, _init_language, _init_extra_language] = _applyDecs(this, [], [[(0, _di.inject)(_metadata.default), 0, "metadata"], [(0, _di.inject)(_viewHelper.default), 0, "viewHelper"], [(0, _di.inject)(_aclManager.default), 0, "acl"], [(0, _di.inject)(_language.default), 0, "language"]]).e;
    }
    constructor() {
      _init_extra_language(this);
    }
    /**
     * @private
     * @type {Metadata}
     */
    metadata = _init_metadata(this);

    /**
     * @private
     * @type {ViewHelper}
     */
    viewHelper = (_init_extra_metadata(this), _init_viewHelper(this));

    /**
     * @private
     * @type {AclManager}
     */
    acl = (_init_extra_viewHelper(this), _init_acl(this));

    /**
     * @private
     * @type {Language}
     */
    language = (_init_extra_acl(this), _init_language(this));

    /**
     * @param {module:view} view
     * @param {string} type
     * @param {function(Promise): void} waitFunc
     * @param {function(Object): void} addFunc
     * @param {function(string): void} showFunc
     * @param {function(string): void} hideFunc
     * @param {{listenToViewModelSync?: boolean}} [options]
     */
    setup(view, type, waitFunc, addFunc, showFunc, hideFunc, options) {
      options = options || {};
      const actionList = [];

      // noinspection JSUnresolvedReference
      const scope = view.scope || view.model.entityType;
      if (!scope) {
        throw new Error();
      }
      const actionDefsList = [...(this.metadata.get(['clientDefs', 'Global', type + 'ActionList']) || []), ...(this.metadata.get(['clientDefs', scope, type + 'ActionList']) || [])];
      actionDefsList.forEach(item => {
        if (typeof item === 'string') {
          item = {
            name: item
          };
        }
        item = Espo.Utils.cloneDeep(item);
        const name = item.name;
        if (!item.label) {
          item.html = this.language.translate(name, 'actions', scope);
        }
        item.data = item.data || {};
        const handlerName = item.handler || item.data.handler;
        if (handlerName && !item.data.handler) {
          item.data.handler = handlerName;
        }
        addFunc(item);
        if (!Espo.Utils.checkActionAvailability(this.viewHelper, item)) {
          return;
        }
        if (!Espo.Utils.checkActionAccess(this.acl, view.model, item, true)) {
          item.hidden = true;
        }
        actionList.push(item);
        if (!handlerName) {
          return;
        }
        if (!item.initFunction && !item.checkVisibilityFunction) {
          return;
        }
        waitFunc(new Promise(resolve => {
          Espo.loader.require(handlerName, Handler => {
            const handler = new Handler(view);
            if (item.initFunction) {
              handler[item.initFunction].call(handler);
            }
            if (item.checkVisibilityFunction) {
              const isNotVisible = !handler[item.checkVisibilityFunction].call(handler);
              if (isNotVisible) {
                hideFunc(item.name);
              }
            }
            item.handlerInstance = handler;
            resolve();
          });
        }));
      });
      if (!actionList.length) {
        return;
      }
      const onSync = () => {
        actionList.forEach(item => {
          if (item.handlerInstance && item.checkVisibilityFunction) {
            const isNotVisible = !item.handlerInstance[item.checkVisibilityFunction].call(item.handlerInstance);
            if (isNotVisible) {
              hideFunc(item.name);
              return;
            }
          }
          if (Espo.Utils.checkActionAccess(this.acl, view.model, item, true)) {
            showFunc(item.name);
            return;
          }
          hideFunc(item.name);
        });
      };
      if (options.listenToViewModelSync) {
        view.listenTo(view, 'model-sync', () => onSync());
        return;
      }
      view.listenTo(view.model, 'sync', () => onSync());
    }
  }
  var _default = _exports.default = ActionItemSetupHelper;
});

define("helpers/record/misc/sticky-bar", ["exports", "jquery"], function (_exports, _jquery) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _jquery = _interopRequireDefault(_jquery);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /**
   * @internal
   */
  class StickyBarHelper {
    /**
     * @param {import('views/record/detail').default} view
     * @param {boolean} stickButtonsFormBottomSelector
     * @param {boolean} stickButtonsContainerAllTheWay
     * @param {number} numId
     */
    constructor(view, stickButtonsFormBottomSelector, stickButtonsContainerAllTheWay, numId) {
      this.view = view;
      this.stickButtonsFormBottomSelector = stickButtonsFormBottomSelector;
      this.stickButtonsContainerAllTheWay = stickButtonsContainerAllTheWay;
      this.numId = numId;
      this.themeManager = view.getThemeManager();
      this.$el = view.$el;
    }
    init() {
      const $containers = this.$el.find('.detail-button-container');
      const $container = this.$el.find('.detail-button-container.record-buttons');
      if (!$container.length) {
        return;
      }
      const navbarHeight = this.themeManager.getParam('navbarHeight') * this.themeManager.getFontSizeFactor();
      const screenWidthXs = this.themeManager.getParam('screenWidthXs');
      const isSmallScreen = (0, _jquery.default)(window.document).width() < screenWidthXs;
      const getOffsetTop = (/** JQuery */$element) => {
        let element = /** @type {HTMLElement} */$element.get(0);
        let value = 0;
        while (element) {
          value += !isNaN(element.offsetTop) ? element.offsetTop : 0;
          element = element.offsetParent;
        }
        if (isSmallScreen) {
          return value;
        }
        return value - navbarHeight;
      };
      let stickTop = getOffsetTop($container);
      const blockHeight = $container.outerHeight();
      stickTop -= 5; // padding;

      const $block = (0, _jquery.default)('<div>').css('height', blockHeight + 'px').html('&nbsp;').hide().insertAfter($container);
      let $middle = this.view.getMiddleView().$el;
      const $window = (0, _jquery.default)(window);
      const $navbarRight = (0, _jquery.default)('#navbar .navbar-right');
      if (this.stickButtonsFormBottomSelector) {
        const $bottom = this.$el.find(this.stickButtonsFormBottomSelector);
        if ($bottom.length) {
          $middle = $bottom;
        }
      }
      $window.off('scroll.detail-' + this.numId);
      $window.on('scroll.detail-' + this.numId, () => {
        const edge = $middle.position().top + $middle.outerHeight(false) - blockHeight;
        const scrollTop = $window.scrollTop();
        if (scrollTop >= edge && !this.stickButtonsContainerAllTheWay) {
          $containers.hide();
          $navbarRight.removeClass('has-sticked-bar');
          $block.show();
          return;
        }
        if (isSmallScreen && (0, _jquery.default)('#navbar .navbar-body').hasClass('in')) {
          return;
        }
        if (scrollTop > stickTop) {
          if (!$containers.hasClass('stick-sub')) {
            $containers.addClass('stick-sub');
            $block.show();
          }
          $navbarRight.addClass('has-sticked-bar');
          $containers.show();
          return;
        }
        if ($containers.hasClass('stick-sub')) {
          $containers.removeClass('stick-sub');
          $navbarRight.removeClass('has-sticked-bar');
          $block.hide();
        }
        $containers.show();
      });
    }
  }
  var _default = _exports.default = StickyBarHelper;
});

define("views/record/detail", ["exports", "views/record/base", "view-record-helper", "helpers/action-item-setup", "helpers/record/misc/sticky-bar"], function (_exports, _base, _viewRecordHelper, _actionItemSetup, _stickyBar) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _base = _interopRequireDefault(_base);
  _viewRecordHelper = _interopRequireDefault(_viewRecordHelper);
  _actionItemSetup = _interopRequireDefault(_actionItemSetup);
  _stickyBar = _interopRequireDefault(_stickyBar);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/record/detail */

  /**
   * A detail record view.
   */
  class DetailRecordView extends _base.default {
    /**
     * @typedef {Object} module:views/record/detail~options
     *
     * @property {module:model} model A model.
     * @property {string} [scope] A scope.
     * @property {string} [layoutName] A layout name.
     * @property {module:views/record/detail~panelDefs[]} [detailLayout] A detail layout.
     * @property {boolean} [readOnly] Read-only.
     * @property {string} [rootUrl]
     * @property {string} [returnUrl]
     * @property {boolean} [returnAfterCreate]
     * @property {boolean} [editModeDisabled]
     * @property {boolean} [confirmLeaveDisabled]
     * @property {boolean} [isWide]
     * @property {string|null} [sideView]
     * @property {string|null} [bottomView]
     * @property {string} [inlineEditDisabled] Disable inline edit.
     * @property {boolean} [buttonsDisabled] Disable buttons.
     * @property {string} [navigateButtonsDisabled]
     * @property {Object} [dynamicLogicDefs]
     * @property {module:view-record-helper} [recordHelper] A record helper. For a form state management.
     * @property {Object.<string, *>} [attributes]
     * @property {module:views/record/detail~button[]} [buttonList] Buttons.
     * @property {module:views/record/detail~dropdownItem[]} [dropdownItemList] Dropdown items.
     * @property {Object.<string, *>} [dataObject] Additional data.
     * @property {Record} [rootData] Data from the root view.
     * @property {boolean} [shortcutKeysEnabled] Enable shortcut keys.
     */

    /**
     * @param {module:views/record/detail~options | Object.<string, *>} options Options.
     */
    constructor(options) {
      super(options);
    }

    /** @inheritDoc */
    template = 'record/detail';

    /** @inheritDoc */
    type = 'detail';

    /**
     * A layout name. Can be overridden by an option parameter.
     *
     * @protected
     * @type {string}
     */
    layoutName = 'detail';

    /**
     * Panel definitions.
     *
     * @typedef {Object} module:views/record/detail~panelDefs
     * @property {string} [label] A translatable label.
     * @property {string} [customLabel] A custom label.
     * @property {string} [name] A name. Useful to be able to show/hide by a name.
     * @property {'default'|'success'|'danger'|'warning'|'info'} [style] A style.
     * @property {boolean} [tabBreak] Is a tab-break.
     * @property {string} [tabLabel] A tab label. If starts with `$`, a translation
     *   of the `tabs` category is used.
     * @property {module:views/record/detail~rowDefs[]} [rows] Rows.
     * @property {module:views/record/detail~rowDefs[]} [columns] Columns.
     * @property {string} [noteText] A note text.
     * @property {'success'|'danger'|'warning'|'info'} [noteStyle] A note style.
     */

    /**
     * A row.
     *
     * @typedef {Array<module:views/record/detail~cellDefs|false>} module:views/record/detail~rowDefs
     */

    /**
     * Cell definitions.
     *
     * @typedef {Object} module:views/record/detail~cellDefs
     * @property {string} [name] A name (usually a field name).
     * @property {string|module:views/fields/base} [view] An overridden field view name or a view instance.
     * @property {string} [type] An overridden field type.
     * @property {boolean} [readOnly] Read-only.
     * @property {boolean} [inlineEditDisabled] Disable inline edit.
     * @property {Object.<string, *>} [params] Overridden field parameters.
     * @property {Object.<string, *>} [options] Field view options.
     * @property {string} [labelText] A label text (not-translatable).
     * @property {boolean} [noLabel] No label.
     * @property {string} [label] A translatable label (using the `fields` category).
     * @property {1|2|3|4} [span] A width.
     */

    /**
     * A layout. If null, then will be loaded from the backend (using the `layoutName` value).
     * Can be overridden by an option parameter.
     *
     * @protected
     * @type {module:views/record/detail~panelDefs[]|null}
     */
    detailLayout = null;

    /**
     * A fields mode.
     *
     * @protected
     * @type {'detail'|'edit'|'list'}
     */
    fieldsMode = 'detail';

    /**
     * A current mode. Only for reading.
     *
     * @protected
     * @type {'detail'|'edit'}
     */
    mode = 'detail';

    /**
     * @private
     */
    gridLayout = null;

    /**
     * Disable buttons. Can be overridden by an option parameter.
     *
     * @protected
     * @type {boolean}
     */
    buttonsDisabled = false;

    /**
     * Is record new. Only for reading.
     *
     * @protected
     */
    isNew = false;

    /**
     * A button. Handled by an `action{Name}` method, a click handler or a handler class.
     *
     * @typedef module:views/record/detail~button
     *
     * @property {string} name A name.
     * @property {string} [label] A label.
     * @property {string} [labelTranslation] A label translation path.
     * @property {string} [html] An HTML.
     * @property {string} [text] A text.
     * @property {'default'|'danger'|'success'|'warning'|'primary'} [style] A style.
     * @property {boolean} [hidden] Hidden.
     * @property {string} [title] A title (not translatable).
     * @property {boolean} [disabled] Disabled.
     * @property {function()} [onClick] A click handler.
     */

    /**
     * A dropdown item. Handled by an `action{Name}` method, a click handler or a handler class.
     *
     * @typedef module:views/record/detail~dropdownItem
     *
     * @property {string} name A name.
     * @property {string} [label] A label.
     * @property {string} [labelTranslation] A label translation path.
     * @property {string} [html] An HTML.
     * @property {string} [text] A text.
     * @property {boolean} [hidden] Hidden.
     * @property {Object.<string, string>} [data] Data attributes.
     * @property {string} [title] A title (not translatable).
     * @property {boolean} [disabled] Disabled.
     * @property {function()} [onClick] A click handler.
     * @property {number} [groupIndex] A group index.
     */

    /**
     * A button list.
     *
     * @protected
     * @type {module:views/record/detail~button[]}
     */
    buttonList = [{
      name: 'edit',
      label: 'Edit',
      title: 'Ctrl+Space'
    }];

    /**
     * A dropdown item list.
     *
     * @protected
     * @type {Array<module:views/record/detail~dropdownItem>}
     */
    dropdownItemList = [{
      name: 'delete',
      label: 'Remove',
      groupIndex: 0
    }];

    /**
     * A button list for edit mode.
     *
     * @protected
     * @type {module:views/record/detail~button[]}
     */
    buttonEditList = [{
      name: 'save',
      label: 'Save',
      style: 'primary',
      edit: true,
      title: 'Ctrl+Enter'
    }, {
      name: 'cancelEdit',
      label: 'Cancel',
      edit: true,
      title: 'Esc'
    }];

    /**
     * A dropdown item list for edit mode.
     *
     * @protected
     * @type {module:views/record/detail~dropdownItem[]}
     */
    dropdownEditItemList = [];

    /**
     * All action items disabled;
     *
     * @protected
     */
    allActionItemsDisabled = false;

    /**
     * A DOM element ID. Only for reading.
     *
     * @private
     * @type {string|null}
     */
    id = null;

    /**
     * A return-URL. Can be overridden by an option parameter.
     *
     * @protected
     * @type {string|null}
     */
    returnUrl = null;

    /**
     * A return dispatch params. Can be overridden by an option parameter.
     *
     * @protected
     * @type {Object|null}
     */
    returnDispatchParams = null;

    /**
     * A middle view name.
     *
     * @protected
     */
    middleView = 'views/record/detail-middle';

    /**
     * A side view name.
     *
     * @protected
     */
    sideView = 'views/record/detail-side';

    /**
     * A bottom view name.
     *
     * @protected
     */
    bottomView = 'views/record/detail-bottom';

    /**
     * Disable a side view. Can be overridden by an option parameter.
     *
     * @protected
     */
    sideDisabled = false;

    /**
     * Disable a bottom view. Can be overridden by an option parameter.
     *
     * @protected
     */
    bottomDisabled = false;

    /**
     * @protected
     */
    gridLayoutType = 'record';

    /**
     * Disable edit mode. Can be overridden by an option parameter.
     *
     * @protected
     */
    editModeDisabled = false;

    /**
     * Disable navigate (prev, next) buttons. Can be overridden by an option parameter.
     *
     * @protected
     */
    navigateButtonsDisabled = false;

    /**
     * Read-only. Can be overridden by an option parameter.
     */
    readOnly = false;

    /**
     * Middle view expanded to full width (no side view).
     * Can be overridden by an option parameter.
     *
     * @protected
     */
    isWide = false;

    /**
     * Enable a duplicate action.
     *
     * @protected
     */
    duplicateAction = true;

    /**
     * Enable a self-assign action.
     *
     * @protected
     */
    selfAssignAction = false;

    /**
     * Enable a print-pdf action.
     *
     * @protected
     */
    printPdfAction = true;

    /**
     * Enable a convert-currency action.
     *
     * @protected
     */
    convertCurrencyAction = true;

    /**
     * Enable a save-and-continue-editing action.
     *
     * @protected
     */
    saveAndContinueEditingAction = true;

    /**
     * Disable the inline-edit. Can be overridden by an option parameter.
     *
     * @protected
     */
    inlineEditDisabled = false;

    /**
     * Disable a portal layout usage. Can be overridden by an option parameter.
     *
     * @protected
     */
    portalLayoutDisabled = false;

    /**
     * A panel soft-locked type.
     *
     * @typedef {'default'|'acl'|'delimiter'|'dynamicLogic'
     * } module:views/record/detail~panelSoftLockedType
     */

    /**
     * @private
     * @type {module:views/record/detail~panelSoftLockedType[]}
     */
    panelSoftLockedTypeList = ['default', 'acl', 'delimiter', 'dynamicLogic'];

    /**
     * Dynamic logic. Can be overridden by an option parameter.
     *
     * @protected
     * @type {Object}
     * @todo Add typedef.
     */
    dynamicLogicDefs = {};

    /**
     * Disable confirm leave-out processing.
     *
     * @protected
     */
    confirmLeaveDisabled = false;

    /**
     * @protected
     */
    setupHandlerType = 'record/detail';

    /**
     * @protected
     */
    currentTab = 0;

    /**
     * @protected
     * @type {Object.<string,*>|null}
     */
    middlePanelDefs = null;

    /**
     * @protected
     * @type {Object.<string,*>[]|null}
     */
    middlePanelDefsList = null;

    /**
     * @protected
     * @type {JQuery|null}
     */
    $middle = null;

    /**
     * @protected
     * @type {JQuery|null}
     */
    $bottom = null;

    /**
     * @private
     * @type {JQuery|null}
     */
    $detailButtonContainer = null;

    /** @private */
    blockUpdateWebSocketPeriod = 500;

    /**
     * @internal
     * @protected
     */
    stickButtonsFormBottomSelector;

    /**
     * @protected
     * @type {string}
     */
    dynamicHandlerClassName;

    /**
     * Disable access control.
     *
     * @protected
     * @type {boolean}
     */
    accessControlDisabled;

    /**
     * @protected
     * @type {boolean}
     */
    inlineEditModeIsOn = false;

    /**
     * A Ctrl+Enter shortcut action.
     *
     * @protected
     * @type {?string}
     */
    shortcutKeyCtrlEnterAction = 'save';

    /**
     * Additional data. Passed to sub-views and fields.
     *
     * @protected
     * @type {Object.<string, *>}
     * @since 9.0.0
     */
    dataObject;

    /**
     * Data from the root view.
     *
     * @protected
     * @type {Record}
     * @since 9.0.0
     */
    rootData;

    /**
     * A shortcut-key => action map.
     *
     * @protected
     * @type {?Object.<string, string|function (KeyboardEvent): void>}
     */
    shortcutKeys = {
      /** @this DetailRecordView */
      'Control+Enter': function (e) {
        this.handleShortcutKeyCtrlEnter(e);
      },
      /** @this DetailRecordView */
      'Control+Alt+Enter': function (e) {
        this.handleShortcutKeyCtrlAltEnter(e);
      },
      /** @this DetailRecordView */
      'Control+KeyS': function (e) {
        this.handleShortcutKeyCtrlS(e);
      },
      /** @this DetailRecordView */
      'Control+Space': function (e) {
        this.handleShortcutKeyCtrlSpace(e);
      },
      /** @this DetailRecordView */
      'Escape': function (e) {
        this.handleShortcutKeyEscape(e);
      },
      /** @this DetailRecordView */
      'Control+Backslash': function (e) {
        this.handleShortcutKeyControlBackslash(e);
      },
      /** @this DetailRecordView */
      'Control+ArrowLeft': function (e) {
        this.handleShortcutKeyControlArrowLeft(e);
      },
      /** @this DetailRecordView */
      'Control+ArrowRight': function (e) {
        this.handleShortcutKeyControlArrowRight(e);
      }
    };

    /**
     * @inheritDoc
     */
    events = {
      /** @this DetailRecordView */
      'click .button-container .action': function (e) {
        const target = /** @type {HTMLElement} */e.currentTarget;
        let actionItems = undefined;
        if (target.classList.contains('detail-action-item')) {
          actionItems = [...this.buttonList, ...this.dropdownItemList];
        } else if (target.classList.contains('edit-action-item')) {
          actionItems = [...this.buttonEditList, ...this.dropdownEditItemList];
        }
        Espo.Utils.handleAction(this, e.originalEvent, target, {
          actionItems: actionItems
        });
      },
      /** @this DetailRecordView */
      'click [data-action="showMoreDetailPanels"]': function () {
        this.showMoreDetailPanels();
      },
      /** @this DetailRecordView */
      'click .middle-tabs > button': function (e) {
        const tab = parseInt($(e.currentTarget).attr('data-tab'));
        this.selectTab(tab);
      }
    };

    /**
     * An `edit` action.
     */
    actionEdit() {
      if (!this.editModeDisabled) {
        this.setEditMode();
        this.focusOnFirstDiv();
        $(window).scrollTop(0);
        return;
      }
      const options = {
        id: this.model.id,
        model: this.model.clone()
      };
      if (this.options.rootUrl) {
        options.rootUrl = this.options.rootUrl;
      }
      if (this.inlineEditModeIsOn) {
        options.attributes = this.getChangedAttributes();
        this.resetModelChanges();
      }
      this.getRouter().navigate(`#${this.scope}/edit/${this.model.id}`, {
        trigger: false
      });
      this.getRouter().dispatch(this.scope, 'edit', options);
    }

    // noinspection JSUnusedGlobalSymbols
    actionDelete() {
      this.delete();
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * A `save` action.
     *
     * @param {{options?: module:views/record/base~saveOptions}} [data] Data.
     * @return Promise
     */
    actionSave(data) {
      data = data || {};
      const modeBeforeSave = this.mode;
      const promise = this.save(data.options).catch(reason => {
        if (modeBeforeSave === this.MODE_EDIT && ['error', 'cancel'].includes(reason)) {
          this.setEditMode();
        }
        return Promise.reject(reason);
      });
      if (!this.lastSaveCancelReason || this.lastSaveCancelReason === 'notModified') {
        this.setDetailMode();
        this.focusOnFirstDiv();
        $(window).scrollTop(0);
      }
      return promise;
    }
    actionCancelEdit() {
      this.cancelEdit();
      this.focusOnFirstDiv();
      $(window).scrollTop(0);
    }
    focusOnFirstDiv() {
      const element = /** @type {HTMLElement} */this.$el.find('> div').get(0);
      if (element) {
        element.focus({
          preventScroll: true
        });
      }
    }

    /**
     * A `save-and-continue-editing` action.
     */
    actionSaveAndContinueEditing(data) {
      data = data || {};
      this.save(data.options).catch(() => {});
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * A `self-assign` action.
     */
    actionSelfAssign() {
      const attributes = {
        assignedUserId: this.getUser().id,
        assignedUserName: this.getUser().get('name')
      };
      if ('getSelfAssignAttributes' in this) {
        const attributesAdditional = this.getSelfAssignAttributes();
        if (attributesAdditional) {
          for (const i in attributesAdditional) {
            attributes[i] = attributesAdditional[i];
          }
        }
      }
      this.model.save(attributes, {
        patch: true
      }).then(() => {
        Espo.Ui.success(this.translate('Self-Assigned'));
      });
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * A `convert-currency` action.
     */
    actionConvertCurrency() {
      this.createView('modalConvertCurrency', 'views/modals/convert-currency', {
        entityType: this.entityType,
        model: this.model
      }, view => {
        view.render();
        this.listenToOnce(view, 'after:update', attributes => {
          let isChanged = false;
          for (const a in attributes) {
            if (attributes[a] !== this.model.get(a)) {
              isChanged = true;
              break;
            }
          }
          if (!isChanged) {
            Espo.Ui.warning(this.translate('notUpdated', 'messages'));
            return;
          }
          this.model.fetch().then(() => {
            Espo.Ui.success(this.translate('done', 'messages'));
          });
        });
      });
    }

    /**
     * Compose attribute values for a self-assignment.
     *
     * @protected
     * @return {Object.<string,*>|null}
     */
    getSelfAssignAttributes() {
      return null;
    }

    /**
     * Set up action items.
     *
     * @protected
     */
    setupActionItems() {
      if (this.model.isNew()) {
        this.isNew = true;
        this.removeActionItem('delete');
      } else if (this.getMetadata().get(['clientDefs', this.scope, 'removeDisabled'])) {
        this.removeActionItem('delete');
      }
      if (this.duplicateAction) {
        if (this.getAcl().check(this.entityType, 'create') && !this.getMetadata().get(['clientDefs', this.scope, 'duplicateDisabled'])) {
          this.addDropdownItem({
            label: 'Duplicate',
            name: 'duplicate',
            groupIndex: 0
          });
        }
      }
      if (this.selfAssignAction) {
        if (this.getAcl().check(this.entityType, 'edit') && !~this.getAcl().getScopeForbiddenFieldList(this.entityType).indexOf('assignedUser') && !this.getUser().isPortal()) {
          if (this.model.has('assignedUserId')) {
            this.addDropdownItem({
              label: 'Self-Assign',
              name: 'selfAssign',
              hidden: !!this.model.get('assignedUserId'),
              groupIndex: 0
            });
            this.listenTo(this.model, 'change:assignedUserId', () => {
              if (!this.model.get('assignedUserId')) {
                this.showActionItem('selfAssign');
              } else {
                this.hideActionItem('selfAssign');
              }
            });
          }
        }
      }
      if (this.type === this.TYPE_DETAIL && this.printPdfAction) {
        let printPdfAction = true;
        if (!~(this.getHelper().getAppParam('templateEntityTypeList') || []).indexOf(this.entityType)) {
          printPdfAction = false;
        }
        if (printPdfAction) {
          this.addDropdownItem({
            label: 'Print to PDF',
            name: 'printPdf',
            groupIndex: 6
          });
        }
      }
      if (this.type === this.TYPE_DETAIL && this.convertCurrencyAction) {
        if (this.getAcl().check(this.entityType, 'edit') && !this.getMetadata().get(['clientDefs', this.scope, 'convertCurrencyDisabled'])) {
          const currencyFieldList = this.getFieldManager().getEntityTypeFieldList(this.entityType, {
            type: 'currency',
            acl: 'edit'
          });
          if (currencyFieldList.length) {
            this.addDropdownItem({
              label: 'Convert Currency',
              name: 'convertCurrency',
              groupIndex: 5
            });
          }
        }
      }
      if (this.type === this.TYPE_DETAIL && this.getMetadata().get(['scopes', this.scope, 'hasPersonalData'])) {
        if (this.getAcl().getPermissionLevel('dataPrivacyPermission') === 'yes') {
          this.dropdownItemList.push({
            label: 'View Personal Data',
            name: 'viewPersonalData',
            groupIndex: 4
          });
        }
      }
      if (this.type === this.TYPE_DETAIL && this.getMetadata().get(['scopes', this.scope, 'stream'])) {
        this.addDropdownItem({
          label: 'View Followers',
          name: 'viewFollowers',
          groupIndex: 4
        });
      }
      if (this.type === this.TYPE_DETAIL) {
        const actionItemSetup = new _actionItemSetup.default();
        actionItemSetup.setup(this, this.type, promise => this.wait(promise), item => this.addDropdownItem(item), name => this.showActionItem(name), name => this.hideActionItem(name));
        if (this.saveAndContinueEditingAction) {
          this.dropdownEditItemList.push({
            name: 'saveAndContinueEditing',
            label: 'Save & Continue Editing',
            title: 'Ctrl+S'
          });
        }
      }
    }

    /**
     * Disable action items.
     */
    disableActionItems() {
      // noinspection JSDeprecatedSymbols
      this.disableButtons();
    }

    /**
     * Enable action items.
     */
    enableActionItems() {
      // noinspection JSDeprecatedSymbols
      this.enableButtons();
    }

    /**
     * Hide a button or dropdown action item.
     *
     * @param {string} name A name.
     */
    hideActionItem(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.hidden = true;
          break;
        }
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          item.hidden = true;
          break;
        }
      }
      for (const item of this.dropdownEditItemList) {
        if (item.name === name) {
          item.hidden = true;
          break;
        }
      }
      for (const item of this.buttonEditList) {
        if (item.name === name) {
          item.hidden = true;
          break;
        }
      }
      if (this.isRendered()) {
        this.$detailButtonContainer.find('li > .action[data-action="' + name + '"]').parent().addClass('hidden');
        this.$detailButtonContainer.find('button.action[data-action="' + name + '"]').addClass('hidden');
        if (this.isDropdownItemListEmpty()) {
          this.$dropdownItemListButton.addClass('hidden');
        }
        if (this.isDropdownEditItemListEmpty()) {
          this.$dropdownEditItemListButton.addClass('hidden');
        }
        this.adjustButtons();
      }
    }

    /**
     * Show a button or dropdown action item.
     *
     * @param {string} name A name.
     */
    showActionItem(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.hidden = false;
          break;
        }
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          item.hidden = false;
          break;
        }
      }
      for (const item of this.dropdownEditItemList) {
        if (item.name === name) {
          item.hidden = false;
          break;
        }
      }
      for (const item of this.buttonEditList) {
        if (item.name === name) {
          item.hidden = false;
          break;
        }
      }
      if (this.isRendered()) {
        this.$detailButtonContainer.find('li > .action[data-action="' + name + '"]').parent().removeClass('hidden');
        this.$detailButtonContainer.find('button.action[data-action="' + name + '"]').removeClass('hidden');
        if (!this.isDropdownItemListEmpty()) {
          this.$dropdownItemListButton.removeClass('hidden');
        }
        if (!this.isDropdownEditItemListEmpty()) {
          this.$dropdownEditItemListButton.removeClass('hidden');
        }
        this.adjustButtons();
      }
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Disable a button or dropdown action item.
     *
     * @param {string} name A name.
     */
    disableActionItem(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.disabled = true;
          break;
        }
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          item.disabled = true;
          break;
        }
      }
      for (const item of this.dropdownEditItemList) {
        if (item.name === name) {
          item.disabled = true;
          break;
        }
      }
      for (const item of this.buttonEditList) {
        if (item.name === name) {
          item.disabled = true;
          break;
        }
      }
      if (this.isRendered()) {
        this.$detailButtonContainer.find('li > .action[data-action="' + name + '"]').parent().addClass('disabled').attr('disabled', 'disabled');
        this.$detailButtonContainer.find('button.action[data-action="' + name + '"]').addClass('disabled').attr('disabled', 'disabled');
      }
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Enable a button or dropdown action item.
     *
     * @param {string} name A name.
     */
    enableActionItem(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.disabled = false;
          break;
        }
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          item.disabled = false;
          break;
        }
      }
      for (const item of this.dropdownEditItemList) {
        if (item.name === name) {
          item.disabled = false;
          break;
        }
      }
      for (const item of this.buttonEditList) {
        if (item.name === name) {
          item.disabled = false;
          break;
        }
      }
      if (this.isRendered()) {
        this.$detailButtonContainer.find('li > .action[data-action="' + name + '"]').parent().removeClass('disabled').removeAttr('disabled');
        this.$detailButtonContainer.find('button.action[data-action="' + name + '"]').removeClass('disabled').removeAttr('disabled');
      }
    }

    /**
     * Whether an action item is visible and not disabled.
     *
     * @param {string} name An action item name.
     */
    hasAvailableActionItem(name) {
      if (this.allActionItemsDisabled) {
        return false;
      }
      if (this.type === this.TYPE_DETAIL && this.mode === this.MODE_EDIT) {
        const hasButton = this.buttonEditList.findIndex(item => item.name === name && !item.disabled && !item.hidden) !== -1;
        if (hasButton) {
          return true;
        }
        return this.dropdownEditItemList.findIndex(item => item.name === name && !item.disabled && !item.hidden) !== -1;
      }
      const hasButton = this.buttonList.findIndex(item => item.name === name && !item.disabled && !item.hidden) !== -1;
      if (hasButton) {
        return true;
      }
      return this.dropdownItemList.findIndex(item => item.name === name && !item.disabled && !item.hidden) !== -1;
    }

    /**
     * Show a panel.
     *
     * @param {string} name A panel name.
     * @param {module:views/record/detail~panelSoftLockedType} [softLockedType='default']
     *   A soft-locked type.
     */
    showPanel(name, softLockedType) {
      if (this.recordHelper.getPanelStateParam(name, 'hiddenLocked')) {
        return;
      }
      softLockedType = softLockedType || 'default';
      const softLockedParam = 'hidden' + Espo.Utils.upperCaseFirst(softLockedType) + 'Locked';
      this.recordHelper.setPanelStateParam(name, softLockedParam, false);
      if (softLockedType === 'dynamicLogic' && this.recordHelper.getPanelStateParam(name, 'hidden') === false) {
        return;
      }
      for (let i = 0; i < this.panelSoftLockedTypeList.length; i++) {
        const iType = this.panelSoftLockedTypeList[i];
        if (iType === softLockedType) {
          continue;
        }
        const iParam = 'hidden' + Espo.Utils.upperCaseFirst(iType) + 'Locked';
        if (this.recordHelper.getPanelStateParam(name, iParam)) {
          return;
        }
      }
      const middleView = this.getMiddleView();
      if (middleView) {
        middleView.showPanelInternal(name);
      }
      const bottomView = this.getBottomView();
      if (bottomView) {
        if ('showPanel' in bottomView) {
          bottomView.showPanel(name);
        }
      } else if (this.bottomView) {
        this.once('ready', () => {
          const view = this.getBottomView();
          if (view) {
            if ('processShowPanel' in view) {
              view.processShowPanel(name);
              return;
            }
            if ('showPanel' in view) {
              view.showPanel(name);
            }
          }
        });
      }
      const sideView = this.getSideView();
      if (sideView) {
        if ('showPanel' in sideView) {
          sideView.showPanel(name);
        }
      } else if (this.sideView) {
        this.once('ready', () => {
          const view = this.getSideView();
          if (view) {
            if ('processShowPanel' in view) {
              view.processShowPanel(name);
              return;
            }
            if ('showPanel' in view) {
              view.showPanel(name);
            }
          }
        });
      }
      this.recordHelper.setPanelStateParam(name, 'hidden', false);
      if (this.middlePanelDefs[name]) {
        this.controlTabVisibilityShow(this.middlePanelDefs[name].tabNumber);
        this.adjustMiddlePanels();
      }
      this.recordHelper.trigger('panel-show');
    }

    /**
     * Hide a panel.
     *
     * @param {string} name A panel name.
     * @param {boolean} [locked=false] Won't be able to un-hide.
     * @param {module:views/record/detail~panelSoftLockedType} [softLockedType='default']
     *   A soft-locked type.
     */
    hidePanel(name, locked, softLockedType) {
      softLockedType = softLockedType || 'default';
      if (locked) {
        this.recordHelper.setPanelStateParam(name, 'hiddenLocked', true);
      }
      const softLockedParam = 'hidden' + Espo.Utils.upperCaseFirst(softLockedType) + 'Locked';
      this.recordHelper.setPanelStateParam(name, softLockedParam, true);
      if (softLockedType === 'dynamicLogic' && this.recordHelper.getPanelStateParam(name, 'hidden') === true) {
        return;
      }
      const middleView = this.getMiddleView();
      if (middleView) {
        middleView.hidePanelInternal(name);
      }
      const bottomView = this.getBottomView();
      if (bottomView) {
        if ('hidePanel' in bottomView) {
          bottomView.hidePanel(name);
        }
      } else if (this.bottomView) {
        this.once('ready', () => {
          const view = this.getBottomView();
          if (view) {
            if ('processHidePanel' in view) {
              view.processHidePanel(name);
              return;
            }
            if ('hidePanel' in view) {
              view.hidePanel(name);
            }
          }
        });
      }
      const sideView = this.getSideView();
      if (sideView) {
        if ('hidePanel' in sideView) {
          sideView.hidePanel(name);
        }
      } else if (this.sideView) {
        this.once('ready', () => {
          const view = this.getSideView();
          if (view) {
            if ('processHidePanel' in view) {
              view.processHidePanel(name);
              return;
            }
            if ('hidePanel' in view) {
              view.hidePanel(name);
            }
          }
        });
      }
      this.recordHelper.setPanelStateParam(name, 'hidden', true);
      if (this.middlePanelDefs[name]) {
        this.controlTabVisibilityHide(this.middlePanelDefs[name].tabNumber);
        this.adjustMiddlePanels();
      }
    }
    afterRender() {
      this.$middle = this.$el.find('.middle');
      if (this.bottomView) {
        this.$bottom = this.$el.find('.bottom');
      }
      this.initElementReferences();
      this.adjustMiddlePanels();
      this.adjustButtons();
      this.initStickableButtonsContainer();
      this.initFieldsControlBehaviour();
    }

    /**
     * @private
     */
    initFieldsControlBehaviour() {
      const fields = this.getFieldViews();
      let fieldInEditMode = null;
      for (const field in fields) {
        const fieldView = fields[field];
        this.listenTo(fieldView, 'edit', view => {
          if (fieldInEditMode && fieldInEditMode.isEditMode()) {
            fieldInEditMode.inlineEditClose();
          }
          fieldInEditMode = view;
        });
        this.listenTo(fieldView, 'inline-edit-on', () => {
          this.inlineEditModeIsOn = true;
        });
        this.listenTo(fieldView, 'inline-edit-off', o => {
          o = o || {};
          if (o.all) {
            return;
          }
          this.inlineEditModeIsOn = false;
          this.setIsNotChanged();
        });
        this.listenTo(fieldView, 'after:inline-edit-off', o => {
          if (this.updatedAttributes && !o.noReset) {
            this.resetModelChanges();
          }
        });
      }
    }

    /** @private */
    initStickableButtonsContainer() {
      const helper = new _stickyBar.default(this, this.stickButtonsFormBottomSelector, this.stickButtonsContainerAllTheWay, this.numId);
      helper.init();
    }
    fetch() {
      let data = super.fetch();
      if (this.hasView('side')) {
        const view = this.getSideView();
        if ('fetch' in view) {
          data = _.extend(data, view.fetch());
        }
      }
      if (this.hasView('bottom')) {
        const view = this.getBottomView();
        if ('fetch' in view) {
          data = _.extend(data, view.fetch());
        }
      }
      return data;
    }
    setEditMode() {
      this.trigger('before:set-edit-mode');
      this.inlineEditModeIsOn = false;
      this.$el.find('.record-buttons').addClass('hidden');
      this.$el.find('.edit-buttons').removeClass('hidden');
      return new Promise(resolve => {
        const fields = this.getFieldViews(true);
        const promiseList = [];
        for (const field in fields) {
          const fieldView = fields[field];
          if (fieldView.readOnly) {
            continue;
          }
          if (fieldView.isEditMode()) {
            fieldView.fetchToModel();
            fieldView.removeInlineEditLinks();
            fieldView.setIsInlineEditMode(false);
          }
          promiseList.push(fieldView.setEditMode().then(() => {
            return fieldView.render();
          }));
        }
        this.mode = this.MODE_EDIT;
        this.trigger('after:set-edit-mode');
        this.trigger('after:mode-change', this.MODE_EDIT);
        Promise.all(promiseList).then(() => resolve());
      });
    }
    setDetailMode() {
      this.trigger('before:set-detail-mode');
      this.$el.find('.edit-buttons').addClass('hidden');
      this.$el.find('.record-buttons').removeClass('hidden');
      this.inlineEditModeIsOn = false;
      return new Promise(resolve => {
        const fields = this.getFieldViews(true);
        const promiseList = [];
        for (const field in fields) {
          const fieldView = fields[field];
          if (!fieldView.isDetailMode()) {
            if (fieldView.isEditMode()) {
              fieldView.trigger('inline-edit-off', {
                all: true
              });
            }
            promiseList.push(fieldView.setDetailMode().then(() => fieldView.render()));
          }
        }
        this.mode = this.MODE_DETAIL;
        this.trigger('after:set-detail-mode');
        this.trigger('after:mode-change', this.MODE_DETAIL);
        Promise.all(promiseList).then(() => resolve());
      });
    }
    cancelEdit() {
      this.resetModelChanges();
      this.setDetailMode();
      this.setIsNotChanged();
    }

    /**
     * Whether in edit mode.
     * @return {boolean}
     */
    isEditMode() {
      return this.mode === 'edit';
    }
    resetModelChanges() {
      let skipReRender = true;
      if (this.updatedAttributes) {
        this.attributes = this.updatedAttributes;
        this.updatedAttributes = null;
        skipReRender = false;
      }
      const attributes = this.model.attributes;
      for (const attr in attributes) {
        if (!(attr in this.attributes)) {
          this.model.unset(attr);
        }
      }
      this.model.set(this.attributes, {
        skipReRender: skipReRender
      });
    }
    delete() {
      this.confirm({
        message: this.translate('removeRecordConfirmation', 'messages', this.scope),
        confirmText: this.translate('Remove')
      }, () => {
        this.trigger('before:delete');
        this.trigger('delete');
        Espo.Ui.notify(' ... ');
        const collection = this.model.collection;
        this.model.destroy({
          wait: true
        }).then(() => {
          if (collection) {
            if (collection.total > 0) {
              collection.total--;
            }
          }
          this.model.set('deleted', true, {
            silent: true
          });
          Espo.Ui.success(this.translate('Removed'), {
            suppress: true
          });
          this.trigger('after:delete');
          this.exit('delete');
        });
      });
    }

    /**
     * Get field views.
     *
     * @param {boolean} [withHidden] With hidden.
     * @return {Object.<string, module:views/fields/base>}
     */
    getFieldViews(withHidden) {
      const fields = {};
      if (this.hasView('middle')) {
        if ('getFieldViews' in this.getMiddleView()) {
          _.extend(fields, Espo.Utils.clone(this.getMiddleView().getFieldViews()));
        }
      }
      if (this.hasView('side')) {
        if ('getFieldViews' in this.getSideView()) {
          _.extend(fields, this.getSideView().getFieldViews(withHidden));
        }
      }
      if (this.hasView('bottom')) {
        if ('getFieldViews' in this.getBottomView()) {
          _.extend(fields, this.getBottomView().getFieldViews(withHidden));
        }
      }
      return fields;
    }

    /**
     * Get a field view.
     *
     * @param {string} name A field name.
     * @return {module:views/fields/base|null}
     */
    getFieldView(name) {
      let view;
      if (this.hasView('middle')) {
        view = (this.getMiddleView().getFieldViews() || {})[name];
      }
      if (!view && this.hasView('side')) {
        view = (this.getSideView().getFieldViews(true) || {})[name];
      }
      if (!view && this.hasView('bottom')) {
        view = (this.getBottomView().getFieldViews(true) || {})[name];
      }
      return view || null;
    }

    // @todo Remove.
    handleDataBeforeRender(data) {}
    data() {
      let navigateButtonsEnabled = !this.navigateButtonsDisabled && !!this.model.collection;
      let previousButtonEnabled = false;
      let nextButtonEnabled = false;
      if (navigateButtonsEnabled) {
        if (this.indexOfRecord > 0 || this.model.collection.offset) {
          previousButtonEnabled = true;
        }
        const total = this.model.collection.total !== undefined ? this.model.collection.total : this.model.collection.length;
        if (this.indexOfRecord < total - 1 - this.model.collection.offset) {
          nextButtonEnabled = true;
        } else {
          if (total === -1) {
            nextButtonEnabled = true;
          } else if (total === -2) {
            if (this.indexOfRecord < this.model.collection.length - 1 - this.model.collection.offset) {
              nextButtonEnabled = true;
            }
          }
        }
        if (!previousButtonEnabled && !nextButtonEnabled) {
          navigateButtonsEnabled = false;
        }
      }
      const hasMiddleTabs = this.hasTabs();
      const middleTabDataList = hasMiddleTabs ? this.getMiddleTabDataList() : [];
      return {
        scope: this.scope,
        entityType: this.entityType,
        buttonList: this.buttonList,
        buttonEditList: this.buttonEditList,
        dropdownItemList: this.getDropdownItemDataList(),
        dropdownEditItemList: this.dropdownEditItemList,
        dropdownItemListEmpty: this.isDropdownItemListEmpty(),
        dropdownEditItemListEmpty: this.isDropdownEditItemListEmpty(),
        buttonsDisabled: this.buttonsDisabled,
        id: this.id,
        isWide: this.isWide,
        isSmall: this.type === 'editSmall' || this.type === 'detailSmall',
        navigateButtonsEnabled: navigateButtonsEnabled,
        previousButtonEnabled: previousButtonEnabled,
        nextButtonEnabled: nextButtonEnabled,
        hasMiddleTabs: hasMiddleTabs,
        middleTabDataList: middleTabDataList
      };
    }

    /**
     * @private
     * @return {Array<module:views/record/detail~dropdownItem|false>}
     */
    getDropdownItemDataList() {
      /** @type {Array<module:views/record/detail~dropdownItem[]>} */
      const dropdownGroups = [];
      this.dropdownItemList.forEach(item => {
        // For bc.
        if (item === false) {
          return;
        }
        const index = (item.groupIndex === undefined ? 9999 : item.groupIndex) + 100;
        if (dropdownGroups[index] === undefined) {
          dropdownGroups[index] = [];
        }
        dropdownGroups[index].push(item);
      });
      const dropdownItemList = [];
      dropdownGroups.forEach(list => {
        list.forEach(it => dropdownItemList.push(it));
        dropdownItemList.push(false);
      });
      return dropdownItemList;
    }
    init() {
      this.entityType = this.model.entityType || this.model.name || 'Common';
      this.scope = this.options.scope || this.entityType;
      this.layoutName = this.options.layoutName || this.layoutName;
      this.detailLayout = this.options.detailLayout || this.detailLayout;
      this.type = this.options.type || this.type;
      this.buttonList = this.options.buttonList || this.buttonList;
      this.dropdownItemList = this.options.dropdownItemList || this.dropdownItemList;
      this.buttonList = Espo.Utils.cloneDeep(this.buttonList);
      this.buttonEditList = Espo.Utils.cloneDeep(this.buttonEditList);
      this.dropdownItemList = Espo.Utils.cloneDeep(this.dropdownItemList);
      this.dropdownEditItemList = Espo.Utils.cloneDeep(this.dropdownEditItemList);
      this.returnAfterCreate = this.options.returnAfterCreate;
      this.returnUrl = this.options.returnUrl || this.returnUrl;
      this.returnDispatchParams = this.options.returnDispatchParams || this.returnDispatchParams;
      this.exit = this.options.exit || this.exit;
      if (this.shortcutKeys) {
        this.shortcutKeys = Espo.Utils.cloneDeep(this.shortcutKeys);
      }
    }
    isDropdownItemListEmpty() {
      if (this.dropdownItemList.length === 0) {
        return true;
      }
      let isEmpty = true;
      this.dropdownItemList.forEach(item => {
        if (!item.hidden) {
          isEmpty = false;
        }
      });
      return isEmpty;
    }
    isDropdownEditItemListEmpty() {
      if (this.dropdownEditItemList.length === 0) {
        return true;
      }
      let isEmpty = true;
      this.dropdownEditItemList.forEach(item => {
        if (!item.hidden) {
          isEmpty = false;
        }
      });
      return isEmpty;
    }
    setup() {
      if (typeof this.model === 'undefined') {
        throw new Error('Model has not been injected into record view.');
      }
      this.recordHelper = this.options.recordHelper || new _viewRecordHelper.default(this.defaultFieldStates, this.defaultPanelStates);
      this._initInlineEditSave();
      const collection = this.collection = this.model.collection;
      if (collection) {
        this.listenTo(this.model, 'destroy', () => {
          collection.remove(this.model.id);
          collection.trigger('sync', {});
        });
        if ('indexOfRecord' in this.options) {
          this.indexOfRecord = this.options.indexOfRecord;
        } else {
          this.indexOfRecord = collection.indexOf(this.model);
        }
      }

      /** @type {Object.<string,*>|null} */
      this.middlePanelDefs = {};

      /** @type {Object.<string,*>[]} */
      this.middlePanelDefsList = [];
      if (this.getUser().isPortal() && !this.portalLayoutDisabled) {
        if (this.getMetadata().get(['clientDefs', this.scope, 'additionalLayouts', this.layoutName + 'Portal'])) {
          this.layoutName += 'Portal';
        }
      }
      this.numId = Math.floor(Math.random() * 10000 + 1);

      // For testing purpose.
      $(window).on('fetch-record.' + this.cid, () => this.handleRecordUpdate());
      this.once('remove', () => {
        if (this.isChanged) {
          this.resetModelChanges();
        }
        this.setIsNotChanged();
        $(window).off('scroll.detail-' + this.numId);
        $(window).off('fetch-record.' + this.cid);
      });
      this.id = Espo.Utils.toDom(this.entityType) + '-' + Espo.Utils.toDom(this.type) + '-' + this.numId;
      this.isNew = this.model.isNew();
      if (!this.editModeDisabled) {
        if ('editModeDisabled' in this.options) {
          this.editModeDisabled = this.options.editModeDisabled;
        }
      }
      this.confirmLeaveDisabled = this.options.confirmLeaveDisabled || this.confirmLeaveDisabled;
      this.buttonsDisabled = this.options.buttonsDisabled || this.buttonsDisabled;

      // for backward compatibility
      // @todo remove
      if ('buttonsPosition' in this.options && !this.options.buttonsPosition) {
        this.buttonsDisabled = true;
      }
      if ('isWide' in this.options) {
        this.isWide = this.options.isWide;
      }
      if ('sideView' in this.options) {
        this.sideView = this.options.sideView;
      }
      if ('bottomView' in this.options) {
        this.bottomView = this.options.bottomView;
      }
      this.sideDisabled = this.options.sideDisabled || this.sideDisabled;
      this.bottomDisabled = this.options.bottomDisabled || this.bottomDisabled;
      this.readOnly = this.options.readOnly || this.readOnly;
      if (!this.readOnly && !this.isNew) {
        this.readOnly = this.getMetadata().get(['clientDefs', this.scope, 'editDisabled']) || false;
      }
      if (this.getMetadata().get(['clientDefs', this.scope, 'createDisabled'])) {
        this.duplicateAction = false;
      }
      if ((this.getConfig().get('currencyList') || []).length <= 1) {
        this.convertCurrencyAction = false;
      }
      this.readOnlyLocked = this.readOnly;
      this.inlineEditDisabled = this.inlineEditDisabled || this.getMetadata().get(['clientDefs', this.scope, 'inlineEditDisabled']) || false;
      this.inlineEditDisabled = this.options.inlineEditDisabled || this.inlineEditDisabled;
      this.navigateButtonsDisabled = this.options.navigateButtonsDisabled || this.navigateButtonsDisabled;
      this.portalLayoutDisabled = this.options.portalLayoutDisabled || this.portalLayoutDisabled;
      this.dynamicLogicDefs = this.options.dynamicLogicDefs || this.dynamicLogicDefs;
      this.accessControlDisabled = this.options.accessControlDisabled || this.accessControlDisabled;
      this.dataObject = this.options.dataObject || {};
      this.rootData = this.options.rootData || {};
      this.setupActionItems();
      this.setupBeforeFinal();
      this.setupDynamicBehavior();
      this.on('after:render', () => {
        this.initElementReferences();
      });
      if (!this.isNew && !!this.getHelper().webSocketManager && this.getMetadata().get(['scopes', this.entityType, 'object'])) {
        this.subscribeToWebSocket();
        this.once('remove', () => {
          if (this.isSubscribedToWebSocket) {
            this.unsubscribeFromWebSocket();
          }
        });
      }
      this.wait(this.getHelper().processSetupHandlers(this, this.setupHandlerType));
      this.initInlineEditDynamicWithLogicInteroperability();
      this.forcePatchAttributeDependencyMap = this.getMetadata().get(['clientDefs', this.scope, 'forcePatchAttributeDependencyMap']) || {};
    }
    setupBeforeFinal() {
      if (!this.accessControlDisabled) {
        this.manageAccess();
      }
      this.attributes = this.model.getClonedAttributes();
      if (this.options.attributes) {
        this.model.set(this.options.attributes);
      }
      this.listenTo(this.model, 'sync', () => {
        this.attributes = this.model.getClonedAttributes();
      });
      this.listenTo(this.model, 'change', (m, o) => {
        if (o.sync) {
          for (const attribute in m.attributes) {
            if (!m.hasChanged(attribute)) {
              continue;
            }
            this.attributes[attribute] = Espo.Utils.cloneDeep(m.get(attribute));
          }
          return;
        }
        if (this.mode === this.MODE_EDIT || this.inlineEditModeIsOn) {
          this.setIsChanged();
        }
      });
    }

    /**
     * @protected
     */
    setupDynamicBehavior() {
      const dependencyDefs = Espo.Utils.clone(this.getMetadata().get(['clientDefs', this.entityType, 'formDependency']) || {});

      // noinspection JSDeprecatedSymbols
      this.dependencyDefs = _.extend(dependencyDefs, this.dependencyDefs);
      this.initDependency();
      const dynamicLogic = Espo.Utils.clone(this.getMetadata().get(['clientDefs', this.entityType, 'dynamicLogic']) || {});
      this.dynamicLogicDefs = _.extend(dynamicLogic, this.dynamicLogicDefs);
      this.initDynamicLogic();
      this.setupFieldLevelSecurity();
      this.initDynamicHandler();
    }

    /**
     * @private
     */
    _initInlineEditSave() {
      this.listenTo(this.recordHelper, 'inline-edit-save', (field, o) => {
        this.inlineEditSave(field, o);
      });
    }

    /**
     * @param {string} field
     * @param {module:views/record/base~saveOptions} [options]
     */
    inlineEditSave(field, options) {
      const view = this.getFieldView(field);
      if (!view) {
        throw new Error(`No field '${field}'.`);
      }
      options = _.extend({
        inline: true,
        field: field,
        afterValidate: () => {
          if (options.bypassClose) {
            return;
          }
          view.inlineEditClose(true);
        }
      }, options || {});
      this.save(options).then(() => {
        view.trigger('after:inline-save');
        view.trigger('after:save');
        if (options.bypassClose) {
          view.initialAttributes = this.model.getClonedAttributes();
        }
      }).catch(reason => {
        if (reason === 'notModified') {
          if (options.bypassClose) {
            return;
          }
          view.inlineEditClose(true);
          return;
        }
        if (reason === 'error') {
          if (options.bypassClose) {
            return;
          }
          const initialAttributes = {
            ...view.initialAttributes
          };
          view.inlineEdit().then(() => view.initialAttributes = initialAttributes);
        }
      });
    }

    /**
     * @private
     */
    initInlineEditDynamicWithLogicInteroperability() {
      let blockEdit = false;
      const process = (type, field) => {
        if (!this.inlineEditModeIsOn || this.editModeDisabled) {
          return;
        }
        if (blockEdit) {
          return;
        }
        if (type === 'required') {
          const fieldView = this.getFieldView(field);
          if (fieldView.validateRequired) {
            fieldView.suspendValidationMessage();
            try {
              if (!fieldView.validateRequired()) {
                return;
              }
            } catch (e) {}
          }
        }
        blockEdit = true;
        setTimeout(() => blockEdit = false, 300);
        setTimeout(() => {
          this.setEditMode();
          this.getFieldViewList().forEach(view => view.removeInlineEditLinks());
        }, 10);
      };
      this.on('set-field-required', field => process('required', field));
      this.on('set-field-option-list', field => process('options', field));
      this.on('reset-field-option-list', field => process('options', field));
    }

    /**
     * @private
     */
    initDynamicHandler() {
      const dynamicHandlerClassName = this.dynamicHandlerClassName || this.getMetadata().get(['clientDefs', this.scope, 'dynamicHandler']);
      const init = dynamicHandler => {
        this.listenTo(this.model, 'change', (model, o) => {
          if ('onChange' in dynamicHandler) {
            dynamicHandler.onChange.call(dynamicHandler, model, o);
          }
          const changedAttributes = model.changedAttributes();
          for (const attribute in changedAttributes) {
            const methodName = 'onChange' + Espo.Utils.upperCaseFirst(attribute);
            if (methodName in dynamicHandler) {
              dynamicHandler[methodName].call(dynamicHandler, model, changedAttributes[attribute], o);
            }
          }
        });
        if ('init' in dynamicHandler) {
          dynamicHandler.init();
        }
      };
      if (dynamicHandlerClassName) {
        this.wait(new Promise(resolve => {
          Espo.loader.require(dynamicHandlerClassName, DynamicHandler => {
            const dynamicHandler = this.dynamicHandler = new DynamicHandler(this);
            init(dynamicHandler);
            resolve();
          });
        }));
      }
      const handlerList = this.getMetadata().get(['clientDefs', this.scope, 'dynamicHandlerList']) || [];
      if (handlerList.length) {
        const self = this;
        const promiseList = [];
        handlerList.forEach(className => {
          promiseList.push(new Promise(resolve => {
            Espo.loader.require(className, DynamicHandler => {
              resolve(new DynamicHandler(self));
            });
          }));
        });
        this.wait(Promise.all(promiseList).then(list => {
          list.forEach(dynamicHandler => {
            init(dynamicHandler);
          });
        }));
      }
    }
    setupFinal() {
      this.build();
      if (this.shortcutKeys && this.options.shortcutKeysEnabled) {
        this.events['keydown.record-detail'] = e => {
          const key = Espo.Utils.getKeyFromKeyEvent(e);
          if (typeof this.shortcutKeys[key] === 'function') {
            this.shortcutKeys[key].call(this, e.originalEvent);
            return;
          }
          const actionName = this.shortcutKeys[key];
          if (!actionName) {
            return;
          }
          e.preventDefault();
          e.stopPropagation();
          const methodName = 'action' + Espo.Utils.upperCaseFirst(actionName);
          if (typeof this[methodName] === 'function') {
            this[methodName]();
            return;
          }
          this[actionName]();
        };
      }
      if (!this.options.focusForCreate) {
        this.once('after:render', () => this.focusOnFirstDiv());
      }
    }
    setIsChanged() {
      this.isChanged = true;
      if (this.confirmLeaveDisabled) {
        return;
      }
      this.setConfirmLeaveOut(true);
    }
    setIsNotChanged() {
      this.isChanged = false;
      if (this.confirmLeaveDisabled) {
        return;
      }
      this.setConfirmLeaveOut(false);
    }
    switchToModelByIndex(indexOfRecord) {
      const collection = this.model.collection || this.collection;
      if (!collection) {
        return;
      }
      const model = collection.at(indexOfRecord);
      if (!model) {
        console.error("Model is not found in collection by index.");
        return;
      }
      const id = model.id;
      const scope = this.entityType || this.scope;
      this.getRouter().navigate('#' + scope + '/view/' + id, {
        trigger: false
      });
      this.getRouter().dispatch(scope, 'view', {
        id: id,
        model: model,
        indexOfRecord: indexOfRecord,
        rootUrl: this.options.rootUrl
      });
    }
    actionPrevious() {
      this.model.abortLastFetch();
      if (!this.model.collection) {
        return;
      }
      const collection = this.model.collection;
      if (this.indexOfRecord <= 0 && !collection.offset) {
        return;
      }
      if (this.indexOfRecord === 0 && collection.offset > 0 && collection.maxSize) {
        collection.offset = Math.max(0, collection.offset - collection.maxSize);
        collection.fetch().then(() => {
          const indexOfRecord = collection.length - 1;
          if (indexOfRecord < 0) {
            return;
          }
          this.switchToModelByIndex(indexOfRecord);
        });
        return;
      }
      const indexOfRecord = this.indexOfRecord - 1;
      this.switchToModelByIndex(indexOfRecord);
    }
    actionNext() {
      this.model.abortLastFetch();
      if (!this.model.collection) {
        return;
      }
      const collection = this.model.collection;
      if (!(this.indexOfRecord < collection.total - 1 - collection.offset) && collection.total >= 0) {
        return;
      }
      if (collection.total === -2 && this.indexOfRecord >= collection.length - 1 - collection.offset) {
        return;
      }
      const indexOfRecord = this.indexOfRecord + 1;
      if (indexOfRecord <= collection.length - 1 - collection.offset) {
        this.switchToModelByIndex(indexOfRecord);
        return;
      }
      collection.fetch({
        more: true,
        remove: false
      }).then(() => {
        this.switchToModelByIndex(indexOfRecord);
      });
    }

    // noinspection JSUnusedGlobalSymbols
    actionViewPersonalData() {
      this.createView('viewPersonalData', 'views/personal-data/modals/personal-data', {
        model: this.model
      }, view => {
        view.render();
        this.listenToOnce(view, 'erase', () => {
          this.clearView('viewPersonalData');
          this.model.fetch();
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    actionViewFollowers(data) {
      const viewName = this.getMetadata().get(['clientDefs', this.entityType, 'relationshipPanels', 'followers', 'viewModalView']) || this.getMetadata().get(['clientDefs', 'User', 'modalViews', 'relatedList']) || 'views/modals/followers-list';
      const selectDisabled = !this.getUser().isAdmin() && this.getAcl().getPermissionLevel('followerManagementPermission') === 'no' && this.getAcl().getPermissionLevel('portalPermission') === 'no';
      const options = {
        model: this.model,
        link: 'followers',
        scope: 'User',
        title: this.translate('Followers'),
        filtersDisabled: true,
        url: this.entityType + '/' + this.model.id + '/followers',
        createDisabled: true,
        selectDisabled: selectDisabled,
        rowActionsView: 'views/user/record/row-actions/relationship-followers'
      };
      if (data.viewOptions) {
        for (const item in data.viewOptions) {
          options[item] = data.viewOptions[item];
        }
      }
      Espo.Ui.notify(' ... ');
      this.createView('modalRelatedList', viewName, options, view => {
        Espo.Ui.notify(false);
        view.render();
        this.listenTo(view, 'action', (event, element) => {
          Espo.Utils.handleAction(this, event, element);
        });
        this.listenToOnce(view, 'close', () => {
          this.clearView('modalRelatedList');
        });
        view.listenTo(this.model, 'after:relate:followers', () => {
          this.model.fetch();
        });
        view.listenTo(this.model, 'after:unrelate:followers', () => {
          this.model.fetch();
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    actionPrintPdf() {
      this.createView('pdfTemplate', 'views/modals/select-template', {
        entityType: this.entityType
      }, view => {
        view.render();
        this.listenToOnce(view, 'select', model => {
          this.clearView('pdfTemplate');
          window.open('?entryPoint=pdf&entityType=' + this.entityType + '&entityId=' + this.model.id + '&templateId=' + model.id, '_blank');
        });
      });
    }
    afterSave() {
      if (this.isNew) {
        Espo.Ui.success(this.translate('Created'));
      } else {
        Espo.Ui.success(this.translate('Saved'));
      }
      this.enableActionItems();
      this.setIsNotChanged();
      setTimeout(() => {
        this.unblockUpdateWebSocket();
      }, this.blockUpdateWebSocketPeriod);
    }
    beforeSave() {
      Espo.Ui.notify(this.translate('saving', 'messages'));
      this.blockUpdateWebSocket();
    }
    beforeBeforeSave() {
      this.disableActionItems();
    }
    afterSaveError() {
      this.enableActionItems();
    }
    afterNotModified() {
      const msg = this.translate('notModified', 'messages');
      Espo.Ui.warning(msg);
      this.enableActionItems();
      this.setIsNotChanged();
    }
    afterNotValid() {
      Espo.Ui.error(this.translate('Not valid'));
      this.enableActionItems();
    }

    /**
     * @protected
     * @param duplicates
     * @param {Object} options
     * @param {function} resolve
     * @param {function} reject
     * @return {boolean}
     */
    errorHandlerDuplicate(duplicates, options, resolve, reject) {
      Espo.Ui.notify(false);
      this.createView('duplicate', 'views/modals/duplicate', {
        scope: this.entityType,
        duplicates: duplicates,
        model: this.model
      }, view => {
        view.render();
        this.listenToOnce(view, 'save', () => {
          this.actionSave({
            options: {
              headers: {
                'X-Skip-Duplicate-Check': 'true'
              }
            }
          }).then(() => resolve()).catch(() => reject('error'));
        });
        this.listenToOnce(view, 'cancel', () => reject('cancel'));
      });
      return true;
    }

    // noinspection JSUnusedGlobalSymbols
    errorHandlerModified(data, options) {
      Espo.Ui.notify(false);
      const versionNumber = data.versionNumber;
      const values = data.values || {};
      const attributeList = Object.keys(values);
      const diffAttributeList = [];
      attributeList.forEach(attribute => {
        if (this.attributes[attribute] !== values[attribute]) {
          diffAttributeList.push(attribute);
        }
      });
      if (diffAttributeList.length === 0) {
        setTimeout(() => {
          this.model.set('versionNumber', versionNumber, {
            silent: true
          });
          this.attributes.versionNumber = versionNumber;
          if (options.inline && options.field) {
            this.inlineEditSave(options.field);
            return;
          }
          this.actionSave();
        }, 5);
        return;
      }
      this.createView('dialog', 'views/modals/resolve-save-conflict', {
        model: this.model,
        attributeList: diffAttributeList,
        currentAttributes: Espo.Utils.cloneDeep(this.model.attributes),
        originalAttributes: Espo.Utils.cloneDeep(this.attributes),
        actualAttributes: Espo.Utils.cloneDeep(values)
      }).then(view => {
        view.render();
        this.listenTo(view, 'resolve', () => {
          this.model.set('versionNumber', versionNumber, {
            silent: true
          });
          this.attributes.versionNumber = versionNumber;
          for (const attribute in values) {
            this.setInitialAttributeValue(attribute, values[attribute]);
          }
        });
      });
    }

    /**
     * Get a middle view.
     *
     * @return {module:views/record/detail-middle}
     */
    getMiddleView() {
      return this.getView('middle');
    }

    /**
     * Get a side view.
     *
     * @protected
     * @return {module:views/record/detail-side}
     */
    getSideView() {
      return this.getView('side');
    }

    /**
     * Get a bottom view.
     *
     * @protected
     * @return {module:views/record/detail-bottom}
     */
    getBottomView() {
      return this.getView('bottom');
    }
    setReadOnly() {
      if (!this.readOnlyLocked) {
        this.readOnly = true;
      }
      const bottomView = this.getBottomView();
      if (bottomView && 'setReadOnly' in bottomView) {
        bottomView.setReadOnly();
      }
      const sideView = this.getSideView();
      if (sideView && 'setReadOnly' in sideView) {
        sideView.setReadOnly();
      }
      this.getFieldList().forEach(field => {
        this.setFieldReadOnly(field);
      });
    }
    setNotReadOnly(onlyNotSetAsReadOnly) {
      if (!this.readOnlyLocked) {
        this.readOnly = false;
      }
      const bottomView = this.getBottomView();
      if (bottomView && 'setNotReadOnly' in bottomView) {
        bottomView.setNotReadOnly(onlyNotSetAsReadOnly);
      }
      const sideView = this.getSideView();
      if (sideView && 'setNotReadOnly' in sideView) {
        sideView.setNotReadOnly(onlyNotSetAsReadOnly);
      }
      this.getFieldList().forEach(field => {
        if (onlyNotSetAsReadOnly) {
          if (this.recordHelper.getFieldStateParam(field, 'readOnly')) {
            return;
          }
        }
        this.setFieldNotReadOnly(field);
      });
    }
    manageAccessEdit(second) {
      if (this.isNew) {
        return;
      }
      const editAccess = this.getAcl().checkModel(this.model, 'edit', true);
      if (!editAccess || this.readOnlyLocked) {
        this.readOnly = true;
        this.hideActionItem('edit');
        if (this.selfAssignAction) {
          this.hideActionItem('selfAssign');
        }
      } else {
        this.showActionItem('edit');
        if (this.selfAssignAction) {
          this.hideActionItem('selfAssign');
          if (this.model.has('assignedUserId')) {
            if (!this.model.get('assignedUserId')) {
              this.showActionItem('selfAssign');
            }
          }
        }
        if (!this.readOnlyLocked) {
          if (this.readOnly && second) {
            if (this.isReady) {
              this.setNotReadOnly(true);
            } else {
              this.on('ready', () => this.setNotReadOnly(true));
            }
          }
          this.readOnly = false;
        }
      }
      if (editAccess === null) {
        this.listenToOnce(this.model, 'sync', () => {
          this.model.trigger('acl-edit-ready');
          this.manageAccessEdit(true);
        });
      }
    }
    manageAccessDelete() {
      if (this.isNew) {
        return;
      }
      const deleteAccess = this.getAcl().checkModel(this.model, 'delete', true);
      if (!deleteAccess) {
        this.hideActionItem('delete');
      } else {
        this.showActionItem('delete');
      }
      if (deleteAccess === null) {
        this.listenToOnce(this.model, 'sync', () => {
          this.manageAccessDelete(true);
        });
      }
    }
    manageAccessStream() {
      if (this.isNew) {
        return;
      }
      if (~['no', 'own'].indexOf(this.getAcl().getLevel('User', 'read')) && this.getAcl().getPermissionLevel('portalPermission') === 'no') {
        this.hideActionItem('viewFollowers');
        return;
      }
      const streamAccess = this.getAcl().checkModel(this.model, 'stream', true);
      if (!streamAccess) {
        this.hideActionItem('viewFollowers');
      } else {
        this.showActionItem('viewFollowers');
      }
      if (streamAccess === null) {
        this.listenToOnce(this.model, 'sync', () => {
          this.manageAccessStream(true);
        });
      }
    }
    manageAccess() {
      this.manageAccessEdit();
      this.manageAccessDelete();
      this.manageAccessStream();
    }

    /**
     * Add a button.
     *
     * @param {module:views/record/detail~button} o
     * @param {boolean} [toBeginning]
     */
    addButton(o, toBeginning) {
      const name = o.name;
      if (!name) {
        return;
      }
      for (const item of this.buttonList) {
        if (item.name === name) {
          return;
        }
      }
      toBeginning ? this.buttonList.unshift(o) : this.buttonList.push(o);
    }

    /**
     * Add a dropdown item.
     *
     * @param {module:views/record/detail~dropdownItem} o
     * @param {boolean} [toBeginning]
     */
    addDropdownItem(o, toBeginning) {
      if (!o) {
        // For bc.
        return;
      }
      const name = o.name;
      if (!name) {
        return;
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          return;
        }
      }
      toBeginning ? this.dropdownItemList.unshift(o) : this.dropdownItemList.push(o);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Add an 'edit' mode button.
     *
     * @param {module:views/record/detail~button} o
     * @param {boolean} [toBeginning]
     */
    addButtonEdit(o, toBeginning) {
      const name = o.name;
      if (!name) {
        return;
      }
      for (const item of this.buttonEditList) {
        if (item.name === name) {
          return;
        }
      }
      toBeginning ? this.buttonEditList.unshift(o) : this.buttonEditList.push(o);
    }

    /**
     * @deprecated Use `enableActionItems`.
     */
    enableButtons() {
      this.allActionItemsDisabled = false;
      this.$el.find(".button-container .actions-btn-group .action").removeAttr('disabled').removeClass('disabled');
      this.$el.find(".button-container .actions-btn-group .dropdown-toggle").removeAttr('disabled').removeClass('disabled');
      this.buttonList.filter(item => item.disabled).forEach(item => {
        this.$detailButtonContainer.find(`button.action[data-action="${item.name}"]`).addClass('disabled').attr('disabled', 'disabled');
      });
      this.buttonEditList.filter(item => item.disabled).forEach(item => {
        this.$detailButtonContainer.find(`button.action[data-action="${item.name}"]`).addClass('disabled').attr('disabled', 'disabled');
      });
      this.dropdownItemList.filter(item => item.disabled).forEach(item => {
        this.$detailButtonContainer.find(`li > .action[data-action="${item.name}"]`).parent().addClass('disabled').attr('disabled', 'disabled');
      });
      this.dropdownEditItemList.filter(item => item.disabled).forEach(item => {
        this.$detailButtonContainer.find(`li > .action[data-action="${item.name}"]`).parent().addClass('disabled').attr('disabled', 'disabled');
      });
    }

    /**
     * @deprecated Use `disableActionItems`.
     */
    disableButtons() {
      this.allActionItemsDisabled = true;
      this.$el.find(".button-container .actions-btn-group .action").attr('disabled', 'disabled').addClass('disabled');
      this.$el.find(".button-container .actions-btn-group .dropdown-toggle").attr('disabled', 'disabled').addClass('disabled');
    }

    /**
     * Remove a button or dropdown item.
     *
     * @param {string} name A name.
     */
    removeActionItem(name) {
      // noinspection JSDeprecatedSymbols
      this.removeButton(name);
    }

    /**
     * @deprecated Use `removeActionItem`.
     *
     * @param {string} name A name.
     */
    removeButton(name) {
      for (const [i, item] of this.buttonList.entries()) {
        if (item.name === name) {
          this.buttonList.splice(i, 1);
          break;
        }
      }
      for (const [i, item] of this.dropdownItemList.entries()) {
        if (item.name === name) {
          this.dropdownItemList.splice(i, 1);
          break;
        }
      }
      if (!this.isRendered()) {
        return;
      }
      const $container = this.$el.find('.detail-button-container');
      const $action = $container.find(`ul > li > a.action[data-action="${name}"]`);
      if ($action.length) {
        $action.parent().remove();
        $container.find(`ul > .divider:last-child`).remove();
        return;
      }
      const $button = $container.find(`button.action[data-action="${name}"]`);
      if ($button.length) {
        $button.remove();
      }
    }

    /**
     * Convert a detail layout to an internal layout.
     *
     * @protected
     * @param {module:views/record/detail~panelDefs[]} simplifiedLayout A detail layout.
     * @return {Object[]}
     */
    convertDetailLayout(simplifiedLayout) {
      const layout = [];
      const el = this.getSelector() || '#' + this.id;
      this.panelFieldListMap = {};
      let tabNumber = -1;
      for (let p = 0; p < simplifiedLayout.length; p++) {
        const item = simplifiedLayout[p];
        const panel = {};
        const tabBreak = item.tabBreak || p === 0;
        if (tabBreak) {
          tabNumber++;
        }
        if ('customLabel' in item) {
          panel.label = item.customLabel;
          if (panel.label) {
            panel.label = this.translate(panel.label, 'panelCustomLabels', this.entityType);
          }
        } else {
          panel.label = item.label || null;
          if (panel.label) {
            panel.label = panel.label[0] === '$' ? this.translate(panel.label.substring(1), 'panels', this.entityType) : this.translate(panel.label, 'labels', this.entityType);
          }
        }
        panel.name = item.name || 'panel-' + p.toString();
        panel.style = item.style || 'default';
        panel.rows = [];
        panel.tabNumber = tabNumber;
        panel.noteText = item.noteText;
        panel.noteStyle = item.noteStyle || 'info';
        if (panel.noteText) {
          if (panel.noteText.startsWith('$') && !panel.noteText.includes(' ')) {
            const label = panel.noteText.substring(1);
            panel.noteText = this.translate(label, 'panelNotes', this.entityType);
          }
          panel.noteText = this.getHelper().transformMarkdownText(panel.noteText);
        }
        this.middlePanelDefs[panel.name] = {
          name: panel.name,
          style: panel.style,
          tabNumber: panel.tabNumber,
          tabBreak: tabBreak,
          tabLabel: item.tabLabel
        };
        this.middlePanelDefsList.push(this.middlePanelDefs[panel.name]);

        // noinspection JSUnresolvedReference
        if (item.dynamicLogicVisible && this.dynamicLogic) {
          this.dynamicLogic.addPanelVisibleCondition(panel.name, item.dynamicLogicVisible);
        }

        // noinspection JSUnresolvedReference
        if (item.dynamicLogicStyled && this.dynamicLogic) {
          this.dynamicLogic.addPanelStyledCondition(panel.name, item.dynamicLogicStyled);
        }

        // noinspection JSUnresolvedReference
        if (item.hidden && tabNumber === 0) {
          panel.hidden = true;
          this.hidePanel(panel.name);
          this.underShowMoreDetailPanelList = this.underShowMoreDetailPanelList || [];
          this.underShowMoreDetailPanelList.push(panel.name);
        }
        let lType = 'rows';
        if (item.columns) {
          lType = 'columns';
          panel.columns = [];
        }
        if (panel.name) {
          this.panelFieldListMap[panel.name] = [];
        }
        for (const [i, itemI] of item[lType].entries()) {
          const row = [];
          for (const cellDefs of itemI) {
            if (cellDefs === false) {
              row.push(false);
              continue;
            }
            let view = cellDefs.view;
            let name = cellDefs.name;
            if (!name && view && typeof view === 'object') {
              name = view.name;
            }
            if (!name) {
              console.warn(`No 'name' specified in detail layout cell.`);
              continue;
            }
            let selector;
            if (view && typeof view === 'object') {
              view.model = this.model;
              view.mode = this.fieldsMode;
              if (this.readOnly) {
                view.setReadOnly();
              }
              selector = `.field[data-name="${name}"]`;
            }
            if (panel.name) {
              this.panelFieldListMap[panel.name].push(name);
            }
            const type = cellDefs.type || this.model.getFieldType(name) || 'base';
            view = view || this.model.getFieldParam(name, 'view') || this.getFieldManager().getViewName(type);
            const o = {
              fullSelector: el + ' .middle .field[data-name="' + name + '"]',
              defs: {
                name: name,
                params: cellDefs.params || {}
              },
              mode: this.fieldsMode
            };
            if (this.readOnly) {
              o.readOnly = true;
            }
            if (cellDefs.readOnly) {
              o.readOnly = true;
              o.readOnlyLocked = true;
            }
            if (this.readOnlyLocked) {
              o.readOnlyLocked = true;
            }
            if (this.inlineEditDisabled || cellDefs.inlineEditDisabled) {
              o.inlineEditDisabled = true;
            }

            // noinspection JSUnresolvedReference
            let fullWidth = cellDefs.fullWidth || false;
            if (!fullWidth) {
              if (item[lType][i].length === 1) {
                fullWidth = true;
              }
            }
            if (this.recordHelper.getFieldStateParam(name, 'hidden')) {
              o.disabled = true;
            }
            if (this.recordHelper.getFieldStateParam(name, 'hiddenLocked')) {
              o.disabledLocked = true;
            }
            if (this.recordHelper.getFieldStateParam(name, 'readOnly')) {
              o.readOnly = true;
            }
            if (!o.readOnlyLocked && this.recordHelper.getFieldStateParam(name, 'readOnlyLocked')) {
              o.readOnlyLocked = true;
            }
            if (this.recordHelper.getFieldStateParam(name, 'required') !== null) {
              o.defs.params = o.defs.params || {};
              o.defs.params.required = this.recordHelper.getFieldStateParam(name, 'required');
            }
            if (this.recordHelper.hasFieldOptionList(name)) {
              o.customOptionList = this.recordHelper.getFieldOptionList(name);
            }
            o.validateCallback = () => this.validateField(name);
            o.recordHelper = this.recordHelper;
            o.dataObject = this.dataObject;
            if (cellDefs.options) {
              for (const optionName in cellDefs.options) {
                if (typeof o[optionName] !== 'undefined') {
                  continue;
                }
                o[optionName] = cellDefs.options[optionName];
              }
            }
            const cell = {
              name: name + 'Field',
              view: view,
              field: name,
              fullSelector: el + ' .middle .field[data-name="' + name + '"]',
              fullWidth: fullWidth,
              options: o
            };
            if (selector) {
              cell.selector = selector;
            }
            if ('labelText' in cellDefs) {
              o.labelText = cellDefs.labelText;
              cell.customLabel = cellDefs.labelText;
            }
            if ('customLabel' in cellDefs) {
              cell.customLabel = cellDefs.customLabel;
            }
            if ('label' in cellDefs) {
              cell.label = cellDefs.label;
            }
            if (view && typeof view === 'object' && !cell.customLabel && !cell.label && view.getLabelText()) {
              cell.customLabel = view.getLabelText();
            }
            if ('customCode' in cellDefs) {
              cell.customCode = cellDefs.customCode;
            }
            if ('noLabel' in cellDefs) {
              cell.noLabel = cellDefs.noLabel;
            }
            if ('span' in cellDefs) {
              cell.span = cellDefs.span;
            }
            row.push(cell);
          }
          panel[lType].push(row);
        }
        layout.push(panel);
      }
      return layout;
    }

    /**
     * @private
     * @param {function(Object[]): void}callback
     */
    getGridLayout(callback) {
      if (this.gridLayout !== null) {
        callback(this.gridLayout);
        return;
      }
      if (this.detailLayout) {
        this.gridLayout = {
          type: this.gridLayoutType,
          layout: this.convertDetailLayout(this.detailLayout)
        };
        callback(this.gridLayout);
        return;
      }
      this.getHelper().layoutManager.get(this.entityType, this.layoutName, detailLayout => {
        if (typeof this.modifyDetailLayout === 'function') {
          detailLayout = Espo.Utils.cloneDeep(detailLayout);
          this.modifyDetailLayout(detailLayout);
        }
        this.detailLayout = detailLayout;
        this.gridLayout = {
          type: this.gridLayoutType,
          layout: this.convertDetailLayout(this.detailLayout)
        };
        callback(this.gridLayout);
      });
    }

    /**
     * Create a side view.
     *
     * @protected
     */
    createSideView() {
      const el = this.getSelector() || '#' + this.id;
      this.createView('side', this.sideView, {
        model: this.model,
        scope: this.scope,
        fullSelector: el + ' .side',
        type: this.type,
        readOnly: this.readOnly,
        inlineEditDisabled: this.inlineEditDisabled,
        recordHelper: this.recordHelper,
        recordViewObject: this,
        isReturn: this.options.isReturn,
        dataObject: this.dataObject
      });
    }

    /**
     * Create a middle view.
     *
     * @protected
     */
    createMiddleView(callback) {
      const el = this.getSelector() || '#' + this.id;
      this.waitForView('middle');
      this.getGridLayout(layout => {
        if (this.hasTabs() && this.options.isReturn && this.isStoredTabForThisRecord()) {
          this.selectStoredTab();
        }
        this.createView('middle', this.middleView, {
          model: this.model,
          scope: this.scope,
          type: this.type,
          layoutDefs: layout,
          fullSelector: el + ' .middle',
          layoutData: {
            model: this.model
          },
          recordHelper: this.recordHelper,
          recordViewObject: this,
          panelFieldListMap: this.panelFieldListMap
        }, callback);
      });
    }

    /**
     * Create a bottom view.
     *
     * @protected
     */
    createBottomView() {
      const el = this.getSelector() || '#' + this.id;
      this.createView('bottom', this.bottomView, {
        model: this.model,
        scope: this.scope,
        fullSelector: el + ' .bottom',
        readOnly: this.readOnly,
        type: this.type,
        inlineEditDisabled: this.inlineEditDisabled,
        recordHelper: this.recordHelper,
        recordViewObject: this,
        portalLayoutDisabled: this.portalLayoutDisabled,
        isReturn: this.options.isReturn,
        dataObject: this.dataObject
      });
    }

    /**
     * Create views.
     *
     * @protected
     * @param {function(module:views/record/detail-middle): void} [callback]
     */
    build(callback) {
      if (!this.sideDisabled && this.sideView) {
        this.createSideView();
      }
      if (this.middleView) {
        this.createMiddleView(callback);
      }
      if (!this.bottomDisabled && this.bottomView) {
        this.createBottomView();
      }
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Called after create.
     *
     * @return {boolean} True if redirecting is processed.
     */
    exitAfterCreate() {
      if (!this.returnAfterCreate && this.model.id) {
        const url = '#' + this.scope + '/view/' + this.model.id;
        this.getRouter().navigate(url, {
          trigger: false
        });
        this.getRouter().dispatch(this.scope, 'view', {
          id: this.model.id,
          rootUrl: this.options.rootUrl,
          model: this.model,
          isAfterCreate: true
        });
        return true;
      }
      return false;
    }

    /**
     * Called after save or cancel. By default, redirects a page. Can be overridden in options.
     *
     * @param {string|'create'|'save'|'cancel'|'delete'} [after] Name of an action after which #exit is invoked.
     */
    exit(after) {
      if (after) {
        const methodName = 'exitAfter' + Espo.Utils.upperCaseFirst(after);
        if (methodName in this) {
          const result = this[methodName]();
          if (result) {
            return;
          }
        }
      }
      let url;
      let options;
      if (this.returnUrl) {
        url = this.returnUrl;
      } else {
        if (after === 'delete') {
          url = this.options.rootUrl || '#' + this.scope;
          if (this.options.rootUrl) {
            this.getRouter().navigate(url, {
              trigger: true
            });
            return;
          }
          this.getRouter().navigate(url, {
            trigger: false
          });
          this.getRouter().dispatch(this.scope, null, {
            isReturn: true
          });
          return;
        }
        if (this.model.id) {
          url = `#${this.scope}/view/${this.model.id}`;
          if (!this.returnDispatchParams) {
            this.getRouter().navigate(url, {
              trigger: false
            });
            options = {
              id: this.model.id,
              model: this.model
            };
            if (this.options.rootUrl) {
              options.rootUrl = this.options.rootUrl;
            }
            this.getRouter().dispatch(this.scope, 'view', options);
          }
        } else {
          url = this.options.rootUrl || '#' + this.scope;
        }
      }
      if (this.returnDispatchParams) {
        const controller = this.returnDispatchParams.controller;
        const action = this.returnDispatchParams.action;
        options = this.returnDispatchParams.options || {};
        this.getRouter().navigate(url, {
          trigger: false
        });
        this.getRouter().dispatch(controller, action, options);
        return;
      }
      this.getRouter().navigate(url, {
        trigger: true
      });
    }
    subscribeToWebSocket() {
      const topic = 'recordUpdate.' + this.entityType + '.' + this.model.id;
      this.recordUpdateWebSocketTopic = topic;
      this.isSubscribedToWebSocket = true;
      this.getHelper().webSocketManager.subscribe(topic, () => {
        this.handleRecordUpdate();
      });
    }
    unsubscribeFromWebSocket() {
      if (!this.isSubscribedToWebSocket) {
        return;
      }
      this.getHelper().webSocketManager.unsubscribe(this.recordUpdateWebSocketTopic);
    }
    handleRecordUpdate() {
      if (this.updateWebSocketIsBlocked) {
        return;
      }
      if (this.inlineEditModeIsOn || this.mode === this.MODE_EDIT) {
        const m = this.model.clone();
        m.fetch().then(() => {
          if (this.inlineEditModeIsOn || this.mode === this.MODE_EDIT) {
            this.updatedAttributes = Espo.Utils.cloneDeep(m.attributes);
          }
        });
        return;
      }
      this.model.fetch({
        highlight: true
      });
    }
    blockUpdateWebSocket(toUnblock) {
      this.updateWebSocketIsBlocked = true;
      if (toUnblock) {
        setTimeout(() => {
          this.unblockUpdateWebSocket();
        }, this.blockUpdateWebSocketPeriod);
      }
    }
    unblockUpdateWebSocket() {
      this.updateWebSocketIsBlocked = false;
    }

    /**
     * Show more detail panels.
     */
    showMoreDetailPanels() {
      this.hidePanel('showMoreDelimiter');
      this.underShowMoreDetailPanelList.forEach(item => {
        this.showPanel(item);
      });
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @protected
     * @return {Number}
     */
    getTabCount() {
      if (!this.hasTabs()) {
        return 0;
      }
      let count = 1;
      (this.detailLayout || []).forEach(item => {
        if (item.tabBreak) {
          count++;
        }
      });
      return count;
    }

    /**
     * @protected
     * @return {boolean}
     */
    hasTabs() {
      if (typeof this._hasMiddleTabs !== 'undefined') {
        return this._hasMiddleTabs;
      }
      if (!this.detailLayout) {
        return false;
      }
      for (const item of this.detailLayout) {
        if (item.tabBreak) {
          this._hasMiddleTabs = true;
          return true;
        }
      }
      this._hasMiddleTabs = false;
      return false;
    }

    /**
     * @private
     * @return {{label: string}[]}
     */
    getMiddleTabDataList() {
      const currentTab = this.currentTab;
      const panelDataList = this.middlePanelDefsList;
      return panelDataList.filter((item, i) => i === 0 || item.tabBreak).map((item, i) => {
        let label = item.tabLabel;
        let hidden = false;
        if (i > 0) {
          hidden = panelDataList.filter(panel => panel.tabNumber === i).findIndex(panel => !this.recordHelper.getPanelStateParam(panel.name, 'hidden')) === -1;
        }
        if (!label) {
          label = i === 0 ? this.translate('Overview') : (i + 1).toString();
        } else if (label.substring(0, 7) === '$label:') {
          label = this.translate(label.substring(7), 'labels', this.scope);
        } else if (label[0] === '$') {
          label = this.translate(label.substring(1), 'tabs', this.scope);
        }
        return {
          label: label,
          isActive: currentTab === i,
          hidden: hidden
        };
      });
    }

    /**
     * Select a tab.
     *
     * @protected
     * @param {Number} tab
     */
    selectTab(tab) {
      this.currentTab = tab;
      $('.popover.in').removeClass('in');
      this.whenRendered().then(() => {
        this.$el.find('.middle-tabs > button').removeClass('active');
        this.$el.find(`.middle-tabs > button[data-tab="${tab}"]`).addClass('active');
        this.$el.find('.middle > .panel[data-tab]').addClass('tab-hidden');
        this.$el.find(`.middle > .panel[data-tab="${tab}"]`).removeClass('tab-hidden');
        this.adjustMiddlePanels();
        this.recordHelper.trigger('panel-show');
      });
      this.storeTab();
    }

    /**
     * @private
     */
    storeTab() {
      const key = 'tab_middle';
      const keyRecord = 'tab_middle_record';
      this.getSessionStorage().set(key, this.currentTab);
      this.getSessionStorage().set(keyRecord, this.entityType + '_' + this.model.id);
    }

    /**
     * @private
     */
    selectStoredTab() {
      const key = 'tab_middle';
      const tab = this.getSessionStorage().get(key);
      if (tab > 0) {
        this.selectTab(tab);
      }
    }

    /**
     * @private
     */
    isStoredTabForThisRecord() {
      const keyRecord = 'tab_middle_record';
      return this.getSessionStorage().get(keyRecord) === this.entityType + '_' + this.model.id;
    }

    /**
     * @inheritDoc
      */
    onInvalid(invalidFieldList) {
      if (!this.hasTabs()) {
        return;
      }
      const tabList = [];
      for (const field of invalidFieldList) {
        const view = this.getMiddleView().getFieldView(field);
        if (!view) {
          continue;
        }
        const tabString = view.$el.closest('.panel.tab-hidden').attr('data-tab');
        const tab = parseInt(tabString);
        if (tabList.indexOf(tab) !== -1) {
          continue;
        }
        tabList.push(tab);
      }
      if (!tabList.length) {
        return;
      }
      const $tabs = this.$el.find('.middle-tabs');
      tabList.forEach(tab => {
        const $tab = $tabs.find(`> [data-tab="${tab.toString()}"]`);
        $tab.addClass('invalid');
        $tab.one('click', () => {
          $tab.removeClass('invalid');
        });
      });
    }

    /**
     * @private
     */
    controlTabVisibilityShow(tab) {
      if (!this.hasTabs() || tab === 0) {
        return;
      }
      if (this.isBeingRendered()) {
        this.once('after:render', () => this.controlTabVisibilityShow(tab));
        return;
      }
      this.$el.find(`.middle-tabs > [data-tab="${tab.toString()}"]`).removeClass('hidden');
    }

    /**
     * @private
     */
    controlTabVisibilityHide(tab) {
      if (!this.hasTabs() || tab === 0) {
        return;
      }
      if (this.isBeingRendered()) {
        this.once('after:render', () => this.controlTabVisibilityHide(tab));
        return;
      }
      const panelList = this.middlePanelDefsList.filter(panel => panel.tabNumber === tab);
      const allIsHidden = panelList.findIndex(panel => !this.recordHelper.getPanelStateParam(panel.name, 'hidden')) === -1;
      if (!allIsHidden) {
        return;
      }
      const $tab = this.$el.find(`.middle-tabs > [data-tab="${tab.toString()}"]`);
      $tab.addClass('hidden');
      if (this.currentTab === tab) {
        this.selectTab(0);
      }
    }

    /**
     * @private
     */
    adjustMiddlePanels() {
      if (!this.isRendered() || !this.$middle.length) {
        return;
      }
      const $panels = this.$middle.find('> .panel');
      const $bottomPanels = this.$bottom ? this.$bottom.find('> .panel') : null;
      $panels.removeClass('first').removeClass('last').removeClass('in-middle');
      const $visiblePanels = $panels.filter(`:not(.tab-hidden):not(.hidden)`);
      $visiblePanels.each((i, el) => {
        const $el = $(el);
        if (i === $visiblePanels.length - 1) {
          if ($bottomPanels && $bottomPanels.first().hasClass('sticked')) {
            if (i === 0) {
              $el.addClass('first');
              return;
            }
            $el.addClass('in-middle');
            return;
          }
          if (i === 0) {
            return;
          }
          $el.addClass('last');
          return;
        }
        if (i > 0 && i < $visiblePanels.length - 1) {
          $el.addClass('in-middle');
          return;
        }
        if (i === 0) {
          $el.addClass('first');
        }
      });
    }

    /**
     * @private
     */
    adjustButtons() {
      const $buttons = this.$detailButtonContainer.filter('.record-buttons').find('button.btn');
      $buttons.removeClass('radius-left').removeClass('radius-right');
      const $buttonsVisible = $buttons.filter('button:not(.hidden)');
      $buttonsVisible.first().addClass('radius-left');
      $buttonsVisible.last().addClass('radius-right');
      this.adjustEditButtons();
    }

    /**
     * @private
     */
    adjustEditButtons() {
      const $buttons = this.$detailButtonContainer.filter('.edit-buttons').find('button.btn');
      $buttons.removeClass('radius-left').removeClass('radius-right');
      const $buttonsVisible = $buttons.filter('button:not(.hidden)');
      $buttonsVisible.first().addClass('radius-left');
      $buttonsVisible.last().addClass('radius-right');
    }

    /**
     * @private
     */
    initElementReferences() {
      if (this.$detailButtonContainer && this.$detailButtonContainer.length) {
        return;
      }
      this.$detailButtonContainer = this.$el.find('.detail-button-container');
      this.$dropdownItemListButton = this.$detailButtonContainer.find('.dropdown-item-list-button');
      this.$dropdownEditItemListButton = this.$detailButtonContainer.find('.dropdown-edit-item-list-button');
    }

    /**
     * @protected
     */
    focusForEdit() {
      this.$el.find('.field:not(.hidden) .form-control:not([disabled])').first().focus();
    }

    /**
     * @protected
     */
    focusForCreate() {
      this.$el.find('.form-control:not([disabled])').first().focus();
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyCtrlEnter(e) {
      const action = this.shortcutKeyCtrlEnterAction;
      if (this.inlineEditModeIsOn || this.buttonsDisabled || !action) {
        return;
      }
      if (this.mode !== this.MODE_EDIT) {
        return;
      }
      if (!this.hasAvailableActionItem(action)) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      if (document.activeElement instanceof HTMLInputElement) {
        // Fields may need to fetch data first.
        document.activeElement.dispatchEvent(new Event('change', {
          bubbles: true
        }));
      }
      const methodName = 'action' + Espo.Utils.upperCaseFirst(action);
      this[methodName]();
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyCtrlS(e) {
      if (this.inlineEditModeIsOn || this.buttonsDisabled) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      if (this.mode !== this.MODE_EDIT) {
        return;
      }
      if (!this.saveAndContinueEditingAction) {
        return;
      }
      if (!this.hasAvailableActionItem('saveAndContinueEditing')) {
        return;
      }
      this.actionSaveAndContinueEditing();
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyCtrlSpace(e) {
      if (this.inlineEditModeIsOn || this.buttonsDisabled) {
        return;
      }
      if (this.type !== this.TYPE_DETAIL || this.mode !== this.MODE_DETAIL) {
        return;
      }
      if (e.target.tagName === 'TEXTAREA' || e.target.tagName === 'INPUT') {
        return;
      }
      if (!this.hasAvailableActionItem('edit')) {
        return;
      }
      $(e.currentTarget);
      e.preventDefault();
      e.stopPropagation();
      this.actionEdit();
      if (!this.editModeDisabled) {
        setTimeout(() => this.focusForEdit(), 200);
      }
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyEscape(e) {
      if (this.inlineEditModeIsOn || this.buttonsDisabled) {
        return;
      }
      if (this.type !== this.TYPE_DETAIL || this.mode !== this.MODE_EDIT) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();

      // Fetching a currently edited form element.
      this.model.set(this.fetch());
      if (this.isChanged) {
        this.confirm(this.translate('confirmLeaveOutMessage', 'messages')).then(() => this.actionCancelEdit());
        return;
      }
      this.actionCancelEdit();
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyCtrlAltEnter(e) {}

    /**
     * @public
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyControlBackslash(e) {
      if (!this.hasTabs()) {
        return;
      }
      const $buttons = this.$el.find('.middle-tabs > button:not(.hidden)');
      if ($buttons.length === 1) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      let index = $buttons.toArray().findIndex(el => $(el).hasClass('active'));
      index++;
      if (index >= $buttons.length) {
        index = 0;
      }
      const $tab = $($buttons.get(index));
      const tab = parseInt($tab.attr('data-tab'));
      this.selectTab(tab);
      if (this.mode === this.MODE_EDIT) {
        setTimeout(() => {
          this.$middle.find(`.panel[data-tab="${tab}"] .cell:not(.hidden)`).first().focus();
        }, 50);
        return;
      }
      this.$el.find(`.middle-tabs button[data-tab="${tab}"]`).focus();
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyControlArrowLeft(e) {
      if (this.inlineEditModeIsOn || this.buttonsDisabled) {
        return;
      }
      if (this.navigateButtonsDisabled) {
        return;
      }
      if (this.type !== this.TYPE_DETAIL || this.mode !== this.MODE_DETAIL) {
        return;
      }
      if (e.target.tagName === 'TEXTAREA' || e.target.tagName === 'INPUT') {
        return;
      }
      const $button = this.$el.find('button[data-action="previous"]');
      if (!$button.length || $button.hasClass('disabled')) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      this.actionPrevious();
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyControlArrowRight(e) {
      if (this.inlineEditModeIsOn || this.buttonsDisabled) {
        return;
      }
      if (this.navigateButtonsDisabled) {
        return;
      }
      if (this.type !== this.TYPE_DETAIL || this.mode !== this.MODE_DETAIL) {
        return;
      }
      if (e.target.tagName === 'TEXTAREA' || e.target.tagName === 'INPUT') {
        return;
      }
      const $button = this.$el.find('button[data-action="next"]');
      if (!$button.length || $button.hasClass('disabled')) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      this.actionNext();
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Get a current mode.
     *
     * @since 8.0.0
     * @return {'detail'|'edit'}
     */
    getMode() {
      return this.mode;
    }
  }
  var _default = _exports.default = DetailRecordView;
});

define("views/fields/base", ["exports", "view", "ui/select", "jquery"], function (_exports, _view, _select, _jquery) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  _select = _interopRequireDefault(_select);
  _jquery = _interopRequireDefault(_jquery);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/base */

  /**
   * A base field view. Can be in different modes. Each mode uses a separate template.
   *
   * @todo Document events.
   * @template TParams
   */
  class BaseFieldView extends _view.default {
    /**
     * @typedef {Object} module:views/fields/base~options
     * @property {string} name A field name.
     * @property {module:model} [model] A model.
     * @property {module:views/fields/base~params | Object.<string, *>} [params] Parameters.
     * @property {boolean} [inlineEditDisabled] Disable inline edit.
     * @property {boolean} [readOnly] Read-only.
     * @property {string} [labelText] A custom label text.
     * @property {'detail'|'edit'|'list'|'search'} [mode] A mode.
     * @property {import('view-record-helper').default} [recordHelper] A record helper.
     */

    /**
     * @typedef {Object} module:views/fields/base~params
     * @property {boolean} [inlineEditDisabled] Disable inline edit.
     * @property {boolean} [readOnly] Read-only.
     */

    /**
     * @param {module:views/fields/base~options | Object.<string, *>} options Options.
     */
    constructor(options) {
      super(options);
      this.name = options.name;
      this.labelText = options.labelText;
    }

    /**
     * A field type.
     *
     * @type {string}
     */
    type = 'base';

    /**
     * List mode template.
     *
     * @protected
     * @type {string}
     */
    listTemplate = 'fields/base/list';

    // noinspection JSUnusedGlobalSymbols
    /**
     * List-link mode template.
     *
     * @protected
     * @type {string}
     */
    listLinkTemplate = 'fields/base/list-link';

    /**
     * Detail mode template.
     *
     * @protected
     * @type {string}
     */
    detailTemplate = 'fields/base/detail';

    /**
     * Edit mode template.
     *
     * @protected
     * @type {string}
     */
    editTemplate = 'fields/base/edit';

    /**
     * Search mode template.
     *
     * @protected
     * @type {string}
     */
    searchTemplate = 'fields/base/search';

    // noinspection JSUnusedGlobalSymbols
    /**
     * @protected
     * @type {string}
     */
    listTemplateContent;

    // noinspection JSUnusedGlobalSymbols
    /**
     * @protected
     * @type {string}
     */
    detailTemplateContent;

    // noinspection JSUnusedGlobalSymbols
    /**
     * @protected
     * @type {string}
     */
    editTemplateContent;

    /**
     * A validation list. A function returning true if non-valid, or a name.
     * For the latter, there should be a `validate{Name}` method in the class.
     *
     * Functions are supported as of v8.3.
     *
     * @type {Array<(function (): boolean)|string>}
     */
    validations = ['required'];

    /**
     * @const
     */
    MODE_LIST = 'list';

    /**
     * @const
     */
    MODE_LIST_LINK = 'listLink';

    /**
     * @const
     */
    MODE_DETAIL = 'detail';

    /**
     * @const
     */
    MODE_EDIT = 'edit';

    /**
     * @const
     */
    MODE_SEARCH = 'search';

    /**
     * A field name.
     *
     * @type {string}
     */
    name;

    /**
     * A field parameter list. To be used for custom fields not defined in metadata > fields.
     *
     * @type {string[]}
     * @since 9.0.0
     */
    paramList;

    /**
     * Definitions.
     *
     * @type {Object}
     */
    defs = null;

    /**
     * Field params.
     *
     * @type {TParams & module:views/fields/base~params & Object.<string, *>}
     */
    params = null;

    /**
     * A mode.
     *
     * @type {'list'|'listLink'|'detail'|'edit'|'search'}
     */
    mode = 'detail';

    /**
     * Search params.
     *
     * @type {Object.<string,*>|null}
     */
    searchParams = null;

    /**
     * Inline edit disabled.
     *
     * @type {boolean}
     */
    inlineEditDisabled = false;

    /**
     * Field is disabled.
     *
     * @type {boolean}
     */
    disabled = false;

    /**
     * Field is read-only.
     *
     * @type {boolean}
     */
    readOnly = false;

    /**
     * A label text.
     *
     * @type {string}
     * @protected
     */
    labelText;

    /**
     * @type {string[]|null}
     */
    attributeList = null;

    /**
     * Attribute values before edit.
     *
     * @type {Object.<string, *>|{}}
     */
    initialAttributes = null;

    /**
     * @const
     */
    VALIDATION_POPOVER_TIMEOUT = 3000;

    /**
     * @type {(function():boolean)}
     * @private
     * @internal
     */
    validateCallback;

    /**
     * An element selector to point validation popovers to.
     *
     * @type {string}
     * @protected
     */
    validationElementSelector;

    /**
     * A view-record helper.
     *
     * @type {module:view-record-helper}
     */
    recordHelper;

    /**
     * @type {JQuery|null}
     * @private
     * @internal
     */
    $label = null;

    /**
     * A form element.
     *
     * @type {JQuery|null}
     * @protected
     */
    $element = null;

    /**
     * Is searchable once a search filter is added (no need to type or selecting anything).
     * Actual for search mode.
     *
     * @public
     * @type {boolean}
     */
    initialSearchIsNotIdle = false;

    /**
     * An entity type.
     *
     * @private
     * @type {string|null}
     */
    entityType = null;

    /**
     * A last validation message;
     *
     * @type {?string}
     */
    lastValidationMessage = null;

    /**
     * Additional data.
     *
     * @type {Object.<string, *>}
     */
    dataObject;

    /**
     * Is the field required.
     *
     * @returns {boolean}
     */
    isRequired() {
      return this.params.required;
    }

    /**
     * Get a cell element. Available only after the view is  rendered.
     *
     * @private
     * @returns {JQuery}
     */
    get$cell() {
      return this.$el.parent();
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @deprecated
     * @returns {JQuery}
     */
    getCellElement() {
      return this.get$cell();
    }

    /**
     * Is in inline-edit mode.
     *
     * @return {boolean}
     */
    isInlineEditMode() {
      return !!this._isInlineEditMode;
    }

    /**
     * Set disabled.
     *
     * @param {boolean} [locked] Won't be able to set back.
     */
    setDisabled(locked) {
      this.disabled = true;
      if (locked) {
        this.disabledLocked = true;
      }
    }

    /**
     * Set not-disabled.
     */
    setNotDisabled() {
      if (this.disabledLocked) {
        return;
      }
      this.disabled = false;
    }

    /**
     * Set required.
     */
    setRequired() {
      this.params.required = true;
      if (this.isEditMode()) {
        if (this.isRendered()) {
          this.showRequiredSign();
        } else {
          this.once('after:render', () => {
            this.showRequiredSign();
          });
        }
      }
    }

    /**
     * Set not required.
     */
    setNotRequired() {
      this.params.required = false;
      this.get$cell().removeClass('has-error');
      if (this.isEditMode()) {
        if (this.isRendered()) {
          this.hideRequiredSign();
        } else {
          this.once('after:render', () => {
            this.hideRequiredSign();
          });
        }
      }
    }

    /**
     * Set read-only.
     *
     * @param {boolean} [locked] Won't be able to set back.
     * @return {Promise}
     */
    setReadOnly(locked) {
      if (this.readOnlyLocked) {
        return Promise.reject();
      }
      this.readOnly = true;
      if (locked) {
        this.readOnlyLocked = true;
      }
      if (this.isEditMode()) {
        if (this.isInlineEditMode()) {
          return this.inlineEditClose();
        }
        return this.setDetailMode().then(() => this.reRender());
      }
      return Promise.resolve();
    }

    /**
     * Set not read only.
     */
    setNotReadOnly() {
      if (this.readOnlyLocked) {
        return;
      }
      this.readOnly = false;
    }

    /**
     * Get a label element. Available only after the view is rendered.
     *
     * @return {JQuery}
     */
    getLabelElement() {
      if (this.$label && this.$label.get(0) && !document.contains(this.$label.get(0))) {
        this.$label = undefined;
      }
      if (!this.$label || !this.$label.length) {
        this.$label = this.$el.parent().children('label');
      }
      return this.$label;
    }

    /**
     * Hide field and label. Available only after the view is rendered.
     */
    hide() {
      this.$el.addClass('hidden');
      const $cell = this.get$cell();
      $cell.children('label').addClass('hidden');
      $cell.addClass('hidden-cell');
    }

    /**
     * Show field and label. Available only after the view is rendered.
     */
    show() {
      this.$el.removeClass('hidden');
      const $cell = this.get$cell();
      $cell.children('label').removeClass('hidden');
      $cell.removeClass('hidden-cell');
    }

    /**
     * @inheritDoc
     * @return {Record<string, *>}
     */
    data() {
      const data = {
        scope: this.model.entityType || this.model.name,
        name: this.name,
        defs: this.defs,
        params: this.params,
        value: this.getValueForDisplay()
      };
      if (this.isSearchMode()) {
        data.searchParams = this.searchParams;
        data.searchData = this.searchData;
        data.searchValues = this.getSearchValues();
        data.searchType = this.getSearchType();
        data.searchTypeList = this.getSearchTypeList();
      }
      return data;
    }

    /**
     * Get a value for display. Is available by using a `{value}` placeholder in templates.
     *
     * @return {*}
     */
    getValueForDisplay() {
      return this.model.get(this.name);
    }

    /**
     * Is in list, detail or list-link mode.
     *
     * @returns {boolean}
     */
    isReadMode() {
      return this.mode === this.MODE_LIST || this.mode === this.MODE_DETAIL || this.mode === this.MODE_LIST_LINK;
    }

    /**
     * Is in list or list-link mode.
     *
     * @returns {boolean}
     */
    isListMode() {
      return this.mode === this.MODE_LIST || this.mode === this.MODE_LIST_LINK;
    }

    /**
     * Is in detail mode.
     *
     * @returns {boolean}
     */
    isDetailMode() {
      return this.mode === this.MODE_DETAIL;
    }

    /**
     * Is in edit mode.
     *
     * @returns {boolean}
     */
    isEditMode() {
      return this.mode === this.MODE_EDIT;
    }

    /**
     * Is in search mode.
     *
     * @returns {boolean}
     */
    isSearchMode() {
      return this.mode === this.MODE_SEARCH;
    }

    /**
     * Set detail mode.
     *
     * @returns {Promise}
     */
    setDetailMode() {
      return this.setMode(this.MODE_DETAIL) || Promise.resolve();
    }

    /**
     * Set edit mode.
     *
     * @returns {Promise}
     */
    setEditMode() {
      return this.setMode(this.MODE_EDIT) || Promise.resolve();
    }

    /**
     * Set a mode.
     *
     * @internal
     * @returns {Promise}
     */
    setMode(mode) {
      const modeIsChanged = this.mode !== mode && this.mode;
      const modeBefore = this.mode;
      this.mode = mode;
      const property = mode + 'Template';
      if (!(property in this)) {
        this[property] = 'fields/' + Espo.Utils.camelCaseToHyphen(this.type) + '/' + this.mode;
      }
      if (!this._hasTemplateContent) {
        this.setTemplate(this[property]);
      }
      const contentProperty = mode + 'TemplateContent';
      if (!this._hasTemplateContent) {
        if (contentProperty in this && this[contentProperty] != null) {
          this.setTemplateContent(this[contentProperty]);
        }
      }
      if (modeIsChanged) {
        if (modeBefore) {
          this.trigger('mode-changed');
        }
        return this._onModeSet();
      }
      return Promise.resolve();
    }

    /**
     * Called on mode change and on value change before re-rendering.
     * To be used for additional initialization that depends on field
     * values or mode.
     *
     * @protected
     * @returns {Promise|undefined}
     */
    prepare() {}

    /**
     * @private
     * @returns {Promise}
     */
    _onModeSet() {
      if (this.isListMode()) {
        return this.onListModeSet() || Promise.resolve();
      }
      if (this.isDetailMode()) {
        return this.onDetailModeSet() || Promise.resolve();
      }
      if (this.isEditMode()) {
        return this.onEditModeSet() || Promise.resolve();
      }
      return Promise.resolve();
    }

    /**
     * Additional initialization for the detail mode.
     *
     * @protected
     * @returns {Promise|undefined}
     */
    onDetailModeSet() {
      return this.prepare();
    }

    /**
     * Additional initialization for the edit mode.
     *
     * @protected
     * @returns {Promise|undefined}
     */
    onEditModeSet() {
      return this.prepare();
    }

    /**
     * Additional initialization for the list mode.
     *
     * @protected
     * @returns {Promise|undefined}
     */
    onListModeSet() {
      return this.prepare();
    }

    /** @inheritDoc */
    init() {
      this.validations = Espo.Utils.clone(this.validations);
      this.searchTypeList = Espo.Utils.clone(this.searchTypeList);
      this._hasTemplateContent = !!this.templateContent;
      this.defs = this.options.defs || {};
      this.name = this.options.name || this.defs.name;
      this.params = this.options.params || this.defs.params || {};
      this.validateCallback = this.options.validateCallback;
      this.fieldType = this.model.getFieldParam(this.name, 'type') || this.type;
      this.entityType = this.model.entityType || this.model.name;
      this.recordHelper = this.options.recordHelper;
      this.dataObject = Espo.Utils.clone(this.options.dataObject || {});
      if (!this.labelText) {
        this.labelText = this.translate(this.name, 'fields', this.entityType);
      }
      const paramList = this.paramList || this.getFieldManager().getParamList(this.type).map(it => it.name);
      paramList.forEach(name => {
        if (name in this.params) {
          return;
        }
        this.params[name] = this.model.getFieldParam(this.name, name);
        if (typeof this.params[name] === 'undefined') {
          this.params[name] = null;
        }
      });
      const additionalParamList = ['inlineEditDisabled'];
      additionalParamList.forEach(item => {
        this.params[item] = this.model.getFieldParam(this.name, item) || null;
      });
      this.readOnly = this.readOnly || this.params.readOnly || this.model.getFieldParam(this.name, 'readOnly') || this.model.getFieldParam(this.name, 'clientReadOnly');
      if (!this.model.isNew() && this.model.getFieldParam(this.name, 'readOnlyAfterCreate')) {
        this.readOnly = true;
      }
      this.readOnlyLocked = this.options.readOnlyLocked || this.readOnly;
      this.inlineEditDisabled = this.options.inlineEditDisabled || this.params.inlineEditDisabled || this.inlineEditDisabled;
      this.readOnly = this.readOnlyLocked || this.options.readOnly || false;
      this.tooltip = this.options.tooltip || this.params.tooltip || this.model.getFieldParam(this.name, 'tooltip') || this.tooltip;
      if (this.options.readOnlyDisabled) {
        this.readOnly = false;
      }
      this.disabledLocked = this.options.disabledLocked || false;
      this.disabled = this.disabledLocked || this.options.disabled || this.disabled;
      let mode = this.options.mode || this.mode || this.MODE_DETAIL;
      if (mode === this.MODE_EDIT && this.readOnly) {
        mode = this.MODE_DETAIL;
      }
      this.mode = undefined;
      this.wait(this.setMode(mode));
      if (this.isSearchMode()) {
        this.searchParams = _.clone(this.options.searchParams || {});
        this.searchData = {};
        this.setupSearch();
        this.events['keydown.' + this.cid] = /** JQueryKeyEventObject */e => {
          if (Espo.Utils.getKeyFromKeyEvent(e) === 'Control+Enter') {
            this.trigger('search');
          }
        };
      }
      this.on('highlight', () => {
        const $cell = this.get$cell();
        $cell.addClass('highlighted');
        $cell.addClass('transition');
        setTimeout(() => {
          $cell.removeClass('highlighted');
        }, 3000);
        setTimeout(() => {
          $cell.removeClass('transition');
        }, 3000 + 2000);
      });
      this.on('invalid', () => {
        const $cell = this.get$cell();
        $cell.addClass('has-error');
        this.$el.one('click', () => {
          $cell.removeClass('has-error');
        });
        this.once('render', () => {
          $cell.removeClass('has-error');
        });
      });
      this.on('after:render', () => {
        if (this.isEditMode()) {
          if (this.hasRequiredMarker()) {
            this.showRequiredSign();
            return;
          }
          this.hideRequiredSign();
          return;
        }
        if (this.hasRequiredMarker()) {
          this.hideRequiredSign();
        }
        if (this.isSearchMode()) {
          const $searchType = this.$el.find('select.search-type');
          if ($searchType.length) {
            _select.default.init($searchType, {
              matchAnyWord: true
            });
          }
        }
      });
      if ((this.isDetailMode() || this.isEditMode()) && this.tooltip) {
        this.initTooltip();
      }
      if (this.isDetailMode()) {
        if (!this.inlineEditDisabled) {
          this.listenToOnce(this, 'after:render', () => this.initInlineEdit());
        }
      }
      if (!this.isSearchMode()) {
        this.attributeList = this.getAttributeList(); // for backward compatibility, to be removed

        this.listenTo(this.model, 'change', (model, options) => {
          if (options.ui) {
            return;
          }
          let changed = false;
          for (const attribute of this.getAttributeList()) {
            if (model.hasChanged(attribute)) {
              changed = true;
              break;
            }
          }
          if (!changed) {
            return;
          }
          if (options.fromField === this.name) {
            return;
          }
          if (options.skipReRenderInEditMode && this.isEditMode()) {
            return;
          }
          if (options.skipReRender) {
            return;
          }
          const reRender = () => {
            if (!this.isRendered() && !this.isBeingRendered()) {
              return;
            }
            this.reRender();
            if (options.highlight) {
              this.trigger('highlight');
            }
          };
          if (!this.isReady) {
            this.once('ready', () => {
              const promise = this.prepare();
              if (promise) {
                promise.then(() => reRender());
              }
            });
            return;
          }
          const promise = this.prepare();
          if (promise) {
            promise.then(() => reRender());
            return;
          }
          reRender();
        });
        this.listenTo(this, 'change', () => {
          const attributes = this.fetch();
          this.model.set(attributes, {
            ui: true,
            fromView: this
          });
        });
      }
    }
    highlight() {
      const $cell = this.get$cell();
      $cell.addClass('highlighted');
    }

    /** @inheritDoc */
    setupFinal() {
      this.wait(this._onModeSet());
    }

    /**
     * @internal
     * @private
     */
    initTooltip() {
      let $a;
      this.once('after:render', () => {
        $a = (0, _jquery.default)('<a>').attr('role', 'button').attr('tabindex', '-1').addClass('text-muted field-info').append((0, _jquery.default)('<span>').addClass('fas fa-info-circle'));
        const $label = this.getLabelElement();
        $label.append(' ');
        this.getLabelElement().append($a);
        let tooltipText = this.options.tooltipText || this.tooltipText;
        if (!tooltipText && typeof this.tooltip === 'string') {
          const [scope, field] = this.tooltip.includes('.') ? this.tooltip.split('.') : [this.entityType, this.tooltip];
          tooltipText = this.translate(field, 'tooltips', scope);
        }
        tooltipText = tooltipText || this.translate(this.name, 'tooltips', this.entityType) || '';
        tooltipText = this.getHelper().transformMarkdownText(tooltipText, {
          linksInNewTab: true
        }).toString();
        Espo.Ui.popover($a, {
          placement: 'bottom',
          content: tooltipText,
          preventDestroyOnRender: true
        }, this);
      });
    }

    /**
     * Show a required-field sign.
     *
     * @private
     */
    showRequiredSign() {
      const $label = this.getLabelElement();
      let $sign = $label.find('span.required-sign');
      if ($label.length && !$sign.length) {
        const $text = $label.find('span.label-text');
        (0, _jquery.default)('<span class="required-sign"> *</span>').insertAfter($text);
        $sign = $label.find('span.required-sign');
      }
      $sign.show();
    }

    /**
     * Hide a required-field sign.
     *
     * @private
     */
    hideRequiredSign() {
      const $label = this.getLabelElement();
      const $sign = $label.find('span.required-sign');
      $sign.hide();
    }

    /**
     * Get search-params data.
     *
     * @protected
     * @return {Object.<string,*>}
     */
    getSearchParamsData() {
      return this.searchParams.data || {};
    }

    /**
     * Get search values.
     *
     * @protected
     * @return {Object.<string,*>}
     */
    getSearchValues() {
      return this.getSearchParamsData().values || {};
    }

    /**
     * Get a current search type.
     *
     * @protected
     * @return {string}
     */
    getSearchType() {
      return this.getSearchParamsData().type || this.searchParams.type;
    }

    /**
     * Get the search type list.
     *
     * @protected
     * @returns {string[]}
     */
    getSearchTypeList() {
      return this.searchTypeList;
    }

    /**
     * @private
     * @internal
     */
    initInlineEdit() {
      const $cell = this.get$cell();
      const $editLink = (0, _jquery.default)('<a>').attr('role', 'button').addClass('pull-right inline-edit-link hidden').append((0, _jquery.default)('<span>').addClass('fas fa-pencil-alt fa-sm'));
      if ($cell.length === 0) {
        this.listenToOnce(this, 'after:render', () => this.initInlineEdit());
        return;
      }
      $cell.prepend($editLink);
      $editLink.on('click', () => this.inlineEdit());
      $cell.on('mouseenter', e => {
        e.stopPropagation();
        if (this.disabled || this.readOnly) {
          return;
        }
        if (this.isDetailMode()) {
          $editLink.removeClass('hidden');
        }
      }).on('mouseleave', e => {
        e.stopPropagation();
        if (this.isDetailMode()) {
          $editLink.addClass('hidden');
        }
      });
      this.on('after:render', () => {
        if (!this.isDetailMode()) {
          $editLink.addClass('hidden');
        }
      });
    }

    /**
     * Initializes a form element reference.
     *
     * @protected
     */
    initElement() {
      this.$element = this.$el.find('[data-name="' + this.name + '"]');
      if (!this.$element.length) {
        this.$element = this.$el.find('[name="' + this.name + '"]');
      }
      if (!this.$element.length) {
        this.$element = this.$el.find('.main-element');
      }
      if (this.isEditMode()) {
        this.$element.on('change', () => {
          this.trigger('change');
        });
      }
    }

    /** @inheritDoc */
    afterRender() {
      if (this.isEditMode() || this.isSearchMode()) {
        this.initElement();
      }
      if (this.isReadMode()) {
        this.afterRenderRead();
      }
      if (this.isListMode()) {
        this.afterRenderList();
      }
      if (this.isDetailMode()) {
        this.afterRenderDetail();
      }
      if (this.isEditMode()) {
        this.afterRenderEdit();
      }
      if (this.isSearchMode()) {
        this.afterRenderSearch();
      }
    }

    /**
     * Called after the view is rendered in list or read mode.
     *
     * @protected
     */
    afterRenderRead() {}

    /**
     * Called after the view is rendered in list mode.
     *
     * @protected
     */
    afterRenderList() {}

    /**
     * Called after the view is rendered in detail mode.
     *
     * @protected
     */
    afterRenderDetail() {}

    /**
     * Called after the view is rendered in edit mode.
     *
     * @protected
     */
    afterRenderEdit() {}

    /**
     * Called after the view is rendered in search mode.
     *
     * @protected
     */
    afterRenderSearch() {}

    /**
     * Initialization.
     */
    setup() {}

    /**
     * Initialization for search mode.
     *
     * @protected
     */
    setupSearch() {}

    /**
     * Get list of model attributes that relate to the field.
     * Changing of any attributes makes the field to re-render.
     *
     * @return {string[]}
     */
    getAttributeList() {
      return this.getFieldManager().getAttributeList(this.fieldType, this.name);
    }

    /**
     * Invoke inline-edit saving.
     *
     * @param {{[bypassClose]: boolean}} [options]
     */
    inlineEditSave(options) {
      options = options || {};
      if (this.recordHelper) {
        this.recordHelper.trigger('inline-edit-save', this.name, options);
        return;
      }

      // Code below supposed not to be executed.

      let data = this.fetch();
      const model = this.model;
      const prev = this.initialAttributes;
      model.set(data, {
        silent: true
      });
      data = model.attributes;
      let attrs = false;
      for (const attr in data) {
        if (_.isEqual(prev[attr], data[attr])) {
          continue;
        }
        (attrs || (attrs = {}))[attr] = data[attr];
      }
      if (!attrs) {
        this.inlineEditClose();
      }
      const isInvalid = this.validateCallback ? this.validateCallback() : this.validate();
      if (isInvalid) {
        Espo.Ui.error(this.translate('Not valid'));

        // @todo Revise.
        model.set(prev, {
          silent: true
        });
        return;
      }
      Espo.Ui.notify(this.translate('saving', 'messages'));
      model.save(/** @type Object */attrs, {
        patch: true
      }).then(() => {
        this.trigger('after:inline-save');
        this.trigger('after:save');
        model.trigger('after:save');
        Espo.Ui.success(this.translate('Saved'));
      }).catch(() => {
        Espo.Ui.error(this.translate('Error occurred'));

        // @todo Revise.
        model.set(prev, {
          silent: true
        });
        this.reRender();
      });
      if (!options.bypassClose) {
        this.inlineEditClose(true);
      }
    }

    /**
     * @public
     */
    removeInlineEditLinks() {
      const $cell = this.get$cell();
      $cell.find('.inline-save-link').remove();
      $cell.find('.inline-cancel-link').remove();
      $cell.find('.inline-edit-link').addClass('hidden');
    }

    /**
     * @private
     */
    addInlineEditLinks() {
      const $cell = this.get$cell();
      const saveLink = document.createElement('a');
      saveLink.role = 'button';
      saveLink.tabIndex = -1;
      saveLink.title = this.translate('Update') + ' · ' + 'Ctrl+Enter';
      saveLink.innerHTML = `<span class="fas fa-check"></span>`;
      saveLink.classList.add('inline-save-link');
      const cancelLink = document.createElement('a');
      cancelLink.role = 'button';
      cancelLink.tabIndex = -1;
      cancelLink.title = this.translate('Cancel') + ' · ' + 'Esc';
      cancelLink.innerHTML = `<span class="fas fa-arrow-right-to-bracket"></span>`;
      cancelLink.classList.add('inline-cancel-link');
      $cell.prepend(saveLink);
      $cell.prepend(cancelLink);
      $cell.find('.inline-edit-link').addClass('hidden');
      saveLink.onclick = () => this.inlineEditSave();
      cancelLink.onclick = () => this.inlineEditClose();
    }

    /**
     * @public
     * @param {boolean} value
     * @internal
     */
    setIsInlineEditMode(value) {
      this._isInlineEditMode = value;
    }

    /**
     * Exist inline-edit mode.
     *
     * @param {boolean} [noReset]
     * @return {Promise}
     */
    inlineEditClose(noReset) {
      this.trigger('inline-edit-off', {
        noReset: noReset
      });
      this.$el.off('keydown.inline-edit');
      this._isInlineEditMode = false;
      if (!this.isEditMode()) {
        return Promise.resolve();
      }
      if (!noReset) {
        this.model.set(this.initialAttributes, {
          skipReRenderInEditMode: true
        });
      }
      const promise = this.setDetailMode().then(() => this.reRender(true)).then(() => this.removeInlineEditLinks());
      this.trigger('after:inline-edit-off', {
        noReset: noReset
      });
      return promise;
    }

    /**
     * Switch to inline-edit mode.
     *
     * @return {Promise}
     */
    inlineEdit() {
      this.trigger('edit', this);
      this.initialAttributes = this.model.getClonedAttributes();
      this._isInlineEditMode = true;
      const promise = this.setEditMode().then(() => this.reRender(true)).then(() => this.addInlineEditLinks()).then(() => {
        this.$el.on('keydown.inline-edit', e => {
          const key = Espo.Utils.getKeyFromKeyEvent(e);
          if (key === 'Control+Enter') {
            e.stopPropagation();
            if (document.activeElement instanceof HTMLInputElement) {
              // Fields may need to fetch data first.
              document.activeElement.dispatchEvent(new Event('change', {
                bubbles: true
              }));
            }
            this.fetchToModel();
            this.inlineEditSave();
            setTimeout(() => {
              this.get$cell().focus();
            }, 100);
            return;
          }
          if (key === 'Escape') {
            e.stopPropagation();
            this.inlineEditClose().then(() => {
              this.get$cell().focus();
            });
            return;
          }
          if (key === 'Control+KeyS') {
            e.preventDefault();
            e.stopPropagation();
            this.fetchToModel();
            this.inlineEditSave({
              bypassClose: true
            });
          }
        });
        setTimeout(() => this.focusOnInlineEdit(), 10);
      });
      this.trigger('inline-edit-on');
      return promise;
    }

    /**
     * @protected
     */
    focusOnInlineEdit() {
      const $element = this.$element && this.$element.length ? this.$element : this.$el.find('.form-control').first();
      if (!$element) {
        return;
      }
      $element.first().focus();
    }

    /**
     * Suspend a validation message.
     *
     * @internal
     * @param {number} [time=200]
     */
    suspendValidationMessage(time) {
      this.validationMessageSuspended = true;
      setTimeout(() => this.validationMessageSuspended = false, time || 200);
    }

    /**
     * Show a validation message.
     *
     * @param {string} message A message.
     * @param {string|JQuery|Element} [target] A target element or selector.
     * @param {module:view} [view] A child view that contains the target. The closest view should to passed.
     *   Should be omitted if there is no child views or the target is not rendered by a child view.
     */
    showValidationMessage(message, target, view) {
      if (this.validationMessageSuspended) {
        return;
      }
      let $el;
      target = target || this.validationElementSelector || '.main-element';
      if (typeof target === 'string' || target instanceof String) {
        $el = this.$el.find(target);
      } else {
        $el = (0, _jquery.default)(target);
      }
      if (!$el.length && this.$element) {
        $el = this.$element;
      }
      if (!$el.length) {
        $el = this.$el;
      }
      if ($el.length) {
        const rect = $el.get(0).getBoundingClientRect();
        this.lastValidationMessage = message;
        if (rect.top === 0 && rect.bottom === 0 && rect.left === 0) {
          return;
        }
      }
      this._popoverMap = this._popoverMap || new WeakMap();
      const element = $el.get(0);
      if (!element) {
        return;
      }
      if (this._popoverMap.has(element)) {
        try {
          this._popoverMap.get(element).detach();
        } catch (e) {}
      }
      const popover = Espo.Ui.popover($el, {
        placement: 'bottom',
        container: 'body',
        content: this.getHelper().transformMarkdownText(message).toString(),
        trigger: 'manual',
        noToggleInit: true,
        noHideOnOutsideClick: true
      }, view || this);
      popover.show();
      this._popoverMap.set(element, popover);
      $el.closest('.field').one('mousedown click', () => popover.destroy());
      this.once('render remove', () => popover.destroy());
      this._timeoutMap = this._timeoutMap || new WeakMap();
      if (this._timeoutMap.has(element)) {
        clearTimeout(this._timeoutMap.get(element));
      }
      const timeout = setTimeout(() => {
        popover.destroy();
      }, this.VALIDATION_POPOVER_TIMEOUT);
      this._timeoutMap.set(element, timeout);
    }

    /**
     * Validate field values.
     *
     * @return {boolean} True if not valid.
     */
    validate() {
      this.lastValidationMessage = null;
      for (const item of this.validations) {
        let notValid = false;
        if (typeof item === 'function') {
          notValid = item();
        } else {
          const method = 'validate' + Espo.Utils.upperCaseFirst(item);
          notValid = this[method].call(this);
        }
        if (notValid) {
          this.trigger('invalid');
          return true;
        }
      }
      return false;
    }

    /**
     * Get a label text.
     *
     * @returns {string}
     */
    getLabelText() {
      return this.labelText;
    }

    /**
     * Validate required.
     *
     * @return {boolean}
     */
    validateRequired() {
      if (this.isRequired()) {
        if (this.model.get(this.name) === '' || this.model.get(this.name) === null) {
          const msg = this.translate('fieldIsRequired', 'messages').replace('{field}', this.getLabelText());
          this.showValidationMessage(msg);
          return true;
        }
      }
    }

    /**
     * Defines whether the field should have a required-marker rendered.
     *
     * @protected
     * @return {boolean}
     */
    hasRequiredMarker() {
      return this.isRequired();
    }

    /**
     * Fetch field values to the model.
     */
    fetchToModel() {
      this.model.set(this.fetch(), {
        silent: true
      });
    }

    /**
     * Fetch field values from DOM.
     *
     * @return {Object.<string, *>}
     */
    fetch() {
      if (!this.$element.length) {
        return {};
      }
      const data = {};
      data[this.name] = this.$element.val().trim();
      return data;
    }

    /**
     * Fetch search data from DOM.
     *
     * @return {Object.<string, *>|null}
     */
    fetchSearch() {
      const value = this.$element.val().toString().trim();
      if (value) {
        return {
          type: 'equals',
          value: value
        };
      }
      return null;
    }

    /**
     * Fetch a search type from DOM.
     *
     * @return {string}
     */
    fetchSearchType() {
      return this.$el.find('select.search-type').val();
    }
  }
  var _default = _exports.default = BaseFieldView;
});

define("ui/autocomplete", ["exports", "jquery", "handlebars"], function (_exports, _jquery, _handlebars) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _jquery = _interopRequireDefault(_jquery);
  _handlebars = _interopRequireDefault(_handlebars);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /**
   * An autocomplete.
   */
  class Autocomplete {
    /** @module ui/autocomplete */

    /**
     * @typedef {Object} module:ui/autocomplete~item
     * @property {string} value
     */

    /**
     * @typedef {{
     *     name?: string,
     *     forceHide?: boolean,
     *     lookup?: string[],
     *     lookupFunction?: function (string): Promise<Array<module:ui/autocomplete~item & Record>>,
     *     minChars?: Number,
     *     formatResult?: function (module:ui/autocomplete~item & Record): string,
     *     onSelect?: function (module:ui/autocomplete~item & Record): void,
     *     beforeRender?: function (HTMLElement): void,
     *     triggerSelectOnValidInput?: boolean,
     *     autoSelectFirst?: boolean,
     *     handleFocusMode?: 1|2|3,
     *     focusOnSelect?: boolean,
     * }} module:ui/autocomplete~options
     */

    /**
     * @param {HTMLInputElement} element
     * @param {module:ui/autocomplete~options} options
     */
    constructor(element, options) {
      /** @private */
      this.$element = (0, _jquery.default)(element);
      this.$element.on('keydown', e => {
        if (e.code === 'Tab' && !this.$element.val()) {
          e.stopImmediatePropagation();
        }
      });
      const lookup = options.lookupFunction ? (query, done) => {
        options.lookupFunction(query).then(items => {
          done({
            suggestions: items
          });
        });
      } : options.lookup;
      const lookupFilter = !options.lookupFunction ? (/** (module:ui/autocomplete~item */suggestion, /** string */query, /** string */queryLowerCase) => {
        if (suggestion.value.toLowerCase().indexOf(queryLowerCase) === 0) {
          return suggestion.value.length !== queryLowerCase.length;
        }
        return false;
      } : undefined;
      const $modalBody = this.$element.closest('.modal-body');
      this.$element.autocomplete({
        beforeRender: $container => {
          if (options.beforeRender) {
            options.beforeRender($container.get(0));
          }
          if (this.$element.hasClass('input-sm')) {
            $container.addClass('small');
          }
          if (options.forceHide) {
            // Prevent an issue that suggestions are shown and not hidden
            // when clicking outside the window and then focusing back on the document.
            if (this.$element.get(0) !== document.activeElement) {
              setTimeout(() => this.$element.autocomplete('hide'), 30);
            }
          }
        },
        lookup: lookup,
        minChars: options.minChars || 0,
        noCache: true,
        autoSelectFirst: options.autoSelectFirst,
        appendTo: $modalBody.length ? $modalBody : 'body',
        forceFixPosition: true,
        formatResult: item => {
          if (options.formatResult) {
            return options.formatResult(item);
          }
          return _handlebars.default.Utils.escapeExpression(item.value);
        },
        lookupFilter: lookupFilter,
        onSelect: item => {
          if (options.onSelect) {
            options.onSelect(item);
          }
          if (options.focusOnSelect) {
            this.$element.focus();
          }
        },
        triggerSelectOnValidInput: options.triggerSelectOnValidInput || false
      });
      this.$element.attr('autocomplete', 'espo-' + (options.name || 'dummy'));
      if (options.handleFocusMode) {
        this.initHandleFocus(options);
      }
    }

    /**
     * @private
     * @param {module:ui/autocomplete~options} options
     */
    initHandleFocus(options) {
      this.$element.off('focus.autocomplete');
      this.$element.on('focus', () => {
        if (options.handleFocusMode === 1) {
          if (this.$element.val()) {
            return;
          }
          this.$element.autocomplete('onValueChange');
          return;
        }
        if (this.$element.val()) {
          // noinspection JSUnresolvedReference
          this.$element.get(0).select();
          return;
        }
        this.$element.autocomplete('onFocus');
      });
      if (options.handleFocusMode === 3) {
        this.$element.on('change', () => this.$element.val(''));
      }
    }

    /**
     * Dispose.
     */
    dispose() {
      this.$element.autocomplete('dispose');
    }

    /**
     * Hide.
     */
    hide() {
      this.$element.autocomplete('hide');
    }

    /**
     * Clear.
     */
    clear() {
      this.$element.autocomplete('clear');
    }
  }
  var _default = _exports.default = Autocomplete;
});

define("helpers/reg-exp-pattern", ["exports", "di", "metadata", "language"], function (_exports, _di, _metadata, _language) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _metadata = _interopRequireDefault(_metadata);
  _language = _interopRequireDefault(_language);
  let _init_metadata, _init_extra_metadata, _init_language, _init_extra_language;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * A regular expression pattern helper.
   */
  class RegExpPatternHelper {
    static {
      [_init_metadata, _init_extra_metadata, _init_language, _init_extra_language] = _applyDecs(this, [], [[(0, _di.inject)(_metadata.default), 0, "metadata"], [(0, _di.inject)(_language.default), 0, "language"]]).e;
    }
    constructor() {
      _init_extra_language(this);
    }
    /**
     * @private
     * @type {Metadata}
     */
    metadata = _init_metadata(this);

    /**
     * @private
     * @type {Language}
     */
    language = (_init_extra_metadata(this), _init_language(this));

    /**
     *
     * @param {string} pattern
     * @param {string|null} value
     * @param {string} [field]
     * @param {string} [entityType]
     * @return {{message: string}|null}
     */
    validate(pattern, value, field, entityType) {
      if (value === '' || value === null) {
        return null;
      }
      let messageKey = 'fieldNotMatchingPattern';
      if (pattern[0] === '$') {
        const patternName = pattern.slice(1);
        const foundPattern = this.metadata.get(['app', 'regExpPatterns', patternName, 'pattern']);
        if (foundPattern) {
          messageKey += '$' + patternName;
          pattern = foundPattern;
        }
      }
      const regExp = new RegExp('^' + pattern + '$');
      if (regExp.test(value)) {
        return null;
      }
      let message = this.language.translate(messageKey, 'messages').replace('{pattern}', pattern);
      if (field && entityType) {
        message = message.replace('{field}', this.language.translate(field, 'fields', entityType));
      }
      return {
        message: message
      };
    }
  }
  var _default = _exports.default = RegExpPatternHelper;
});

define("views/record/edit", ["exports", "views/record/detail"], function (_exports, _detail) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _detail = _interopRequireDefault(_detail);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/record/edit */

  /**
   * An edit-record view. Used for create and edit.
   */
  class EditRecordView extends _detail.default {
    /** @inheritDoc */
    template = 'record/edit';

    /** @inheritDoc */
    type = 'edit';
    /** @inheritDoc */
    fieldsMode = 'edit';
    /** @inheritDoc */
    mode = 'edit';

    /**
     * @inheritDoc
     * @type {module:views/record/detail~button[]}
     */
    buttonList = [{
      name: 'save',
      label: 'Save',
      style: 'primary',
      title: 'Ctrl+Enter'
    }, {
      name: 'cancel',
      label: 'Cancel',
      title: 'Esc'
    }];
    /** @inheritDoc */
    dropdownItemList = [];
    /** @inheritDoc */
    sideView = 'views/record/edit-side';
    /** @inheritDoc */
    bottomView = 'views/record/edit-bottom';
    /** @inheritDoc */
    duplicateAction = false;
    /** @inheritDoc */
    saveAndContinueEditingAction = true;
    /** @inheritDoc */
    saveAndNewAction = true;
    /** @inheritDoc */
    setupHandlerType = 'record/edit';

    /**
     * @param {
     *     module:views/record/detail~options |
     *     {
     *         duplicateSourceId?: string,
     *         focusForCreate?: boolean,
     *     }
     * } options Options.
     */
    constructor(options) {
      super(options);
    }

    /** @inheritDoc */
    actionSave(data) {
      data = data || {};
      const isNew = this.isNew;
      return this.save(data.options).then(() => {
        if (this.options.duplicateSourceId) {
          this.returnUrl = null;
        }
        this.exit(isNew ? 'create' : 'save');
      }).catch(reason => Promise.reject(reason));
    }

    /**
     * A `cancel` action.
     */
    actionCancel() {
      this.cancel();
    }

    /**
     * Cancel.
     */
    cancel() {
      if (this.isChanged) {
        this.resetModelChanges();
      }
      this.setIsNotChanged();
      this.exit('cancel');
    }

    /** @inheritDoc */
    setupBeforeFinal() {
      /** @type {Promise|undefined} */
      let promise = undefined;
      if (this.model.isNew()) {
        promise = this.populateDefaults();
      }
      if (!promise) {
        // Attributes are yet not ready.
        super.setupBeforeFinal();
      }
      if (promise) {
        this.wait(promise);

        // @todo Revise. Possible race condition issues.
        promise.then(() => super.setupBeforeFinal());
      }
      if (this.model.isNew()) {
        this.once('after:render', () => {
          this.model.set(this.fetch(), {
            silent: true
          });
        });
      }
      if (this.options.focusForCreate) {
        this.once('after:render', () => {
          if (this.$el.closest('.modal').length) {
            setTimeout(() => this.focusForCreate(), 50);
            return;
          }
          this.focusForCreate();
        });
      }
      this.setupHighlight();
    }

    /** @inheritDoc */
    setupActionItems() {
      super.setupActionItems();
      if (this.saveAndContinueEditingAction && this.getAcl().checkScope(this.entityType, 'edit')) {
        this.dropdownItemList.push({
          name: 'saveAndContinueEditing',
          label: 'Save & Continue Editing',
          title: 'Ctrl+S'
        });
      }
      if (this.isNew && this.saveAndNewAction && this.getAcl().checkScope(this.entityType, 'create')) {
        this.dropdownItemList.push({
          name: 'saveAndNew',
          label: 'Save & New',
          title: 'Ctrl+Alt+Enter'
        });
      }
    }

    /**
     * A `save-and-create-new` action.
     */
    actionSaveAndNew(data) {
      data = data || {};
      const proceedCallback = () => {
        Espo.Ui.success(this.translate('Created'));
        this.getRouter().dispatch(this.scope, 'create', {
          rootUrl: this.options.rootUrl,
          focusForCreate: !!data.focusForCreate
        });
        this.getRouter().navigate('#' + this.scope + '/create', {
          trigger: false
        });
      };
      this.save(data.options).then(proceedCallback).catch(() => {});
      if (this.lastSaveCancelReason === 'notModified') {
        proceedCallback();
      }
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyEscape(e) {
      if (this.buttonsDisabled) {
        return;
      }
      if (this.buttonList.findIndex(item => item.name === 'cancel' && !item.hidden) === -1) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      const focusedFieldView = this.getFocusedFieldView();
      if (focusedFieldView) {
        this.model.set(focusedFieldView.fetch());
      }
      if (this.isChanged) {
        this.confirm(this.translate('confirmLeaveOutMessage', 'messages')).then(() => this.actionCancel());
        return;
      }
      this.actionCancel();
    }

    /**
     * @protected
     * @param {KeyboardEvent} e
     */
    handleShortcutKeyCtrlAltEnter(e) {
      if (this.buttonsDisabled) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      if (!this.saveAndNewAction) {
        return;
      }
      if (!this.hasAvailableActionItem('saveAndNew')) {
        return;
      }
      this.actionSaveAndNew({
        focusForCreate: true
      });
    }

    /** @private */
    setupHighlight() {
      if (!this.options.highlightFieldList) {
        return;
      }
      this.on('after:render', () => {
        const fieldList = /** @type {string[]} */this.options.highlightFieldList;
        fieldList.map(it => this.getFieldView(it)).filter(view => view).forEach(view => view.highlight());
      });
    }
  }
  var _default = _exports.default = EditRecordView;
});

define("views/fields/varchar", ["exports", "views/fields/base", "helpers/reg-exp-pattern", "ui/autocomplete"], function (_exports, _base, _regExpPattern, _autocomplete) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _base = _interopRequireDefault(_base);
  _regExpPattern = _interopRequireDefault(_regExpPattern);
  _autocomplete = _interopRequireDefault(_autocomplete);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/varchar */

  /**
   * A varchar field.
   *
   * @extends BaseFieldView<module:views/fields/varchar~params>
   */
  class VarcharFieldView extends _base.default {
    /**
     * @typedef {Object} module:views/fields/varchar~options
     * @property {
     *     module:views/fields/varchar~params &
     *     module:views/fields/base~params &
     *     Record
     * } [params] Parameters.
     */

    /**
     * @typedef {Object} module:views/fields/varchar~params
     * @property {number} [maxLength] A max length.
     * @property {string[]} [options] Select options.
     * @property {boolean} [required] Required.
     * @property {string} [optionsPath] An options metadata path.
     * @property {boolean} [noSpellCheck] Disable spell check.
     * @property {string} [pattern] A validation pattern. If starts with `$`, then a predefined pattern is used.
     * @property {boolean} [copyToClipboard] To display a Copy-to-clipboard button.
     */

    /**
     * @param {
     *     module:views/fields/varchar~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }
    type = 'varchar';
    listTemplate = 'fields/varchar/list';
    detailTemplate = 'fields/varchar/detail';
    searchTemplate = 'fields/varchar/search';
    searchTypeList = ['startsWith', 'contains', 'equals', 'endsWith', 'like', 'notContains', 'notEquals', 'notLike', 'isEmpty', 'isNotEmpty'];

    /**
     * @inheritDoc
     * @type {Array<(function (): boolean)|string>}
     */
    validations = ['required', 'pattern'];

    /**
     * Use an autocomplete requesting data from the backend.
     *
     * @protected
     * @type {boolean}
     */
    useAutocompleteUrl = false;

    /**
     * No spell-check.
     *
     * @protected
     * @type {boolean}
     */
    noSpellCheck = false;
    setup() {
      this.setupOptions();
      this.noSpellCheck = this.noSpellCheck || this.params.noSpellCheck;
      if (this.params.optionsPath) {
        this.params.options = Espo.Utils.clone(this.getMetadata().get(this.params.optionsPath) || []);
      }
      if (this.options.customOptionList) {
        this.setOptionList(this.options.customOptionList);
      }
      if (this.mode === this.MODE_DETAIL) {
        if (this.params.copyToClipboard) {
          this.events['click [data-action="copyToClipboard"]'] = () => this.copyToClipboard();
        }
      }
    }

    /**
     * Set up options.
     */
    setupOptions() {}

    /**
     * Set options.
     *
     * @param {string[]} optionList Options.
     */
    setOptionList(optionList) {
      if (!this.originalOptionList) {
        this.originalOptionList = this.params.options || [];
      }
      this.params.options = Espo.Utils.clone(optionList);
      if (this.isEditMode()) {
        if (this.isRendered()) {
          this.reRender();
        }
      }
    }

    /**
     * Reset options.
     */
    resetOptionList() {
      if (this.originalOptionList) {
        this.params.options = Espo.Utils.clone(this.originalOptionList);
      }
      if (this.isEditMode()) {
        if (this.isRendered()) {
          this.reRender();
        }
      }
    }

    /**
     * @protected
     */
    copyToClipboard() {
      const value = this.model.get(this.name);
      navigator.clipboard.writeText(value).then(() => {
        Espo.Ui.success(this.translate('Copied to clipboard'));
      });
    }

    // noinspection JSUnusedLocalSymbols
    /**
     * Compose an autocomplete URL.
     *
     * @param {string} q A query.
     * @return {string}
     */
    getAutocompleteUrl(q) {
      return '';
    }

    /**
     * @return {module:ui/autocomplete~item[]}
     */
    transformAutocompleteResult(response) {
      const responseParsed = typeof response === 'string' ? JSON.parse(response) : response;
      const list = [];
      responseParsed.list.forEach(item => {
        list.push({
          value: item.name || item.id,
          attributes: item
        });
      });
      return list;
    }
    setupSearch() {
      this.events['change select.search-type'] = e => {
        const type = $(e.currentTarget).val();
        this.handleSearchType(type);
      };
    }
    data() {
      const data = super.data();
      if (this.model.get(this.name) !== null && this.model.get(this.name) !== '' && this.model.has(this.name)) {
        data.isNotEmpty = true;
      }
      data.valueIsSet = this.model.has(this.name);
      if (this.isSearchMode()) {
        if (typeof this.searchParams.value === 'string') {
          this.searchData.value = this.searchParams.value;
        }
        if (this.searchParams.data && typeof this.searchParams.data.value === 'string') {
          this.searchData.value = this.searchParams.data.value;
        }
        if (!this.searchParams.value && !this.searchParams.data) {
          this.searchData.value = null;
        }
      }
      data.noSpellCheck = this.noSpellCheck;
      data.copyToClipboard = this.params.copyToClipboard;
      data.textClass = null;
      return data;
    }
    handleSearchType(type) {
      if (~['isEmpty', 'isNotEmpty'].indexOf(type)) {
        this.$el.find('input.main-element').addClass('hidden');
        return;
      }
      this.$el.find('input.main-element').removeClass('hidden');
    }
    afterRender() {
      super.afterRender();
      if (this.isSearchMode()) {
        const type = this.$el.find('select.search-type').val();
        this.handleSearchType(type);
      }
      if ((this.isEditMode() || this.isSearchMode()) && (this.params.options && this.params.options.length || this.useAutocompleteUrl)) {
        let lookupFunction = this.getAutocompleteLookupFunction();
        if (this.useAutocompleteUrl) {
          lookupFunction = query => {
            return Espo.Ajax.getRequest(this.getAutocompleteUrl(query)).then(response => this.transformAutocompleteResult(response));
          };
        }
        const autocomplete = new _autocomplete.default(this.$element.get(0), {
          name: this.name,
          triggerSelectOnValidInput: true,
          autoSelectFirst: true,
          handleFocusMode: 1,
          focusOnSelect: true,
          onSelect: () => this.trigger('change'),
          lookup: this.params.options,
          lookupFunction: lookupFunction
        });
        this.once('render remove', () => autocomplete.dispose());
      }
      if (this.isSearchMode()) {
        this.$el.find('select.search-type').on('change', () => {
          this.trigger('change');
        });
        this.$element.on('input', () => {
          this.trigger('change');
        });
      }
    }

    // noinspection JSUnusedGlobalSymbols
    validatePattern() {
      const pattern = this.params.pattern;
      return this.fieldValidatePattern(this.name, pattern);
    }

    /**
     * Used by other field views.
     *
     * @param {string} name
     * @param {string} [pattern]
     */
    fieldValidatePattern(name, pattern) {
      pattern = pattern || this.model.getFieldParam(name, 'pattern');
      /** @var {string|null} value */
      const value = this.model.get(name);
      if (!pattern) {
        return false;
      }
      const helper = new _regExpPattern.default();
      const result = helper.validate(pattern, value, name, this.entityType);
      if (!result) {
        return false;
      }
      const message = result.message.replace('{field}', this.getLanguage().translate(this.getLabelText()));
      this.showValidationMessage(message, '[data-name="' + name + '"]');
      return true;
    }

    /** @inheritDoc */
    fetch() {
      const data = {};
      const value = this.$element.val().trim();
      data[this.name] = value || null;
      return data;
    }

    /** @inheritDoc */
    fetchSearch() {
      const type = this.fetchSearchType() || 'startsWith';
      if (['isEmpty', 'isNotEmpty'].includes(type)) {
        if (type === 'isEmpty') {
          return {
            type: 'or',
            value: [{
              type: 'isNull',
              attribute: this.name
            }, {
              type: 'equals',
              attribute: this.name,
              value: ''
            }],
            data: {
              type: type
            }
          };
        }

        /** @type {Record[]} */
        const value = [{
          type: 'isNotNull',
          attribute: this.name,
          value: null
        }];
        if (!this.model.getFieldParam(this.name, 'notStorable')) {
          value.push({
            type: 'notEquals',
            attribute: this.name,
            value: ''
          });
        }
        return {
          type: 'and',
          value: value,
          data: {
            type: type
          }
        };
      }
      const value = this.$element.val().toString().trim();
      if (!value) {
        return null;
      }
      return {
        value: value,
        type: type,
        data: {
          type: type
        }
      };
    }
    getSearchType() {
      return this.getSearchParamsData().type || this.searchParams.typeFront || this.searchParams.type;
    }

    /**
     * Get an autocomplete lookup function.
     *
     * @protected
     * @return {function (string): Promise<Array<module:ui/autocomplete~item & Record>>|undefined}
     */
    getAutocompleteLookupFunction() {
      return undefined;
    }
  }
  var _default = _exports.default = VarcharFieldView;
});

define("ui/multi-select", ["exports", "lib!selectize"], function (_exports, _libSelectize) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _libSelectize = _interopRequireDefault(_libSelectize);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module module:ui/multi-select */

  /**
   * @typedef module:ui/multi-select~Options
   * @type {Object}
   * @property {{value: string, text: string}[]} items
   * @property {string} [delimiter=':,:']
   * @property {boolean} [restoreOnBackspace=false]
   * @property {boolean} [removeButton=true]
   * @property {boolean} [draggable=false]
   * @property {boolean} [selectOnTab=false]
   * @property {boolean} [matchAnyWord=false]
   * @property {boolean} [allowCustomOptions=false]
   * @property {function (string): {value: string, text: string}|null} [create]
   */

  /**
   * @module ui/multi-select
   */
  const MultiSelect = {
    /**
     * @param {Element|JQuery} element An element.
     * @param {module:ui/multi-select~Options} options Options.
     */
    init: function (element, options) {
      const $el = $(element);
      options = MultiSelect.applyDefaultOptions(options);
      const plugins = [];
      if (options.removeButton) {
        plugins.push('remove_button');
      }
      if (options.draggable) {
        plugins.push('drag_drop');
      }
      if (options.restoreOnBackspace) {
        MultiSelect.loadRestoreOnBackspacePlugin();
        plugins.push('restore_on_backspace_espo');
      }
      MultiSelect.loadBypassCtrlEnterPlugin();
      plugins.push('bypass_ctrl_enter');
      const selectizeOptions = {
        options: options.items,
        plugins: plugins,
        delimiter: options.delimiter,
        labelField: 'text',
        valueField: 'value',
        searchField: ['text'],
        highlight: false,
        selectOnTab: options.selectOnTab
      };
      if (!options.matchAnyWord) {
        // noinspection JSUnresolvedReference
        /** @this Selectize */
        selectizeOptions.score = function (search) {
          // noinspection JSUnresolvedReference
          const score = this.getScoreFunction(search);
          search = search.toLowerCase();
          return function (item) {
            if (item.text.toLowerCase().indexOf(search) === 0) {
              return score(item);
            }
            return 0;
          };
        };
      }
      if (options.matchAnyWord) {
        /** @this Selectize */
        selectizeOptions.score = function (search) {
          // noinspection JSUnresolvedReference
          const score = this.getScoreFunction(search);
          search = search.toLowerCase();
          return function (item) {
            const text = item.text.toLowerCase();
            if (!text.split(' ').find(item => item.startsWith(search)) && !text.startsWith(search)) {
              return 0;
            }
            return score(item);
          };
        };
      }
      if (options.allowCustomOptions) {
        selectizeOptions.persist = false;
        selectizeOptions.create = options.create;
        // noinspection JSUnusedGlobalSymbols
        selectizeOptions.render = {
          option_create: data => {
            return $('<div>').addClass('create').append($('<span>').text(data.input).addClass('text-bold')).append('&hellip;').get(0).outerHTML;
          }
        };
      }
      $el.selectize(selectizeOptions);
    },
    /**
     * Focus.
     *
     * @param {Element|JQuery} element An element.
     */
    focus: function (element) {
      const $el = $(element);
      if (!$el[0] || !$el[0].selectize) {
        return;
      }
      const selectize = $el[0].selectize;
      selectize.focus();
    },
    /**
     * @private
     * @param {module:ui/multi-select~Options} options
     * @return {module:ui/multi-select~Options}
     */
    applyDefaultOptions: function (options) {
      options = Espo.Utils.clone(options);
      const defaults = {
        removeButton: true,
        draggable: false,
        selectOnTab: false,
        delimiter: ':,:',
        matchAnyWord: false,
        allowCustomOptions: false
      };
      for (const key in defaults) {
        if (key in options) {
          continue;
        }
        options[key] = defaults[key];
      }
      return options;
    },
    /**
     * @private
     */
    loadBypassCtrlEnterPlugin: function () {
      if ('bypass_ctrl_enter' in _libSelectize.default.plugins) {
        return;
      }
      const IS_MAC = /Mac/.test(navigator.userAgent);
      _libSelectize.default.define('bypass_ctrl_enter', function () {
        const self = this;
        this.onKeyDown = function () {
          const original = self.onKeyDown;
          return function (e) {
            if (e.code === 'Enter' && (IS_MAC ? e.metaKey : e.ctrlKey)) {
              return;
            }
            return original.apply(this, arguments);
          };
        }();
      });
    },
    /**
     * @private
     */
    loadRestoreOnBackspacePlugin: function () {
      if ('restore_on_backspace_espo' in _libSelectize.default.plugins) {
        return;
      }
      _libSelectize.default.define('restore_on_backspace_espo', function (options) {
        options.text = options.text || function (option) {
          return option[this.settings.labelField];
        };
        const self = this;
        this.onKeyDown = function () {
          const original = self.onKeyDown;
          return function (e) {
            let index, option;
            if (e.code === 'Backspace' && this.$control_input.val() === '' && !this.$activeItems.length) {
              index = this.caretPos - 1;
              if (index >= 0 && index < this.items.length) {
                option = this.options[this.items[index]];
                option = {
                  value: option.value,
                  $order: option.$order,
                  text: option.value
                };

                // noinspection JSUnresolvedReference
                if (this.deleteSelection(e)) {
                  // noinspection JSUnresolvedReference
                  this.setTextboxValue(options.text.apply(this, [option]));
                  this.refreshOptions(true);
                }
                e.preventDefault();
                return;
              }
            }
            return original.apply(this, arguments);
          };
        }();
      });
    }
  };
  var _default = _exports.default = MultiSelect;
});

define("views/modal", ["exports", "view"], function (_exports, _view) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/modal */

  /**
   * A base modal view. Can be extended or used directly.
   *
   * @see https://docs.espocrm.com/development/modal/
   */
  class ModalView extends _view.default {
    /**
     * A button or dropdown action item.
     *
     * @typedef {Object} module:views/modal~Button
     *
     * @property {string} name A name.
     * @property {string} [label] A label. To be translated
     *   (with a scope defined in the `scope` class property).
     * @property {string} [text] A text (not translated).
     * @property {string} [labelTranslation] A label translation path.
     * @property {string} [html] HTML.
     * @property {boolean} [pullLeft=false] Deprecated. Use the `position` property.
     * @property {'left'|'right'} [position='left'] A position.
     * @property {'default'|'danger'|'success'|'warning'|'info'} [style='default'] A style.
     * @property {boolean} [hidden=false] Is hidden.
     * @property {boolean} [disabled=false] Disabled.
     * @property {function(module:ui.Dialog): void} [onClick] Called on click. If not defined, then
     * the `action<Name>` class method will be called.
     * @property {string} [className] An additional class name.
     * @property {string} [title] A title text.
     * @property {'primary'|'danger'|'success'|'warning'|'text'} [style] A style.
     * @property {string} [iconHtml] An icon HTML.
     * @property {string} [iconClass] An icon class.
     * @property {number} [groupIndex] A group index. Only for the dropdown.
     */

    /**
     * @typedef {Object} module:views/modal~Options
     * @property {string} [headerText] A header text.
     * @property {HTMLElement} [headerElement] A header element.
     * @property {'static'|boolean} [backdrop] A backdrop.
     * @property {module:views/modal~Button} [buttonList] Buttons.
     * @property {module:views/modal~Button} [dropdownItemList] Buttons.
     */

    /**
     * @param {module:views/modal~Options | Record} [options] Options.
     */
    constructor(options) {
      super(options);
    }

    /**
     * A CSS name.
     *
     * @protected
     */
    cssName = 'modal-dialog';

    /**
     * A class-name. Use `'dialog dialog-record'` for modals containing a record form.
     *
     * @protected
     */
    className = 'dialog';

    /**
     * @protected
     * @deprecated Use `headerHtml`
     */
    header;

    /**
     * A header HTML. Beware of XSS.
     *
     * @protected
     * @type {string|null}
     */
    headerHtml;

    /**
     * A header JQuery instance.
     *
     * @protected
     * @type {JQuery}
     */
    $header;

    /**
     * A header element.
     *
     * @protected
     * @type {Element}
     */
    headerElement;

    /**
     * A header text.
     *
     * @protected
     * @type {string}
     */
    headerText;

    /**
     * A dialog instance.
     *
     * @protected
     * @type {Espo.Ui.Dialog}
     */
    dialog;

    /**
     * A container selector.
     *
     * @protected
     * @type {string}
     */
    containerSelector = '';

    /**
     * A scope name. Used when translating button labels.
     *
     * @type {string|null}
     */
    scope = null;

    /**
     * A backdrop.
     *
     * @protected
     * @type {'static'|boolean}
     */
    backdrop = 'static';

    /**
     * Buttons.
     *
     * @protected
     * @type {module:views/modal~Button[]}
     */
    buttonList = [];

    /**
     * Dropdown action items.
     *
     * @protected
     * @type {Array<module:views/modal~Button|false>}
     */
    dropdownItemList = [];

    /**
     * @deprecated Use `buttonList`.
     * @protected
     * @todo Remove.
     */
    buttons = [];

    /**
     * A width. Do not use.
     * @todo Consider removing.
     *
     * @protected
     * @type {number|null}
     */
    width = null;

    /**
     * Not used.
     *
     * @deprecated
     */
    fitHeight = false;

    /**
     * To disable fitting to a window height.
     *
     * @protected
     * @type {boolean}
     */
    noFullHeight = false;

    /**
     * Disable the ability to close by pressing the `Esc` key.
     *
     * @protected
     * @type {boolean}
     */
    escapeDisabled = false;

    /**
     * Is draggable.
     *
     * @protected
     * @type {boolean}
     */
    isDraggable = false;

    /**
     * Is collapsable.
     *
     * @protected
     * @type {boolean}
     */
    isCollapsable = false;

    /**
     * Is collapsed. Do not change value. Only for reading.
     *
     * @protected
     * @type {boolean}
     */
    isCollapsed = false;

    /**
     * @type {HTMLElement}
     * @protected
     */
    bodyElement;

    /**
     * @inheritDoc
     */
    events = {
      /** @this module:views/modal */
      'click .action': function (e) {
        Espo.Utils.handleAction(this, e.originalEvent, e.currentTarget);
      },
      /** @this module:views/modal */
      'click [data-action="collapseModal"]': function () {
        this.collapse();
      }
    };

    /**
     * @protected
     * @type {boolean|null}
     */
    footerAtTheTop = null;

    /**
     * A shortcut-key => action map.
     *
     * @protected
     * @type {?Object.<string, string|function (KeyboardEvent): void>}
     */
    shortcutKeys = null;

    /**
     * @protected
     * @type {HTMLElement}
     * @since 9.0.0
     */
    containerElement;

    /**
     * @inheritDoc
     */
    init() {
      const id = this.cssName + '-container-' + Math.floor(Math.random() * 10000 + 1).toString();
      this.containerSelector = '#' + id;
      this.header = this.options.header || this.header;
      this.headerHtml = this.options.headerHtml || this.headerHtml;
      this.$header = this.options.$header || this.$header;
      this.headerElement = this.options.headerElement || this.headerElement;
      this.headerText = this.options.headerText || this.headerText;
      this.backdrop = this.options.backdrop || this.backdrop;
      this.setSelector(this.containerSelector);
      this.buttonList = this.options.buttonList || this.buttonList;
      this.dropdownItemList = this.options.dropdownItemList || this.dropdownItemList;
      this.buttonList = Espo.Utils.cloneDeep(this.buttonList);
      this.dropdownItemList = Espo.Utils.cloneDeep(this.dropdownItemList);
      if (this.shortcutKeys) {
        this.shortcutKeys = Espo.Utils.cloneDeep(this.shortcutKeys);
      }
      this.on('render', () => {
        if (this.dialog) {
          this.dialog.close();
        }

        // Otherwise, re-render won't work.
        this.element = undefined;
        this.isCollapsed = false;
        $(this.containerSelector).remove();
        $('<div />').css('display', 'none').attr('id', id).addClass('modal-container').appendTo('body');
        let modalBodyDiffHeight = 92;
        if (this.getThemeManager().getParam('modalBodyDiffHeight') !== null) {
          modalBodyDiffHeight = this.getThemeManager().getParam('modalBodyDiffHeight');
        }
        let headerHtml = this.headerHtml || this.header;
        if (this.$header && this.$header.length) {
          headerHtml = this.$header.get(0).outerHTML;
        }
        if (this.headerElement) {
          headerHtml = this.headerElement.outerHTML;
        }
        if (this.headerText) {
          headerHtml = Handlebars.Utils.escapeExpression(this.headerText);
        }
        const footerAtTheTop = this.footerAtTheTop !== null ? this.footerAtTheTop : this.getThemeManager().getParam('modalFooterAtTheTop');
        this.dialog = new Espo.Ui.Dialog({
          backdrop: this.backdrop,
          header: headerHtml,
          container: this.containerSelector,
          body: '',
          buttonList: this.getDialogButtonList(),
          dropdownItemList: this.getDialogDropdownItemList(),
          width: this.width,
          keyboard: !this.escapeDisabled,
          fitHeight: this.fitHeight,
          draggable: this.isDraggable,
          className: this.className,
          bodyDiffHeight: modalBodyDiffHeight,
          footerAtTheTop: footerAtTheTop,
          fullHeight: !this.noFullHeight && this.getThemeManager().getParam('modalFullHeight'),
          screenWidthXs: this.getThemeManager().getParam('screenWidthXs'),
          fixedHeaderHeight: this.fixedHeaderHeight,
          closeButton: !this.noCloseButton,
          collapseButton: this.isCollapsable,
          onRemove: () => this.onDialogClose(),
          onBackdropClick: () => this.onBackdropClick()
        });
        this.containerElement = document.querySelector(this.containerSelector);
        this.setElement(this.containerSelector + ' .body');
        this.bodyElement = this.element;

        // @todo Review that the element is set back to the container afterwards.
        //     Force keeping set to the body?
      });
      this.on('after:render', () => {
        // Trick to delegate events for the whole modal.
        this.element = undefined;
        this.setElement(this.containerSelector);
        $(this.containerSelector).show();
        this.dialog.show();
        if (this.fixedHeaderHeight && this.flexibleHeaderFontSize) {
          this.adjustHeaderFontSize();
        }
        this.adjustButtons();
        if (!this.noFullHeight) {
          this.initBodyScrollListener();
        }
        if (this.getParentView()) {
          this.getParentView().trigger('modal-shown');
        }
      });
      this.once('remove', () => {
        if (this.dialog) {
          this.dialog.close();
        }
        $(this.containerSelector).remove();
      });
    }
    setupFinal() {
      if (this.shortcutKeys) {
        this.events['keydown.modal-base'] = e => {
          const key = Espo.Utils.getKeyFromKeyEvent(e);
          if (typeof this.shortcutKeys[key] === 'function') {
            this.shortcutKeys[key].call(this, e.originalEvent);
            return;
          }
          const actionName = this.shortcutKeys[key];
          if (!actionName) {
            return;
          }
          if (this.hasActionItem(actionName) && !this.hasAvailableActionItem(actionName)) {
            return;
          }
          e.preventDefault();
          e.stopPropagation();
          const methodName = 'action' + Espo.Utils.upperCaseFirst(actionName);
          if (typeof this[methodName] === 'function') {
            this[methodName]();
            return;
          }
          this[actionName]();
        };
      }
    }

    /**
     * Get a button list for a dialog.
     *
     * @private
     * @return {module:ui.Dialog~Button[]}
     */
    getDialogButtonList() {
      const buttonListExt = [];

      // @todo remove it as deprecated.
      this.buttons.forEach(item => {
        const o = Espo.Utils.clone(item);
        if (!('text' in o) && 'label' in o) {
          o.text = this.getLanguage().translate(o.label);
        }
        buttonListExt.push(o);
      });
      this.buttonList.forEach(item => {
        let o = {};
        if (typeof item === 'string') {
          o.name = /** @type string */item;
        } else if (typeof item === 'object') {
          o = item;
        } else {
          return;
        }
        if (!o.text) {
          if (o.labelTranslation) {
            o.text = this.getLanguage().translatePath(o.labelTranslation);
          } else if ('label' in o) {
            o.text = this.translate(o.label, 'labels', this.scope);
          } else {
            o.text = this.translate(o.name, 'modalActions', this.scope);
          }
        }
        if (o.iconHtml && !o.html) {
          o.html = o.iconHtml + '<span>' + this.getHelper().escapeString(o.text) + '</span>';
        } else if (o.iconClass && !o.html) {
          o.html = `<span class="${o.iconClass}"></span>` + '<span>' + this.getHelper().escapeString(o.text) + '</span>';
        }
        o.onClick = o.onClick || ((d, e) => {
          const handler = o.handler || (o.data || {}).handler;
          Espo.Utils.handleAction(this, e.originalEvent, e.currentTarget, {
            action: o.name,
            handler: handler,
            actionFunction: o.actionFunction
          });
        });
        buttonListExt.push(o);
      });
      return buttonListExt;
    }

    /**
     * Get a dropdown item list for a dialog.
     *
     * @private
     * @return {Array<module:ui.Dialog~Button|false>}
     */
    getDialogDropdownItemList() {
      const dropdownItemListExt = [];
      this.dropdownItemList.forEach(item => {
        let o = {};
        if (typeof item === 'string') {
          o.name = /** @type string */item;
        } else if (typeof item === 'object') {
          o = item;
        } else {
          return;
        }
        if (!o.text) {
          if (o.labelTranslation) {
            o.text = this.getLanguage().translatePath(o.labelTranslation);
          } else if ('label' in o) {
            o.text = this.translate(o.label, 'labels', this.scope);
          } else {
            o.text = this.translate(o.name, 'modalActions', this.scope);
          }
        }
        o.onClick = o.onClick || ((d, e) => {
          // noinspection ES6ConvertLetToConst
          let handler = o.handler || (o.data || {}).handler;
          Espo.Utils.handleAction(this, e.originalEvent, e.currentTarget, {
            action: o.name,
            handler: handler,
            actionFunction: o.actionFunction
          });
        });
        dropdownItemListExt.push(o);
      });

      /** @type {Array<module:ui.Dialog~Button[]>} */
      const dropdownGroups = [];
      dropdownItemListExt.forEach(item => {
        // For bc.
        if (item === false) {
          return;
        }
        const index = (item.groupIndex === undefined ? 9999 : item.groupIndex) + 100;
        if (dropdownGroups[index] === undefined) {
          dropdownGroups[index] = [];
        }
        dropdownGroups[index].push(item);
      });
      const dropdownItemList = [];
      dropdownGroups.forEach(list => {
        list.forEach(it => dropdownItemList.push(it));
        dropdownItemList.push(false);
      });
      return dropdownItemList;
    }

    /** @private */
    updateDialog() {
      if (!this.dialog) {
        return;
      }
      this.dialog.setActionItems(this.getDialogButtonList(), this.getDialogDropdownItemList());
    }

    /** @private */
    onDialogClose() {
      if (!this.isBeingRendered() && !this.isCollapsed) {
        this.trigger('close');
        this.remove();
      }
    }

    /**
     * @protected
     */
    onBackdropClick() {}

    /**
     * A `cancel` action.
     */
    actionCancel() {
      this.trigger('cancel');
      this.close();
    }

    /**
     * A `close` action.
     */
    actionClose() {
      this.actionCancel();
    }

    /**
     * Close a dialog.
     */
    close() {
      this.dialog.close();
      if (!this.getParentView()) {
        return;
      }
      const key = this.getParentView().getViewKey(this);
      if (key) {
        this.getParentView().clearView(key);
      }
    }

    /**
     * Disable a button.
     *
     * @param {string} name A button name.
     */
    disableButton(name) {
      this.buttonList.forEach(d => {
        if (d.name !== name) {
          return;
        }
        d.disabled = true;
      });
      if (!this.isRendered()) {
        return;
      }
      if (!this.containerElement) {
        return;
      }
      $(this.containerElement).find(`footer button[data-name="${name}"]`).addClass('disabled').attr('disabled', 'disabled');
    }

    /**
     * Enable a button.
     *
     * @param {string} name A button name.
     */
    enableButton(name) {
      this.buttonList.forEach(d => {
        if (d.name !== name) {
          return;
        }
        d.disabled = false;
      });
      if (!this.isRendered()) {
        return;
      }
      if (!this.containerElement) {
        return;
      }
      $(this.containerElement).find('footer button[data-name="' + name + '"]').removeClass('disabled').removeAttr('disabled');
    }

    /**
     * Add a button.
     *
     * @param {module:views/modal~Button} o Button definitions.
     * @param {boolean|string} [position=false] True prepends, false appends. If a string
     *   then will be added after a button with a corresponding name.
     * @param {boolean} [doNotReRender=false] Do not re-render.
     */
    addButton(o, position, doNotReRender) {
      let index = -1;
      this.buttonList.forEach((item, i) => {
        if (item.name === o.name) {
          index = i;
        }
      });
      if (~index) {
        return;
      }
      if (position === true) {
        this.buttonList.unshift(o);
      } else if (typeof position === 'string') {
        index = -1;
        this.buttonList.forEach((item, i) => {
          if (item.name === position) {
            index = i;
          }
        });
        if (~index) {
          this.buttonList.splice(index, 0, o);
        } else {
          this.buttonList.push(o);
        }
      } else {
        this.buttonList.push(o);
      }
      if (!doNotReRender && this.isRendered()) {
        this.reRenderFooter();
      }
    }

    /**
     * Add a dropdown item.
     *
     * @param {module:views/modal~Button} o Button definitions.
     * @param {boolean} [toBeginning=false] To prepend.
     * @param {boolean} [doNotReRender=false] Do not re-render.
     */
    addDropdownItem(o, toBeginning, doNotReRender) {
      if (!o) {
        // For bc.
        return;
      }
      const name = o.name;
      if (!name) {
        return;
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          return;
        }
      }
      toBeginning ? this.dropdownItemList.unshift(o) : this.dropdownItemList.push(o);
      if (!doNotReRender && this.isRendered()) {
        this.reRenderFooter();
      }
    }

    /** @private */
    reRenderFooter() {
      if (!this.dialog) {
        return;
      }
      this.updateDialog();
      const $footer = this.dialog.getFooter();
      $(this.containerElement).find('footer.modal-footer').empty().append($footer);
      this.dialog.initButtonEvents();
    }

    /**
     * Remove a button or a dropdown action item.
     *
     * @param {string} name A name.
     * @param {boolean} [doNotReRender=false] Do not re-render.
     */
    removeButton(name, doNotReRender) {
      let index = -1;
      for (const [i, item] of this.buttonList.entries()) {
        if (item.name === name) {
          index = i;
          break;
        }
      }
      if (~index) {
        this.buttonList.splice(index, 1);
      }
      for (const [i, item] of this.dropdownItemList.entries()) {
        if (item.name === name) {
          this.dropdownItemList.splice(i, 1);
          break;
        }
      }
      if (this.isRendered()) {
        $(this.containerElement).find(`.modal-footer [data-name="${name}"]`).remove();
      }
      if (!doNotReRender && this.isRendered()) {
        this.reRender();
      }
    }

    /**
     * @deprecated Use `showActionItem`.
     *
     * @protected
     * @param {string} name
     */
    showButton(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.hidden = false;
          break;
        }
      }
      if (!this.isRendered()) {
        return;
      }
      if (!this.containerElement) {
        return;
      }
      $(this.containerElement).find(`footer button[data-name="${name}"]`).removeClass('hidden');
      this.adjustButtons();
    }

    /**
     * @deprecated Use `hideActionItem`.
     *
     * @protected
     * @param {string} name
     */
    hideButton(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.hidden = true;
          break;
        }
      }
      if (!this.isRendered()) {
        return;
      }
      if (!this.containerElement) {
        return;
      }
      $(this.containerElement).find(`footer button[data-name="${name}"]`).addClass('hidden');
      this.adjustButtons();
    }

    /**
     * Show an action item (button or dropdown item).
     *
     * @param {string} name A name.
     */
    showActionItem(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.hidden = false;
          break;
        }
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          item.hidden = false;
          break;
        }
      }
      if (!this.isRendered()) {
        return;
      }
      if (!this.containerElement) {
        return;
      }
      const $el = $(this.containerElement);
      $el.find(`footer button[data-name="${name}"]`).removeClass('hidden');
      $el.find(`footer li > a[data-name="${name}"]`).parent().removeClass('hidden');
      if (!this.isDropdownItemListEmpty()) {
        const $dropdownGroup = $el.find('footer .main-btn-group > .btn-group');
        $dropdownGroup.removeClass('hidden');
        $dropdownGroup.find('> button').removeClass('hidden');
      }
      this.adjustButtons();
    }

    /**
     * Hide an action item (button or dropdown item).
     *
     * @param {string} name A name.
     */
    hideActionItem(name) {
      for (const item of this.buttonList) {
        if (item.name === name) {
          item.hidden = true;
          break;
        }
      }
      for (const item of this.dropdownItemList) {
        if (item.name === name) {
          item.hidden = true;
          break;
        }
      }
      if (!this.isRendered()) {
        return;
      }
      const $el = $(this.containerElement);
      $el.find(`footer button[data-name="${name}"]`).addClass('hidden');
      $el.find(`footer li > a[data-name="${name}"]`).parent().addClass('hidden');
      if (this.isDropdownItemListEmpty()) {
        const $dropdownGroup = $el.find('footer .main-btn-group > .btn-group');
        $dropdownGroup.addClass('hidden');
        $dropdownGroup.find('> button').addClass('hidden');
      }
      this.adjustButtons();
    }

    /**
     * Whether an action item exists (hidden, disabled or not).
     *
     * @param {string} name An action item name.
     */
    hasActionItem(name) {
      const hasButton = this.buttonList.findIndex(item => item.name === name) !== -1;
      if (hasButton) {
        return true;
      }
      return this.dropdownItemList.findIndex(item => item.name === name) !== -1;
    }

    /**
     * Whether an action item is visible and not disabled.
     *
     * @param {string} name An action item name.
     */
    hasAvailableActionItem(name) {
      const hasButton = this.buttonList.findIndex(item => item.name === name && !item.disabled && !item.hidden) !== -1;
      if (hasButton) {
        return true;
      }
      return this.dropdownItemList.findIndex(item => item.name === name && !item.disabled && !item.hidden) !== -1;
    }

    /**
     * @private
     * @return {boolean}
     */
    isDropdownItemListEmpty() {
      if (this.dropdownItemList.length === 0) {
        return true;
      }
      let isEmpty = true;
      this.dropdownItemList.forEach(item => {
        if (!item.hidden) {
          isEmpty = false;
        }
      });
      return isEmpty;
    }

    /**
     * @private
     * @param {number} [step=0]
     */
    adjustHeaderFontSize(step) {
      step = step || 0;
      if (!step) {
        this.fontSizePercentage = 100;
      }
      if (!this.containerElement) {
        return;
      }
      const $titleText = $(this.containerElement).find('.modal-title > .modal-title-text');
      const containerWidth = $titleText.parent().width();
      let textWidth = 0;
      $titleText.children().each((i, el) => {
        textWidth += $(el).outerWidth(true);
      });
      if (containerWidth < textWidth) {
        if (step > 5) {
          const $title = $(this.containerElement).find('.modal-title');
          $title.attr('title', $titleText.text());
          $title.addClass('overlapped');
          $titleText.children().each((i, el) => {
            $(el).removeAttr('title');
          });
          return;
        }
        this.fontSizePercentage -= 4;
        $(this.containerElement).find('.modal-title .font-size-flexible').css('font-size', this.fontSizePercentage + '%');
        this.adjustHeaderFontSize(step + 1);
      }
    }

    /**
     * Collapse.
     */
    collapse() {
      this.beforeCollapse().then(data => {
        if (!this.getParentView()) {
          throw new Error("Can't collapse w/o parent view.");
        }
        this.isCollapsed = true;
        data = data || {};
        let title;
        if (data.title) {
          title = data.title;
        } else {
          const $title = $(this.containerElement).find('.modal-header .modal-title .modal-title-text');
          title = $title.text();
        }
        this.dialog.close();
        let masterView = this;
        while (masterView.getParentView()) {
          masterView = masterView.getParentView();
        }
        this.unchainFromParent();
        new Promise(resolve => {
          if (masterView.hasView('collapsedModalBar')) {
            resolve(masterView.getView('collapsedModalBar'));
            return;
          }
          masterView.createView('collapsedModalBar', 'views/collapsed-modal-bar', {
            fullSelector: 'body > .collapsed-modal-bar'
          }).then(view => resolve(view));
        }).then(barView => {
          barView.addModalView(this, {
            title: title
          });
        });
      });
    }
    unchainFromParent() {
      const key = this.getParentView().getViewKey(this);
      this.getParentView().unchainView(key);
    }

    /**
     * Called before collapse. Can be extended to execute some logic, e.g. save form data.
     *
     * @protected
     * @return {Promise}
     */
    beforeCollapse() {
      return new Promise(resolve => resolve());
    }

    /** @private */
    adjustButtons() {
      this.adjustLeftButtons();
      this.adjustRightButtons();
    }

    /** @private */
    adjustLeftButtons() {
      const $buttons = $(this.containerElement).find('footer.modal-footer > .main-btn-group button.btn');
      $buttons.removeClass('radius-left').removeClass('radius-right');
      const $buttonsVisible = $buttons.filter('button:not(.hidden)');
      $buttonsVisible.first().addClass('radius-left');
      $buttonsVisible.last().addClass('radius-right');
    }

    /** @private */
    adjustRightButtons() {
      const $buttons = $(this.containerElement).find('footer.modal-footer > .additional-btn-group button.btn:not(.btn-text)');
      $buttons.removeClass('radius-left').removeClass('radius-right').removeClass('margin-right');
      const $buttonsVisible = $buttons.filter('button:not(.hidden)');
      $buttonsVisible.first().addClass('radius-left');
      $buttonsVisible.last().addClass('radius-right');
      if ($buttonsVisible.last().next().hasClass('btn-text')) {
        $buttonsVisible.last().addClass('margin-right');
      }
    }

    /**
     * @private
     */
    initBodyScrollListener() {
      const $body = $(this.containerElement).find('> .dialog > .modal-dialog > .modal-content > .modal-body');
      const $footer = $body.parent().find('> .modal-footer');
      if (!$footer.length) {
        return;
      }
      $body.off('scroll.footer-shadow');
      $body.on('scroll.footer-shadow', () => {
        if ($body.scrollTop()) {
          $footer.addClass('shadowed');
          return;
        }
        $footer.removeClass('shadowed');
      });
    }
  }
  var _default = _exports.default = ModalView;
});

define("views/record/edit-for-modal", ["exports", "views/record/edit"], function (_exports, _edit) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _edit = _interopRequireDefault(_edit);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/record/edit-for-modal */

  /**
   * An edit-record view to used for custom forms.
   */
  class EditForModalRecordView extends _edit.default {
    bottomView = null;
    sideView = null;
    buttonsDisabled = true;
    isWide = true;
    accessControlDisabled = true;
    confirmLeaveDisabled = true;
  }
  var _default = _exports.default = EditForModalRecordView;
});

define("views/fields/enum", ["exports", "views/fields/base", "ui/multi-select", "ui/select"], function (_exports, _base, _multiSelect, _select) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _base = _interopRequireDefault(_base);
  _multiSelect = _interopRequireDefault(_multiSelect);
  _select = _interopRequireDefault(_select);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/enumeration */

  /**
   * An enum field (select-box).
   *
   * @extends BaseFieldView<module:views/fields/enumeration~params>
   */
  class EnumFieldView extends _base.default {
    /**
     * @typedef {Object} module:views/fields/enumeration~options
     * @property {
     *     module:views/fields/enumeration~params &
     *     module:views/fields/base~params &
     *     Object.<string, *>
     * } [params] Parameters.
     */

    /**
     * @typedef {Object} module:views/fields/enumeration~params
     * @property {string[]} [options] Select options.
     * @property {boolean} [required] Required.
     * @property {string} [translation] A translation string. E.g. `Global.scopeNames`.
     * @property {boolean} [displayAsLabel] Display as label.
     * @property {string|'state'} [labelType] A label type.
     * @property {'regular'|'state'} [labelType] A label type.
     * @property {string} [optionsReference] A reference to options. E.g. `Account.industry`.
     * @property {string} [optionsPath] An options metadata path.
     * @property {boolean} [isSorted] To sort options.
     * @property {Object.<string, 'warning'|'danger'|'success'|'info'|'primary'>} [style] A style map.
     * @property {Object.<string, string>} [translatedOptions] Option translations.
     */

    /**
     * @param {
     *     module:views/fields/enumeration~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }
    type = 'enum';
    listTemplate = 'fields/enum/list';
    listLinkTemplate = 'fields/enum/list-link';
    detailTemplate = 'fields/enum/detail';
    editTemplate = 'fields/enum/edit';
    searchTemplate = 'fields/enum/search';
    translatedOptions = null;

    /**
     * @todo Remove? Always treat as true.
     */
    fetchEmptyValueAsNull = true;
    searchTypeList = ['anyOf', 'noneOf', 'isEmpty', 'isNotEmpty'];
    validationElementSelector = '.selectize-control';

    // noinspection JSCheckFunctionSignatures
    /** @inheritDoc */
    data() {
      const data = super.data();
      data.translatedOptions = this.translatedOptions;
      const value = this.model.get(this.name);
      if (this.isReadMode() && this.styleMap) {
        data.style = this.styleMap[value || ''] || 'default';
      }
      if (this.isReadMode()) {
        if (!this.params.displayAsLabel) {
          data.class = 'text';
        } else {
          if (this.params.labelType === 'state') {
            data.class = 'label label-md label-state label';
          } else {
            data.class = data.style && data.style !== 'default' ? 'label label-md label' : 'text';
          }
        }
      }
      const translationKey = value || '';
      if (typeof value !== 'undefined' && value !== null && value !== '' || translationKey === '' && translationKey in (this.translatedOptions || {}) && (this.translatedOptions || {})[translationKey] !== '') {
        data.isNotEmpty = true;
      }
      data.valueIsSet = this.model.has(this.name);
      if (data.isNotEmpty) {
        data.valueTranslated = this.translatedOptions ? this.translatedOptions[translationKey] || value : this.getLanguage().translateOption(translationKey, this.name, this.entityType);
      }

      // noinspection JSValidateTypes
      return data;
    }
    setup() {
      if (!this.params.options) {
        const methodName = 'get' + Espo.Utils.upperCaseFirst(this.name) + 'Options';
        if (typeof this.model[methodName] === 'function') {
          this.params.options = this.model[methodName].call(this.model);
        }
      }
      let optionsPath = this.params.optionsPath;
      /** @type {?string} */
      const optionsReference = this.params.optionsReference;
      if (!optionsPath && optionsReference) {
        const [refEntityType, refField] = optionsReference.split('.');
        optionsPath = `entityDefs.${refEntityType}.fields.${refField}.options`;
      }
      if (optionsPath) {
        this.params.options = Espo.Utils.clone(this.getMetadata().get(optionsPath)) || [];
      }
      this.styleMap = this.params.style || this.model.getFieldParam(this.name, 'style') || {};
      this.setupOptions();
      if ('translatedOptions' in this.options) {
        this.translatedOptions = this.options.translatedOptions;
      }
      if ('translatedOptions' in this.params) {
        this.translatedOptions = this.params.translatedOptions;
      }
      this.setupTranslation();
      if (this.translatedOptions === null) {
        this.translatedOptions = this.getLanguage().translate(this.name, 'options', this.model.name) || {};
        if (this.translatedOptions === this.name) {
          this.translatedOptions = null;
        }
      }
      if (this.params.isSorted && this.translatedOptions) {
        this.params.options = Espo.Utils.clone(this.params.options) || [];
        this.params.options = this.params.options.sort((v1, v2) => {
          return (this.translatedOptions[v1] || v1).localeCompare(this.translatedOptions[v2] || v2);
        });
      }
      if (this.options.customOptionList) {
        this.setOptionList(this.options.customOptionList);
      }
    }
    setupTranslation() {
      let translation = this.params.translation;
      /** @type {?string} */
      const optionsReference = this.params.optionsReference;
      if (!translation && optionsReference) {
        const [refEntityType, refField] = optionsReference.split('.');
        translation = `${refEntityType}.options.${refField}`;
      }
      if (!translation) {
        return;
      }
      this.translatedOptions = null;
      if (!this.params.options) {
        return;
      }
      const obj = this.getLanguage().translatePath(translation);
      const map = {};
      this.params.options.forEach(item => {
        if (typeof obj === 'object' && item in obj) {
          map[item] = obj[item];
          return;
        }
        if (Array.isArray(obj) && typeof item === 'number' && typeof obj[item] !== 'undefined') {
          map[item.toString()] = obj[item];
          return;
        }
        map[item] = item;
      });
      const value = this.model.get(this.name);
      if ((value || value === '') && !(value in map)) {
        if (typeof obj === 'object' && value in obj) {
          map[value] = obj[value];
        }
      }
      this.translatedOptions = map;
    }

    /**
     * Set up options.
     */
    setupOptions() {}

    /**
     * Set translated options.
     *
     * @param {Record} translatedOptions
     * @since 8.4.0
     */
    setTranslatedOptions(translatedOptions) {
      this.translatedOptions = translatedOptions;
    }

    /**
     * Set an option list.
     *
     * @param {string[]} optionList An option list.
     * @return {Promise}
     */
    setOptionList(optionList) {
      const previousOptions = this.params.options;
      if (!this.originalOptionList) {
        this.originalOptionList = this.params.options;
      }
      const newOptions = Espo.Utils.clone(optionList) || [];
      this.params.options = newOptions;
      const isChanged = !_(previousOptions).isEqual(optionList);
      if (!this.isEditMode() || !isChanged) {
        return Promise.resolve();
      }
      let triggerChange = false;
      const currentValue = this.model.get(this.name);
      if (!newOptions.includes(currentValue) && this.isReady) {
        this.model.set(this.name, newOptions[0] ?? null, {
          silent: true
        });
        triggerChange = true;
      }
      return this.reRender().then(() => {
        if (triggerChange) {
          this.trigger('change');
        }
      });
    }

    /**
     * Reset a previously set option list.
     *
     * @return {Promise}
     */
    resetOptionList() {
      if (!this.originalOptionList) {
        return Promise.resolve();
      }
      const previousOptions = this.params.options;
      this.params.options = Espo.Utils.clone(this.originalOptionList);
      const isChanged = !_(previousOptions).isEqual(this.originalOptionList);
      if (!this.isEditMode() || !isChanged) {
        return Promise.resolve();
      }
      if (this.isRendered()) {
        return this.reRender();
      }
      return Promise.resolve();
    }
    setupSearch() {
      this.events = _.extend({
        'change select.search-type': e => {
          this.handleSearchType($(e.currentTarget).val());
        }
      }, this.events || {});
    }
    handleSearchType(type) {
      const $inputContainer = this.$el.find('div.input-container');
      if (~['anyOf', 'noneOf'].indexOf(type)) {
        $inputContainer.removeClass('hidden');
      } else {
        $inputContainer.addClass('hidden');
      }
    }
    afterRender() {
      super.afterRender();
      if (this.isSearchMode()) {
        this.$element = this.$el.find('.main-element');
        const type = this.$el.find('select.search-type').val();
        this.handleSearchType(type);
        const valueList = this.getSearchParamsData().valueList || this.searchParams.value || [];
        this.$element.val(valueList.join(':,:'));
        const items = [];
        (this.params.options || []).forEach(value => {
          let label = this.getLanguage().translateOption(value, this.name, this.scope);
          if (this.translatedOptions) {
            if (value in this.translatedOptions) {
              label = this.translatedOptions[value];
            }
          }
          if (label === '') {
            return;
          }
          items.push({
            value: value,
            text: label
          });
        });

        /** @type {module:ui/multi-select~Options} */
        const multiSelectOptions = {
          items: items,
          delimiter: ':,:',
          matchAnyWord: true
        };
        _multiSelect.default.init(this.$element, multiSelectOptions);
        this.$el.find('.selectize-dropdown-content').addClass('small');
        this.$el.find('select.search-type').on('change', () => this.trigger('change'));
        this.$element.on('change', () => this.trigger('change'));
      }
      if (this.isEditMode() || this.isSearchMode()) {
        _select.default.init(this.$element, {
          matchAnyWord: true
        });
      }
    }
    focusOnInlineEdit() {
      _select.default.focus(this.$element);
    }
    validateRequired() {
      if (this.isRequired()) {
        if (!this.model.get(this.name)) {
          const msg = this.translate('fieldIsRequired', 'messages').replace('{field}', this.getLabelText());
          this.showValidationMessage(msg);
          return true;
        }
      }
    }
    fetch() {
      let value = this.$element.val();
      if (this.fetchEmptyValueAsNull && !value) {
        value = null;
      }
      const data = {};
      data[this.name] = value;
      return data;
    }
    parseItemForSearch(item) {
      return item;
    }
    fetchSearch() {
      const type = this.fetchSearchType();
      let list = this.$element.val().split(':,:');
      if (list.length === 1 && list[0] === '') {
        list = [];
      }
      list.forEach((item, i) => {
        list[i] = this.parseItemForSearch(item);
      });
      if (type === 'anyOf') {
        if (list.length === 0) {
          return {
            type: 'any',
            data: {
              type: 'anyOf',
              valueList: list
            }
          };
        }
        return {
          type: 'in',
          value: list,
          data: {
            type: 'anyOf',
            valueList: list
          }
        };
      }
      if (type === 'noneOf') {
        if (list.length === 0) {
          return {
            type: 'any',
            data: {
              type: 'noneOf',
              valueList: list
            }
          };
        }
        return {
          type: 'or',
          value: [
          // Don't change order.
          {
            type: 'notIn',
            value: list,
            attribute: this.name
          }, {
            type: 'isNull',
            attribute: this.name
          }],
          data: {
            type: 'noneOf',
            valueList: list
          }
        };
      }
      if (type === 'isEmpty') {
        return {
          type: 'or',
          value: [{
            type: 'isNull',
            attribute: this.name
          }, {
            type: 'equals',
            value: '',
            attribute: this.name
          }],
          data: {
            type: 'isEmpty'
          }
        };
      }
      if (type === 'isNotEmpty') {
        const value = [{
          type: 'isNotNull',
          attribute: this.name
        }];
        if (!this.model.getFieldParam(this.name, 'notStorable')) {
          value.push({
            type: 'notEquals',
            value: '',
            attribute: this.name
          });
        }
        return {
          type: 'and',
          value: value,
          data: {
            type: 'isNotEmpty'
          }
        };
      }
      return null;
    }
    getSearchType() {
      return this.getSearchParamsData().type || 'anyOf';
    }
  }
  var _default = _exports.default = EnumFieldView;
});

define("views/fields/colorpicker", ["exports", "views/fields/varchar"], function (_exports, _varchar) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _varchar = _interopRequireDefault(_varchar);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class ColorpickerFieldView extends _varchar.default {
    type = 'varchar';
    detailTemplate = 'fields/colorpicker/detail';
    listTemplate = 'fields/colorpicker/detail';
    editTemplate = 'fields/colorpicker/edit';
    setup() {
      super.setup();
      this.params.maxLength = 7;
      this.wait(Espo.loader.requirePromise('lib!bootstrap-colorpicker'));
    }
    afterRender() {
      super.afterRender();
      if (this.isEditMode()) {
        const isModal = !!this.$el.closest('.modal').length;

        // noinspection JSUnresolvedReference
        this.$element.parent().colorpicker({
          format: 'hex',
          container: isModal ? this.$el : false,
          sliders: {
            saturation: {
              maxLeft: 200,
              maxTop: 200
            },
            hue: {
              maxTop: 200
            },
            alpha: {
              maxTop: 200
            }
          }
        });
        if (isModal) {
          this.$el.find('.colorpicker').css('position', 'relative').addClass('pull-right');
        }
        this.$element.on('change', () => {
          if (this.$element.val() === '') {
            this.$el.find('.input-group-addon > i').css('background-color', 'transparent');
          }
        });
      }
    }
  }
  var _default = _exports.default = ColorpickerFieldView;
});

define("views/wysiwyg/modals/edit-table", ["exports", "views/modal", "views/record/edit-for-modal", "model", "views/fields/enum", "views/fields/varchar", "views/fields/colorpicker"], function (_exports, _modal, _editForModal, _model, _enum, _varchar, _colorpicker) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _modal = _interopRequireDefault(_modal);
  _editForModal = _interopRequireDefault(_editForModal);
  _model = _interopRequireDefault(_model);
  _enum = _interopRequireDefault(_enum);
  _varchar = _interopRequireDefault(_varchar);
  _colorpicker = _interopRequireDefault(_colorpicker);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class EditTableModalView extends _modal.default {
    templateContent = `
        <div class="record no-side-margin">{{{record}}}</div>
    `;
    /**
     * @param {{
     *     params: {
     *         align: null|'left'|'center'|'right',
     *         width: null|string,
     *         height: null|string,
     *         borderWidth: null|string,
     *         borderColor: null|string,
     *         cellPadding: null|string,
     *         backgroundColor: null|string,
     *    },
     *    onApply: function({
     *         align: null|'left'|'center'|'right',
     *         width: null|string,
     *         height: null|string,
     *         borderWidth: null|string,
     *         borderColor: null|string,
     *         cellPadding: null|string,
     *         backgroundColor: null|string,
     *    }),
     * }} options
     */
    constructor(options) {
      super(options);
      this.params = options.params;
      this.onApply = options.onApply;
    }
    setup() {
      this.addButton({
        name: 'apply',
        style: 'primary',
        label: 'Apply',
        onClick: () => this.apply()
      });
      this.addButton({
        name: 'cancel',
        label: 'Cancel',
        onClick: () => this.close()
      });
      this.shortcutKeys = {
        'Control+Enter': () => this.apply()
      };
      this.model = new _model.default({
        align: this.params.align,
        width: this.params.width,
        height: this.params.height,
        borderWidth: this.params.borderWidth,
        borderColor: this.params.borderColor,
        cellPadding: this.params.cellPadding,
        backgroundColor: this.params.backgroundColor
      });
      this.recordView = new _editForModal.default({
        model: this.model,
        detailLayout: [{
          rows: [[{
            view: new _varchar.default({
              name: 'width',
              labelText: this.translate('width', 'wysiwygLabels'),
              params: {
                maxLength: 12
              }
            })
          }, {
            view: new _varchar.default({
              name: 'height',
              labelText: this.translate('height', 'wysiwygLabels'),
              params: {
                maxLength: 12
              }
            })
          }], [{
            view: new _varchar.default({
              name: 'borderWidth',
              labelText: this.translate('borderWidth', 'wysiwygLabels'),
              params: {
                maxLength: 12
              }
            })
          }, {
            view: new _colorpicker.default({
              name: 'borderColor',
              labelText: this.translate('borderColor', 'wysiwygLabels')
            })
          }], [{
            view: new _varchar.default({
              name: 'cellPadding',
              labelText: this.translate('cellPadding', 'wysiwygLabels'),
              params: {
                maxLength: 12
              }
            })
          }, {
            view: new _colorpicker.default({
              name: 'backgroundColor',
              labelText: this.translate('backgroundColor', 'wysiwygLabels')
            })
          }], [{
            view: new _enum.default({
              name: 'align',
              labelText: this.translate('align', 'wysiwygLabels'),
              params: {
                options: ['', 'left', 'center', 'right'],
                translation: 'Global.wysiwygOptions.align'
              }
            })
          }, false]]
        }]
      });
      this.assignView('record', this.recordView, '.record');
    }
    apply() {
      if (this.recordView.validate()) {
        return;
      }
      let borderWidth = this.model.attributes.borderWidth;
      let cellPadding = this.model.attributes.cellPadding;
      let width = this.model.attributes.width;
      let height = this.model.attributes.height;
      if (/^\d+$/.test(borderWidth)) {
        borderWidth += 'px';
      }
      if (/^\d+$/.test(cellPadding)) {
        cellPadding += 'px';
      }
      if (/^\d+$/.test(width)) {
        width += 'px';
      }
      if (/^\d+$/.test(height)) {
        height += 'px';
      }
      this.onApply({
        align: this.model.attributes.align,
        width: width,
        height: height,
        borderWidth: borderWidth,
        borderColor: this.model.attributes.borderColor,
        cellPadding: cellPadding,
        backgroundColor: this.model.attributes.backgroundColor
      });
      this.close();
    }
  }
  var _default = _exports.default = EditTableModalView;
});

define("views/wysiwyg/modals/edit-cell", ["exports", "views/modal", "views/record/edit-for-modal", "model", "views/fields/varchar", "views/fields/colorpicker", "views/fields/enum"], function (_exports, _modal, _editForModal, _model, _varchar, _colorpicker, _enum) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _modal = _interopRequireDefault(_modal);
  _editForModal = _interopRequireDefault(_editForModal);
  _model = _interopRequireDefault(_model);
  _varchar = _interopRequireDefault(_varchar);
  _colorpicker = _interopRequireDefault(_colorpicker);
  _enum = _interopRequireDefault(_enum);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class EditCellModalView extends _modal.default {
    templateContent = `
        <div class="record no-side-margin">{{{record}}}</div>
    `;
    /**
     * @param {{
     *     params: {
     *         width: null|string,
     *         height: null|height,
     *         backgroundColor: null|string,
     *         verticalAlign: null|'top'|'middle'|'bottom',
     *    },
     *    onApply: function({
     *         width: null|string,
     *         height: null|height,
     *         backgroundColor: null|string,
     *         verticalAlign: null|'top'|'middle'|'bottom',
     *    }),
     * }} options
     */
    constructor(options) {
      super(options);
      this.params = options.params;
      this.onApply = options.onApply;
    }
    setup() {
      this.addButton({
        name: 'apply',
        style: 'primary',
        label: 'Apply',
        onClick: () => this.apply()
      });
      this.addButton({
        name: 'cancel',
        label: 'Cancel',
        onClick: () => this.close()
      });
      this.shortcutKeys = {
        'Control+Enter': () => this.apply()
      };
      this.model = new _model.default({
        width: this.params.width,
        height: this.params.height,
        backgroundColor: this.params.backgroundColor,
        verticalAlign: this.params.verticalAlign
      });
      this.recordView = new _editForModal.default({
        model: this.model,
        detailLayout: [{
          rows: [[{
            view: new _varchar.default({
              name: 'width',
              labelText: this.translate('width', 'wysiwygLabels'),
              params: {
                maxLength: 12
              }
            })
          }, {
            view: new _varchar.default({
              name: 'height',
              labelText: this.translate('height', 'wysiwygLabels'),
              params: {
                maxLength: 12
              }
            })
          }], [{
            view: new _enum.default({
              name: 'verticalAlign',
              labelText: this.translate('verticalAlign', 'wysiwygLabels'),
              params: {
                options: ['', 'top', 'middle', 'bottom'],
                translation: 'Global.wysiwygOptions.verticalAlign'
              }
            })
          }, {
            view: new _colorpicker.default({
              name: 'backgroundColor',
              labelText: this.translate('backgroundColor', 'wysiwygLabels')
            })
          }]]
        }]
      });
      this.assignView('record', this.recordView, '.record');
    }
    apply() {
      if (this.recordView.validate()) {
        return;
      }
      let width = this.model.attributes.width;
      if (/^\d+$/.test(width)) {
        width += 'px';
      }
      let height = this.model.attributes.height;
      if (/^\d+$/.test(height)) {
        height += 'px';
      }
      this.onApply({
        width: width,
        height: height,
        backgroundColor: this.model.attributes.backgroundColor,
        verticalAlign: this.model.attributes.verticalAlign
      });
      this.close();
    }
  }
  var _default = _exports.default = EditCellModalView;
});

define("views/record/list/settings", ["exports", "view"], function (_exports, _view) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class RecordListSettingsView extends _view.default {
    // language=Handlebars
    templateContent = `
        {{#if toDisplay}}
        <div class="btn-group">
            <a
                role="button"
                class="btn btn-text dropdown-toggle"
                data-toggle="dropdown"
                tabindex="0"
            ><span class="caret"></span></a>
            <ul class="dropdown-menu pull-right">
                {{#if dataList.length}}
                <li class="dropdown-header">{{fieldsLabel}}</li>
                    {{#each dataList}}
                        <li>
                            <a
                                role="button"
                                tabindex="0"
                                data-action="toggleColumn"
                                data-name="{{name}}"
                            ><span class="check-icon fas fa-check pull-right{{#if hidden}} hidden{{/if}}"></span><div>{{label}}</div></a>
                        </li>
                    {{/each}}
                {{/if}}
            {{#if hasColumnResize}}
                <li class="divider"></li>
                <li>
                    <a
                        role="button"
                        tabindex="0"
                        data-action="toggleColumnResize"
                    >
                        <span class="check-icon fas fa-check pull-right {{#unless columnResize}} hidden {{/unless}}"></span>
                        <div>{{translate 'Column Resize'}}</div></a>
                </li>
            {{/if}}
            {{#if isNotDefault}}
                <li class="divider"></li>
                <li>
                    <a
                        role="button"
                        tabindex="0"
                        data-action="resetToDefault"
                    >{{translate 'Reset'}}</a>
                </li>
            {{/if}}
            </ul>
        </div>
        {{/if}}
    `;
    data() {
      const columnResize = this.helper.getColumnResize();
      const dataList = this.getDataList();
      const hasColumnResize = this.columnResize && (columnResize || this.isColumnResizeApplicable());
      const isNotDefault = dataList.find(item => item.hiddenDefault !== item.hidden) !== undefined || Object.keys(this.helper.getColumnWidthMap()).length > 0;
      return {
        dataList: dataList,
        toDisplay: dataList.length > 0 || columnResize,
        isNotDefault: isNotDefault,
        fieldsLabel: this.translate('Fields'),
        hasColumnResize: hasColumnResize,
        columnResize: columnResize
      };
    }

    /**
     * @typedef {Object} RecordListSettingsView~onChangeOptions
     * @property {'resetToDefault'|'toggleColumn'|'toggleColumnResize'} action An action.
     * @property {string} [column] A column.
     */

    /**
     * @param {{
     *     layoutProvider: function(): {
     *         name: string,
     *         width?: number,
     *         widthPx?: number,
     *         label?: string,
     *         customLabel?: string,
     *         noLabel?: boolean,
     *         hidden?: boolean,
     *     }[],
     *     helper: import('helpers/list/settings').default,
     *     entityType: string,
     *     onChange: function(RecordListSettingsView~onChangeOptions),
     *     columnResize?: boolean,
     * }} options
     */
    constructor(options) {
      super();
      this.layoutProvider = options.layoutProvider;
      this.helper = options.helper;
      this.entityType = options.entityType;
      this.onChange = options.onChange;
      this.columnResize = options.columnResize || false;
    }
    setup() {
      this.addActionHandler('toggleColumn', (e, target) => this.toggleColumn(target.dataset.name));
      this.addActionHandler('toggleColumnResize', () => this.toggleColumnResize());
      this.addActionHandler('resetToDefault', () => this.resetToDefault());

      /** @private */
      this.onColumnWidthChangeBind = this.onColumnWidthChange.bind(this);
      this.helper.subscribeToColumnWidthChange(this.onColumnWidthChangeBind);
      if (window.innerWidth < this.getThemeManager().getParam('screenWidthXs')) {
        this.columnResize = false;
      }
    }
    onRemove() {
      this.helper.unsubscribeFromColumnWidthChange(this.onColumnWidthChangeBind);
    }

    /**
     * @private
     */
    onColumnWidthChange() {
      this.reRender();
    }

    /**
     * @private
     * @return {{
     *     hidden: boolean,
     *     hiddenDefault: boolean,
     *     name: string,
     *     label: string,
     * }[]}
     */
    getDataList() {
      const list = this.layoutProvider() || [];
      const map = this.helper.getHiddenColumnMap() || {};
      return list.filter(item => item.name && !item.link && !item.noLabel && !item.customLabel).map(item => {
        const label = item.label || item.name;
        const hidden = item.name in map ? map[item.name] : !!item.hidden;
        return {
          name: item.name,
          label: this.translate(label, 'fields', this.entityType),
          hidden: hidden,
          hiddenDefault: !!item.hidden
        };
      });
    }

    /**
     * @private
     * @return {boolean}
     */
    isColumnResizeApplicable() {
      const list = this.layoutProvider().filter(it => {
        return !this.helper.isColumnHidden(it.name, it.hidden);
      });
      if (!list || list.length <= 1) {
        return false;
      }
      if (!list.find(it => !it.widthPx && !it.width)) {
        return false;
      }
      return !!list.find(it => it.widthPx || it.width);
    }

    /**
     * @private
     * @param {string} name
     */
    toggleColumn(name) {
      const map = /** @type {Object.<string, boolean>} */this.helper.getHiddenColumnMap() || {};
      const item = this.getDataList().find(item => item.name === name);
      const defaultValue = item ? item.hiddenDefault : false;
      map[name] = !(name in map ? map[name] : defaultValue);
      this.helper.storeHiddenColumnMap(map);
      this.onChange({
        action: 'toggleColumn',
        column: name
      });
    }

    /**
     * @private
     */
    toggleColumnResize() {
      const value = !this.helper.getColumnResize();
      this.helper.storeColumnResize(value);
      this.onChange({
        action: 'toggleColumnResize'
      });
    }

    /**
     * @private
     */
    resetToDefault() {
      this.helper.clearHiddenColumnMap();
      this.helper.clearColumnWidthMap();
      this.onChange({
        action: 'resetToDefault'
      });
    }
  }
  var _default = _exports.default = RecordListSettingsView;
});

define("helpers/record-modal", ["exports", "di", "metadata", "acl-manager"], function (_exports, _di, _metadata, _aclManager) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _metadata = _interopRequireDefault(_metadata);
  _aclManager = _interopRequireDefault(_aclManager);
  let _init_metadata, _init_extra_metadata, _init_acl, _init_extra_acl;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * A record-modal helper.
   */
  class RecordModalHelper {
    static {
      [_init_metadata, _init_extra_metadata, _init_acl, _init_extra_acl] = _applyDecs(this, [], [[(0, _di.inject)(_metadata.default), 0, "metadata"], [(0, _di.inject)(_aclManager.default), 0, "acl"]]).e;
    }
    constructor() {
      _init_extra_acl(this);
    }
    /**
     * @private
     * @type {Metadata}
     */
    metadata = _init_metadata(this);

    /**
     * @private
     * @type {AclManager}
     */
    acl = (_init_extra_metadata(this), _init_acl(this));

    /**
     * @param {module:view} view
     * @param {{
     *   id: string,
     *   scope: string,
     *   model?: module:model,
     *   editDisabled?: boolean,
     *   rootUrl?: string,
     * }} params
     * @return {Promise}
     */
    showDetail(view, params) {
      const id = params.id;
      const scope = params.scope;
      const model = params.model;
      if (!id || !scope) {
        console.error("Bad data.");
        return Promise.reject();
      }
      if (model && !this.acl.checkScope(model.entityType, 'read')) {
        return Promise.reject();
      }
      const viewName = this.metadata.get(['clientDefs', scope, 'modalViews', 'detail']) || 'views/modals/detail';
      Espo.Ui.notify(' ... ');
      const options = {
        scope: scope,
        model: model,
        id: id,
        quickEditDisabled: params.editDisabled,
        rootUrl: params.rootUrl
      };
      return view.createView('modal', viewName, options, modalView => {
        modalView.render().then(() => Espo.Ui.notify(false));
        view.listenToOnce(modalView, 'remove', () => {
          view.clearView('modal');
        });
      });
    }
  }
  var _default = _exports.default = RecordModalHelper;
});

define("helpers/mass-action", ["exports", "di", "models/settings", "models/user"], function (_exports, _di, _settings, _user) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _settings = _interopRequireDefault(_settings);
  _user = _interopRequireDefault(_user);
  let _init_config, _init_extra_config, _init_user, _init_extra_user;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * A mass-action helper.
   */
  class MassActionHelper {
    static {
      [_init_config, _init_extra_config, _init_user, _init_extra_user] = _applyDecs(this, [], [[(0, _di.inject)(_settings.default), 0, "config"], [(0, _di.inject)(_user.default), 0, "user"]]).e;
    }
    /**
     * @private
     * @type {Settings}
     */
    config = _init_config(this);

    /**
     * @private
     * @type {User}
     */
    user = (_init_extra_config(this), _init_user(this));

    /**
     * @param {module:view} view A view.
     */
    constructor(view) {
      _init_extra_user(this);
      /**
       * @private
       * @type {module:view}
       */
      this.view = view;
    }

    /**
     * Check whether an action should be run in idle.
     *
     * @param {number} [totalCount] A total record count.
     * @returns {boolean}
     */
    checkIsIdle(totalCount) {
      if (this.user.isPortal()) {
        return false;
      }
      if (typeof totalCount === 'undefined') {
        totalCount = this.view.options.totalCount;
      }
      if (typeof totalCount === 'undefined' && this.view.collection) {
        totalCount = this.view.collection.total;
      }
      return totalCount === -1 || totalCount > this.config.get('massActionIdleCountThreshold');
    }

    /**
     * Process.
     *
     * @param {string} id An ID.
     * @param {string} action An action.
     * @returns {Promise<module:view>} Resolves with a dialog view.
     *   The view emits the 'close:success' event.
     */
    process(id, action) {
      Espo.Ui.notify(false);
      return new Promise(resolve => {
        const entityType = this.view.scope || this.view.entityType;
        this.view.createView('dialog', 'views/modals/mass-action', {
          id: id,
          action: action,
          scope: entityType
        }).then(view => {
          view.render();
          resolve(view);
          this.view.listenToOnce(view, 'success', data => {
            resolve(data);
            this.view.listenToOnce(view, 'close', () => {
              view.trigger('close:success', data);
            });
          });
        });
      });
    }
  }
  var _default = _exports.default = MassActionHelper;
});

define("helpers/export", ["exports", "di", "models/settings", "models/user"], function (_exports, _di, _settings, _user) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _settings = _interopRequireDefault(_settings);
  _user = _interopRequireDefault(_user);
  let _init_config, _init_extra_config, _init_user, _init_extra_user;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * An export helper.
   */
  class ExportHelper {
    static {
      [_init_config, _init_extra_config, _init_user, _init_extra_user] = _applyDecs(this, [], [[(0, _di.inject)(_settings.default), 0, "config"], [(0, _di.inject)(_user.default), 0, "user"]]).e;
    }
    /**
     * @private
     * @type {Settings}
     */
    config = _init_config(this);

    /**
     * @private
     * @type {User}
     */
    user = (_init_extra_config(this), _init_user(this));

    /**
     * @param {module:view} view A view.
     */
    constructor(view) {
      _init_extra_user(this);
      /**
       * @private
       * @type {module:view}
       */
      this.view = view;
    }

    /**
     * Check whether an export should be run in idle.
     *
     * @param {number} totalCount A total record count.
     * @returns {boolean}
     */
    checkIsIdle(totalCount) {
      if (this.user.isPortal()) {
        return false;
      }
      if (typeof totalCount === 'undefined') {
        totalCount = this.view.options.totalCount;
      }
      return totalCount === -1 || totalCount > this.config.get('exportIdleCountThreshold');
    }

    /**
     * Process export.
     *
     * @param {string} id An ID.
     * @returns {Promise<module:view>} Resolves with a dialog view.
     *   The view emits the 'close:success' event.
     */
    process(id) {
      Espo.Ui.notify(false);
      return new Promise(resolve => {
        this.view.createView('dialog', 'views/export/modals/idle', {
          id: id
        }).then(view => {
          view.render();
          resolve(view);
          this.view.listenToOnce(view, 'success', data => {
            resolve(data);
            this.view.listenToOnce(view, 'close', () => {
              view.trigger('close:success', data);
            });
          });
        });
      });
    }
  }
  var _default = _exports.default = ExportHelper;
});

define("helpers/record/list/column-width-control", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class ListColumnWidthControlHelper {
    /**
     * A min width in pixels.
     *
     * @private
     * @type {number}
     */
    minWidth = 30;

    /**
     * @param {{
     *     view?: import('views/record/list').default,
     *     helper: import('helpers/list/settings').default,
     *     layoutProvider?: function(): {
     *     name: string,
     *     width?: number,
     *     widthPx?: number,
     *     hidden?: boolean,
     * }[]
     * }} options
     */
    constructor(options) {
      /** @private */
      this.view = options.view;
      /** @private */
      this.helper = options.helper;
      /** @private */
      this.layoutProvider = options.layoutProvider;
    }

    /**
     * Adjust widths.
     *
     * @param {{
     *     tableWidth?: number,
     *     staticWidth?: number,
     * }} [options]
     * @return {boolean}
     */
    adjust(options = {}) {
      let tableWidthData;
      if (options.tableWidth === undefined || options.staticWidth === undefined) {
        tableWidthData = this.getTableWidths();
      }
      const tableWidth = options.tableWidth === undefined ? tableWidthData.table : options.tableWidth;
      const staticWidth = options.staticWidth === undefined ? tableWidthData.static : options.staticWidth;
      const widthMap = this.helper.getColumnWidthMap();

      /**
       * @type {{
       *     name: string,
       *     width: {
       *         value: number,
       *         unit: 'px'|'%',
       *     }|null,
       *     isCustom: boolean,
       *     widthPx: number|null,
       * }[]}
       */
      const list = this.layoutProvider().filter(it => !this.helper.isColumnHidden(it.name, it.hidden)).map(it => {
        let widthItem = widthMap[it.name];
        const isCustom = !!widthItem;
        if (!widthItem) {
          widthItem = null;
          if (it.width) {
            widthItem = {
              value: it.width,
              unit: '%'
            };
          } else if (it.widthPx) {
            widthItem = {
              value: it.widthPx,
              unit: 'px'
            };
          }
        }
        let widthPx = null;
        if (widthItem) {
          if (widthItem.unit === 'px') {
            widthPx = widthItem.value;
          } else {
            widthPx = tableWidth * (widthItem.value / 100.0);
          }
        }
        return {
          name: it.name,
          width: widthItem,
          isCustom: isCustom,
          widthPx: widthPx
        };
      });
      const flexColumnCount = list.filter(it => !it.width).length;
      const extraWidth = flexColumnCount * this.minWidth;
      let sumWidth = 0;
      list.filter(it => it.widthPx).forEach(it => sumWidth += it.widthPx);
      if (tableWidth - extraWidth - staticWidth - sumWidth >= 0) {
        return true;
      }
      const listSorted = list.filter(it => it.widthPx && it.width).sort((a, b) => b.widthPx - a.widthPx);
      if (!listSorted.length) {
        return true;
      }
      const item = listSorted[0];
      const reduceWidthPx = 10;
      if (item.widthPx < reduceWidthPx) {
        return true;
      }

      /** @type {{value: number, unit: 'px'|'%'}} */
      let newWidth;
      if (item.width.unit === 'px') {
        newWidth = {
          value: item.width.value - reduceWidthPx,
          unit: 'px'
        };
      } else {
        const factor = Math.pow(10, 4);
        const reducePercent = Math.floor(factor * (reduceWidthPx / tableWidth) * 100) / factor;
        newWidth = {
          value: item.width.value - reducePercent,
          unit: '%'
        };
      }
      const map = this.helper.getColumnWidthMap();
      map[item.name] = newWidth;
      this.helper.storeColumnWidthMap(map);
      this.adjust({
        tableWidth,
        staticWidth
      });
      return false;
    }

    /**
     * @private
     * @return {{
     *     table: number,
     *     static: number,
     * }|null}
     */
    getTableWidths() {
      const tableElement = this.view.element.querySelector('.list > table');
      if (!tableElement) {
        return null;
      }
      const tableWidth = tableElement.clientWidth;
      let staticWidth = 0;
      tableElement.querySelectorAll(':scope > thead > tr > th').forEach(th => {
        if (!th.classList.contains('field-header-cell') || th.classList.contains('action-cell')) {
          staticWidth += th.clientWidth;
        }
      });
      return {
        table: tableWidth,
        static: staticWidth
      };
    }
  }
  _exports.default = ListColumnWidthControlHelper;
});

define("helpers/record/list/column-resize", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /**
   * @internal
   */
  class ListColumnResizeHelper {
    /**
     * @type {{
     *     startX: number,
     *     startWidth: number,
     *     thElement: HTMLTableCellElement,
     *     name: string,
     *     inPx: boolean,
     *     onRight: boolean,
     *     newWidth: number|null,
     *     thElements: HTMLTableCellElement[],
     * }}
     * @private
     */
    item;

    /**
     * A min width in pixels.
     *
     * @private
     * @type {number}
     */
    minWidth = 30;
    static selector = 'table > thead > tr > th > .column-resizer';

    /**
     * @param {import('views/record/list').default} view
     * @param {import('helpers/list/settings').default} helper
     */
    constructor(view, helper) {
      /** @private */
      this.view = view;
      /** @private */
      this.helper = helper;

      /**
       * @private
       * @type {number}
       */
      this.fontSizeFactor = view.getThemeManager().getFontSizeFactor();
      this.onPointerUpBind = this.onPointerUp.bind(this);
      this.onPointerMoveBind = this.onPointerMove.bind(this);
      view.addHandler('pointerdown', ListColumnResizeHelper.selector, (/** PointerEvent */e, target) => {
        this.onPointerDown(e, target);
      });
    }

    /**
     * @private
     * @param {PointerEvent} event
     * @param {HTMLElement} target
     */
    onPointerDown(event, target) {
      if (!event.isPrimary) {
        return;
      }
      this.startResizeInit(event, target);
      window.addEventListener('pointerup', this.onPointerUpBind);
      window.addEventListener('pointermove', this.onPointerMoveBind);
    }

    /**
     * @private
     * @param {PointerEvent} event
     * @param {HTMLElement} target
     */
    startResizeInit(event, target) {
      const th = /** @type {HTMLTableCellElement} */target.parentNode;
      const thElements = [...th.parentNode.querySelectorAll(':scope > th.field-header-cell')].filter(it => !it.style.width);
      this.item = {
        startX: event.clientX,
        startWidth: th.clientWidth,
        thElement: th,
        name: th.dataset.name,
        inPx: th.style.width && th.style.width.endsWith('px'),
        onRight: target.classList.contains('column-resizer-right'),
        newWidth: null,
        thElements: thElements
      };
      document.body.style.cursor = 'col-resize';
      const trElement = this.item.thElement.closest('tr');
      trElement.classList.add('being-column-resized');
      this.item.thElement.classList.add('being-resized');
    }

    /**
     * @private
     * @param {number} width
     */
    isWidthOk(width) {
      if (width < this.minWidth * this.fontSizeFactor) {
        return false;
      }
      for (const th of this.item.thElements) {
        if (th.style.width) {
          continue;
        }
        if (th.clientWidth < this.minWidth * this.fontSizeFactor) {
          return false;
        }
      }
      return true;
    }

    /**
     * @private
     * @param {PointerEvent} event
     */
    onPointerMove(event) {
      let diff = event.clientX - this.item.startX;
      if (!this.item.onRight) {
        diff *= -1;
      }
      const width = this.item.startWidth + diff;
      if (!this.isWidthOk(width)) {
        return;
      }
      const previousWidth = this.item.newWidth;
      const previousStyleWidth = this.item.thElement.style.width;
      this.item.newWidth = width;
      this.item.thElement.style.width = width.toString() + 'px';
      if (!this.isWidthOk(width)) {
        if (previousWidth) {
          this.item.newWidth = previousWidth;
        }
        this.item.thElement.style.width = previousStyleWidth;
      }
    }

    /**
     * @private
     */
    onPointerUp() {
      window.removeEventListener('pointermove', this.onPointerMoveBind);
      window.removeEventListener('pointerup', this.onPointerUpBind);
      document.body.style.cursor = '';
      const width = this.item.newWidth;
      if (width === null) {
        this.disableResizingState();
        return;
      }
      let unit = 'px';
      let value = width;
      if (!this.item.inPx) {
        const tableElement = this.item.thElement.closest('table');
        const tableWidth = tableElement.clientWidth;
        const factor = Math.pow(10, 4);
        const widthPercents = width / tableWidth;
        const widthPercentsRounded = Math.floor(factor * widthPercents * 100) / factor;
        this.item.thElement.style.width = widthPercentsRounded.toString() + '%';
        unit = '%';
        value = widthPercentsRounded;
      }
      if (this.item.inPx) {
        value = value / this.fontSizeFactor;
      }
      this.helper.storeColumnWidth(this.item.name, {
        value: value,
        unit: unit
      });
      this.disableResizingState();
    }

    /**
     * @private
     */
    disableResizingState() {
      const trElement = this.item.thElement.closest('tr');
      trElement.classList.remove('being-column-resized');
      this.item.thElement.classList.remove('being-resized');
      this.item = undefined;
    }
  }
  _exports.default = ListColumnResizeHelper;
});

define("helpers/misc/mailto", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class MailtoHelper {
    /**
     * @param {import('models/settings').default} config
     * @param {import('models/preferences').default} preferences
     * @param {import('acl-manager').default} acl
     */
    constructor(config, preferences, acl) {
      this.config = config;
      this.preferences = preferences;
      this.acl = acl;
    }

    /**
     * Whether mailto should be used.
     *
     * @return {boolean}
     */
    toUse() {
      return this.config.get('emailForceUseExternalClient') || this.preferences.get('emailUseExternalClient') || !this.acl.checkScope('Email', 'create');
    }

    /**
     * Compose a mailto link.
     *
     * @param {Record} attributes
     * @return {string}
     */
    composeLink(attributes) {
      let link = 'mailto:';
      link += (attributes.to || '').split(';').join(',');
      const params = {};
      if (attributes.cc) {
        params.cc = attributes.cc.split(';').join(',');
      }
      let bcc = this.config.get('outboundEmailBccAddress');
      if (attributes.bcc) {
        if (!bcc) {
          bcc = '';
        } else {
          bcc += ';';
        }
        bcc += attributes.bcc;
      }
      if (bcc) {
        params.bcc = bcc.split(';').join(',');
      }
      if (attributes.name) {
        params.subject = attributes.name;
      }
      if (attributes.body) {
        params.body = /** @type {string} */attributes.body;
        if (attributes.isHtml) {
          params.body = this.htmlToPlain(params.body);
        }
        if (params.body.length > 700) {
          params.body = params.body.substring(0, 700) + '...';
        }
      }
      if (attributes.inReplyTo) {
        params['In-Reply-To'] = attributes.inReplyTo;
      }
      let part = '';
      for (const key in params) {
        if (part !== '') {
          part += '&';
        } else {
          part += '?';
        }
        part += key + '=' + encodeURIComponent(params[key]);
      }
      link += part;
      return link;
    }

    /**
     * @private
     * @param {string} text
     * @returns {string}
     */
    htmlToPlain(text) {
      text = text || '';
      let value = text.replace(/<br\s*\/?>/mg, '\n');
      value = value.replace(/<\/p\s*\/?>/mg, '\n\n');
      const $div = $('<div>').html(value);
      $div.find('style').remove();
      $div.find('link[ref="stylesheet"]').remove();
      value = $div.text();
      return value;
    }
  }
  var _default = _exports.default = MailtoHelper;
});

define("helpers/list/settings", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class ListSettingsHelper {
    /**
     * @typedef {Object} ListSettingsHelper~columnWidth
     * @property {number} value A value.
     * @property {'px'|'%'} unit A unit.
     */

    /**
     * @param {string} entityType
     * @param {string} type
     * @param {string} userId
     * @param {module:storage} storage
     */
    constructor(entityType, type, userId, storage) {
      /** @private */
      this.storage = storage;

      /** @private */
      this.layoutColumnsKey = `${type}-${entityType}-${userId}`;

      /**
       * @private
       * @type {Object.<string, boolean>}
       */
      this.hiddenColumnMapCache = undefined;

      /**
       * @private
       * @type {Object.<string, ListSettingsHelper~columnWidth>}
       */
      this.columnWidthMapCache = undefined;

      /**
       * @private
       * @type {boolean|undefined}
       */
      this.columnResize = undefined;

      /**
       * @private
       * @type {function()[]}
       */
      this.columnWidthChangeFunctions = [];
    }

    /**
     * Get a stored hidden column map.
     *
     * @return {Object.<string, boolean>}
     */
    getHiddenColumnMap() {
      if (this.hiddenColumnMapCache) {
        return this.hiddenColumnMapCache;
      }
      this.hiddenColumnMapCache = this.storage.get('listHiddenColumns', this.layoutColumnsKey) || {};
      return this.hiddenColumnMapCache;
    }

    /**
     * Is a column hidden.
     *
     * @param {string} name A name.
     * @param {boolean} [hidden] Is hidden by default.
     * @return {boolean}
     * @since 9.0.0
     */
    isColumnHidden(name, hidden) {
      const hiddenMap = this.getHiddenColumnMap();
      if (hiddenMap[name]) {
        return true;
      }
      if (!hidden) {
        return false;
      }
      if (!(name in hiddenMap)) {
        return true;
      }
      return hiddenMap[name];
    }

    /**
     * Is column resize enabled.
     *
     * @return {boolean}
     * @since 9.0.0
     */
    getColumnResize() {
      if (this.columnResize === undefined) {
        this.columnResize = this.storage.get('listColumnResize', this.layoutColumnsKey) || false;
      }
      return this.columnResize;
    }

    /**
     * Store column width editable.
     *
     * @param {boolean} columnResize
     */
    storeColumnResize(columnResize) {
      this.columnResize = columnResize;
      this.storage.set('listColumnResize', this.layoutColumnsKey, columnResize);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Clear column width editable.
     */
    clearColumnResize() {
      this.columnResize = undefined;
      this.storage.clear('listColumnResize', this.layoutColumnsKey);
    }

    /**
     * Store a hidden column map.
     *
     * @param {Object.<string, boolean>} map
     */
    storeHiddenColumnMap(map) {
      this.hiddenColumnMapCache = undefined;
      this.storage.set('listHiddenColumns', this.layoutColumnsKey, map);
    }

    /**
     * Clear a hidden column map in the storage.
     */
    clearHiddenColumnMap() {
      this.hiddenColumnMapCache = undefined;
      this.storage.clear('listHiddenColumns', this.layoutColumnsKey);
    }

    /**
     * Get a stored column width map.
     *
     * @return {Object.<string, ListSettingsHelper~columnWidth>}
     */
    getColumnWidthMap() {
      if (this.columnWidthMapCache) {
        return this.columnWidthMapCache;
      }
      this.columnWidthMapCache = this.storage.get('listColumnsWidths', this.layoutColumnsKey) || {};
      return this.columnWidthMapCache;
    }

    /**
     * Store a column width map.
     *
     * @param {Object.<string, ListSettingsHelper~columnWidth>} map
     */
    storeColumnWidthMap(map) {
      this.columnWidthMapCache = undefined;
      this.storage.set('listColumnsWidths', this.layoutColumnsKey, map);
    }

    /**
     * Clear a column width map in the storage.
     */
    clearColumnWidthMap() {
      this.columnWidthMapCache = undefined;
      this.storage.clear('listColumnsWidths', this.layoutColumnsKey);
    }

    /**
     * Set a column width.
     *
     * @param {string} name A column name.
     * @param {ListSettingsHelper~columnWidth} width Width data.
     */
    storeColumnWidth(name, width) {
      if (!this.columnWidthMapCache) {
        this.columnWidthMapCache = {};
      }
      this.columnWidthMapCache[name] = width;
      this.storeColumnWidthMap(this.columnWidthMapCache);
      for (const handler of this.columnWidthChangeFunctions) {
        handler();
      }
    }

    /**
     * Subscribe to a column width change.
     *
     * @param {function()} handler A handler.
     */
    subscribeToColumnWidthChange(handler) {
      this.columnWidthChangeFunctions.push(handler);
    }

    /**
     * Unsubscribe from a column width change.
     *
     * @param {function()} handler A handler.
     */
    unsubscribeFromColumnWidthChange(handler) {
      const index = this.columnWidthChangeFunctions.findIndex(it => handler === it);
      if (!~index) {
        return;
      }
      this.columnWidthChangeFunctions.splice(index, 1);
    }
  }
  var _default = _exports.default = ListSettingsHelper;
});

define("helpers/list/select-provider", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class SelectProvider {
    /**
     * @param {module:layout-manager} layoutManager
     * @param {module:metadata} metadata
     * @param {module:field-manager} fieldManager
     */
    constructor(layoutManager, metadata, fieldManager) {
      this.layoutManager = layoutManager;
      this.metadata = metadata;
      this.fieldManager = fieldManager;
    }

    /**
     * Get select attributes.
     *
     * @param {string} entityType
     * @param {string} [layoutName='list']
     * @return {Promise<string[]>}
     */
    get(entityType, layoutName) {
      return new Promise(resolve => {
        this.layoutManager.get(entityType, layoutName || 'list', layout => {
          const list = this.getFromLayout(entityType, layout);
          resolve(list);
        });
      });
    }

    /**
     * Get select attributes from a layout.
     *
     * @param {string} entityType
     * @param {module:views/record/list~columnDefs[]} listLayout
     * @return {string[]}
     */
    getFromLayout(entityType, listLayout) {
      let list = [];
      listLayout.forEach(item => {
        if (!item.name) {
          return;
        }
        const field = item.name;
        const fieldType = this.metadata.get(['entityDefs', entityType, 'fields', field, 'type']);
        if (!fieldType) {
          return;
        }
        list = [this.fieldManager.getEntityTypeFieldAttributeList(entityType, field), ...list];
      });
      return list;
    }
  }
  var _default = _exports.default = SelectProvider;
});

define("helpers/list/misc/sticky-bar", ["exports", "jquery", "bullbone"], function (_exports, _jquery, _bullbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _jquery = _interopRequireDefault(_jquery);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /**
   * @internal
   *
   * @mixes Bull.Events
   */
  class StickyBarHelper {
    /** @private */
    $bar;
    /** @private */
    $scrollable;
    /** @private */
    $window;
    /** @private */
    $navbarRight;
    /** @private */
    $middle;
    /** @private */
    _isReady = false;

    /**
     * @param {import('views/record/list').default} view
     * @param {{force?: boolean}} options
     */
    constructor(view, options = {}) {
      this.view = view;

      /**
       * @private
       * @type {import('theme-manager').default}
       */

      this.themeManager = this.view.getThemeManager();
      this.$el = view.$el;

      /** @private */
      this.force = options.force || false;
      this.init();
    }
    init() {
      this.$bar = this.$el.find('.sticked-bar');
      this.$middle = this.$el.find('> .list');
      if (!this.$middle.get(0)) {
        return;
      }
      this.$window = (0, _jquery.default)(window);
      this.$scrollable = this.$window;
      this.$navbarRight = (0, _jquery.default)('#navbar .navbar-right');
      this.isModal = !!this.$el.closest('.modal-body').length;
      this.isSmallWindow = (0, _jquery.default)(window.document).width() < this.themeManager.getParam('screenWidthXs');
      if (this.isModal) {
        this.$scrollable = this.$el.closest('.modal-body');
        this.$navbarRight = this.$scrollable.parent().find('.modal-footer');
      }
      if (!this.force) {
        this.$scrollable.off(`scroll.list-${this.view.cid}`);
        this.$scrollable.on(`scroll.list-${this.view.cid}`, () => this._controlSticking());
        this.$window.off(`resize.list-${this.view.cid}`);
        this.$window.on(`resize.list-${this.view.cid}`, () => this._controlSticking());
      }
      this.listenTo(this.view, 'check', () => {
        if (this.view.getCheckedIds().length === 0 && !this.view.isAllResultChecked()) {
          return;
        }
        this._controlSticking();
      });
      this._isReady = true;
    }
    _getMiddleTop() {
      if (this._middleTop !== undefined && this._middleTop >= 0) {
        return this._middleTop;
      }
      this._middleTop = this._getOffsetTop(this.$middle.get(0));
      return this._middleTop;
    }
    _getButtonsTop() {
      if (this._buttonsTop !== undefined && this._buttonsTop >= 0) {
        return this._buttonsTop;
      }
      this._buttonsTop = this._getOffsetTop(this.$el.find('.list-buttons-container').get(0));
      return this._buttonsTop;
    }

    /**
     * @private
     */
    _controlSticking() {
      if (!this.view.toShowStickyBar()) {
        return;
      }
      if (this.isSmallWindow && (0, _jquery.default)('#navbar .navbar-body').hasClass('in')) {
        return;
      }
      const scrollTop = this.$scrollable.scrollTop();
      const stickTop = !this.force ? this._getButtonsTop() : 0;
      const edge = this._getMiddleTop() + this.$middle.outerHeight(true);
      const hide = () => {
        this.$bar.addClass('hidden');
        this.$navbarRight.removeClass('has-sticked-bar');
      };
      const show = () => {
        this.$bar.removeClass('hidden');
        this.$navbarRight.addClass('has-sticked-bar');
      };
      if (scrollTop >= edge) {
        hide();
        return;
      }
      if (scrollTop > stickTop || this.force) {
        show();
        return;
      }
      hide();
    }

    /**
     * @private
     * @param {HTMLElement} element
     */
    _getOffsetTop(element) {
      if (!element) {
        return 0;
      }
      const navbarHeight = this.themeManager.getParam('navbarHeight') * this.themeManager.getFontSizeFactor();
      const withHeader = !this.isSmallWindow && !this.isModal;
      let offsetTop = 0;
      do {
        if (element.classList.contains('modal-body')) {
          break;
        }
        if (!isNaN(element.offsetTop)) {
          offsetTop += element.offsetTop;
        }
        element = element.offsetParent;
      } while (element);
      if (withHeader) {
        offsetTop -= navbarHeight;
      }
      if (!this.isModal) {
        // padding
        offsetTop -= 5;
      }
      return offsetTop;
    }
    hide() {
      this.$bar.addClass('hidden');
    }
    destroy() {
      this.stopListening(this.view, 'check');
      if (!this._isReady) {
        return;
      }
      this.$window.off(`resize.list-${this.view.cid}`);
      this.$scrollable.off(`scroll.list-${this.view.cid}`);
    }
  }
  Object.assign(StickyBarHelper.prototype, _bullbone.Events);
  var _default = _exports.default = StickyBarHelper;
});

define("exceptions", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module exceptions */

  Espo.Exceptions = Espo.Exceptions || {};

  /**
   * An access-denied exception.
   *
   * @param {string} [message] A message.
   * @class
   */
  Espo.Exceptions.AccessDenied = function (message) {
    this.message = message;
    Error.apply(this, arguments);
  };
  Espo.Exceptions.AccessDenied.prototype = new Error();
  Espo.Exceptions.AccessDenied.prototype.name = 'AccessDenied';

  /**
   * A not-found exception.
   *
   * @param {string} [message] A message.
   * @class
   */
  Espo.Exceptions.NotFound = function (message) {
    this.message = message;
    Error.apply(this, arguments);
  };
  Espo.Exceptions.NotFound.prototype = new Error();
  Espo.Exceptions.NotFound.prototype.name = 'NotFound';
  var _default = _exports.default = Espo.Exceptions;
});

define("views/record/list", ["exports", "view", "helpers/mass-action", "helpers/export", "helpers/record-modal", "helpers/list/select-provider", "views/record/list/settings", "helpers/list/settings", "helpers/list/misc/sticky-bar", "helpers/record/list/column-resize", "helpers/record/list/column-width-control"], function (_exports, _view, _massAction, _export, _recordModal, _selectProvider, _settings, _settings2, _stickyBar, _columnResize, _columnWidthControl) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  _massAction = _interopRequireDefault(_massAction);
  _export = _interopRequireDefault(_export);
  _recordModal = _interopRequireDefault(_recordModal);
  _selectProvider = _interopRequireDefault(_selectProvider);
  _settings = _interopRequireDefault(_settings);
  _settings2 = _interopRequireDefault(_settings2);
  _stickyBar = _interopRequireDefault(_stickyBar);
  _columnResize = _interopRequireDefault(_columnResize);
  _columnWidthControl = _interopRequireDefault(_columnWidthControl);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/record/list */

  /**
   * A record-list view. Renders and processes list items, actions.
   */
  class ListRecordView extends _view.default {
    /**
     * A row action.
     *
     * @typedef {Object} module:views/record/list~rowAction
     *
     * @property {string} action An action.
     * @property {string} [label] A label.
     * @property {string} [link] A link.
     * @property {string} [text] A text.
     * @property {Object.<string, string|number|boolean>} [data] Data attributes.
     * @property {number} [groupIndex] A group index.
     */

    /**
     * List view options.
     *
     * @typedef {Record} module:views/record/list~options
     * @property {import('collection').default} collection A collection.
     * @property {module:views/record/list~columnDefs[]} [listLayout] A layout.
     * @property {string|'list'|'listSmall'} [type] A type.
     * @property {string} [layoutName] A layout name.
     * @property {boolean} [checkboxes] To show row checkboxes.
     * @property {boolean} [selectable] Clicking on the record link will trigger the 'select' event.
     * @property {boolean} [skipBuildRows] Do not build rows on initialization. Use when the collection will be fetched
     *    afterward.
     * @property {boolean} [buttonsDisabled] Disable buttons.
     * @property {boolean} [checkAllResultDisabled] Disable select-all-results.
     * @property {boolean} [pagination] To enable the pagination.
     * @property {boolean} [headerDisabled] Disable the header.
     * @property {boolean} [noDataDisabled] Disable the no-data label (when no results).
     * @property {string} [rowActionsView] A row actions view.
     * @property {boolean} [rowActionsDisabled] Disable row actions.
     * @property {boolean} [showMore] The show-more button.
     * @property {boolean} [keepCurrentRootUrl] Keep a current root URL.
     * @property {boolean} [stickyBarDisabled] Disable the sticky bar.
     * @property {boolean} [forceStickyBar] To make bar sticky regardless of scrolling.
     * @property {boolean} [massActionsDisabled] Disable mass actions.
     * @property {module:views/record/list~dropdownItem[]} [dropdownItemList] Dropdown items.
     * @property {string[]} [mandatorySelectAttributeList] Mandatory select attributes. Attributes to be selected
     *     regardless being in the layout.
     * @property {boolean} [editDisabled] Disable edit.
     * @property {boolean} [removeDisabled] Disable remove.
     * @property {boolean} [showCount] To show a record count.
     * @property {boolean} [forceDisplayTopBar] Force displaying the top bar even if empty.
     * @property {boolean} [unlinkMassAction] Enable the 'unlink' mass-action.
     * @property {Record} [rowActionsOptions] Row-actions options.
     * @property {string[]} [additionalRowActionList] Additional row-action list.
     * @property {boolean} [settingsEnabled] Enable settings dropdown.
     * @property {import('helpers/list/settings').default} [settingsHelper] A settings helper.
     * @property {boolean} [displayTotalCount] Display total count.
     * @property {Record} [rootData] Root data.
     * @property {boolean} [columnResize] Column resize. Actual only if the settings is enabled.
     */

    /**
     * @param {module:views/record/list~options | Record} options Options.
     */
    constructor(options) {
      super(options);
    }

    /** @inheritDoc */
    template = 'record/list';

    /**
     * A type. Can be 'list', 'listSmall'.
     */
    type = 'list';

    /** @inheritDoc */
    name = 'list';

    // noinspection JSUnusedGlobalSymbols
    /**
     * A presentation type.
     */
    presentationType = 'table';

    /**
     * If true checkboxes will be shown. Can be overridden by an option parameter.
     *
     * @protected
     */
    checkboxes = true;

    /**
     * If true clicking on the record link will trigger the 'select' event with model passed.
     * Can be overridden by an option parameter.
     */
    selectable = false;

    /**
     * A row-actions view. A dropdown on the right side.
     *
     * @protected
     * @type {string}
     */
    rowActionsView = 'views/record/row-actions/default';

    /**
     * Disable row-actions. Can be overridden by an option parameter.
     */
    rowActionsDisabled = false;

    /**
     * An entity type. Set automatically.
     *
     * @type {string|null}
     */
    entityType = null;

    /**
     * A scope. Set automatically.
     *
     * @type {?string}
     */
    scope = null;

    /** @protected */
    _internalLayoutType = 'list-row';

    /**
     * A selector to a list container.
     *
     * @protected
     */
    listContainerEl = '.list > table > tbody';

    /**
     * To show number of records. Can be overridden by an option parameter.
     *
     * @protected
     */
    showCount = true;

    /** @protected */
    rowActionsColumnWidth = 25;

    /** @protected */
    checkboxColumnWidth = 40;

    /**
     * A button. Handled by a class method `action{Name}` or a handler.
     *
     * @typedef {Object} module:views/record/list~button
     *
     * @property {string} name A name.
     * @property {string} label A label. To be translated in a current scope.
     * @property {'default'|'danger'|'warning'|'success'} [style] A style.
     * @property {boolean} [hidden] Hidden.
     * @property {function()} [onClick] A click handler.
     */

    /**
     * A button list.
     *
     * @protected
     * @type {module:views/record/list~button[]}
     */
    buttonList = [];

    /**
     * A dropdown item. Handled by a class method `action{Name}` or a handler.
     *
     * @typedef {Object} module:views/record/list~dropdownItem
     *
     * @property {string} name A name.
     * @property {string} [label] A label. To be translated in a current scope.
     * @property {string} [html] An HTML.
     * @property {boolean} [hidden] Hidden.
     * @property {function()} [onClick] A click handler.
     */

    /**
     * A dropdown item list. Can be overridden by an option parameter.
     *
     * @protected
     * @type {module:views/record/list~dropdownItem[]}
     */
    dropdownItemList = [];

    /**
     * Disable the header. Can be overridden by an option parameter.
     *
     * @protected
     */
    headerDisabled = false;

    /**
     * Disable mass actions. Can be overridden by an option parameter.
     *
     * @protected
     */
    massActionsDisabled = false;

    /**
     * Disable a portal layout usage. Can be overridden by an option parameter.
     *
     * @protected
     */
    portalLayoutDisabled = false;

    /**
     * Mandatory select attributes. Can be overridden by an option parameter.
     * Attributes to be selected regardless being in the layout.
     *
     * @protected
     * @type {string[]|null}
     */
    mandatorySelectAttributeList = null;

    /**
     * A layout name. If null, a value from `type` property will be used.
     * Can be overridden by an option parameter.
     *
     * @protected
     * @type {string|null}
     */
    layoutName = null;

    /**
     * A scope name for layout loading. If null, an entity type of collection will be used.
     * Can be overridden by an option parameter.
     *
     * @protected
     * @type {string|null}
     */
    layoutScope = null;

    /**
     * To disable field-level access check for a layout.
     * Can be overridden by an option parameter.
     *
     * @protected
     */
    layoutAclDisabled = false;

    /**
     * A setup-handler type.
     *
     * @protected
     */
    setupHandlerType = 'record/list';

    /**
     * @internal
     * @private
     */
    checkboxesDisabled = false;

    /**
     * Force displaying the top bar even if empty. Can be overridden by an option parameter.
     * @protected
     */
    forceDisplayTopBar = false;

    /**
     * Where to display the pagination. Can be overridden by an option parameter.
     *
     * @protected
     * @type {boolean}
     */
    pagination = false;

    /**
     * To display a table header with column names. Can be overridden by an option parameter.
     *
     * @protected
     * @type {boolean}
     */
    header = true;

    /**
     * A show-more button. Can be overridden by an option parameter.
     *
     * @protected
     */
    showMore = true;

    /**
     * Column resize.
     *
     * @protected
     * @type {boolean}
     * @since 9.0.0
     */
    columnResize = true;

    /**
     * A mass-action list.
     *
     * @protected
     * @type {string[]}
     */
    massActionList = ['remove', 'merge', 'massUpdate', 'export'];

    /**
     * A mass-action list available when selecting all results.
     *
     * @protected
     * @type {string[]}
     */
    checkAllResultMassActionList = ['remove', 'massUpdate', 'export'];

    /**
     * A forced mass-action list.
     *
     * @protected
     * @type {?string[]}
     */
    forcedCheckAllResultMassActionList = null;

    /**
     * Disable quick-detail (viewing a record in modal)
     *
     * @protected
     */
    quickDetailDisabled = false;

    /**
     * Disable quick-edit (editing a record in modal)
     *
     * @protected
     */
    quickEditDisabled = false;

    /**
     * Force settings.
     *
     * @protected
     * @type {boolean}
     */
    forceSettings = false;

    /**
     * Disable settings.
     *
     * @protected
     * @type {boolean}
     */
    settingsDisabled = false;

    /**
     * Column definitions.
     *
     * @typedef module:views/record/list~columnDefs
     * @type {Object}
     * @property {string} name A name (usually a field name).
     * @property {string} [view] An overridden field view name.
     * @property {number} [width] A width in percents.
     * @property {number} [widthPx] A width in pixels.
     * @property {boolean} [link] To use `listLink` mode (link to the detail view).
     * @property {'left'|'right'} [align] An alignment.
     * @property {string} [type] An overridden field type.
     * @property {Object.<string, *>} [params] Overridden field parameters.
     * @property {Object.<string, *>} [options] Field view options.
     * @property {string} [label] A label.
     * @property {boolean} [notSortable] Not sortable.
     * @property {boolean} [hidden] Hidden by default.
     * @property {boolean} [noLabel] No label.
     * @property {string} [customLabel] A custom label.
     */

    /**
     * A list layout. Can be overridden by an option parameter.
     * If null, then will be loaded from the backend (using the `layoutName` value).
     *
     * @protected
     * @type {module:views/record/list~columnDefs[]|null}
     */
    listLayout = null;

    /** @private */
    _internalLayout = null;

    /**
     * A list of record IDs currently selected. Only for reading.
     *
     * @protected
     * @type {string[]|null}
     */
    checkedList = null;

    /**
     * Whether all results currently selected. Only for reading.
     *
     * @protected
     */
    allResultIsChecked = false;

    /**
     * Disable the ability to select all results. Can be overridden by an option parameter.
     *
     * @protected
     */
    checkAllResultDisabled = false;

    /**
     * Disable buttons. Can be overridden by an option parameter.
     *
     * @protected
     */
    buttonsDisabled = false;

    /**
     * Disable edit. Can be overridden by an option parameter.
     *
     * @protected
     */
    editDisabled = false;

    /**
     * Disable remove. Can be overridden by an option parameter.
     *
     * @protected
     */
    removeDisabled = false;

    /**
     * Disable a stick-bar. Can be overridden by an option parameter.
     *
     * @protected
     */
    stickyBarDisabled = false;

    /**
     * To show sticky bar regardless of scrolling.
     *
     * @protected
     */
    forceStickyBar = false;

    /**
     * Disable the follow/unfollow mass action.
     *
     * @protected
     */
    massFollowDisabled = false;

    /**
     * Disable the print-pdf mass action.
     *
     * @protected
     */
    massPrintPdfDisabled = false;

    /**
     * Disable the convert-currency mass action.
     *
     * @protected
     */
    massConvertCurrencyDisabled = false;

    /**
     * Disable mass-update.
     *
     * @protected
     */
    massUpdateDisabled = false;

    /**
     * Disable export.
     *
     * @protected
     */
    exportDisabled = false;

    /**
     * Disable merge.
     *
     * @protected
     */
    mergeDisabled = false;

    /**
     * Disable the no-data label (when no result).
     *
     * @protected
     */
    noDataDisabled = false;

    /**
     * Disable pagination.
     *
     * @protected
     */
    paginationDisabled = false;

    /** @private */
    _$focusedCheckbox = null;

    /**
     * @protected
     * @type {?JQuery}
     */
    $selectAllCheckbox = null;

    /** @private */
    _disabledCheckboxes = false;

    /**
     * Mass-action definitions.
     *
     * @typedef {Object} module:views/record/list~massActionItem
     * @property {string} [name] A name.
     * @property {number} [groupIndex] A group index.
     * @property {string} [handler] A handler.
     * @property {string} [initFunction] An init function.
     * @property {string} [actionFunction] An action function.
     * @property {string} [configCheck] A config check.
     * @property {string} [aclScope] An ACL scope to check.
     * @property {string} [acl] An access action to check.
     * @property {string} [url]
     * @property {boolean} [bypassConfirmation] To skip confirmation.
     * @property {string} [confirmationMessage] A confirmation message.
     * @property {string} [waitMessage] A wait message.
     * @property {string} [successMessage] A success message.
     * @property {boolean} [hidden] Is hidden.
     */

    /**
     * @private
     * @type {Object.<string, module:views/record/list~massActionItem>}
     */
    massActionDefs;

    /**
     * Data to pass to record views.
     *
     * @protected
     * @type {Object.<string, *>}
     * @since 9.0.0
     */
    rootData;

    /** @private */
    _additionalRowActionList;

    /**
     * @private
     * @type {import('helpers/record/list/column-resize').default}
     */
    _columnResizeHelper;

    /** @inheritDoc */
    events = {
      /**
       * @param {JQueryMouseEventObject} e
       * @this ListRecordView
       */
      'auxclick a.link': function (e) {
        const isCombination = e.button === 1 && (e.ctrlKey || e.metaKey);
        if (!isCombination) {
          return;
        }
        const $target = $(e.currentTarget);
        const id = $target.attr('data-id');
        if (!id) {
          return;
        }
        if (this.quickDetailDisabled) {
          return;
        }
        const $menu = $target.parent().closest(`[data-id="${id}"]`).find(`ul.list-row-dropdown-menu[data-id="${id}"]`);
        const $quickView = $menu.find(`a[data-action="quickView"]`);
        if ($menu.length && !$quickView.length) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        this.actionQuickView({
          id: id
        });
      },
      /** @this ListRecordView */
      'click [data-action="showMore"]': async function () {
        await this.showMoreRecords();
        this.focusOnList();
      },
      'mousedown a.sort': function (e) {
        e.preventDefault();
      },
      /**
       * @param {JQueryKeyEventObject} e
       * @this module:views/record/list
       */
      'click a.sort': function (e) {
        const field = $(e.currentTarget).data('name');
        this.toggleSort(field);
      },
      /**
       * @param {JQueryKeyEventObject} e
       * @this ListRecordView
       */
      'click .pagination a[data-page]': function (e) {
        const page = $(e.currentTarget).data('page');
        if ($(e.currentTarget).parent().hasClass('disabled')) {
          return;
        }
        this.goToPage(page);
      },
      /** @this ListRecordView */
      'mousedown input.record-checkbox': function () {
        const $focused = $(document.activeElement);
        this._$focusedCheckbox = null;
        if ($focused.length && $focused.get(0).tagName === 'INPUT' && $focused.hasClass('record-checkbox')) {
          this._$focusedCheckbox = $focused;
        }
      },
      /**
       * @param {JQueryKeyEventObject} e
       * @this ListRecordView
       */
      'click input.record-checkbox': function (e) {
        if (this._disabledCheckboxes) {
          return;
        }
        const $target = $(e.currentTarget);
        const $from = this._$focusedCheckbox;
        if (e.shiftKey && $from) {
          const $checkboxes = this.$el.find('input.record-checkbox');
          const start = $checkboxes.index($target);
          const end = $checkboxes.index($from);
          const checked = $from.prop('checked');
          $checkboxes.slice(Math.min(start, end), Math.max(start, end) + 1).each((i, el) => {
            const $el = $(el);
            $el.prop('checked', checked);
            this.checkboxClick($el, checked);
          });
          return;
        }
        this.checkboxClick($target, $target.is(':checked'));
      },
      /**
       * @param {JQueryKeyEventObject} e
       * @this module:views/record/list
       */
      'click .select-all': function (e) {
        if (this._disabledCheckboxes) {
          return;
        }

        // noinspection JSUnresolvedReference
        this.selectAllHandler(e.currentTarget.checked);
      },
      /** @this ListRecordView */
      'click .action': function (e) {
        Espo.Utils.handleAction(this, e.originalEvent, e.currentTarget, {
          actionItems: [...this.buttonList, ...this.dropdownItemList],
          className: 'list-action-item'
        });
      },
      /** @this ListRecordView */
      'click .checkbox-dropdown [data-action="selectAllResult"]': function () {
        if (this._disabledCheckboxes) {
          return;
        }
        this.selectAllResult();
      },
      /**
       * @param {JQueryKeyEventObject} e
       * @this ListRecordView
       */
      'click .actions-menu a.mass-action': function (e) {
        const $el = $(e.currentTarget);
        const action = $el.data('action');
        const method = 'massAction' + Espo.Utils.upperCaseFirst(action);
        e.preventDefault();
        e.stopPropagation();
        const $parent = $el.closest('.dropdown-menu').parent();

        // noinspection JSUnresolvedReference
        $parent.find('.actions-button[data-toggle="dropdown"]').dropdown('toggle').focus();
        if (method in this) {
          this[method]();
          return;
        }
        this.massAction(action);
      },
      /** @this ListRecordView */
      'click a.reset-custom-order': function () {
        this.resetCustomOrder();
      }
    };
    focusOnList() {
      const element = /** @type {HTMLElement} */this.$el.find('.list').get(0);
      if (!element) {
        return;
      }
      element.focus({
        preventScroll: true
      });
    }

    /**
     * @private
     * @param {'first'|'last'|'next'|'previous'|'current'} page
     */
    goToPage(page) {
      Espo.Ui.notify(' ... ');
      const onSync = () => {
        Espo.Ui.notify(false);
        this.trigger('after:paginate');
        this.focusOnList();
      };
      if (page === 'current') {
        this.collection.fetch().then(() => onSync());
        this.deactivate();
        return;
      }
      if (page === 'next') {
        this.collection.nextPage().then(() => onSync());
      } else if (page === 'previous') {
        this.collection.previousPage().then(() => onSync());
      } else if (page === 'last') {
        this.collection.lastPage().then(() => onSync());
      } else if (page === 'first') {
        this.collection.firstPage().then(() => onSync());
      }
      this.trigger('paginate');
      this.deactivate();
    }

    /**
     * @param {JQuery} $checkbox
     * @param {boolean} checked
     * @private
     */
    checkboxClick($checkbox, checked) {
      const id = $checkbox.attr('data-id');
      if (checked) {
        this.checkRecord(id, $checkbox);
        return;
      }
      this.uncheckRecord(id, $checkbox);
    }
    resetCustomOrder() {
      this.collection.offset = 0;
      this.collection.resetOrderToDefault();
      this.collection.trigger('order-changed');
      this.collection.fetch().then(() => {
        this.trigger('sort', {
          orderBy: this.collection.orderBy,
          order: this.collection.order
        });
      });
    }

    /**
     * @param {string} orderBy
     * @protected
     */
    toggleSort(orderBy) {
      let asc = true;
      if (orderBy === this.collection.orderBy && this.collection.order === 'asc') {
        asc = false;
      }
      const order = asc ? 'asc' : 'desc';
      Espo.Ui.notify(' ... ');
      const maxSizeLimit = this.getConfig().get('recordListMaxSizeLimit') || 200;
      while (this.collection.length > maxSizeLimit) {
        this.collection.pop();
      }
      this.collection.offset = 0;
      this.collection.sort(orderBy, order).then(() => {
        Espo.Ui.notify(false);
        this.trigger('sort', {
          orderBy: orderBy,
          order: order
        });
      });
      this.collection.trigger('order-changed');
      this.deactivate();
    }

    /**
     * @return {boolean}
     */
    toShowStickyBar() {
      return this.getCheckedIds().length > 0 || this.isAllResultChecked() || this.pagination;
    }

    /** @private */
    initStickyBar() {
      this._stickyBarHelper = new _stickyBar.default(this, {
        force: this.forceStickyBar
      });
    }

    /** @protected */
    showActions() {
      this.$el.find('.actions-button').removeClass('hidden');
      if (!this.options.stickyBarDisabled && !this.stickyBarDisabled && this.massActionList.length) {
        if (!this._stickyBarHelper) {
          this.initStickyBar();
        }
      }
    }

    /** @protected */
    hideActions() {
      this.$el.find('.actions-button').addClass('hidden');
      if (this._stickyBarHelper && (!this.pagination || this.forceStickyBar)) {
        this._stickyBarHelper.hide();
      }
    }

    /** @protected */
    selectAllHandler(isChecked) {
      this.checkedList = [];
      if (isChecked) {
        this.$el.find('input.record-checkbox').prop('checked', true);
        this.showActions();
        this.collection.models.forEach(model => {
          this.checkedList.push(model.id);
        });
        this.$el.find('.list > table tbody tr').addClass('active');
      } else {
        if (this.allResultIsChecked) {
          this.unselectAllResult();
        }
        this.$el.find('input.record-checkbox').prop('checked', false);
        this.hideActions();
        this.$el.find('.list > table tbody tr').removeClass('active');
      }
      this.trigger('check');
    }

    /** @inheritDoc */
    data() {
      const moreCount = this.collection.total - this.collection.length - this.collection.offset;
      let checkAllResultDisabled = this.checkAllResultDisabled;
      if (!this.massActionsDisabled) {
        if (!this.checkAllResultMassActionList.length) {
          checkAllResultDisabled = true;
        }
      }
      const displayTotalCount = this.displayTotalCount && this.collection.total > 0 && !this.pagination;
      let topBar = this.forceDisplayTopBar || this.collection.length && (this.pagination || this.checkboxes || this.buttonList.length && !this.buttonsDisabled || this.dropdownItemList.length && !this.buttonsDisabled || displayTotalCount);
      if (!topBar && this.pagination && !this.collection.length && this.collection.offset > 0) {
        topBar = true;
      }
      if (this.forceStickyBar) {
        topBar = false;
      }
      const checkboxes = this.checkboxes && this.massActionList.length;
      const displayActionsButtonGroup = checkboxes || this.buttonList.length || this.dropdownItemList.length;
      const hasStickyBar = this.forceStickyBar || displayActionsButtonGroup || this.pagination;
      const noDataDisabled = this.noDataDisabled || this._renderEmpty;
      const rowDataList = this.rowList ? this.rowList.map(id => {
        return {
          id: id,
          isStarred: this.hasStars && this.collection.get(id) ? this.collection.get(id).attributes.isStarred : false
        };
      }) : [];
      const checkboxColumnWidth = (this.checkboxColumnWidth * this._fontSizeFactor).toString() + 'px';
      return {
        scope: this.scope,
        collectionLength: this.collection.models.length,
        entityType: this.entityType,
        header: this.header,
        hasColumnResize: this._hasColumnResize(),
        headerDefs: this._getHeaderDefs(),
        hasPagination: this.hasPagination(),
        showMoreActive: this.collection.hasMore(),
        showMoreEnabled: this.showMore,
        showCount: this.showCount && this.collection.total > 0,
        moreCount: moreCount,
        checkboxes: this.checkboxes,
        massActionDataList: this.getMassActionDataList(),
        rowList: this.rowList,
        // For bc.
        rowDataList: rowDataList,
        topBar: topBar,
        checkAllResultDisabled: checkAllResultDisabled,
        buttonList: this.buttonList,
        dropdownItemList: this.dropdownItemList,
        displayTotalCount: displayTotalCount,
        displayActionsButtonGroup: displayActionsButtonGroup,
        totalCountFormatted: this.getNumberUtil().formatInt(this.collection.total),
        moreCountFormatted: this.getNumberUtil().formatInt(moreCount),
        checkboxColumnWidth: checkboxColumnWidth,
        noDataDisabled: noDataDisabled,
        hasStickyBar: hasStickyBar
      };
    }

    /** @inheritDoc */
    init() {
      this.type = this.options.type || this.type;
      this.listLayout = this.options.listLayout || this.listLayout;
      this.layoutName = this.options.layoutName || this.layoutName || this.type;
      this.layoutScope = this.options.layoutScope || this.layoutScope;
      this.layoutAclDisabled = this.options.layoutAclDisabled || this.layoutAclDisabled;
      this.headerDisabled = this.options.headerDisabled || this.headerDisabled;
      this.noDataDisabled = this.options.noDataDisabled || this.noDataDisabled;
      if (!this.headerDisabled) {
        this.header = _.isUndefined(this.options.header) ? this.header : this.options.header;
      } else {
        this.header = false;
      }
      this.pagination = this.options.pagination == null ? this.pagination : this.options.pagination;
      if (this.paginationDisabled) {
        this.pagination = false;
      }
      if (this.options.columnResize !== undefined) {
        this.columnResize = this.options.columnResize;
      }
      this.checkboxes = _.isUndefined(this.options.checkboxes) ? this.checkboxes : this.options.checkboxes;
      this.selectable = _.isUndefined(this.options.selectable) ? this.selectable : this.options.selectable;
      this.checkboxesDisabled = this.options.checkboxes === false;
      this.rowActionsView = _.isUndefined(this.options.rowActionsView) ? this.rowActionsView : this.options.rowActionsView;
      this.showMore = _.isUndefined(this.options.showMore) ? this.showMore : this.options.showMore;
      this.massActionsDisabled = this.options.massActionsDisabled || this.massActionsDisabled;
      this.portalLayoutDisabled = this.options.portalLayoutDisabled || this.portalLayoutDisabled;
      if (this.massActionsDisabled && !this.selectable) {
        this.checkboxes = false;
      }
      this.rowActionsDisabled = this.options.rowActionsDisabled || this.rowActionsDisabled;
      this.dropdownItemList = Espo.Utils.cloneDeep(this.options.dropdownItemList || this.dropdownItemList);
      if ('buttonsDisabled' in this.options) {
        this.buttonsDisabled = this.options.buttonsDisabled;
      }
      if ('checkAllResultDisabled' in this.options) {
        this.checkAllResultDisabled = this.options.checkAllResultDisabled;
      }
      this.rootData = this.options.rootData || {};
      this._fontSizeFactor = this.getThemeManager().getFontSizeFactor();
    }

    /**
     * Get a record entity type (scope).
     *
     * @param {string} id A record ID.
     * @return {string}
     */
    getModelScope(id) {
      return this.scope;
    }

    /**
     * Select all results.
     */
    selectAllResult() {
      this.allResultIsChecked = true;
      this.hideActions();
      this.$el.find('input.record-checkbox').prop('checked', true).attr('disabled', 'disabled');
      this.$selectAllCheckbox.prop('checked', true);
      this.massActionList.forEach(item => {
        if (!this.checkAllResultMassActionList.includes(item)) {
          this.$el.find(`div.list-buttons-container .actions-menu li a.mass-action[data-action="${item}"]`).parent().addClass('hidden');
        }
      });
      if (this.checkAllResultMassActionList.length) {
        this.showActions();
      }
      this.$el.find('.list > table tbody tr').removeClass('active');
      this.trigger('select-all-results');
    }

    /**
     * Unselect all results.
     */
    unselectAllResult() {
      this.allResultIsChecked = false;
      this.$el.find('input.record-checkbox').prop('checked', false).removeAttr('disabled');
      this.$selectAllCheckbox.prop('checked', false);
      this.massActionList.forEach(item => {
        if (!this.checkAllResultMassActionList.includes(item) && !(this.massActionDefs[item] || {}).hidden) {
          this.$el.find(`div.list-buttons-container .actions-menu li a.mass-action[data-action="${item}"]`).parent().removeClass('hidden');
        }
      });
    }

    /** @protected */
    deactivate() {
      if (this.$el) {
        this.$el.find(".pagination a").addClass('disabled');
        this.$el.find("a.sort").addClass('disabled');
      }
    }

    /**
     * Process export.
     *
     * @param {Object<string,*>} [data]
     * @param {string} [url='Export'] An API URL.
     * @param {string[]} [fieldList] A field list.
     */
    export(data, url, fieldList) {
      if (!data) {
        data = {
          entityType: this.entityType
        };
        if (this.allResultIsChecked) {
          data.where = this.collection.getWhere();
          data.searchParams = this.collection.data || null;
          data.searchData = this.collection.data || {}; // for bc;
        } else {
          data.ids = this.checkedList;
        }
      }
      url = url || 'Export';
      const o = {
        scope: this.entityType
      };
      if (fieldList) {
        o.fieldList = fieldList;
      } else {
        const layoutFieldList = [];
        (this.listLayout || []).forEach(item => {
          if (item.name) {
            layoutFieldList.push(item.name);
          }
        });
        o.fieldList = layoutFieldList;
      }
      const helper = new _export.default(this);
      const idle = this.allResultIsChecked && helper.checkIsIdle(this.collection.total);
      const proceedDownload = attachmentId => {
        window.location = `${this.getBasePath()}?entryPoint=download&id=${attachmentId}`;
      };
      this.createView('dialogExport', 'views/export/modals/export', o, view => {
        view.render();
        this.listenToOnce(view, 'proceed', dialogData => {
          if (!dialogData.exportAllFields) {
            data.attributeList = dialogData.attributeList;
            data.fieldList = dialogData.fieldList;
          }
          data.idle = idle;
          data.format = dialogData.format;
          data.params = dialogData.params;
          Espo.Ui.notify(this.translate('pleaseWait', 'messages'));
          Espo.Ajax.postRequest(url, data, {
            timeout: 0
          }).then(/** Object.<string, *> */response => {
            Espo.Ui.notify(false);
            if (response.exportId) {
              helper.process(response.exportId).then(view => {
                this.listenToOnce(view, 'download', id => proceedDownload(id));
              });
              return;
            }
            if (!response.id) {
              throw new Error("No attachment-id.");
            }
            proceedDownload(response.id);
          });
        });
      });
    }

    /**
     * Process a mass-action.
     *
     * @param {string} name An action.
     */
    massAction(name) {
      const defs = this.massActionDefs[name] || {};
      const handler = defs.handler;
      if (handler) {
        const method = defs.actionFunction || 'action' + Espo.Utils.upperCaseFirst(name);
        const data = {
          entityType: this.entityType,
          action: name,
          params: this.getMassActionSelectionPostData()
        };
        Espo.loader.require(handler, Handler => {
          const handler = new Handler(this);
          handler[method].call(handler, data);
        });
        return;
      }
      const bypassConfirmation = defs.bypassConfirmation || false;
      const confirmationMsg = defs.confirmationMessage || 'confirmation';
      const acl = defs.acl;
      const aclScope = defs.aclScope;
      const proceed = () => {
        if (acl || aclScope) {
          if (!this.getAcl().check(aclScope || this.scope, acl)) {
            Espo.Ui.error(this.translate('Access denied'));
            return;
          }
        }
        const idList = [];
        const data = {};
        if (this.allResultIsChecked) {
          data.where = this.collection.getWhere();
          data.searchParams = this.collection.data || {};
          data.selectData = data.searchData; // for bc;
          data.byWhere = true; // for bc
        } else {
          data.idList = idList; // for bc
          data.ids = idList;
        }
        for (const i in this.checkedList) {
          idList.push(this.checkedList[i]);
        }
        data.entityType = this.entityType;
        const waitMessage = defs.waitMessage || 'pleaseWait';
        Espo.Ui.notify(this.translate(waitMessage, 'messages', this.scope));
        const url = defs.url;
        Espo.Ajax.postRequest(url, data).then(/** Object.<string, *> */result => {
          const successMessage = result.successMessage || defs.successMessage || 'done';
          this.collection.fetch().then(() => {
            let message = this.translate(successMessage, 'messages', this.scope);
            if ('count' in result) {
              message = message.replace('{count}', result.count);
            }
            Espo.Ui.success(message);
          });
        });
      };
      if (!bypassConfirmation) {
        this.confirm(this.translate(confirmationMsg, 'messages', this.scope), proceed, this);
      } else {
        proceed.call(this);
      }
    }
    getMassActionSelectionPostData() {
      const data = {};
      if (this.allResultIsChecked) {
        data.where = this.collection.getWhere();
        data.searchParams = this.collection.data || {};
        data.selectData = this.collection.data || {}; // for bc;
        data.byWhere = true; // for bc;
      } else {
        data.ids = [];
        for (const i in this.checkedList) {
          data.ids.push(this.checkedList[i]);
        }
      }
      return data;
    }

    // noinspection JSUnusedGlobalSymbols
    massActionRecalculateFormula() {
      let ids = false;
      const allResultIsChecked = this.allResultIsChecked;
      if (!allResultIsChecked) {
        ids = this.checkedList;
      }
      this.confirm({
        message: this.translate('recalculateFormulaConfirmation', 'messages'),
        confirmText: this.translate('Yes')
      }, () => {
        Espo.Ui.notify(this.translate('pleaseWait', 'messages'));
        const params = this.getMassActionSelectionPostData();
        const helper = new _massAction.default(this);
        const idle = !!params.searchParams && helper.checkIsIdle(this.collection.total);
        Espo.Ajax.postRequest('MassAction', {
          entityType: this.entityType,
          action: 'recalculateFormula',
          params: params,
          idle: idle
        }).then(result => {
          result = result || {};
          const final = () => {
            this.collection.fetch().then(() => {
              Espo.Ui.success(this.translate('Done'));
              if (allResultIsChecked) {
                this.selectAllResult();
                return;
              }
              ids.forEach(id => {
                this.checkRecord(id);
              });
            });
          };
          if (result.id) {
            helper.process(result.id, 'recalculateFormula').then(view => {
              this.listenToOnce(view, 'close:success', () => final());
            });
            return;
          }
          final();
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    massActionRemove() {
      if (!this.getAcl().check(this.entityType, 'delete')) {
        Espo.Ui.error(this.translate('Access denied'));
        return false;
      }
      this.confirm({
        message: this.translate('removeSelectedRecordsConfirmation', 'messages', this.scope),
        confirmText: this.translate('Remove')
      }, () => {
        Espo.Ui.notify(' ... ');
        const helper = new _massAction.default(this);
        const params = this.getMassActionSelectionPostData();
        const idle = !!params.searchParams && helper.checkIsIdle(this.collection.total);
        Espo.Ajax.postRequest('MassAction', {
          entityType: this.entityType,
          action: 'delete',
          params: params,
          idle: idle
        }).then(result => {
          result = result || {};
          const afterAllResult = count => {
            if (!count) {
              Espo.Ui.warning(this.translate('noRecordsRemoved', 'messages'));
              return;
            }
            this.unselectAllResult();
            this.collection.fetch().then(() => {
              const msg = count === 1 ? 'massRemoveResultSingle' : 'massRemoveResult';
              Espo.Ui.success(this.translate(msg, 'messages').replace('{count}', count));
            });
            this.collection.trigger('after:mass-remove');
            Espo.Ui.notify(false);
          };
          if (result.id) {
            helper.process(result.id, 'delete').then(view => {
              this.listenToOnce(view, 'close:success', result => afterAllResult(result.count));
            });
            return;
          }
          const count = result.count;
          if (this.allResultIsChecked) {
            afterAllResult(count);
            return;
          }
          const idsRemoved = result.ids || [];
          if (!count) {
            Espo.Ui.warning(this.translate('noRecordsRemoved', 'messages'));
            return;
          }
          idsRemoved.forEach(id => {
            Espo.Ui.notify(false);
            this.collection.trigger('model-removing', id);
            this.removeRecordFromList(id);
            this.uncheckRecord(id, null, true);
          });
          if (this.$selectAllCheckbox.prop('checked')) {
            this.$selectAllCheckbox.prop('checked', false);
            if (this.collection.hasMore()) {
              this.showMoreRecords({
                skipNotify: true
              });
            }
          }
          this.collection.trigger('after:mass-remove');
          const showSuccess = () => {
            const msgKey = count === 1 ? 'massRemoveResultSingle' : 'massRemoveResult';
            const msg = this.translate(msgKey, 'messages').replace('{count}', count);
            Espo.Ui.success(msg);
          };
          showSuccess();
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    massActionPrintPdf() {
      const maxCount = this.getConfig().get('massPrintPdfMaxCount');
      if (maxCount) {
        if (this.checkedList.length > maxCount) {
          const msg = this.translate('massPrintPdfMaxCountError', 'messages').replace('{maxCount}', maxCount.toString());
          Espo.Ui.error(msg);
          return;
        }
      }
      const idList = [];
      for (const i in this.checkedList) {
        idList.push(this.checkedList[i]);
      }
      this.createView('pdfTemplate', 'views/modals/select-template', {
        entityType: this.entityType
      }, view => {
        view.render();
        this.listenToOnce(view, 'select', templateModel => {
          this.clearView('pdfTemplate');
          Espo.Ui.notify(' ... ');
          Espo.Ajax.postRequest('Pdf/action/massPrint', {
            idList: idList,
            entityType: this.entityType,
            templateId: templateModel.id
          }, {
            timeout: 0
          }).then(result => {
            Espo.Ui.notify(false);
            window.open('?entryPoint=download&id=' + result.id, '_blank');
          });
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    massActionFollow() {
      const count = this.checkedList.length;
      const confirmMsg = this.translate('confirmMassFollow', 'messages').replace('{count}', count.toString());
      this.confirm({
        message: confirmMsg,
        confirmText: this.translate('Follow')
      }, () => {
        Espo.Ui.notify(this.translate('pleaseWait', 'messages'));
        Espo.Ajax.postRequest('MassAction', {
          action: 'follow',
          entityType: this.entityType,
          params: this.getMassActionSelectionPostData()
        }).then(result => {
          const resultCount = result.count || 0;
          let msg = 'massFollowResult';
          if (resultCount) {
            if (resultCount === 1) {
              msg += 'Single';
            }
            Espo.Ui.success(this.translate(msg, 'messages').replace('{count}', resultCount.toString()));
            return;
          }
          Espo.Ui.warning(this.translate('massFollowZeroResult', 'messages'));
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    massActionUnfollow() {
      const count = this.checkedList.length;
      const confirmMsg = this.translate('confirmMassUnfollow', 'messages').replace('{count}', count.toString());
      this.confirm({
        message: confirmMsg,
        confirmText: this.translate('Yes')
      }, () => {
        Espo.Ui.notify(this.translate('pleaseWait', 'messages'));
        const params = this.getMassActionSelectionPostData();
        const helper = new _massAction.default(this);
        const idle = !!params.searchParams && helper.checkIsIdle(this.collection.total);
        Espo.Ajax.postRequest('MassAction', {
          action: 'unfollow',
          entityType: this.entityType,
          params: params,
          idle: idle
        }).then(result => {
          const final = count => {
            let msg = 'massUnfollowResult';
            if (!count) {
              Espo.Ui.warning(this.translate('massUnfollowZeroResult', 'messages'));
            }
            if (count === 1) {
              msg += 'Single';
            }
            Espo.Ui.success(this.translate(msg, 'messages').replace('{count}', count));
          };
          if (result.id) {
            helper.process(result.id, 'unfollow').then(view => {
              this.listenToOnce(view, 'close:success', result => final(result.count));
            });
            return;
          }
          final(result.count || 0);
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    massActionMerge() {
      if (!this.getAcl().check(this.entityType, 'edit')) {
        Espo.Ui.error(this.translate('Access denied'));
        return false;
      }
      if (this.checkedList.length < 2) {
        Espo.Ui.error(this.translate('select2OrMoreRecords', 'messages'));
        return;
      }
      if (this.checkedList.length > 4) {
        const msg = this.translate('selectNotMoreThanNumberRecords', 'messages').replace('{number}', '4');
        Espo.Ui.error(msg);
        return;
      }
      this.checkedList.sort();
      const url = '#' + this.entityType + '/merge/ids=' + this.checkedList.join(',');
      this.getRouter().navigate(url, {
        trigger: false
      });
      this.getRouter().dispatch(this.entityType, 'merge', {
        ids: this.checkedList.join(','),
        collection: this.collection
      });
    }

    // noinspection JSUnusedGlobalSymbols
    async massActionMassUpdate() {
      if (!this.getAcl().check(this.entityType, 'edit')) {
        Espo.Ui.error(this.translate('Access denied'));
        return false;
      }
      Espo.Ui.notify(' ... ');
      let ids = false;
      const allResultIsChecked = this.allResultIsChecked;
      if (!allResultIsChecked) {
        ids = this.checkedList;
      }
      const viewName = this.getMetadata().get(['clientDefs', this.entityType, 'modalViews', 'massUpdate']) || 'views/modals/mass-update';

      // noinspection JSValidateTypes
      /** @type {import('views/modals/mass-update').default} view */
      const view = await this.createView('massUpdate', viewName, {
        scope: this.scope,
        entityType: this.entityType,
        ids: ids,
        where: this.collection.getWhere(),
        searchParams: this.collection.data,
        byWhere: this.allResultIsChecked,
        totalCount: this.collection.total
      });
      this.listenToOnce(view, 'after:update', async o => {
        if (o.idle) {
          this.listenToOnce(view, 'close', async () => {
            await this.collection.fetch();
            if (allResultIsChecked) {
              this.selectAllResult();
              return;
            }
            ids.forEach(id => this.checkRecord(id));
          });
          return;
        }
        view.close();
        const count = o.count;
        await this.collection.fetch();
        if (count) {
          const msgKey = count === 1 ? 'massUpdateResultSingle' : 'massUpdateResult';
          const message = this.translate(msgKey, 'messages').replace('{count}', count);
          Espo.Ui.success(message);
        } else {
          Espo.Ui.warning(this.translate('noRecordsUpdated', 'messages'));
        }
        if (allResultIsChecked) {
          this.selectAllResult();
          return;
        }
        ids.forEach(id => this.checkRecord(id));
      });
      Espo.Ui.notify();
      await view.render();
    }

    // noinspection JSUnusedGlobalSymbols
    massActionExport() {
      if (this.getConfig().get('exportDisabled') && !this.getUser().isAdmin()) {
        return;
      }
      this.export();
    }

    // noinspection JSUnusedGlobalSymbols
    massActionUnlink() {
      this.confirm({
        message: this.translate('unlinkSelectedRecordsConfirmation', 'messages'),
        confirmText: this.translate('Unlink')
      }, () => {
        Espo.Ui.notify(' ... ');
        Espo.Ajax.deleteRequest(this.collection.url, {
          ids: this.checkedList
        }).then(() => {
          Espo.Ui.success(this.translate('Unlinked'));
          this.collection.fetch();
          this.model.trigger('after:unrelate');
        });
      });
    }

    // noinspection JSUnusedGlobalSymbols
    massActionConvertCurrency() {
      let ids = false;
      const allResultIsChecked = this.allResultIsChecked;
      if (!allResultIsChecked) {
        ids = this.checkedList;
      }
      this.createView('modalConvertCurrency', 'views/modals/mass-convert-currency', {
        entityType: this.entityType,
        ids: ids,
        where: this.collection.getWhere(),
        searchParams: this.collection.data,
        byWhere: this.allResultIsChecked,
        totalCount: this.collection.total
      }, view => {
        view.render();
        this.listenToOnce(view, 'after:update', o => {
          if (o.idle) {
            this.listenToOnce(view, 'close', () => {
              this.collection.fetch().then(() => {
                if (allResultIsChecked) {
                  this.selectAllResult();
                  return;
                }
                ids.forEach(id => {
                  this.checkRecord(id);
                });
              });
            });
            return;
          }
          const count = o.count;
          this.collection.fetch().then(() => {
            if (count) {
              let msg = 'massUpdateResult';
              if (count === 1) {
                msg = 'massUpdateResultSingle';
              }
              Espo.Ui.success(this.translate(msg, 'messages').replace('{count}', count));
            } else {
              Espo.Ui.warning(this.translate('noRecordsUpdated', 'messages'));
            }
            if (allResultIsChecked) {
              this.selectAllResult();
              return;
            }
            ids.forEach(id => {
              this.checkRecord(id);
            });
          });
        });
      });
    }

    /**
     * Add a mass action.
     *
     * @protected
     * @param {string|module:views/record/list~massActionItem} item An action.
     * @param {boolean} [allResult] To make available for all-result.
     * @param {boolean} [toBeginning] Add to the beginning of the list.
     */
    addMassAction(item, allResult, toBeginning) {
      if (typeof item !== 'string') {
        const name = item.name;
        this.massActionDefs[name] = {
          ...this.massActionDefs[name],
          ...item
        };
        item = name;
      }
      toBeginning ? this.massActionList.unshift(item) : this.massActionList.push(item);
      if (allResult && this.collection.url === this.entityType) {
        toBeginning ? this.checkAllResultMassActionList.unshift(item) : this.checkAllResultMassActionList.push(item);
      }
      if (!this.checkboxesDisabled) {
        this.checkboxes = true;
      }
    }

    /**
     * Remove a mass action.
     *
     * @param {string} item An action.
     */
    removeMassAction(item) {
      let index = this.massActionList.indexOf(item);
      if (~index) {
        this.massActionList.splice(index, 1);
      }
      index = this.checkAllResultMassActionList.indexOf(item);
      if (~index) {
        this.checkAllResultMassActionList.splice(index, 1);
      }
    }

    /**
     * Remove an all-result mass action.
     *
     * @param {string} item An action.
     */
    removeAllResultMassAction(item) {
      const index = this.checkAllResultMassActionList.indexOf(item);
      if (~index) {
        this.checkAllResultMassActionList.splice(index, 1);
      }
    }

    /** @inheritDoc */
    setup() {
      this.addHandler('click', 'a.link', (/** KeyboardEvent */e, target) => {
        if (e.ctrlKey || e.metaKey || e.shiftKey) {
          return;
        }
        e.stopPropagation();
        if (!this.scope || this.selectable) {
          return;
        }
        e.preventDefault();
        this.processLinkClick(target.dataset.id);
      });
      if (typeof this.collection === 'undefined') {
        throw new Error('Collection has not been injected into views/record/list view.');
      }
      this.layoutLoadCallbackList = [];
      this.entityType = this.collection.entityType || null;
      this.scope = this.options.scope || this.entityType;
      this.massActionList = Espo.Utils.clone(this.massActionList);
      this.checkAllResultMassActionList = Espo.Utils.clone(this.checkAllResultMassActionList);
      this.buttonList = Espo.Utils.clone(this.buttonList);
      this.mandatorySelectAttributeList = Espo.Utils.clone(this.options.mandatorySelectAttributeList || this.mandatorySelectAttributeList || []);
      this.forceStickyBar = this.options.forceStickyBar || this.forceStickyBar;
      this.editDisabled = this.options.editDisabled || this.editDisabled || this.getMetadata().get(['clientDefs', this.scope, 'editDisabled']);
      this.removeDisabled = this.options.removeDisabled || this.removeDisabled || this.getMetadata().get(['clientDefs', this.scope, 'removeDisabled']);
      this.setupMassActions();
      if (this.selectable) {
        this.events['click .list a.link'] = e => {
          e.preventDefault();
          const id = $(e.currentTarget).attr('data-id');
          if (id) {
            this.selectModel(id);
          }
          e.stopPropagation();
        };
      }
      if ('showCount' in this.options) {
        this.showCount = this.options.showCount;
      }
      this.displayTotalCount = this.showCount && this.getConfig().get('displayListViewRecordCount');
      if ('displayTotalCount' in this.options) {
        this.displayTotalCount = this.options.displayTotalCount;
      }
      this.forceDisplayTopBar = this.options.forceDisplayTopBar || this.forceDisplayTopBar;
      if (!this.massActionList.length && !this.selectable) {
        this.checkboxes = false;
      }

      /**
       * @private
       * @type {boolean}
       */
      this.hasStars = this.getMetadata().get(`scopes.${this.entityType}.stars`) || false;
      if (this.getUser().isPortal() && !this.portalLayoutDisabled && this.getMetadata().get(['clientDefs', this.scope, 'additionalLayouts', this.layoutName + 'Portal'])) {
        this.layoutName += 'Portal';
      }
      this.setupRowActionDefs();
      this.setupSettings();
      this.wait(this.getHelper().processSetupHandlers(this, this.setupHandlerType));
      this.listenTo(this.collection, 'sync', (c, r, options) => {
        this._renderEmpty = false;
        if (this.hasView('modal') && this.getView('modal').isRendered()) {
          return;
        }
        options = options || {};
        if (options.previousDataList) {
          const currentDataList = this.collection.models.map(model => {
            return Espo.Utils.cloneDeep(model.attributes);
          });
          if (_.isEqual(currentDataList, options.previousDataList) && options.previousTotal === this.collection.total) {
            return;
          }
        }
        if (this.noRebuild) {
          this.noRebuild = null;
          return;
        }
        if (options.noRebuild) {
          this.noRebuild = null;
          return;
        }
        this.checkedList = [];
        this.allResultIsChecked = false;
        this.buildRows(() => {
          this.render();
        });
      });
      this.checkedList = [];
      if (!this.options.skipBuildRows) {
        this.buildRows();
      }
      if (this.pagination) {
        this.createView('pagination', 'views/record/list-pagination', {
          collection: this.collection,
          displayTotalCount: this.displayTotalCount,
          recordView: this
        });
        this.createView('paginationSticky', 'views/record/list-pagination', {
          collection: this.collection,
          displayTotalCount: this.displayTotalCount,
          recordView: this
        });
        this.on('request-page', /** string */page => {
          if (this.collection.isBeingFetched()) {
            return;
          }
          if (page === 'next' && !this.collection.hasNextPage()) {
            return;
          }
          if (page === 'previous' && !this.collection.hasPreviousPage()) {
            return;
          }
          this.goToPage(page);
        });
      }
      this._renderEmpty = this.options.skipBuildRows;
      if (this.columnResize && this._listSettingsHelper) {
        this._columnResizeHelper = new _columnResize.default(this, this._listSettingsHelper);
      }
    }

    /**
     * @private
     * @param {string} id
     */
    processLinkClick(id) {
      const model = this.collection.get(id);
      const scope = this.getModelScope(id);
      const options = {
        id: id,
        model: model
      };
      if (this.options.keepCurrentRootUrl) {
        options.rootUrl = this.getRouter().getCurrentUrl();
      }
      options.rootData = this.rootData;
      this.getRouter().navigate(`#${scope}/view/${id}`, {
        trigger: false
      });
      this.getRouter().dispatch(scope, 'view', options);
    }

    /**
     * @private
     * @param {string} id
     */
    selectModel(id) {
      const model = this.collection.get(id);
      if (this.checkboxes) {
        this.trigger('select', [model]);
        return;
      }
      this.trigger('select', model);
    }
    onRemove() {
      this.destroyStickyBar();
    }
    afterRender() {
      this.destroyStickyBar();
      this.$selectAllCheckbox = this.$el.find('input.select-all');
      if (this.allResultIsChecked) {
        this.selectAllResult();
      } else if (this.checkedList.length) {
        this.checkedList.forEach(id => {
          this.checkRecord(id);
        });
      }
      if (this.pagination && this.$el.find('.list-buttons-container').length) {
        this.initStickyBar();
      }
      if (this._disabledCheckboxes) {
        this.disableCheckboxes();
      }
    }

    /** @private */
    destroyStickyBar() {
      if (this._stickyBarHelper) {
        this._stickyBarHelper.destroy();
      }
      this._stickyBarHelper = null;
    }

    /** @private */
    setupMassActions() {
      if (this.massActionsDisabled) {
        this.massActionList = [];
        this.checkAllResultMassActionList = [];
        this.massActionDefs = {};
        return;
      }
      if (!this.getAcl().checkScope(this.entityType, 'delete')) {
        this.removeMassAction('remove');
        this.removeMassAction('merge');
      }
      if (this.removeDisabled || this.getMetadata().get(['clientDefs', this.scope, 'massRemoveDisabled'])) {
        this.removeMassAction('remove');
      }
      if (!this.getAcl().checkScope(this.entityType, 'edit')) {
        this.removeMassAction('massUpdate');
        this.removeMassAction('merge');
      }
      if (this.getMetadata().get(['clientDefs', this.scope, 'mergeDisabled']) || this.mergeDisabled) {
        this.removeMassAction('merge');
      }
      this.massActionDefs = {
        remove: {
          groupIndex: 0
        },
        merge: {
          groupIndex: 0
        },
        massUpdate: {
          groupIndex: 0
        },
        export: {
          groupIndex: 2
        },
        follow: {
          groupIndex: 4
        },
        unfollow: {
          groupIndex: 4
        },
        convertCurrency: {
          groupIndex: 6
        },
        printToPdf: {
          groupIndex: 8
        },
        ...(this.getMetadata().get(['clientDefs', 'Global', 'massActionDefs']) || {}),
        ...(this.getMetadata().get(['clientDefs', this.scope, 'massActionDefs']) || {})
      };
      const metadataMassActionList = [...(this.getMetadata().get(['clientDefs', 'Global', 'massActionList']) || []), ...(this.getMetadata().get(['clientDefs', this.scope, 'massActionList']) || [])];
      const metadataCheckAllMassActionList = [...(this.getMetadata().get(['clientDefs', 'Global', 'checkAllResultMassActionList']) || []), ...(this.getMetadata().get(['clientDefs', this.scope, 'checkAllResultMassActionList']) || [])];
      metadataMassActionList.forEach(item => {
        const defs = /** @type {Espo.Utils~ActionAccessDefs & Espo.Utils~ActionAvailabilityDefs} */
        this.massActionDefs[item] || {};
        if (!Espo.Utils.checkActionAvailability(this.getHelper(), defs) || !Espo.Utils.checkActionAccess(this.getAcl(), this.entityType, defs)) {
          return;
        }
        this.massActionList.push(item);
      });
      this.checkAllResultMassActionList = this.checkAllResultMassActionList.filter(item => this.massActionList.includes(item));
      metadataCheckAllMassActionList.forEach(item => {
        if (this.collection.url !== this.entityType) {
          return;
        }
        if (~this.massActionList.indexOf(item)) {
          const defs = /** @type {Espo.Utils~ActionAccessDefs & Espo.Utils~ActionAvailabilityDefs} */
          this.massActionDefs[item] || {};
          if (!Espo.Utils.checkActionAvailability(this.getHelper(), defs) || !Espo.Utils.checkActionAccess(this.getAcl(), this.entityType, defs)) {
            return;
          }
          this.checkAllResultMassActionList.push(item);
        }
      });
      metadataMassActionList.concat(metadataCheckAllMassActionList).forEach(action => {
        const defs = this.massActionDefs[action] || {};
        if (!defs.initFunction || !defs.handler) {
          return;
        }
        const viewObject = this;
        this.wait(new Promise(resolve => {
          Espo.loader.require(defs.handler, Handler => {
            const handler = new Handler(viewObject);
            handler[defs.initFunction].call(handler);
            resolve();
          });
        }));
      });
      if (this.getConfig().get('exportDisabled') && !this.getUser().isAdmin() || this.getAcl().getPermissionLevel('exportPermission') === 'no' || this.getMetadata().get(['clientDefs', this.scope, 'exportDisabled']) || this.exportDisabled) {
        this.removeMassAction('export');
      }
      if (this.getAcl().getPermissionLevel('massUpdatePermission') !== 'yes' || this.editDisabled || this.massUpdateDisabled || this.getMetadata().get(['clientDefs', this.scope, 'massUpdateDisabled'])) {
        this.removeMassAction('massUpdate');
      }
      if (!this.massFollowDisabled && this.getMetadata().get(['scopes', this.entityType, 'stream']) && this.getAcl().check(this.entityType, 'stream') || this.getMetadata().get(['clientDefs', this.scope, 'massFollowDisabled'])) {
        this.addMassAction('follow');
        this.addMassAction('unfollow', true);
      }
      if (!this.massPrintPdfDisabled && (this.getHelper().getAppParam('templateEntityTypeList') || []).includes(this.entityType)) {
        this.addMassAction('printPdf');
      }
      if (this.options.unlinkMassAction && this.collection) {
        this.addMassAction('unlink', false, true);
      }
      if (!this.massConvertCurrencyDisabled && !this.getMetadata().get(['clientDefs', this.scope, 'convertCurrencyDisabled']) && this.getConfig().get('currencyList').length > 1 && this.getAcl().checkScope(this.scope, 'edit') && this.getAcl().getPermissionLevel('massUpdatePermission') === 'yes') {
        const currencyFieldList = this.getFieldManager().getEntityTypeFieldList(this.entityType, {
          type: 'currency',
          acl: 'edit'
        });
        if (currencyFieldList.length) {
          this.addMassAction('convertCurrency', true);
        }
      }
      this.setupMassActionItems();
      if (this.getUser().isAdmin()) {
        if (this.getMetadata().get(['formula', this.entityType, 'beforeSaveCustomScript'])) {
          this.addMassAction('recalculateFormula', true);
        }
      }
      if (this.collection.url !== this.entityType) {
        Espo.Utils.clone(this.checkAllResultMassActionList).forEach(item => {
          this.removeAllResultMassAction(item);
        });
      }
      if (this.forcedCheckAllResultMassActionList) {
        this.checkAllResultMassActionList = Espo.Utils.clone(this.forcedCheckAllResultMassActionList);
      }
      if (this.getAcl().getPermissionLevel('massUpdatePermission') !== 'yes') {
        this.removeAllResultMassAction('remove');
      }
      Espo.Utils.clone(this.massActionList).forEach(item => {
        const propName = 'massAction' + Espo.Utils.upperCaseFirst(item) + 'Disabled';
        if (this[propName] || this.options[propName]) {
          this.removeMassAction(item);
        }
      });
    }

    /** @protected */
    setupMassActionItems() {}

    /**
     * @param {module:views/record/list~columnDefs[]} listLayout
     * @return {module:views/record/list~columnDefs[]}
     * @protected
     */
    filterListLayout(listLayout) {
      if (this._cachedFilteredListLayout) {
        return this._cachedFilteredListLayout;
      }
      let forbiddenFieldList = this._cachedScopeForbiddenFieldList = this._cachedScopeForbiddenFieldList || this.getAcl().getScopeForbiddenFieldList(this.entityType, 'read');
      if (this.layoutAclDisabled) {
        forbiddenFieldList = [];
      }

      /*if (!forbiddenFieldList.length) {
          this._cachedFilteredListLayout = listLayout;
            return this._cachedFilteredListLayout;
      }*/

      const filteredListLayout = Espo.Utils.cloneDeep(listLayout);
      const deleteIndexes = [];
      for (const [i, item] of listLayout.entries()) {
        if (item.name && forbiddenFieldList.includes(item.name)) {
          item.customLabel = '';
          item.notSortable = true;
          deleteIndexes.push(i);
        }
      }
      deleteIndexes.reverse().forEach(index => filteredListLayout.splice(index, 1));

      /** @type {Record<Record>} */
      const fieldDefs = this.getMetadata().get(`entityDefs.${this.entityType}.fields`) || {};
      filteredListLayout.forEach(item => {
        if (!item.name || !fieldDefs[item.name]) {
          return;
        }
        if (fieldDefs[item.name].orderDisabled) {
          item.notSortable = true;
        }
      });
      this._cachedFilteredListLayout = filteredListLayout;
      return this._cachedFilteredListLayout;
    }

    /**
     * @protected
     * @param {function(module:views/record/list~columnDefs[]): void} callback A callback.
     */
    _loadListLayout(callback) {
      this.layoutLoadCallbackList.push(callback);
      if (this.layoutIsBeingLoaded) {
        return;
      }
      this.layoutIsBeingLoaded = true;
      const layoutName = this.layoutName;
      const layoutScope = this.layoutScope || this.collection.entityType;
      this.getHelper().layoutManager.get(layoutScope, layoutName, listLayout => {
        const filteredListLayout = this.filterListLayout(listLayout);
        this.layoutLoadCallbackList.forEach(callbackItem => {
          callbackItem(filteredListLayout);
          this.layoutLoadCallbackList = [];
          this.layoutIsBeingLoaded = false;
        });
      });
    }

    /**
     * Get a select-attribute list.
     *
     * @param {function(string[]):void} callback A callback.
     */
    getSelectAttributeList(callback) {
      if (this.scope === null) {
        callback(null);
        return;
      }
      if (this.listLayout) {
        const attributeList = this.fetchAttributeListFromLayout();
        callback(attributeList);
        return;
      }
      this._loadListLayout(listLayout => {
        this.listLayout = listLayout;
        let attributeList = this.fetchAttributeListFromLayout();
        if (this.mandatorySelectAttributeList) {
          attributeList = attributeList.concat(this.mandatorySelectAttributeList);
        }
        callback(attributeList);
      });
    }

    /**
     * @protected
     * @return {string[]}
     */
    fetchAttributeListFromLayout() {
      const selectProvider = new _selectProvider.default(this.getHelper().layoutManager, this.getHelper().metadata, this.getHelper().fieldManager);
      return selectProvider.getFromLayout(this.entityType, this.listLayout);
    }

    /**
     * @private
     */
    _hasColumnResize() {
      return this._listSettingsHelper ? this._listSettingsHelper.getColumnResize() : false;
    }

    /** @protected */
    _getHeaderDefs() {
      const defs = [];
      const resize = this._hasColumnResize();
      const widthMap = this._listSettingsHelper ? this._listSettingsHelper.getColumnWidthMap() : {};

      // noinspection JSIncompatibleTypesComparison
      if (!this.listLayout || !Array.isArray(this.listLayout)) {
        return [];
      }
      let emptyWidthMet = false;
      const visibleColumns = this.listLayout.filter(it => {
        if (!this._listSettingsHelper && it.hidden) {
          return false;
        }
        if (!this._listSettingsHelper) {
          return true;
        }
        if (it.name && this._listSettingsHelper.isColumnHidden(it.name, it.hidden)) {
          return false;
        }
        return true;
      });
      for (const col of visibleColumns) {
        let width = false;
        const itemName = col.name;
        if (itemName && itemName in widthMap) {
          const widthItem = widthMap[itemName];
          width = widthItem.value + widthItem.unit;
        } else if ('width' in col && col.width !== null) {
          width = col.width + '%';
        } else if ('widthPx' in col) {
          width = (col.widthPx * this._fontSizeFactor).toString() + 'px';
        } else {
          emptyWidthMet = true;
        }
        const label = col.label || itemName;
        const item = {
          name: itemName,
          isSortable: !(col.notSortable || false),
          width: width,
          align: 'align' in col ? col.align : false,
          resizable: resize && width && visibleColumns.length > 1,
          resizeOnRight: resize && width && !emptyWidthMet
        };
        if ('customLabel' in col) {
          item.customLabel = col.customLabel;
          item.hasCustomLabel = true;
          item.label = item.customLabel;
        } else {
          item.label = this.translate(label, 'fields', this.collection.entityType);
        }
        if (col.noLabel) {
          item.label = null;
        }
        if (item.isSortable) {
          item.isSorted = this.collection.orderBy === itemName;
          if (item.isSorted) {
            item.isDesc = this.collection.order === 'desc';
          }
        }
        defs.push(item);
      }
      const isCustomSorted = this.collection.orderBy !== this.collection.defaultOrderBy || this.collection.order !== this.collection.defaultOrder;
      if (this.rowActionsView && !this.rowActionsDisabled || isCustomSorted) {
        let html = null;
        if (isCustomSorted) {
          html = $('<a>').attr('role', 'button').attr('tabindex', '0').addClass('reset-custom-order').attr('title', this.translate('Reset')).append($('<span>').addClass('fas fa-times fa-sm')).get(0).outerHTML;
        }
        const width = (this._fontSizeFactor * this.rowActionsColumnWidth).toString() + 'px';
        defs.push({
          width: width,
          html: html,
          className: 'action-cell'
        });
      }
      return defs;
    }

    /** @protected */
    _convertLayout(listLayout, model) {
      model = model || this.collection.prepareModel();
      const layout = [];
      if (this.checkboxes) {
        layout.push({
          name: 'r-checkboxField',
          columnName: 'r-checkbox',
          template: 'record/list-checkbox'
        });
      }
      for (const col of listLayout) {
        const type = col.type || model.getFieldType(col.name) || 'base';
        if (!col.name) {
          continue;
        }
        const item = {
          columnName: col.name,
          name: col.name + 'Field',
          view: col.view || model.getFieldParam(col.name, 'view') || this.getFieldManager().getViewName(type),
          options: {
            defs: {
              name: col.name,
              params: col.params || {}
            },
            mode: 'list'
          }
        };
        if (col.width) {
          item.options.defs.width = col.width;
        }
        if (col.widthPx) {
          item.options.defs.widthPx = col.widthPx;
        }
        if (col.link) {
          item.options.mode = 'listLink';
        }
        if (col.align) {
          item.options.defs.align = col.align;
        }
        if (col.options) {
          for (const optionName in col.options) {
            if (typeof item.options[optionName] !== 'undefined') {
              continue;
            }
            item.options[optionName] = col.options[optionName];
          }
        }
        if (col.name && this._listSettingsHelper) {
          if (this._listSettingsHelper.isColumnHidden(col.name, col.hidden)) {
            continue;
          }
        }
        if (!this._listSettingsHelper && col.hidden) {
          continue;
        }
        layout.push(item);
      }
      if (this.rowActionsView && !this.rowActionsDisabled) {
        layout.push(this.getRowActionsDefs());
      }
      return layout;
    }

    /**
     * Select a record.
     *
     * @param {string} id An ID.
     * @param {JQuery} [$target]
     * @param {boolean} [isSilent] Do not trigger the `check` event.
     */
    checkRecord(id, $target, isSilent) {
      if (this._disabledCheckboxes) {
        return;
      }
      if (!this.collection.get(id)) {
        return;
      }
      $target = $target || this.$el.find('.record-checkbox[data-id="' + id + '"]');
      if ($target.length) {
        $target.get(0).checked = true;
        $target.closest('tr').addClass('active');
      }
      const index = this.checkedList.indexOf(id);
      if (index === -1) {
        this.checkedList.push(id);
      }
      this.handleAfterCheck(isSilent);
    }

    /**
     * Unselect a record.
     *
     * @param {string} id An ID.
     * @param {JQuery} [$target]
     * @param {boolean} [isSilent] Do not trigger the `check` event.
     */
    uncheckRecord(id, $target, isSilent) {
      $target = $target || this.$el.find('.record-checkbox[data-id="' + id + '"]');
      if ($target.get(0)) {
        $target.get(0).checked = false;
        $target.closest('tr').removeClass('active');
      }
      const index = this.checkedList.indexOf(id);
      if (index !== -1) {
        this.checkedList.splice(index, 1);
      }
      this.handleAfterCheck(isSilent);
    }

    /**
     * @protected
     * @param {boolean} [isSilent]
     */
    handleAfterCheck(isSilent) {
      if (this.checkedList.length) {
        this.showActions();
      } else {
        this.hideActions();
      }
      if (this.checkedList.length === this.collection.models.length) {
        this.$el.find('.select-all').prop('checked', true);
      } else {
        this.$el.find('.select-all').prop('checked', false);
      }
      if (!isSilent) {
        this.trigger('check');
      }
    }

    /**
     * Get row-actions defs.
     *
     * @return {Object}
     */
    getRowActionsDefs() {
      const options = {
        defs: {
          params: {}
        },
        additionalActionList: this._additionalRowActionList || [],
        scope: this.scope
      };
      if (this.options.rowActionsOptions) {
        for (const item in this.options.rowActionsOptions) {
          options[item] = this.options.rowActionsOptions[item];
        }
      }
      return {
        columnName: 'buttons',
        name: 'buttonsField',
        view: this.rowActionsView,
        options: options
      };
    }

    /**
     * Is all-result is checked.
     *
     * @return {boolean}
     */
    isAllResultChecked() {
      return this.allResultIsChecked;
    }

    /**
     * Get checked record IDs.
     *
     * @return {string[]}
     */
    getCheckedIds() {
      return Espo.Utils.clone(this.checkedList);
    }

    /**
     * Get selected models.
     *
     * @return {module:model[]}
     */
    getSelected() {
      const list = [];
      this.$el.find('input.record-checkbox:checked').each((i, el) => {
        const id = $(el).attr('data-id');
        const model = this.collection.get(id);
        list.push(model);
      });
      return list;
    }

    /** @protected */
    getInternalLayoutForModel(callback, model) {
      const scope = model.entityType;
      if (this._internalLayout === null) {
        this._internalLayout = {};
      }
      if (!(scope in this._internalLayout)) {
        this._internalLayout[scope] = this._convertLayout(this.listLayout[scope], model);
      }
      callback(this._internalLayout[scope]);
    }

    /** @protected */
    getInternalLayout(callback, model) {
      if (this.scope === null && !Array.isArray(this.listLayout)) {
        if (!model) {
          callback(null);
          return;
        }
        this.getInternalLayoutForModel(callback, model);
        return;
      }
      if (this._internalLayout !== null) {
        callback(this._internalLayout);
        return;
      }
      if (this.listLayout !== null) {
        this._internalLayout = this._convertLayout(this.listLayout);
        callback(this._internalLayout);
        return;
      }
      this._loadListLayout(listLayout => {
        this.listLayout = listLayout;
        this._internalLayout = this._convertLayout(listLayout);
        callback(this._internalLayout);
      });
    }

    /**
     * Compose a cell selector for a layout item.
     *
     * @param {module:model} model A model.
     * @param {Record} item An item.
     * @return {string}
     */
    getCellSelector(model, item) {
      return `${this.getSelector()} ${this.getRowSelector(model.id)} .cell[data-name="${item.columnName}"]`;
    }
    prepareInternalLayout(internalLayout, model) {
      internalLayout.forEach(item => {
        item.el = this.getCellSelector(model, item);
      });
    }

    /**
     * Build a row.
     *
     * @param {number} i An index.
     * @param {module:model} model A model.
     * @param {function(module:view):void} [callback] A callback.
     */
    buildRow(i, model, callback) {
      const key = model.id;
      this.rowList.push(key);
      this.getInternalLayout(internalLayout => {
        internalLayout = Espo.Utils.cloneDeep(internalLayout);
        this.prepareInternalLayout(internalLayout, model);
        const acl = {
          edit: this.getAcl().checkModel(model, 'edit') && !this.editDisabled,
          delete: this.getAcl().checkModel(model, 'delete') && !this.removeDisabled
        };
        this.createView(key, 'views/base', {
          model: model,
          acl: acl,
          rowActionHandlers: this._rowActionHandlers || {},
          selector: this.getRowSelector(key),
          optionsToPass: ['acl', 'rowActionHandlers'],
          layoutDefs: {
            type: this._internalLayoutType,
            layout: internalLayout
          },
          setViewBeforeCallback: this.options.skipBuildRows && !this.isRendered()
        }, callback);
      }, model);
    }

    /**
     * Build rows.
     *
     * @param {function():void} [callback] A callback.
     */
    buildRows(callback) {
      this.checkedList = [];

      /**
       * @internal
       * @type {string[]}
       * @private
       */
      this.rowList = [];
      if (this.collection.length <= 0) {
        if (typeof callback === 'function') {
          callback();
          this.trigger('after:build-rows');
        }
        return;
      }
      this.wait(true);
      const modelList = this.collection.models;
      let counter = 0;
      modelList.forEach((model, i) => {
        this.buildRow(i, model, () => {
          counter++;
          if (counter !== modelList.length) {
            return;
          }
          if (typeof callback === 'function') {
            callback();
          }
          this.wait(false);
          this.trigger('after:build-rows');
        });
      });
    }

    /**
     * Show more records.
     *
     * @param {{skipNotify?: boolean}} [options]
     * @param {module:collection} [collection]
     * @param {JQuery} [$list]
     * @param {JQuery} [$showMore]
     * @param {function(): void} [callback] A callback.
     */
    showMoreRecords(options, collection, $list, $showMore, callback) {
      collection = collection || this.collection;
      $showMore = $showMore || this.$el.find('.show-more');
      $list = $list || this.$el.find(this.listContainerEl);
      options = options || {};
      const $container = this.$el.find('.list');
      $showMore.children('a').addClass('disabled');
      if (!options.skipNotify) {
        Espo.Ui.notify(' ... ');
      }
      const lengthBefore = collection.length;
      const final = () => {
        $showMore.parent().append($showMore);
        if (collection.hasMore()) {
          const moreCount = collection.total - collection.offset - collection.length - collection.lengthCorrection;
          this.$el.find('.more-count').text(this.getNumberUtil().formatInt(moreCount));
          $showMore.removeClass('hidden');
          $container.addClass('has-show-more');
        } else {
          $showMore.remove();
          $container.removeClass('has-show-more');
        }
        $showMore.children('a').removeClass('disabled');
        if (this.allResultIsChecked) {
          this.$el.find('input.record-checkbox').attr('disabled', 'disabled').prop('checked', true);
        }
        if (!options.skipNotify) {
          Espo.Ui.notify(false);
        }
        if (callback) {
          callback.call(this);
        }
        this.trigger('after:show-more', lengthBefore);
      };
      const initialCount = collection.length;
      const success = () => {
        if (!options.skipNotify) {
          Espo.Ui.notify(false);
        }
        $showMore.addClass('hidden');
        $container.removeClass('has-show-more');
        const rowCount = collection.length - initialCount;
        let rowsReady = 0;
        if (collection.length <= initialCount) {
          final();
        }
        for (let i = initialCount; i < collection.length; i++) {
          const model = collection.at(i);
          this.buildRow(i, model, view => {
            const model = view.model;
            const $existingRow = this.getDomRowItem(model.id);
            if ($existingRow && $existingRow.length) {
              $existingRow.remove();
            }
            $list.append(this.getRowContainerHtml(model.id));
            view.render().then(() => {
              rowsReady++;
              if (rowsReady === rowCount) {
                final();
              }
            });
          });
        }
        this.noRebuild = true;
      };
      const onUpdate = (c, /** Record */o) => {
        if (o.changes.merged.length) {
          collection.lengthCorrection += o.changes.merged.length;
        }
      };
      this.listenToOnce(collection, 'update', onUpdate);

      // If using promise callback, then need to pass `noRebuild: true`.
      collection.fetch({
        success: success,
        remove: false,
        more: true
      }).catch(() => this.stopListening(collection, 'update', onUpdate));
    }
    getDomRowItem(id) {
      return null;
    }

    /**
     * Compose a row-container HTML.
     *
     * @param {string} id A record ID.
     * @return {string} HTML.
     */
    getRowContainerHtml(id) {
      return $('<tr>').attr('data-id', id).addClass('list-row').get(0).outerHTML;
    }
    actionQuickView(data) {
      data = data || {};
      const id = data.id;
      if (!id) {
        console.error("No id.");
        return;
      }
      let model = null;
      if (this.collection) {
        model = this.collection.get(id);
      }
      let scope = data.scope;
      if (!scope && model) {
        scope = model.entityType;
      }
      if (!scope) {
        scope = this.scope;
      }
      if (!scope) {
        console.error("No scope.");
        return;
      }
      if (this.quickDetailDisabled) {
        this.getRouter().navigate('#' + scope + '/view/' + id, {
          trigger: true
        });
        return;
      }
      const helper = new _recordModal.default();
      helper.showDetail(this, {
        id: id,
        scope: scope,
        model: model,
        rootUrl: this.options.keepCurrentRootUrl ? this.getRouter().getCurrentUrl() : null,
        editDisabled: this.quickEditDisabled
      }).then(view => {
        if (!model) {
          return;
        }
        this.listenTo(view, 'after:save', model => {
          this.trigger('after:save', model);
        });
        this.listenTo(view, 'after:destroy', model => this.removeRecordFromList(model.id));
      });
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * @param {Object.<string, *>} data
     */
    actionQuickEdit(data) {
      data = data || {};
      const id = data.id;
      if (!id) {
        console.error("No id.");
        return;
      }
      let model = null;
      if (this.collection) {
        model = this.collection.get(id);
      }
      let scope = data.scope;
      if (!scope && model) {
        scope = model.entityType;
      }
      if (!scope) {
        scope = this.scope;
      }
      if (!scope) {
        console.error("No scope.");
        return;
      }
      const viewName = this.getMetadata().get(['clientDefs', scope, 'modalViews', 'edit']) || 'views/modals/edit';
      if (!this.quickEditDisabled) {
        Espo.Ui.notify(' ... ');
        const options = {
          scope: scope,
          id: id,
          model: model,
          fullFormDisabled: data.noFullForm,
          returnUrl: this.getRouter().getCurrentUrl(),
          returnDispatchParams: {
            controller: scope,
            action: null,
            options: {
              isReturn: true
            }
          }
        };
        if (this.options.keepCurrentRootUrl) {
          options.rootUrl = this.getRouter().getCurrentUrl();
        }
        this.createView('modal', viewName, options, view => {
          view.once('after:render', () => {
            Espo.Ui.notify(false);
          });
          view.render();
          this.listenToOnce(view, 'remove', () => {
            this.clearView('modal');
          });
          this.listenToOnce(view, 'after:save', m => {
            const model = this.collection.get(m.id);
            if (model) {
              model.set(m.getClonedAttributes(), {
                sync: true
              });
            }
            this.trigger('after:save', m);
          });
        });
        return;
      }
      const options = {
        id: id,
        model: this.collection.get(id),
        returnUrl: this.getRouter().getCurrentUrl(),
        returnDispatchParams: {
          controller: scope,
          action: null,
          options: {
            isReturn: true
          }
        }
      };
      if (this.options.keepCurrentRootUrl) {
        options.rootUrl = this.getRouter().getCurrentUrl();
      }
      this.getRouter().navigate('#' + scope + '/edit/' + id, {
        trigger: false
      });
      this.getRouter().dispatch(scope, 'edit', options);
    }

    /**
     * Compose a row selector.
     *
     * @param {string} id A record ID.
     * @return {string}
     */
    getRowSelector(id) {
      return `tr.list-row[data-id="${id}"]`;
    }

    // noinspection JSUnusedGlobalSymbols
    actionQuickRemove(data) {
      data = data || {};
      const id = data.id;
      if (!id) {
        return;
      }
      const model = this.collection.get(id);
      if (!this.getAcl().checkModel(model, 'delete')) {
        Espo.Ui.error(this.translate('Access denied'));
        return;
      }
      this.confirm({
        message: this.translate('removeRecordConfirmation', 'messages', this.scope),
        confirmText: this.translate('Remove')
      }, () => {
        this.collection.trigger('model-removing', id);
        this.collection.remove(model);
        Espo.Ui.notify(' ... ');
        model.destroy({
          wait: true,
          fromList: true
        }).then(() => {
          Espo.Ui.success(this.translate('Removed'));
          this.trigger('after:delete', model);
          this.removeRecordFromList(id);
        }).catch(() => {
          // @todo Revert to the same position.
          this.collection.push(model);
        });
      });
    }

    /**
     * @protected
     * @param {string} id An ID.
     */
    removeRecordFromList(id) {
      if (this.collection.total > 0) {
        this.collection.total--;
        this.collection.trigger('update-total');
      }
      this.collection.remove(id);
      this.$el.find('.total-count-span').text(this.collection.total.toString());
      let index = this.checkedList.indexOf(id);
      if (index !== -1) {
        this.checkedList.splice(index, 1);
      }
      const key = id;
      this.clearView(key);
      index = this.rowList.indexOf(key);
      if (~index) {
        this.rowList.splice(index, 1);
      }
      this.removeRowHtml(id);
    }

    /**
     * @protected
     * @param {string} id An ID.
     */
    removeRowHtml(id) {
      this.$el.find(this.getRowSelector(id)).remove();
      if (this.collection.length === 0 && (this.collection.total === 0 || this.collection.total === -2)) {
        this.reRender();
      }
    }

    /**
     * @public
     * @param {string} id An ID.
     * @return {boolean}
     */
    isIdChecked(id) {
      return this.checkedList.indexOf(id) !== -1;
    }
    setupRowActionDefs() {
      this._rowActionHandlers = {};
      const list = this.options.additionalRowActionList;
      if (!list) {
        return;
      }
      this._additionalRowActionList = list;
      const defs = this.getMetadata().get(`clientDefs.${this.scope}.rowActionDefs`) || {};
      const promiseList = list.map(action => {
        /** @type {{handler: string, label?: string, labelTranslation?: string, groupIndex?: number}} */
        const itemDefs = defs[action] || {};
        if (!itemDefs.handler) {
          return Promise.resolve();
        }
        return Espo.loader.requirePromise(itemDefs.handler).then(Handler => {
          this._rowActionHandlers[action] = new Handler(this);
          return true;
        });
      });
      this.wait(Promise.all(promiseList));
    }

    // noinspection JSUnusedGlobalSymbols
    actionRowAction(data) {
      const action = data.actualAction;
      const id = data.id;
      if (!action) {
        return;
      }

      /** @type {{process: function(module:model, string)}} */
      const handler = (this._rowActionHandlers || {})[action];
      if (!handler) {
        console.warn(`No handler for action ${action}.`);
        return;
      }
      const model = this.collection.get(id);
      if (!model) {
        return;
      }
      handler.process(model, action);
    }

    /** @protected */
    setupSettings() {
      if (!this.options.settingsEnabled || !this.collection.entityType || !this.layoutName) {
        return;
      }
      if (!this.forceSettings && !this.getMetadata().get(`scopes.${this.entityType}.object`) || this.getConfig().get('listViewSettingsDisabled')) {
        return;
      }
      if (this.settingsDisabled) {
        return;
      }
      this._listSettingsHelper = this.options.settingsHelper || new _settings2.default(this.entityType, this.layoutName, this.getUser().id, this.getStorage());
      const view = new _settings.default({
        layoutProvider: () => this.listLayout,
        helper: this._listSettingsHelper,
        entityType: this.entityType,
        columnResize: this.columnResize,
        onChange: options => this.afterSettingsChange(options)
      });
      this.assignView('settings', view, '.settings-container');
    }

    /**
     * @protected
     * @param {RecordListSettingsView~onChangeOptions} options
     */
    async afterSettingsChange(options) {
      if (options.action === 'toggleColumnResize') {
        await this.reRender();
        return;
      }
      if (options.action === 'toggleColumn' && !this._listSettingsHelper.getHiddenColumnMap()[options.column] && this._columnResizeHelper) {
        const helper = new _columnWidthControl.default({
          view: this,
          helper: this._listSettingsHelper,
          layoutProvider: () => this.listLayout
        });
        helper.adjust();
      }
      this._internalLayout = null;
      Espo.Ui.notify(' ... ');
      await this.collection.fetch();
      Espo.Ui.notify(false);
    }

    /**
     * Whether the pagination is enabled.
     *
     * @return {boolean}
     */
    hasPagination() {
      return this.pagination;
    }

    /**
     * Hide a mass action. Requires re-render.
     *
     * @protected
     * @param {string} name An action name.
     * @since 8.4.0
     */
    hideMassAction(name) {
      if (!this.massActionDefs[name]) {
        this.massActionDefs[name] = {};
      }
      this.massActionDefs[name].hidden = true;
    }

    /**
     * Show a mass action. Requires re-render.
     *
     * @protected
     * @param {string} name An action name.
     * @since 8.4.0
     */
    showMassAction(name) {
      if (!this.massActionDefs[name]) {
        this.massActionDefs[name] = {};
      }
      this.massActionDefs[name].hidden = false;
    }

    /**
     * @private
     * @return {Array<{name: string, hidden: boolean}|false>}
     */
    getMassActionDataList() {
      /** @type {string[][]} */
      const groups = [];
      this.massActionList.forEach(action => {
        const item = this.massActionDefs[action];

        // For bc.
        if (item === false) {
          return;
        }
        const index = (!item || item.groupIndex === undefined ? 9999 : item.groupIndex) + 100;
        if (groups[index] === undefined) {
          groups[index] = [];
        }
        groups[index].push(action);
      });
      const list = [];
      groups.forEach(subList => {
        subList.forEach(it => list.push(it));
        list.push(false);
      });
      return list.map(name => {
        if (name === false) {
          return false;
        }
        return {
          name,
          hidden: (this.massActionDefs[name] || {}).hidden
        };
      });
    }

    /**
     * Uncheck all.
     *
     * @since 8.4.0
     */
    uncheckAll() {
      if (this.allResultIsChecked) {
        this.unselectAllResult();
      }
      this.checkedList.forEach(id => this.uncheckRecord(id));
    }

    /**
     * To temporarily disable checkboxes.
     *
     * @since 8.4.0
     */
    disableCheckboxes() {
      if (!this.checkboxes) {
        return;
      }
      this._disabledCheckboxes = true;
      this.uncheckAll();
      this.$el.find('input.record-checkbox').attr('disabled', 'disabled');
      if (this.$selectAllCheckbox) {
        this.$selectAllCheckbox.attr('disabled', 'disabled');
      }
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * To enabled temporarily disabled checkboxes.
     *
     * @since 8.4.0
     */
    enableCheckboxes() {
      if (!this.checkboxes) {
        return;
      }
      this._disabledCheckboxes = false;
      this.$el.find('input.record-checkbox').removeAttr('disabled');
      if (this.$selectAllCheckbox) {
        this.$selectAllCheckbox.removeAttr('disabled');
      }
    }

    /**
     * Checkboxes are disabled.
     *
     * @since 9.0.1
     * @return {boolean}
     */
    checkboxesAreDisabled() {
      return this._disabledCheckboxes || !this.checkboxes;
    }

    /**
     * Rebuild the internal layout.
     *
     * @return {Promise}
     * @since 8.4.0
     */
    rebuild() {
      return new Promise(resolve => {
        this._internalLayout = null;
        this.buildRows(() => resolve());
      });
    }
  }
  var _default = _exports.default = ListRecordView;
});

define("views/fields/text", ["exports", "views/fields/base", "helpers/misc/mailto", "views/modal"], function (_exports, _base, _mailto, _modal) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _base = _interopRequireDefault(_base);
  _mailto = _interopRequireDefault(_mailto);
  _modal = _interopRequireDefault(_modal);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/text */

  /**
   * A text field.
   *
   * @extends BaseFieldView<module:views/fields/text~params>
   */
  class TextFieldView extends _base.default {
    /**
     * @typedef {Object} module:views/fields/text~options
     * @property {
     *     module:views/fields/text~params &
     *     module:views/fields/base~params &
     *     Record
     * } [params] Parameters.
     */

    /**
     * @typedef {Object} module:views/fields/text~params
     * @property {boolean} [required] Required.
     * @property {number} [maxLength] A max length.
     * @property {number} [rows] A number of rows.
     * @property {number} [rowsMin] A min number of rows.
     * @property {boolean} [noResize] No resize.
     * @property {boolean} [seeMoreDisabled] Disable 'See-more'.
     * @property {boolean} [autoHeightDisabled] Disable auto-height.
     * @property {number} [cutHeight] A height of cut in pixels.
     * @property {boolean} [displayRawText] Display raw text.
     * @property {boolean} [preview] Display the preview button.
     * @property {string} [attachmentField] An attachment-multiple field to connect with.
     */

    /**
     * @param {
     *     module:views/fields/text~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }
    type = 'text';
    listTemplate = 'fields/text/list';
    detailTemplate = 'fields/text/detail';
    editTemplate = 'fields/text/edit';
    searchTemplate = 'fields/text/search';

    /**
     * Show-more is applied.
     * @type {boolean}
     */
    seeMoreText = false;
    rowsDefault = 50000;
    rowsMin = 2;
    seeMoreDisabled = false;
    cutHeight = 200;
    noResize = false;
    changeInterval = 5;
    shrinkThreshold = 10;
    searchTypeList = ['contains', 'startsWith', 'equals', 'endsWith', 'like', 'notContains', 'notLike', 'isEmpty', 'isNotEmpty'];
    events = {
      /** @this TextFieldView */
      'click [data-action="mailTo"]': function (e) {
        this.mailTo($(e.currentTarget).data('email-address'));
      }
    };

    /** @private */
    _lastLength;

    /** @private */
    maxRows;

    /**
     * @private
     * @type {HTMLElement}
     */
    previewButtonElement;
    setup() {
      super.setup();
      this.addActionHandler('seeMoreText', () => this.seeMore());
      this.maxRows = this.params.rows || this.rowsDefault;
      this.noResize = this.options.noResize || this.params.noResize || this.noResize;
      this.seeMoreDisabled = this.seeMoreDisabled || this.params.seeMoreDisabled;
      this.autoHeightDisabled = this.options.autoHeightDisabled || this.params.autoHeightDisabled || this.autoHeightDisabled;
      if (this.params.cutHeight) {
        this.cutHeight = this.params.cutHeight;
      }
      this.rowsMin = this.options.rowsMin || this.params.rowsMin || this.rowsMin;
      if (this.maxRows < this.rowsMin) {
        this.rowsMin = this.maxRows;
      }
      this.on('remove', () => {
        $(window).off('resize.see-more-' + this.cid);
      });
      if (this.params.preview) {
        this.addHandler('input', 'textarea', (e, /** HTMLTextAreaElement */target) => {
          const text = target.value;
          if (!this.previewButtonElement) {
            return;
          }
          if (text) {
            this.previewButtonElement.classList.remove('hidden');
          } else {
            this.previewButtonElement.classList.add('hidden');
          }
        });
        this.addActionHandler('previewText', () => this.preview());
      }

      /** @private */
      this.controlSeeMoreBind = this.controlSeeMore.bind(this);
      /** @private */
      this.onPasteBind = this.onPaste.bind(this);
    }
    setupSearch() {
      this.events['change select.search-type'] = e => {
        const type = $(e.currentTarget).val();
        this.handleSearchType(type);
      };
    }

    // noinspection JSCheckFunctionSignatures
    data() {
      const data = super.data();
      if (this.model.get(this.name) !== null && this.model.get(this.name) !== '' && this.model.has(this.name)) {
        data.isNotEmpty = true;
      }
      if (this.mode === this.MODE_SEARCH) {
        if (typeof this.searchParams.value === 'string') {
          this.searchData.value = this.searchParams.value;
        }
      }
      if (this.mode === this.MODE_EDIT) {
        data.rows = this.autoHeightDisabled ? this.maxRows : this.rowsMin;
      }
      data.valueIsSet = this.model.has(this.name);
      if (this.isReadMode()) {
        data.isCut = this.isCut();
        if (data.isCut) {
          data.cutHeight = this.cutHeight;
        }
        data.displayRawText = this.params.displayRawText;
      }
      data.htmlValue = undefined;
      data.noResize = this.noResize || !this.autoHeightDisabled && !this.params.rows;
      data.preview = this.params.preview && !this.params.displayRawText;

      // noinspection JSValidateTypes
      return data;
    }
    handleSearchType(type) {
      if (~['isEmpty', 'isNotEmpty'].indexOf(type)) {
        this.$el.find('input.main-element').addClass('hidden');
      } else {
        this.$el.find('input.main-element').removeClass('hidden');
      }
    }
    getValueForDisplay() {
      const text = this.model.get(this.name);
      return text || '';
    }

    /**
     * @public
     * @param {Number} [lastHeight]
     */
    controlTextareaHeight(lastHeight) {
      const scrollHeight = this.$element.prop('scrollHeight');
      const clientHeight = this.$element.prop('clientHeight');
      if (typeof lastHeight === 'undefined' && clientHeight === 0) {
        setTimeout(this.controlTextareaHeight.bind(this), 10);
        return;
      }

      /** @type {HTMLTextAreaElement} */
      const element = this.$element.get(0);
      if (!element || element.value === undefined) {
        return;
      }
      const length = element.value.length;
      if (this._lastLength === undefined) {
        this._lastLength = length;
      }
      if (length > this._lastLength) {
        this._lastLength = length;
      }
      if (clientHeight === lastHeight) {
        // @todo Revise.
        return;
      }
      if (scrollHeight > clientHeight + 1) {
        const rows = element.rows;
        if (this.maxRows && rows >= this.maxRows) {
          return;
        }
        element.rows++;
        this.controlTextareaHeight(clientHeight);
        return;
      }
      if (this.$element.val().length === 0) {
        element.rows = this.rowsMin;
        return;
      }
      const tryShrink = () => {
        const rows = element.rows;
        if (this.rowsMin && rows - 1 <= this.rowsMin) {
          return;
        }
        element.rows--;
        if (element.scrollHeight > element.clientHeight + 1) {
          this.controlTextareaHeight();
          return;
        }
        tryShrink();
      };
      if (length < this._lastLength - this.shrinkThreshold) {
        this._lastLength = length;
        tryShrink();
      }
    }
    isCut() {
      return !this.seeMoreText && !this.seeMoreDisabled;
    }
    controlSeeMore() {
      if (!this.isCut()) {
        return;
      }
      if (this.$text.height() > this.cutHeight) {
        this.$seeMoreContainer.removeClass('hidden');
        this.$textContainer.addClass('cut');
      } else {
        this.$seeMoreContainer.addClass('hidden');
        this.$textContainer.removeClass('cut');
      }
    }
    afterRender() {
      super.afterRender();
      if (this.isReadMode()) {
        $(window).off('resize.see-more-' + this.cid);
        this.$textContainer = this.$el.find('> .complex-text-container');
        this.$text = this.$textContainer.find('> .complex-text');
        this.$seeMoreContainer = this.$el.find('> .see-more-container');
        if (this.isCut()) {
          this.controlSeeMore();
          if (this.model.get(this.name) && this.$text.height() === 0) {
            this.$textContainer.addClass('cut');
            setTimeout(this.controlSeeMore.bind(this), 50);
          }
          this.listenTo(this.recordHelper, 'panel-show', () => this.controlSeeMore());
          this.on('panel-show-propagated', () => this.controlSeeMore());
          $(window).on('resize.see-more-' + this.cid, () => {
            this.controlSeeMore();
          });

          // Can be hidden.
          // @todo Revise stream post with empty text.
          if (this.element) {
            this.element.querySelectorAll('img').forEach(image => {
              image.addEventListener('load', this.controlSeeMoreBind);
            });
          }
        }
      }
      if (this.mode === this.MODE_EDIT) {
        const text = this.getValueForDisplay();
        if (text) {
          this.$element.val(text);
        }
        this.previewButtonElement = this.element ? this.element.querySelector('a[data-action="previewText"]') : undefined;
        const textAreaElement = /** @type {HTMLTextAreaElement} */this.$element.get(0);
        if (this.params.attachmentField && textAreaElement) {
          textAreaElement.removeEventListener('paste', this.onPasteBind);
          textAreaElement.addEventListener('paste', this.onPasteBind);
        }
      }
      if (this.mode === this.MODE_SEARCH) {
        const type = this.$el.find('select.search-type').val();
        this.handleSearchType(type);
        this.$el.find('select.search-type').on('change', () => {
          this.trigger('change');
        });
        this.$element.on('input', () => {
          this.trigger('change');
        });
      }
      if (this.mode === this.MODE_EDIT && !this.autoHeightDisabled) {
        if (!this.autoHeightDisabled) {
          this.controlTextareaHeight();
          this.$element.on('input', () => this.controlTextareaHeight());
        }
        let lastChangeKeydown = new Date();
        const changeKeydownInterval = this.changeInterval * 1000;
        this.$element.on('keydown', () => {
          if (Date.now() - lastChangeKeydown > changeKeydownInterval) {
            this.trigger('change');
            lastChangeKeydown = Date.now();
          }
        });
      }
    }
    fetch() {
      const data = {};
      let value = this.$element.val() || null;
      if (value && value.trim() === '') {
        value = '';
      }
      data[this.name] = value;
      return data;
    }
    fetchSearch() {
      const type = this.fetchSearchType() || 'startsWith';
      if (type === 'isEmpty') {
        return {
          type: 'or',
          value: [{
            type: 'isNull',
            field: this.name
          }, {
            type: 'equals',
            field: this.name,
            value: ''
          }],
          data: {
            type: type
          }
        };
      }
      if (type === 'isNotEmpty') {
        return {
          type: 'and',
          value: [{
            type: 'notEquals',
            field: this.name,
            value: ''
          }, {
            type: 'isNotNull',
            field: this.name,
            value: null
          }],
          data: {
            type: type
          }
        };
      }
      const value = this.$element.val().toString().trim();
      if (value) {
        return {
          value: value,
          type: type
        };
      }
      return false;
    }
    getSearchType() {
      return this.getSearchParamsData().type || this.searchParams.typeFront || this.searchParams.type;
    }
    mailTo(emailAddress) {
      const attributes = {
        status: 'Draft',
        to: emailAddress
      };
      const helper = new _mailto.default(this.getConfig(), this.getPreferences(), this.getAcl());
      if (helper.toUse()) {
        document.location.href = helper.composeLink(attributes);
        return;
      }
      const viewName = this.getMetadata().get('clientDefs.' + this.scope + '.modalViews.compose') || 'views/modals/compose-email';
      Espo.Ui.notify(' ... ');
      this.createView('quickCreate', viewName, {
        attributes: attributes
      }, view => {
        view.render();
        Espo.Ui.notify(false);
      });
    }

    /**
     * Show the preview modal.
     *
     * @since 9.0.0
     * @return {Promise<void>}
     */
    async preview() {
      const view = new _modal.default({
        templateContent: `<div class="complex-text">{{complexText viewObject.options.text linksInNewTab=true}}</div>`,
        text: this.model.attributes[this.name] || '',
        headerText: this.translate('Preview'),
        backdrop: true
      });
      await this.assignView('dialog', view);
      await view.render();
    }

    /**
     * @protected
     * @param {ClipboardEvent} event
     */
    onPaste(event) {
      const items = event.clipboardData.items;
      if (!items) {
        return;
      }
      for (let i = 0; i < items.length; i++) {
        if (!items[i].type.startsWith('image')) {
          continue;
        }
        const blob = items[i].getAsFile();
        this.recordHelper.trigger('upload-files:' + this.params.attachmentField, [blob]);
      }
    }

    /**
     * @return {Promise}
     * @since 9.0.0
     */
    async seeMore() {
      this.seeMoreText = true;
      await this.reRender();
    }
  }
  var _default = _exports.default = TextFieldView;
});

define("views/fields/int", ["exports", "views/fields/base", "autonumeric"], function (_exports, _base, _autonumeric) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _base = _interopRequireDefault(_base);
  _autonumeric = _interopRequireDefault(_autonumeric);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/int */

  /**
   * An integer field.
   *
   * @extends BaseFieldView<module:views/fields/int~params>
   */
  class IntFieldView extends _base.default {
    /**
     * @typedef {Object} module:views/fields/int~options
     * @property {
     *     module:views/fields/int~params &
     *     module:views/fields/base~params &
     *     Record
     * } [params] Parameters.
     */

    /**
     * @typedef {Object} module:views/fields/int~params
     * @property {number} [min] A max value.
     * @property {number} [max] A max value.
     * @property {boolean} [required] Required.
     * @property {boolean} [disableFormatting] Disable formatting.
     */

    /**
     * @param {
     *     module:views/fields/int~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }
    type = 'int';
    listTemplate = 'fields/int/list';
    detailTemplate = 'fields/int/detail';
    editTemplate = 'fields/int/edit';
    searchTemplate = 'fields/int/search';

    /**
     * @inheritDoc
     * @type {Array<(function (): boolean)|string>}
     */
    validations = ['required', 'int', 'range'];
    thousandSeparator = ',';
    searchTypeList = ['isNotEmpty', 'isEmpty', 'equals', 'notEquals', 'greaterThan', 'lessThan', 'greaterThanOrEquals', 'lessThanOrEquals', 'between'];

    /**
     * @type {import('autonumeric').Options}
     * @protected
     */
    autoNumericOptions;

    /**
     * @type {?AutoNumeric}
     * @protected
     */
    autoNumericInstance = null;
    setup() {
      super.setup();
      this.setupMaxLength();
      if (this.getPreferences().has('thousandSeparator')) {
        this.thousandSeparator = this.getPreferences().get('thousandSeparator');
      } else if (this.getConfig().has('thousandSeparator')) {
        this.thousandSeparator = this.getConfig().get('thousandSeparator');
      }
      if (this.params.disableFormatting) {
        this.disableFormatting = true;
      }
    }
    setupFinal() {
      super.setupFinal();
      this.setupAutoNumericOptions();
    }

    /**
     * @protected
     */
    setupAutoNumericOptions() {
      const separator = (!this.disableFormatting ? this.thousandSeparator : null) || '';
      let decimalCharacter = '.';
      if (separator === '.') {
        decimalCharacter = ',';
      }

      // noinspection JSValidateTypes
      this.autoNumericOptions = {
        digitGroupSeparator: separator,
        decimalCharacter: decimalCharacter,
        modifyValueOnWheel: false,
        decimalPlaces: 0,
        selectOnFocus: false,
        formulaMode: true
      };
      if (this.params.max != null && this.params.max > Math.pow(10, 6)) {
        this.autoNumericOptions.maximumValue = this.params.max.toString();
      }
      if (this.params.min != null && this.params.min < -Math.pow(10, 6)) {
        this.autoNumericOptions.minimumValue = this.params.min.toString();
      }
    }
    afterRender() {
      super.afterRender();
      if (this.mode === this.MODE_EDIT) {
        if (this.autoNumericOptions) {
          /** @type {HTMLInputElement} */
          const element = this.$element.get(0);
          this.autoNumericInstance = new _autonumeric.default(element, this.autoNumericOptions);
        }
      }
      if (this.mode === this.MODE_SEARCH) {
        const $searchType = this.$el.find('select.search-type');
        this.handleSearchType($searchType.val());
        this.$el.find('select.search-type').on('change', () => {
          this.trigger('change');
        });
        this.$element.on('input', () => {
          this.trigger('change');
        });
        const $inputAdditional = this.$el.find('input.additional');
        $inputAdditional.on('input', () => {
          this.trigger('change');
        });
        if (this.autoNumericOptions) {
          /** @type {HTMLInputElement} */
          const element1 = this.$element.get(0);
          /** @type {HTMLInputElement} */
          const element2 = $inputAdditional.get(0);
          new _autonumeric.default(element1, this.autoNumericOptions);
          new _autonumeric.default(element2, this.autoNumericOptions);
        }
      }
    }

    // noinspection JSCheckFunctionSignatures
    data() {
      const data = super.data();
      if (this.model.get(this.name) !== null && typeof this.model.get(this.name) !== 'undefined') {
        data.isNotEmpty = true;
      }
      data.valueIsSet = this.model.has(this.name);
      if (this.isSearchMode()) {
        data.value = this.searchParams.value;
        if (this.getSearchType() === 'between') {
          data.value = this.getSearchParamsData().value1 || this.searchParams.value1;
          data.value2 = this.getSearchParamsData().value2 || this.searchParams.value2;
        }
      }
      if (this.isEditMode()) {
        data.value = this.model.get(this.name);
      }

      // noinspection JSValidateTypes
      return data;
    }
    getValueForDisplay() {
      const value = isNaN(this.model.get(this.name)) ? null : this.model.get(this.name);
      return this.formatNumber(value);
    }
    formatNumber(value) {
      if (this.disableFormatting) {
        return value;
      }
      return this.formatNumberDetail(value);
    }
    formatNumberDetail(value) {
      if (value === null) {
        return '';
      }
      let stringValue = value.toString();
      stringValue = stringValue.replace(/\B(?=(\d{3})+(?!\d))/g, this.thousandSeparator);
      return stringValue;
    }
    setupSearch() {
      this.events['change select.search-type'] = e => {
        // noinspection JSUnresolvedReference
        this.handleSearchType($(e.currentTarget).val());
      };
    }
    handleSearchType(type) {
      const $additionalInput = this.$el.find('input.additional');
      const $input = this.$el.find('input[data-name="' + this.name + '"]');
      if (type === 'between') {
        $additionalInput.removeClass('hidden');
        $input.removeClass('hidden');
      } else if (~['isEmpty', 'isNotEmpty'].indexOf(type)) {
        $additionalInput.addClass('hidden');
        $input.addClass('hidden');
      } else {
        $additionalInput.addClass('hidden');
        $input.removeClass('hidden');
      }
    }
    getMaxValue() {
      let maxValue = this.model.getFieldParam(this.name, 'max') || null;
      if (!maxValue && maxValue !== 0) {
        maxValue = null;
      }
      if ('max' in this.params) {
        maxValue = this.params.max;
      }
      return maxValue;
    }
    getMinValue() {
      let minValue = this.model.getFieldParam(this.name, 'min');
      if (!minValue && minValue !== 0) {
        minValue = null;
      }
      if ('min' in this.params) {
        minValue = this.params.min;
      }
      return minValue;
    }
    setupMaxLength() {
      let maxValue = this.getMaxValue();
      if (typeof max !== 'undefined' && max !== null) {
        maxValue = this.formatNumber(maxValue);
        this.params.maxLength = maxValue.toString().length;
      }
    }

    // noinspection JSUnusedGlobalSymbols
    validateInt() {
      const value = this.model.get(this.name);
      if (isNaN(value)) {
        const msg = this.translate('fieldShouldBeInt', 'messages').replace('{field}', this.getLabelText());
        this.showValidationMessage(msg);
        return true;
      }
    }

    // noinspection JSUnusedGlobalSymbols
    validateRange() {
      const value = this.model.get(this.name);
      if (value === null) {
        return false;
      }
      const minValue = this.getMinValue();
      const maxValue = this.getMaxValue();
      if (minValue !== null && maxValue !== null) {
        if (value < minValue || value > maxValue) {
          const msg = this.translate('fieldShouldBeBetween', 'messages').replace('{field}', this.getLabelText()).replace('{min}', minValue).replace('{max}', maxValue);
          this.showValidationMessage(msg);
          return true;
        }
      } else {
        if (minValue !== null) {
          if (value < minValue) {
            const msg = this.translate('fieldShouldBeGreater', 'messages').replace('{field}', this.getLabelText()).replace('{value}', minValue);
            this.showValidationMessage(msg);
            return true;
          }
        } else if (maxValue !== null) {
          if (value > maxValue) {
            const msg = this.translate('fieldShouldBeLess', 'messages').replace('{field}', this.getLabelText()).replace('{value}', maxValue);
            this.showValidationMessage(msg);
            return true;
          }
        }
      }
    }
    validateRequired() {
      if (this.isRequired()) {
        const value = this.model.get(this.name);
        if (value === null || value === false) {
          const msg = this.translate('fieldIsRequired', 'messages').replace('{field}', this.getLabelText());
          this.showValidationMessage(msg);
          return true;
        }
      }
    }
    parse(value) {
      value = value !== '' ? value : null;
      if (value === null) {
        return null;
      }
      value = value.split(this.thousandSeparator).join('');
      if (value.indexOf('.') !== -1 || value.indexOf(',') !== -1) {
        return NaN;
      }
      return parseInt(value);
    }
    fetch() {
      let value = this.$element.val();
      value = this.parse(value);
      const data = {};
      data[this.name] = value;
      return data;
    }
    fetchSearch() {
      const value = this.parse(this.$element.val());
      const type = this.fetchSearchType();
      let data;
      if (isNaN(value)) {
        return false;
      }
      if (type === 'between') {
        const valueTo = this.parse(this.$el.find('input.additional').val());
        if (isNaN(valueTo)) {
          return false;
        }
        data = {
          type: type,
          value: [value, valueTo],
          data: {
            value1: value,
            value2: valueTo
          }
        };
      } else if (type === 'isEmpty') {
        data = {
          type: 'isNull',
          typeFront: 'isEmpty'
        };
      } else if (type === 'isNotEmpty') {
        data = {
          type: 'isNotNull',
          typeFront: 'isNotEmpty'
        };
      } else {
        data = {
          type: type,
          value: value,
          data: {
            value1: value
          }
        };
      }
      return data;
    }
    getSearchType() {
      return this.searchParams.typeFront || this.searchParams.type;
    }
  }
  var _default = _exports.default = IntFieldView;
});

define("ui/datepicker", ["exports", "jquery", "language", "models/settings", "di", "moment"], function (_exports, _jquery, _language, _settings, _di, _moment) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _jquery = _interopRequireDefault(_jquery);
  _language = _interopRequireDefault(_language);
  _settings = _interopRequireDefault(_settings);
  _moment = _interopRequireDefault(_moment);
  let _init_language, _init_extra_language, _init_config, _init_extra_config;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * A datepicker.
   *
   * @since 9.0.0
   */
  class Datepicker {
    static {
      [_init_language, _init_extra_language, _init_config, _init_extra_config] = _applyDecs(this, [], [[(0, _di.inject)(_language.default), 0, "language"], [(0, _di.inject)(_settings.default), 0, "config"]]).e;
    }
    /**
     * @private
     * @type {Language}
     */
    language = _init_language(this);

    /**
     * @private
     * @type {Settings}
     */
    config = (_init_extra_language(this), _init_config(this));

    /**
     * @param {HTMLElement} element
     * @param {{
     *     format: string,
     *     weekStart: number,
     *     todayButton?: boolean,
     *     date?: string,
     *     startDate?: string|undefined,
     *     onChange?: function(),
     *     hasDay?: function(string): boolean,
     *     hasMonth?: function(string): boolean,
     *     hasYear?: function(string): boolean,
     *     onChangeDate?: function(),
     *     onChangeMonth?: function(string),
     *     defaultViewDate?: string,
     * }} options
     */
    constructor(element, options) {
      _init_extra_config(this);
      /**
       * @private
       */
      this.$element = (0, _jquery.default)(element);

      /**
       * @private
       * @type {string}
       */
      this.format = options.format;
      if (element instanceof HTMLInputElement) {
        if (options.date) {
          element.value = options.date;
        }
        let wait = false;
        this.$element.on('change', /** Record */e => {
          if (!wait) {
            if (options.onChange) {
              options.onChange();
            }
            wait = true;
            setTimeout(() => wait = false, 100);
          }
          if (e.isTrigger && document.activeElement !== this.$element.get(0)) {
            this.$element.focus();
          }
        });
        this.$element.on('click', () => this.show());
      } else {
        if (options.date) {
          element.dataset.date = options.date;
        }
      }
      const modalBodyElement = element.closest('.modal-body');
      const language = this.config.get('language');
      const format = options.format;
      const datepickerOptions = {
        autoclose: true,
        todayHighlight: true,
        keyboardNavigation: true,
        assumeNearbyYear: true,
        format: format.toLowerCase(),
        weekStart: options.weekStart,
        todayBtn: options.todayButton || false,
        startDate: options.startDate,
        orientation: 'bottom auto',
        templates: {
          leftArrow: '<span class="fas fa-chevron-left fa-sm"></span>',
          rightArrow: '<span class="fas fa-chevron-right fa-sm"></span>'
        },
        container: modalBodyElement ? (0, _jquery.default)(modalBodyElement) : 'body',
        language: language,
        maxViewMode: 2,
        defaultViewDate: options.defaultViewDate
      };
      if (options.hasDay) {
        datepickerOptions.beforeShowDay = (/** Date */date) => {
          const stringDate = (0, _moment.default)(date).format(this.format);
          return {
            enabled: options.hasDay(stringDate)
          };
        };
      }
      if (options.hasMonth) {
        datepickerOptions.beforeShowMonth = (/** Date */date) => {
          const stringDate = (0, _moment.default)(date).format(this.format);
          return {
            enabled: options.hasMonth(stringDate)
          };
        };
      }
      if (options.hasYear) {
        datepickerOptions.beforeShowYear = (/** Date */date) => {
          const stringDate = (0, _moment.default)(date).format(this.format);
          return {
            enabled: options.hasYear(stringDate)
          };
        };
      }

      // noinspection JSUnresolvedReference
      if (!(language in _jquery.default.fn.datepicker.dates)) {
        // noinspection JSUnresolvedReference
        _jquery.default.fn.datepicker.dates[language] = {
          days: this.language.get('Global', 'lists', 'dayNames'),
          daysShort: this.language.get('Global', 'lists', 'dayNamesShort'),
          daysMin: this.language.get('Global', 'lists', 'dayNamesMin'),
          months: this.language.get('Global', 'lists', 'monthNames'),
          monthsShort: this.language.get('Global', 'lists', 'monthNamesShort'),
          today: this.language.translate('Today'),
          clear: this.language.translate('Clear')
        };
      }
      this.$element.datepicker(datepickerOptions).on('changeDate', () => {
        if (options.onChangeDate) {
          options.onChangeDate();
        }
      }).on('changeMonth', (/** {date: Date} */event) => {
        if (options.onChangeMonth) {
          const dateString = (0, _moment.default)(event.date).startOf('month').format(options.format);
          options.onChangeMonth(dateString);
        }
      });
      if (element.classList.contains('input-group') && !(element instanceof HTMLInputElement)) {
        element.querySelectorAll('input').forEach(input => {
          (0, _jquery.default)(input).on('click', () => (0, _jquery.default)(input).datepicker('show'));
        });
      }
    }

    /**
     * Set a start date.
     *
     * @param {string|undefined} startDate
     */
    setStartDate(startDate) {
      this.$element.datepicker('setStartDate', startDate);
    }

    /**
     * Show.
     */
    show() {
      this.$element.datepicker('show');
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Get the value.
     *
     * @return {string|null}
     */
    getDate() {
      const date = this.$element.datepicker('getDate');
      if (!date) {
        return null;
      }
      return (0, _moment.default)(date).format(this.format);
    }

    /**
     * Refresh.
     */
    refresh() {
      const picker = this.$element.data('datepicker');
      if (!picker) {
        return;
      }
      picker.fill();
    }
  }
  var _default = _exports.default = Datepicker;
});

define("helpers/misc/summernote-custom", ["exports", "jquery", "views/wysiwyg/modals/edit-table", "views/wysiwyg/modals/edit-cell", "handlebars"], function (_exports, _jquery, _editTable, _editCell, _handlebars) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.init = init;
  _jquery = _interopRequireDefault(_jquery);
  _editTable = _interopRequireDefault(_editTable);
  _editCell = _interopRequireDefault(_editCell);
  _handlebars = _interopRequireDefault(_handlebars);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /**
   * @type {{
   *     edit: import('ace-builds').edit,
   *     require: import('ace-builds').require,
   * }}
   */
  let ace;
  function init(langSets) {
    _jquery.default.extend(_jquery.default.summernote.options, {
      espoImage: {
        icon: '<i class="note-icon-picture"/>',
        tooltip: langSets.image.image
      },
      espoLink: {
        icon: '<i class="note-icon-link"/>',
        tooltip: langSets.link.link
      },
      espoTable: {
        icon: '<i class="note-icon-table"/>',
        tooltip: langSets.table.table
      },
      popover: {
        table: [['custom', ['tableParams', 'cellParams']], ['add', ['addRowDown', 'addRowUp', 'addColLeft', 'addColRight']], ['delete', ['deleteRow', 'deleteCol', 'deleteTable']]]
      }
    });
    _jquery.default.extend(_jquery.default.summernote.plugins, {
      'cellParams': function (/** Record */context) {
        const ui = _jquery.default.summernote.ui;
        const options = context.options;
        const view = /** @type {import('view').default} */options.espoView;
        if (!view) {
          return;
        }
        context.memo('button.cellParams', () => {
          return ui.button({
            className: '',
            contents: '<i class="far fa-square fa-sm"/>',
            tooltip: view.translate('cell', 'wysiwygLabels'),
            click: () => {
              context.invoke('cellParams.show');
            }
          }).render();
        });
        this.show = function () {
          const range = context.invoke('editor.getLastRange');
          const $td = (0, _jquery.default)(range.ec).closest('td,th');
          const td = /** @type {HTMLTableCellElement} */$td[0];
          const width = td.style.width;
          const height = td.style.height;
          const backgroundColor = td.style.backgroundColor;
          const verticalAlign = td.style.verticalAlign;
          const params = {
            width,
            height,
            backgroundColor,
            verticalAlign
          };

          //const prevParams = params;

          const modalView = new _editCell.default({
            params,
            headerText: view.translate('cell', 'wysiwygLabels'),
            onApply: params => applyParams(params)
          });
          view.assignView('dialog', modalView).then(() => {
            modalView.render();
          });

          /**
           * @param {{
           *     width: string|null,
           *     height: string|null,
           *     backgroundColor: string|null,
           *     verticalAlign: string|null,
           * }} params
           */
          const applyParams = params => {
            let backgroundColor = td.style.backgroundColor;
            if (backgroundColor === 'transparent') {
              backgroundColor = null;
            }
            td.style.backgroundColor = params.backgroundColor;
            td.style.width = params.width;
            td.style.height = params.height;
            td.style.verticalAlign = params.verticalAlign;
          };
        };
        this.destroy = function () {
          if (view) {
            view.clearView('dialog');
          }
        };
      },
      'tableParams': function (/** Record */context) {
        const ui = _jquery.default.summernote.ui;
        const options = context.options;
        const view = /** @type {import('view').default} */options.espoView;
        if (!view) {
          // Prevents an issue with a collapsed modal.
          // @todo Revise.
          return;
        }
        context.memo('button.tableParams', () => {
          return ui.button({
            className: '',
            contents: '<i class="note-icon-table"/>',
            tooltip: langSets.table.table,
            click: () => {
              context.invoke('tableParams.show');
            }
          }).render();
        });
        this.show = function () {
          const range = context.invoke('editor.getLastRange');
          const $table = (0, _jquery.default)(range.ec).closest('table');
          const table = /** @type {HTMLTableElement} */$table[0];
          let borderWidth = table.style.borderWidth;
          if (borderWidth !== null) {
            // @todo
            for (const /** HTMLTableCellElement */cell of table.querySelectorAll('td, th')) {
              if (cell.style.borderWidth !== borderWidth) {
                borderWidth = null;
                break;
              }
            }
          }
          let backgroundColor = table.style.backgroundColor;
          if (backgroundColor === 'transparent') {
            backgroundColor = null;
          }
          let borderColor = table.style.borderColor;
          if (borderColor !== null) {
            // @todo
            for (const /** HTMLTableCellElement */cell of table.querySelectorAll('td, th')) {
              if (cell.style.borderColor !== borderColor) {
                borderColor = null;
                break;
              }
            }
          }
          let cellPadding = null;
          for (const /** HTMLTableCellElement */cell of table.querySelectorAll('td, th')) {
            if (cellPadding !== null && cell.style.padding !== cellPadding) {
              cellPadding = null;
              break;
            }
            cellPadding = cell.style.padding;
          }
          let align = null;
          if (table.style.marginLeft === 'auto' && table.style.marginRight === '0px') {
            align = 'right';
          } else if (table.style.marginLeft === 'auto' && table.style.marginRight === 'auto') {
            align = 'center';
          } else if (table.style.marginLeft === '0px' && table.style.marginRight === 'auto') {
            align = 'left';
          }
          const width = table.style.width;
          const height = table.style.height;
          const params = {
            align,
            width,
            height,
            borderWidth,
            borderColor,
            cellPadding,
            backgroundColor
          };
          const prevParams = params;
          const modalView = new _editTable.default({
            params,
            headerText: langSets.table.table,
            onApply: params => applyParams(params)
          });
          view.assignView('dialog', modalView).then(() => {
            modalView.render();
          });

          /**
           * @param {{
           *     align: string|null,
           *     width: string|null,
           *     height: string|null,
           *     borderWidth: string|null,
           *     borderColor: string|null,
           *     cellPadding: string|null,
           *     backgroundColor: string|null,
           * }} params
           */
          const applyParams = params => {
            if (params.align === 'left') {
              table.style.marginLeft = '0';
              table.style.marginRight = 'auto';
            } else if (params.align === 'right') {
              table.style.marginLeft = 'auto';
              table.style.marginRight = '0';
            } else if (params.align === 'center') {
              table.style.marginLeft = 'auto';
              table.style.marginRight = 'auto';
            } else if (params.align === null && prevParams.align !== null) {
              table.style.marginLeft = null;
              table.style.marginRight = null;
            }
            table.style.backgroundColor = params.backgroundColor;
            table.style.width = params.width;
            table.style.height = params.height;
            if (params.borderWidth !== null || prevParams.borderWidth !== null) {
              table.style.borderWidth = params.borderWidth;
              table.style.borderStyle = params.borderWidth !== null ? 'solid' : null;
              for (const /** HTMLTableCellElement */cell of table.querySelectorAll('td, th')) {
                cell.style.borderWidth = params.borderWidth;
                cell.style.borderStyle = params.borderWidth !== null ? 'solid' : null;
              }
            }
            if (params.borderColor !== null || prevParams.borderColor !== null) {
              table.style.borderColor = params.borderColor;
              for (const /** HTMLTableCellElement */cell of table.querySelectorAll('td, th')) {
                cell.style.borderColor = params.borderColor;
              }
            }
            if (params.cellPadding !== null || prevParams.padding !== null) {
              for (const /** HTMLTableCellElement */cell of table.querySelectorAll('td, th')) {
                cell.style.padding = params.cellPadding;
              }
            }
          };
        };
        this.destroy = function () {
          if (view) {
            view.clearView('dialog');
          }
        };
      },
      'aceCodeview': function (/** Record */context) {
        const ui = _jquery.default.summernote.ui;
        const options = context.options;
        const lang = options.langInfo;
        /** @type {JQuery} */
        const $editor = context.layoutInfo.editor;
        /** @type {JQuery} */
        const $editable = context.layoutInfo.editable;
        const view = /** @type {module:view} */options.espoView;
        if (!view) {
          return;
        }

        // noinspection SpellCheckingInspection
        let $codable;
        /** @type {import('ace-builds').Ace.Editor} */
        let aceEditor;
        let isActivated = false;
        let isReplaced = false;
        let isBeingActivated = false;
        const EMPTY = '<p>&nbsp;</p>';
        context.destroyAceCodeview = function () {
          if (isActivated) {
            deactivate();
          }
        };
        context.code = function (html) {
          if (html === undefined) {
            let value;
            if (isActivated) {
              value = prepareValue(aceEditor.getValue()) || EMPTY;
              $editable.html(value);
            }
            return isActivated ? value : $editable.html();
          }
          if (isActivated) {
            aceEditor.setValue(html);
          }
          $editable.html(html);
          this.$note.val(html);
          context.triggerEvent('change', html, $editable);
        };
        const id = 'editor-' + Math.random().toString(36).substring(2, 17);
        const prepareValue = input => {
          return input;
        };
        const prepareHtml = (/** string */input) => {
          return prepareValue(input);
        };
        const toggle = () => {
          isActivated ? deactivate() : activate();
          context.triggerEvent('codeview.toggled');
        };
        const deactivate = () => {
          if (!isActivated) {
            return;
          }
          $codable.addClass('hidden');
          let value = prepareValue(aceEditor.getValue()) || EMPTY;
          value = context.invoke('codeview.purify', value);
          const isChange = $editable.html() !== value;
          $editable.html(value);
          $editor.removeClass('codeview');
          if (isChange) {
            context.triggerEvent('change', $editable.html(), $editable);
          }
          $editable.focus();
          context.invoke('toolbar.updateCodeview', false);
          context.invoke('airPopover.updateCodeview', false);
          if (aceEditor) {
            aceEditor.destroy();
            aceEditor = null;
          }
          context.aceEditor = undefined;
          isActivated = false;
        };
        const activate = () => {
          if (isBeingActivated) {
            return;
          }
          isBeingActivated = true;
          context.invoke('toolbar.updateCodeview', true);
          context.invoke('airPopover.updateCodeview', true);
          $editor.addClass('codeview');
          if (!isReplaced) {
            $codable = (0, _jquery.default)('<div>').attr('id', id).css('minHeight', 40 + 'px');
            (0, _jquery.default)(context.layoutInfo.codable).replaceWith($codable);
            context.$aceCodable = $codable;
            isReplaced = true;
          }
          $codable.removeClass('hidden');
          if ($editor.hasClass('fullscreen')) {
            $codable.css('height', $editable.css('height'));
          }
          requireAce().then(() => {
            const html = prepareHtml($editable.html());
            aceEditor = ace.edit(id);
            aceEditor.setValue(html);
            aceEditor.setOptions({
              maxLines: !$editor.hasClass('fullscreen') ? 100000 : null,
              enableLiveAutocompletion: true,
              tabSize: 2,
              useSoftTabs: true
            });
            aceEditor.setOptions({
              fontFamily: 'var(--font-family-monospace)'
            });
            aceEditor.setFontSize('var(--font-size-small)');
            aceEditor.container.style.lineHeight = 'var(--line-height-small)';
            aceEditor.renderer.updateFontSize();
            if (options.isDark) {
              aceEditor.setOptions({
                theme: 'ace/theme/tomorrow_night'
              });
            }
            aceEditor.getSession().setUseWrapMode(true);
            aceEditor.setShowPrintMargin(false);
            aceEditor.getSession().setUseWorker(false);
            aceEditor.commands.removeCommand('find');
            aceEditor.setHighlightActiveLine(false);
            aceEditor.focus();
            try {
              aceEditor.gotoLine(0, 0, false);
            } catch (e) {}
            aceEditor.on('blur', e => {
              context.triggerEvent('blur.codeview', aceEditor.getValue(), e);
            });
            aceEditor.on('change', () => {
              context.triggerEvent('change.codeview', aceEditor.getValue());
            });

            // noinspection JSValidateTypes
            context.aceEditor = aceEditor;
            const modeToRequired = options.handlebars ? 'ace/mode/handlebars' : 'ace/mode/html';
            const Mode = ace.require(modeToRequired).Mode;
            aceEditor.session.setMode(new Mode());
            isActivated = true;
            isBeingActivated = false;
          });
        };
        context.memo('button.aceCodeview', () => {
          return ui.button({
            className: 'btn-codeview note-codeview-keep',
            contents: '<i class="note-icon-code"/>',
            tooltip: lang.options.codeview,
            click: () => toggle()
          }).render();
        });

        /**
         * @return Promise
         */
        const requireAce = function () {
          return Espo.loader.requirePromise('lib!ace').then(lib => {
            ace = /** window.ace */lib;
            const list = [Espo.loader.requirePromise('lib!ace-ext-language_tools')];
            list.push(options.handlebars ? Espo.loader.requirePromise('lib!ace-mode-handlebars') : Espo.loader.requirePromise('lib!ace-mode-html'));
            if (options.isDark) {
              list.push(Espo.loader.requirePromise('lib!ace-theme-tomorrow_night'));
            }
            return Promise.all(list);
          });
        };
      },
      'espoTable': function (context) {
        const ui = _jquery.default.summernote.ui;
        const options = context.options;
        const self = options.espoView;
        const lang = options.langInfo;
        if (!self) {
          return;
        }
        context.memo('button.espoTable', () => {
          return ui.buttonGroup([ui.button({
            className: 'dropdown-toggle',
            contents: ui.dropdownButtonContents(ui.icon(options.icons.table), options),
            tooltip: options.espoTable.tooltip,
            data: {
              toggle: 'dropdown'
            }
          }), ui.dropdown({
            title: lang.table.table,
            className: 'note-table',
            items: ['<div class="note-dimension-picker">', '<div class="note-dimension-picker-mousecatcher" data-event="insertTable" data-value="1x1"></div>', '<div class="note-dimension-picker-highlighted"></div>', '<div class="note-dimension-picker-unhighlighted"></div>', '</div>', '<div class="note-dimension-display">1 x 1</div>'].join('')
          })], {
            callback: $node => {
              const $catcher = $node.find('.note-dimension-picker-mousecatcher');
              const createTable = (colCount, rowCount, options) => {
                const tds = [];
                let tdHTML;
                for (let idxCol = 0; idxCol < colCount; idxCol++) {
                  tds.push('<td>&nbsp;</td>');
                }
                tdHTML = tds.join('\n');
                const trs = [];
                let trHTML;
                for (let idxRow = 0; idxRow < rowCount; idxRow++) {
                  trs.push('<tr>' + tdHTML + '</tr>');
                }
                trHTML = trs.join('\n');
                const $table = (0, _jquery.default)('<table>\n' + trHTML + '</table>');

                /*if (options.tableBorderWidth !== undefined) {
                    $table.attr('border', options.tableBorderWidth);
                }
                  if (options.tableCellPadding !== undefined) {
                    $table.attr('cellpadding', options.tableCellPadding);
                }*/

                $table.css({
                  width: '100%',
                  borderCollapse: 'collapse'
                  //borderSpacing: 0,
                });
                if (options && options.tableClassName) {
                  $table.addClass(options.tableClassName);
                }
                return $table[0];
              };
              $catcher.css({
                width: options.insertTableMaxSize.col + 'em',
                height: options.insertTableMaxSize.row + 'em'
              }).mousedown(() => {
                const $note = context.$note;
                const dims = $catcher.data('value').split('x');
                const range = $note.summernote('editor.getLastRange').deleteContents();
                createTable(dims[0], dims[1], options);
                range.insertNode(createTable(dims[0], dims[1], options));
              }).on('mousemove', event => {
                const PX_PER_EM = 18;
                const $picker = (0, _jquery.default)(event.target.parentNode);
                const $dimensionDisplay = $picker.next();
                const $catcher = $picker.find('.note-dimension-picker-mousecatcher');
                const $highlighted = $picker.find('.note-dimension-picker-highlighted');
                const $unhighlighted = $picker.find('.note-dimension-picker-unhighlighted');
                let posOffset;
                if (event.offsetX === undefined) {
                  const posCatcher = (0, _jquery.default)(event.target).offset();
                  posOffset = {
                    x: event.pageX - posCatcher.left,
                    y: event.pageY - posCatcher.top
                  };
                } else {
                  posOffset = {
                    x: event.offsetX,
                    y: event.offsetY
                  };
                }
                const dim = {
                  c: Math.ceil(posOffset.x / PX_PER_EM) || 1,
                  r: Math.ceil(posOffset.y / PX_PER_EM) || 1
                };
                $highlighted.css({
                  width: dim.c + 'em',
                  height: dim.r + 'em'
                });
                $catcher.data('value', dim.c + 'x' + dim.r);
                if (dim.c > 3 && dim.c < options.insertTableMaxSize.col) {
                  $unhighlighted.css({
                    width: dim.c + 1 + 'em'
                  });
                }
                if (dim.r > 3 && dim.r < options.insertTableMaxSize.row) {
                  $unhighlighted.css({
                    height: dim.r + 1 + 'em'
                  });
                }
                $dimensionDisplay.html(dim.c + ' x ' + dim.r);
              });
            }
          }).render();
        });
      },
      'espoImage': function (context) {
        const ui = _jquery.default.summernote.ui;
        const options = context.options;
        const self = options.espoView;
        const lang = options.langInfo;
        if (!self) {
          return;
        }
        context.memo('button.espoImage', () => {
          const button = ui.button({
            contents: options.espoImage.icon,
            tooltip: options.espoImage.tooltip,
            click() {
              context.invoke('espoImage.show');
            }
          });
          return button.render();
        });
        this.initialize = function () {};
        this.destroy = function () {
          if (!self) {
            return;
          }
          self.clearView('insertImageDialog');
        };
        this.show = function () {
          self.createView('insertImageDialog', 'views/wysiwyg/modals/insert-image', {
            labels: {
              insert: lang.image.insert,
              url: lang.image.url,
              selectFromFiles: lang.image.selectFromFiles
            }
          }, view => {
            view.render();
            self.listenToOnce(view, 'upload', target => {
              self.$summernote.summernote('insertImagesOrCallback', target);
            });
            self.listenToOnce(view, 'insert', target => {
              self.$summernote.summernote('insertImage', target);
            });
            self.listenToOnce(view, 'close', () => {
              self.clearView('insertImageDialog');
              self.fixPopovers();
            });
          });
        };
      },
      // Not used?
      'linkDialog': function (context) {
        const options = context.options;
        const self = options.espoView;
        const lang = options.langInfo;
        if (!self) {
          return;
        }
        this.show = function () {
          const linkInfo = context.invoke('editor.getLinkInfo');
          self.createView('dialogInsertLink', 'views/wysiwyg/modals/insert-link', {
            labels: {
              insert: lang.link.insert,
              openInNewWindow: lang.link.openInNewWindow,
              url: lang.link.url,
              textToDisplay: lang.link.textToDisplay
            },
            linkInfo: linkInfo
          }, view => {
            view.render();
            self.listenToOnce(view, 'insert', data => {
              data.text = _handlebars.default.Utils.escapeExpression(data.text);
              self.$summernote.summernote('createLink', data);
            });
            self.listenToOnce(view, 'close', () => {
              self.clearView('dialogInsertLink');
              self.fixPopovers();
            });
          });
        };
      },
      'espoLink': function (context) {
        const ui = _jquery.default.summernote.ui;
        const options = context.options;
        const self = options.espoView;
        const lang = options.langInfo;
        if (!self) {
          return;
        }
        const isMacLike = /(Mac|iPhone|iPod|iPad)/i.test(navigator.platform);
        context.memo('button.espoLink', function () {
          const button = ui.button({
            contents: options.espoLink.icon,
            tooltip: options.espoLink.tooltip + ' (' + (isMacLike ? 'CMD+K' : 'CTRL+K') + ')',
            click() {
              context.invoke('espoLink.show');
            }
          });
          return button.render();
        });
        this.initialize = function () {
          this.$modalBody = self.$el.closest('.modal-body');
          this.isInModal = this.$modalBody.length > 0;
        };
        this.destroy = function () {
          if (!self) {
            return;
          }
          self.clearView('dialogInsertLink');
        };
        this.show = function () {
          const linkInfo = context.invoke('editor.getLinkInfo');
          const container = this.isInModal ? this.$modalBody.get(0) : window;
          self.createView('dialogInsertLink', 'views/wysiwyg/modals/insert-link', {
            labels: {
              insert: lang.link.insert,
              openInNewWindow: lang.link.openInNewWindow,
              url: lang.link.url,
              textToDisplay: lang.link.textToDisplay
            },
            linkInfo: linkInfo
          }, view => {
            view.render();
            self.listenToOnce(view, 'insert', data => {
              const scrollY = 'scrollY' in container ? container.scrollY : container.scrollTop;
              data.text = _handlebars.default.Utils.escapeExpression(data.text);
              self.$summernote.summernote('createLink', data);
              setTimeout(() => container.scroll(0, scrollY), 20);
            });
            self.listenToOnce(view, 'close', () => {
              self.clearView('dialogInsertLink');
              self.fixPopovers();
            });
          });
        };
      },
      'fullscreen': function (context) {
        const options = context.options;
        const self = options.espoView;
        if (!self) {
          return;
        }
        this.$window = (0, _jquery.default)(window);
        this.$scrollbar = (0, _jquery.default)('html, body');
        this.initialize = function () {
          this.$editor = context.layoutInfo.editor;
          this.$toolbar = context.layoutInfo.toolbar;
          this.$editable = context.layoutInfo.editable;
          this.$modal = self.$el.closest('.modal');
          this.isInModal = this.$modal.length > 0;
        };
        this.resizeTo = function (size) {
          this.$editable.css('height', size.h);
          if (context.$aceCodable) {
            context.$aceCodable.css('height', size.h);
          }
        };
        this.onResize = function () {
          this.resizeTo({
            h: this.$window.height() - this.$toolbar.outerHeight()
          });
        };
        this.isFullscreen = function () {
          return this.$editor.hasClass('fullscreen');
        };
        this.destroy = function () {
          this.$window.off('resize.summernote' + self.cid);
          if (this.isInModal) {
            this.$modal.css('overflow-y', '');
          } else {
            this.$scrollbar.css('overflow', '');
          }
        };
        let maxLines;
        this.toggle = function () {
          this.$editor.toggleClass('fullscreen');
          const aceEditor = /** @type {import('ace-builds').Ace.Editor} */context.aceEditor;
          if (this.isFullscreen()) {
            this.$editable.data('orgHeight', this.$editable.css('height'));
            this.$editable.data('orgMaxHeight', this.$editable.css('maxHeight'));
            this.$editable.css('maxHeight', '');
            this.$window.on('resize.summernote' + self.cid, this.onResize.bind(this)).trigger('resize');
            if (this.isInModal) {
              this.$modal.css('overflow-y', 'hidden');
            } else {
              this.$scrollbar.css('overflow', 'hidden');
            }

            // noinspection JSUnusedGlobalSymbols
            this._isFullscreen = true;
            if (aceEditor) {
              maxLines = aceEditor.getOption('maxLines');
              aceEditor.setOptions({
                maxLines: null
              });
              aceEditor.resize();
            }
          } else {
            this.$window.off('resize.summernote' + self.cid);
            this.resizeTo({
              h: this.$editable.data('orgHeight')
            });
            if (context.$aceCodable) {
              context.$aceCodable.css('height', '');
            }
            this.$editable.css('maxHeight', this.$editable.css('orgMaxHeight'));
            this.$editable.css('height', '');
            if (this.isInModal) {
              this.$modal.css('overflow-y', '');
            } else {
              this.$scrollbar.css('overflow', '');
            }

            // noinspection JSUnusedGlobalSymbols
            this._isFullscreen = false;
            if (aceEditor) {
              aceEditor.setOptions({
                maxLines: 100000
              });
              aceEditor.resize();
            }
          }
          context.invoke('toolbar.updateFullscreen', this.isFullscreen());
        };
      }
    });
  }
});

define("search-manager", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module search-manager */

  /**
   * Search data.
   *
   * @typedef {Object} module:search-manager~data
   *
   * @property {string} [presetName] A preset.
   * @property {string} [textFilter] A text filter.
   * @property {string} [primary] A primary filter.
   * @property {Object.<string, boolean>} [bool] Bool filters.
   * @property {Record<module:search-manager~advancedFilter>} [advanced] Advanced filters (field filters).
   * Contains data needed for both the backend and frontend. Keys are field names.
   */

  /**
   * A where item. Sent to the backend.
   *
   * @typedef {Object} module:search-manager~whereItem
   *
   * @property {string} type A type.
   * @property {string} [attribute] An attribute (field).
   * @property {module:search-manager~whereItem[]|string|number|boolean|null} [value] A value.
   * @property {boolean} [dateTime] Is a date-time item.
   * @property {string} [timeZone] A time-zone.
   */

  /**
   * An advanced filter
   *
   * @typedef {Object} module:search-manager~advancedFilter
   *
   * @property {string} type A type. E.g. `equals`.
   * @property {string} [attribute] An attribute.
   * @property {*} [value] A value.
   * @property {Object.<string, *>} [data] Additional data for UI.
   */

  /**
   * A search manager.
   */
  class SearchManager {
    /**
     * @type {string|null}
     * @private
     */
    timeZone;

    /**
     * @param {module:collection} collection A collection.
     * @param {string|null} type A type. Used for a storage key.
     * @param {module:storage|null} storage A storage.
     * @param {module:date-time|null} dateTime A date-time util.
     * @param {module:search-manager~data|null} [defaultData=null] Default search data.
     * @param {boolean} [emptyOnReset=false] To empty on reset.
     */
    constructor(collection, type, storage, dateTime, defaultData, emptyOnReset) {
      /**
       * @private
       * @type {module:collection}
       */
      this.collection = collection;

      /**
       * An entity type.
       *
       * @public
       * @type {string}
       */
      this.scope = collection.entityType;

      /**
       * @private
       * @type {module:storage|null}
       */
      this.storage = storage;

      /**
       * @private
       * @type {string}
       */
      this.type = type || 'list';

      /**
       * @private
       * @type {module:date-time|null}
       */
      this.dateTime = dateTime;

      /**
       * @private
       * @type {boolean}
       */
      this.emptyOnReset = emptyOnReset;

      /**
       * @private
       * @type {Object}
       */
      this.emptyData = {
        textFilter: '',
        bool: {},
        advanced: {},
        primary: null
      };
      if (defaultData) {
        this.defaultData = defaultData;
        for (const p in this.emptyData) {
          if (!(p in defaultData)) {
            defaultData[p] = Espo.Utils.clone(this.emptyData[p]);
          }
        }
      }

      /**
       * @type {module:search-manager~data}
       * @private
       */
      this.data = Espo.Utils.clone(defaultData) || this.emptyData;
      this.sanitizeData();
    }

    /**
     * @private
     */
    sanitizeData() {
      if (!('advanced' in this.data)) {
        this.data.advanced = {};
      }
      if (!('bool' in this.data)) {
        this.data.bool = {};
      }
      if (!('textFilter' in this.data)) {
        this.data.textFilter = '';
      }
    }

    /**
     * Get a where clause. The where clause to be sent to the backend.
     *
     * @returns {module:search-manager~whereItem[]}
     */
    getWhere() {
      const where = [];
      if (this.data.textFilter && this.data.textFilter !== '') {
        where.push({
          type: 'textFilter',
          value: this.data.textFilter
        });
      }
      if (this.data.bool) {
        const o = {
          type: 'bool',
          value: []
        };
        for (const name in this.data.bool) {
          if (this.data.bool[name]) {
            o.value.push(name);
          }
        }
        if (o.value.length) {
          where.push(o);
        }
      }
      if (this.data.primary) {
        const o = {
          type: 'primary',
          value: this.data.primary
        };
        if (o.value.length) {
          where.push(o);
        }
      }
      if (this.data.advanced) {
        for (const name in this.data.advanced) {
          const defs = this.data.advanced[name];
          if (!defs) {
            continue;
          }
          const part = this.getWherePart(name, defs);
          where.push(part);
        }
      }
      return where;
    }

    /**
     * @private
     */
    getWherePart(name, defs) {
      let attribute = name;
      if (typeof defs !== 'object') {
        console.error('Bad where clause');
        return {};
      }
      if ('where' in defs) {
        return defs.where;
      }
      const type = defs.type;
      let value;
      if (type === 'or' || type === 'and') {
        const a = [];
        value = defs.value || {};
        for (const n in value) {
          a.push(this.getWherePart(n, value[n]));
        }
        return {
          type: type,
          value: a
        };
      }
      if ('field' in defs) {
        // for backward compatibility
        attribute = defs.field;
      }
      if ('attribute' in defs) {
        attribute = defs.attribute;
      }
      if (defs.dateTime || defs.date) {
        const timeZone = this.timeZone !== undefined ? this.timeZone : this.dateTime.getTimeZone();
        const data = {
          type: type,
          attribute: attribute,
          value: defs.value
        };
        if (defs.dateTime) {
          data.dateTime = true;
        }
        if (defs.date) {
          data.date = true;
        }
        if (timeZone) {
          data.timeZone = timeZone;
        }
        return data;
      }
      value = defs.value;
      return {
        type: type,
        attribute: attribute,
        value: value
      };
    }

    /**
     * Load stored data.
     *
     * @returns {module:search-manager}
     */
    loadStored() {
      this.data = this.storage.get(this.type + 'Search', this.scope) || Espo.Utils.clone(this.defaultData) || Espo.Utils.clone(this.emptyData);
      this.sanitizeData();
      return this;
    }

    /**
     * Get data.
     *
     * @returns {module:search-manager~data}
     */
    get() {
      return this.data;
    }

    /**
     * Set advanced filters.
     *
     * @param {Object.<string, module:search-manager~advancedFilter>} advanced Advanced filters.
     *   Pairs of field => advancedFilter.
     */
    setAdvanced(advanced) {
      this.data = Espo.Utils.clone(this.data);
      this.data.advanced = advanced;
    }

    /**
     * Set bool filters.
     *
     * @param {Object.<string, boolean>} bool Bool filters.
     */
    setBool(bool) {
      this.data = Espo.Utils.clone(this.data);
      this.data.bool = bool;
    }

    /**
     * Set a primary filter.
     *
     * @param {string} primary A filter.
     */
    setPrimary(primary) {
      this.data = Espo.Utils.clone(this.data);
      this.data.primary = primary;
    }

    /**
     * Set data.
     *
     * @param {module:search-manager~data} data Data.
     */
    set(data) {
      this.data = data;
      if (this.storage) {
        data = Espo.Utils.clone(data);
        delete data['textFilter'];
        this.storage.set(this.type + 'Search', this.scope, data);
      }
    }
    clearPreset() {
      delete this.data.presetName;
    }

    /**
     * Empty data.
     */
    empty() {
      this.data = Espo.Utils.clone(this.emptyData);
      if (this.storage) {
        this.storage.clear(this.type + 'Search', this.scope);
      }
    }

    /**
     * Reset.
     */
    reset() {
      if (this.emptyOnReset) {
        this.empty();
        return;
      }
      this.data = Espo.Utils.clone(this.defaultData) || Espo.Utils.clone(this.emptyData);
      if (this.storage) {
        this.storage.clear(this.type + 'Search', this.scope);
      }
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Set a time zone. Null will not add a time zone.
     *
     * @type {string|null}
     * @internal Is used. Do not remove.
     */
    setTimeZone(timeZone) {
      this.timeZone = timeZone;
    }
  }
  var _default = _exports.default = SearchManager;
});

define("controller", ["exports", "exceptions", "bullbone", "jquery"], function (_exports, _exceptions, _bullbone, _jquery) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _exceptions = _interopRequireDefault(_exceptions);
  _jquery = _interopRequireDefault(_jquery);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module controller */

  /**
   * @callback module:controller~viewCallback
   * @param {module:view} view A view.
   */

  /**
   * @callback module:controller~masterViewCallback
   * @param {module:views/site/master} view A master view.
   */

  /**
   * A controller. To be extended.
   *
   * @mixes Bull.Events
   */
  class Controller {
    /**
     * @internal
     * @param {Object.<string, *>} params
     * @param {Object} injections
     */
    constructor(params, injections) {
      this.params = params || {};

      /** @type {module:controllers/base} */
      this.baseController = injections.baseController;
      /** @type {Bull.Factory} */
      this.viewFactory = injections.viewFactory;
      /** @type {module:model} */
      this.modelFactory = injections.modelFactory;
      /** @type {module:collection-factory} */
      this.collectionFactory = injections.collectionFactory;
      this._settings = injections.settings || null;
      this._user = injections.user || null;
      this._preferences = injections.preferences || null;
      this._acl = injections.acl || null;
      this._cache = injections.cache || null;
      this._router = injections.router || null;
      this._storage = injections.storage || null;
      this._metadata = injections.metadata || null;
      this._dateTime = injections.dateTime || null;
      this._broadcastChannel = injections.broadcastChannel || null;
      this.set('masterRendered', false);
    }

    /**
     * A default action.
     *
     * @type {string}
     */
    defaultAction = 'index';

    /**
     * A name.
     *
     * @type {string|null}
     */
    name = null;

    /**
     * Params.
     *
     * @type {Object}
     * @private
     */
    params = null;

    /**
     * A view factory.
     *
     * @type {Bull.Factory}
     * @protected
     */
    viewFactory = null;

    /**
     * A model factory.
     *
     * @type {module:model-factory}
     * @protected
     */
    modelFactory = null;

    /**
     * A body view.
     *
     * @public
     * @type {string|null}
     */
    masterView = null;

    /**
     * Set the router.
     *
     * @internal
     * @param {module:router} router
     */
    setRouter(router) {
      this._router = router;
      this.trigger('router-set', router);
    }

    /**
     * @protected
     * @returns {module:models/settings}
     */
    getConfig() {
      return this._settings;
    }

    /**
     * @protected
     * @returns {module:models/user}
     */
    getUser() {
      return this._user;
    }

    /**
     * @protected
     * @returns {module:models/preferences}
     */
    getPreferences() {
      return this._preferences;
    }

    /**
     * @protected
     * @returns {module:acl-manager}
     */
    getAcl() {
      return this._acl;
    }

    /**
     * @protected
     * @returns {module:cache}
     */
    getCache() {
      return this._cache;
    }

    /**
     * @protected
     * @returns {module:router}
     */
    getRouter() {
      return this._router;
    }

    /**
     * @protected
     * @returns {module:storage}
     */
    getStorage() {
      return this._storage;
    }

    /**
     * @protected
     * @returns {module:metadata}
     */
    getMetadata() {
      return this._metadata;
    }

    /**
     * @protected
     * @returns {module:date-time}
     */
    getDateTime() {
      return this._dateTime;
    }

    /**
     * Get a parameter of all controllers.
     *
     * @param {string} key A key.
     * @return {*} Null if a key doesn't exist.
     */
    get(key) {
      if (key in this.params) {
        return this.params[key];
      }
      return null;
    }

    /**
     * Set a parameter for all controllers.
     *
     * @param {string} key A name of a view.
     * @param {*} value
     */
    set(key, value) {
      this.params[key] = value;
    }

    /**
     * Unset a parameter.
     *
     * @param {string} key A key.
     */
    unset(key) {
      delete this.params[key];
    }

    /**
     * Has a parameter.
     *
     * @param {string} key A key.
     * @returns {boolean}
     */
    has(key) {
      return key in this.params;
    }

    /**
     * @param {string} key
     * @param {string} [name]
     * @return {string}
     * @private
     */
    _composeScrollKey(key, name) {
      name = name || this.name;
      return `scrollTop-${name}-${key}`;
    }

    /**
     * @param {string} key
     * @return {string}
     * @private
     */
    _composeMainViewKey(key) {
      return `mainView-${this.name}-${key}`;
    }

    /**
     * Get a stored main view.
     *
     * @param {string} key A key.
     * @returns {module:view|null}
     */
    getStoredMainView(key) {
      return this.get(this._composeMainViewKey(key));
    }

    /**
     * Has a stored main view.
     * @param {string} key
     * @returns {boolean}
     */
    hasStoredMainView(key) {
      return this.has(this._composeMainViewKey(key));
    }

    /**
     * Clear a stored main view.
     * @param {string} key
     */
    clearStoredMainView(key) {
      const view = this.getStoredMainView(key);
      if (view) {
        view.remove(true);
      }
      this.unset(this._composeScrollKey(key));
      this.unset(this._composeMainViewKey(key));
    }

    /**
     * Store a main view.
     *
     * @param {string} key A key.
     * @param {module:view} view A view.
     */
    storeMainView(key, view) {
      this.set(this._composeMainViewKey(key), view);
      this.listenTo(view, 'remove', o => {
        o = o || {};
        if (o.ignoreCleaning) {
          return;
        }
        this.stopListening(view, 'remove');
        this.clearStoredMainView(key);
      });
    }

    /**
     * Check access to an action.
     *
     * @param {string} action An action.
     * @returns {boolean}
     */
    checkAccess(action) {
      return true;
    }

    /**
     * Process access check to the controller.
     */
    handleAccessGlobal() {
      if (!this.checkAccessGlobal()) {
        throw new _exceptions.default.AccessDenied("Denied access to '" + this.name + "'");
      }
    }

    /**
     * Check access to the controller.
     *
     * @returns {boolean}
     */
    checkAccessGlobal() {
      return true;
    }

    /**
     * Check access to an action. Throwing an exception.
     *
     * @param {string} action An action.
     */
    handleCheckAccess(action) {
      if (this.checkAccess(action)) {
        return;
      }
      const msg = action ? "Denied access to action '" + this.name + "#" + action + "'" : "Denied access to scope '" + this.name + "'";
      throw new _exceptions.default.AccessDenied(msg);
    }

    /**
     * Process an action.
     *
     * @param {string} action
     * @param {Object} options
     */
    doAction(action, options) {
      this.handleAccessGlobal();
      action = action || this.defaultAction;
      const method = 'action' + Espo.Utils.upperCaseFirst(action);
      if (!(method in this)) {
        throw new _exceptions.default.NotFound("Action '" + this.name + "#" + action + "' is not found");
      }
      const preMethod = 'before' + Espo.Utils.upperCaseFirst(action);
      const postMethod = 'after' + Espo.Utils.upperCaseFirst(action);
      if (preMethod in this) {
        this[preMethod].call(this, options || {});
      }
      this[method].call(this, options || {});
      if (postMethod in this) {
        this[postMethod].call(this, options || {});
      }
    }

    /**
     * Serve a master view. Render if not already rendered.
     *
     * @param {module:controller~masterViewCallback} callback A callback with a created master view.
     * @private
     */
    master(callback) {
      const entire = this.get('entire');
      if (entire) {
        entire.remove();
        this.set('entire', null);
      }
      const master = this.get('master');
      if (master) {
        callback.call(this, master);
        return;
      }
      const masterView = this.masterView || 'views/site/master';
      this.viewFactory.create(masterView, {
        fullSelector: 'body'
      }, /** module:view */master => {
        this.set('master', master);
        if (this.get('masterRendered')) {
          callback.call(this, master);
          return;
        }
        master.render().then(() => {
          this.set('masterRendered', true);
          callback.call(this, master);
        });
      });
    }

    /**
     * @param {import('views/site/master').default} masterView
     * @private
     */
    _unchainMainView(masterView) {
      if (!masterView.currentViewKey /*||
                                     !this.hasStoredMainView(masterView.currentViewKey)*/) {
        return;
      }
      const currentMainView = masterView.getMainView();
      if (!currentMainView) {
        return;
      }
      currentMainView.propagateEvent('remove', {
        ignoreCleaning: true
      });
      masterView.unchainView('main');
    }

    /**
     * @typedef {Object} module:controller~mainParams
     * @property {boolean} [useStored] Use a stored view if available.
     * @property {string} [key] A stored view key.
     */

    /**
     * Create a main view in the master container and render it.
     *
     * @param {string|module:view} [view] A view name or view instance.
     * @param {Object.<string, *>} [options] Options for a view.
     * @param {module:controller~viewCallback} [callback] A callback with a created view.
     * @param {module:controller~mainParams} [params] Parameters.
     */
    main(view, options, callback, params = {}) {
      const dto = {
        isCanceled: false,
        key: params.key,
        useStored: params.useStored,
        callback: callback
      };
      const selector = '#main';
      const useStored = params.useStored || false;
      const key = params.key;
      this.listenToOnce(this.baseController, 'action', () => dto.isCanceled = true);
      const mainView = view && typeof view === 'object' ? view : undefined;
      const viewName = !mainView ? view || 'views/base' : undefined;
      this.master(masterView => {
        if (dto.isCanceled) {
          return;
        }
        options = options || {};
        options.fullSelector = selector;
        if (useStored && this.hasStoredMainView(key)) {
          const mainView = this.getStoredMainView(key);
          let isActual = true;
          if (mainView && 'isActualForReuse' in mainView && typeof mainView.isActualForReuse === 'function') {
            isActual = mainView.isActualForReuse();
          }
          const lastUrl = mainView && 'lastUrl' in mainView ? mainView.lastUrl : null;
          if (isActual && (!lastUrl || lastUrl === this.getRouter().getCurrentUrl())) {
            this._processMain(mainView, masterView, dto);
            if ('setupReuse' in mainView && typeof mainView.setupReuse === 'function') {
              mainView.setupReuse(options.params || {});
            }
            return;
          }
          this.clearStoredMainView(key);
        }
        if (mainView) {
          this._unchainMainView(masterView);
          masterView.assignView('main', mainView, selector).then(() => {
            dto.isSet = true;
            this._processMain(view, masterView, dto);
          });
          return;
        }
        this.viewFactory.create(viewName, options, view => {
          this._processMain(view, masterView, dto);
        });
      });
    }

    /**
     * @param {import('view').default} mainView
     * @param {import('views/site/master').default} masterView
     * @param {{
     *     isCanceled: boolean,
     *     key?: string,
     *     useStored?: boolean,
     *     callback?: module:controller~viewCallback,
     *     isSet?: boolean,
     * }} dto Data.
     * @private
     */
    _processMain(mainView, masterView, dto) {
      if (dto.isCanceled) {
        return;
      }
      const key = dto.key;
      if (key) {
        this.storeMainView(key, mainView);
      }
      const onAction = () => {
        mainView.cancelRender();
        dto.isCanceled = true;
      };
      mainView.listenToOnce(this.baseController, 'action', onAction);
      if (masterView.currentViewKey) {
        const scrollKey = this._composeScrollKey(masterView.currentViewKey, masterView.currentName);
        this.set(scrollKey, (0, _jquery.default)(window).scrollTop());
        if (!dto.isSet) {
          this._unchainMainView(masterView);
        }
      }
      masterView.currentViewKey = key;
      masterView.currentName = this.name;
      if (!dto.isSet) {
        masterView.setView('main', mainView);
      }
      const afterRender = () => {
        setTimeout(() => mainView.stopListening(this.baseController, 'action', onAction), 500);
        mainView.updatePageTitle();
        const scrollKey = this._composeScrollKey(key);
        if (dto.useStored && this.has(scrollKey)) {
          (0, _jquery.default)(window).scrollTop(this.get(scrollKey));
          return;
        }
        (0, _jquery.default)(window).scrollTop(0);
      };
      if (dto.callback) {
        this.listenToOnce(mainView, 'after:render', afterRender);
        dto.callback.call(this, mainView);
        return;
      }
      mainView.render().then(afterRender);
    }

    /**
     * Show a loading notify-message.
     */
    showLoadingNotification() {
      const master = this.get('master');
      if (!master) {
        return;
      }
      master.showLoadingNotification();
    }

    /**
     * Hide a loading notify-message.
     */
    hideLoadingNotification() {
      const master = this.get('master');
      if (!master) {
        return;
      }
      master.hideLoadingNotification();
    }

    /**
     * Create a view in the BODY element. Use for rendering separate pages without the default navbar and footer.
     * If a callback is not passed, the view will be automatically rendered.
     *
     * @param {string|module:view} view A view name or view instance.
     * @param {Object.<string, *>} [options] Options for a view.
     * @param {module:controller~viewCallback} [callback] A callback with a created view.
     */
    entire(view, options, callback) {
      const masterView = this.get('master');
      if (masterView) {
        masterView.remove();
      }
      this.set('master', null);
      this.set('masterRendered', false);
      if (typeof view === 'object') {
        view.setElement('body');
        this.viewFactory.prepare(view, () => {
          if (!callback) {
            view.render();
            return;
          }
          callback(view);
        });
        return;
      }
      options = options || {};
      options.fullSelector = 'body';
      this.viewFactory.create(view, options, view => {
        this.set('entire', view);
        if (!callback) {
          view.render();
          return;
        }
        callback(view);
      });
    }
  }
  Object.assign(Controller.prototype, _bullbone.Events);

  /** For backward compatibility. */
  Controller.extend = _bullbone.View.extend;
  var _default = _exports.default = Controller;
});

define("views/main", ["exports", "view"], function (_exports, _view) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/main */

  /**
   * A base main view. The detail, edit, list views to be extended from.
   */
  class MainView extends _view.default {
    /**
     * A scope name.
     *
     * @type {string} scope
     */
    scope = '';

    /**
     * A name.
     *
     * @type {string} name
     */
    name = '';

    /**
     * A top-right menu item (button or dropdown action).
     * Handled by a class method `action{Action}`, a click handler or a handler class.
     *
     * @typedef {Object} module:views/main~MenuItem
     *
     * @property {string} [name] A name.
     * @property {string} [action] An action.
     * @property {string} [link] A link.
     * @property {string} [label] A translatable label.
     * @property {string} [labelTranslation] A label translation path.
     * @property {'default'|'danger'|'success'|'warning'} [style] A style. Only for buttons.
     * @property {boolean} [hidden] Hidden.
     * @property {boolean} [disabled] Disabled.
     * @property {Object.<string,string|number|boolean>} [data] Data attribute values.
     * @property {string} [title] A title.
     * @property {string} [iconHtml] An icon HTML.
     * @property {string} [iconClass] An icon class.
     * @property {string} [html] An HTML.
     * @property {string} [text] A text.
     * @property {string} [className] An additional class name. Only for buttons.
     * @property {'create'|'read'|'edit'|'stream'|'delete'} [acl] Access to a record (or a scope if `aclScope` specified)
     *   required for a menu item.
     * @property {string} [aclScope] A scope to check access to with the `acl` parameter.
     * @property {string} [configCheck] A config parameter defining a menu item availability.
     *   If starts with `!`, then the result is negated.
     * @property {module:utils~AccessDefs[]} [accessDataList] Access definitions.
     * @property {string} [handler] A handler.
     * @property {string} [initFunction] An init method in the handler.
     * @property {string} [actionFunction] An action method in the handler.
     * @property {string} [checkVisibilityFunction] A method in the handler that determine whether an item is available.
     * @property {function()} [onClick] A click handler.
     */

    /**
     * Top-right menu definitions.
     *
     * @type {{
     *     buttons: module:views/main~MenuItem[],
     *     dropdown: module:views/main~MenuItem[],
     *     actions: module:views/main~MenuItem[],
     * }} menu
     * @private
     * @internal
     */
    menu = {};

    /**
     * @private
     * @type {JQuery|null}
     */
    $headerActionsContainer = null;

    /**
     * A shortcut-key => action map.
     *
     * @protected
     * @type {?Object.<string, string|function (KeyboardEvent): void>}
     */
    shortcutKeys = null;

    /** @inheritDoc */
    events = {
      /** @this MainView */
      'click .action': function (e) {
        Espo.Utils.handleAction(this, e.originalEvent, e.currentTarget, {
          actionItems: [...this.menu.buttons, ...this.menu.dropdown],
          className: 'main-header-manu-action'
        });
      }
    };
    lastUrl;

    /** @inheritDoc */
    init() {
      this.scope = this.options.scope || this.scope;
      this.menu = {};
      this.options.params = this.options.params || {};
      if (this.name && this.scope) {
        const key = this.name.charAt(0).toLowerCase() + this.name.slice(1);
        this.menu = this.getMetadata().get(['clientDefs', this.scope, 'menu', key]) || {};
      }

      /**
       * @private
       * @type {string[]}
       */
      this.headerActionItemTypeList = ['buttons', 'dropdown', 'actions'];
      this.menu = Espo.Utils.cloneDeep(this.menu);
      let globalMenu = {};
      if (this.name) {
        globalMenu = Espo.Utils.cloneDeep(this.getMetadata().get(['clientDefs', 'Global', 'menu', this.name.charAt(0).toLowerCase() + this.name.slice(1)]) || {});
      }
      this._reRenderHeaderOnSync = false;
      this._menuHandlers = {};
      this.headerActionItemTypeList.forEach(type => {
        this.menu[type] = this.menu[type] || [];
        this.menu[type] = this.menu[type].concat(globalMenu[type] || []);
        const itemList = this.menu[type];
        itemList.forEach(item => {
          const viewObject = this;

          // @todo Set _reRenderHeaderOnSync to true if `acl` is set `ascScope` is not set?
          //     Set _reRenderHeaderOnSync in `addMenuItem` method.

          if ((item.initFunction || item.checkVisibilityFunction) && (item.handler || item.data && item.data.handler)) {
            this.wait(new Promise(resolve => {
              const handler = item.handler || item.data.handler;
              Espo.loader.require(handler, Handler => {
                const handler = new Handler(viewObject);
                const name = item.name || item.action;
                if (name) {
                  this._menuHandlers[name] = handler;
                }
                if (item.initFunction) {
                  handler[item.initFunction].call(handler);
                }
                if (item.checkVisibilityFunction && this.model) {
                  this._reRenderHeaderOnSync = true;
                }
                resolve();
              });
            }));
          }
        });
      });
      if (this.model) {
        this.whenReady().then(() => {
          if (!this._reRenderHeaderOnSync) {
            return;
          }
          this.listenTo(this.model, 'sync', () => {
            if (!this.getHeaderView()) {
              return;
            }
            this.getHeaderView().reRender();
          });
        });
      }
      this.updateLastUrl();
      this.on('after:render-internal', () => {
        this.$headerActionsContainer = this.$el.find('.page-header .header-buttons');
      });
      this.on('header-rendered', () => {
        this.$headerActionsContainer = this.$el.find('.page-header .header-buttons');
        this.adjustButtons();
      });
      this.on('after:render', () => this.adjustButtons());
      if (this.shortcutKeys) {
        this.shortcutKeys = Espo.Utils.cloneDeep(this.shortcutKeys);
      }
    }
    setupFinal() {
      if (this.shortcutKeys) {
        this.events['keydown.main'] = e => {
          const key = Espo.Utils.getKeyFromKeyEvent(e);
          if (typeof this.shortcutKeys[key] === 'function') {
            this.shortcutKeys[key].call(this, e.originalEvent);
            return;
          }
          const actionName = this.shortcutKeys[key];
          if (!actionName) {
            return;
          }
          e.preventDefault();
          e.stopPropagation();
          const methodName = 'action' + Espo.Utils.upperCaseFirst(actionName);
          if (typeof this[methodName] === 'function') {
            this[methodName]();
            return;
          }
          this[actionName]();
        };
      }
    }

    /**
     * Update a last history URL.
     */
    updateLastUrl() {
      this.lastUrl = this.getRouter().getCurrentUrl();
    }

    /**
     * @internal
     * @returns {{
     *     buttons?: module:views/main~MenuItem[],
     *     dropdown?: module:views/main~MenuItem[],
     *     actions?: module:views/main~MenuItem[],
     * }}
     */
    getMenu() {
      if (this.menuDisabled || !this.menu) {
        return {};
      }
      const menu = {};
      this.headerActionItemTypeList.forEach(type => {
        (this.menu[type] || []).forEach(item => {
          if (item === false) {
            menu[type].push(false);
            return;
          }
          item = Espo.Utils.clone(item);
          menu[type] = menu[type] || [];
          if (!Espo.Utils.checkActionAvailability(this.getHelper(), item)) {
            return;
          }
          if (!Espo.Utils.checkActionAccess(this.getAcl(), this.model || this.scope, item)) {
            return;
          }
          if (item.accessDataList) {
            if (!Espo.Utils.checkAccessDataList(item.accessDataList, this.getAcl(), this.getUser())) {
              return;
            }
          }
          item.name = item.name || item.action;
          item.action = item.action || null;
          if (this._menuHandlers[item.name] && item.checkVisibilityFunction) {
            const handler = this._menuHandlers[item.name];
            if (!handler[item.checkVisibilityFunction](item.name)) {
              return;
            }
          }
          if (item.labelTranslation) {
            item.html = this.getHelper().escapeString(this.getLanguage().translatePath(item.labelTranslation));
          }
          menu[type].push(item);
        });
      });
      return menu;
    }

    /**
     * Get a header HTML. To be overridden.
     *
     * @returns {string} HTML.
     */
    getHeader() {
      return '';
    }

    /**
     * Build a header HTML. To be called from the #getHeader method.
     * Beware of XSS.
     *
     * @param {(string|Element|JQuery)[]} itemList A breadcrumb path. Like: Account > Name > edit.
     * @returns {string} HTML
     */
    buildHeaderHtml(itemList) {
      const $itemList = itemList.map(item => {
        return $('<div>').addClass('breadcrumb-item').append(item);
      });
      const $div = $('<div>').addClass('header-breadcrumbs');
      $itemList.forEach(($item, i) => {
        $div.append($item);
        if (i === $itemList.length - 1) {
          return;
        }
        $div.append($('<div>').addClass('breadcrumb-separator').append($('<span>')));
      });
      return $div.get(0).outerHTML;
    }

    /**
     * Get an icon HTML.
     *
     * @returns {string} HTML
     */
    getHeaderIconHtml() {
      return this.getHelper().getScopeColorIconHtml(this.scope);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Action 'showModal'.
     *
     * @todo Revise. To be removed?
     *
     * @param {Object} data
     */
    actionShowModal(data) {
      const view = data.view;
      if (!view) {
        return;
      }
      this.createView('modal', view, {
        model: this.model,
        collection: this.collection
      }, view => {
        view.render();
        this.listenTo(view, 'after:save', () => {
          if (this.model) {
            this.model.fetch();
          }
          if (this.collection) {
            this.collection.fetch();
          }
        });
      });
    }

    /**
     * Update a menu item.
     *
     * @param {string} name An item name.
     * @param {module:views/main~MenuItem} item New item definitions to write.
     * @param {boolean} [doNotReRender=false] Skip re-render.
     *
     * @since 8.2.0
     */
    updateMenuItem(name, item, doNotReRender) {
      const actionItem = this._getHeaderActionItem(name);
      if (!actionItem) {
        return;
      }
      for (const key in item) {
        actionItem[key] = item[key];
      }
      if (doNotReRender) {
        return;
      }
      if (this.isRendered()) {
        this.getHeaderView().reRender();
        return;
      }
      if (this.isBeingRendered()) {
        this.whenRendered().then(() => {
          this.getHeaderView().reRender();
        });
      }
    }

    /**
     * Add a menu item.
     *
     * @param {'buttons'|'dropdown'} type A type.
     * @param {module:views/main~MenuItem|false} item Item definitions.
     * @param {boolean} [toBeginning=false] To beginning.
     * @param {boolean} [doNotReRender=false] Skip re-render.
     */
    addMenuItem(type, item, toBeginning, doNotReRender) {
      if (item) {
        item.name = item.name || item.action || Espo.Utils.generateId();
        const name = item.name;
        let index = -1;
        this.menu[type].forEach((data, i) => {
          data = data || {};
          if (data.name === name) {
            index = i;
          }
        });
        if (~index) {
          this.menu[type].splice(index, 1);
        }
      }
      let method = 'push';
      if (toBeginning) {
        method = 'unshift';
      }
      this.menu[type][method](item);
      if (!doNotReRender && this.isRendered()) {
        this.getHeaderView().reRender();
        return;
      }
      if (!doNotReRender && this.isBeingRendered()) {
        this.once('after:render', () => {
          this.getHeaderView().reRender();
        });
      }
    }

    /**
     * Remove a menu item.
     *
     * @param {string} name An item name.
     * @param {boolean} [doNotReRender] Skip re-render.
     */
    removeMenuItem(name, doNotReRender) {
      let index = -1;
      let type = false;
      this.headerActionItemTypeList.forEach(t => {
        (this.menu[t] || []).forEach((item, i) => {
          item = item || {};
          if (item.name === name) {
            index = i;
            type = t;
          }
        });
      });
      if (~index && type) {
        this.menu[type].splice(index, 1);
      }
      if (!doNotReRender && this.isRendered()) {
        this.getHeaderView().reRender();
        return;
      }
      if (!doNotReRender && this.isBeingRendered()) {
        this.once('after:render', () => {
          this.getHeaderView().reRender();
        });
        return;
      }
      if (doNotReRender && this.isRendered()) {
        this.$headerActionsContainer.find('[data-name="' + name + '"]').remove();
      }
    }

    /**
     * Disable a menu item.
     *
     * @param {string} name A name.
     */
    disableMenuItem(name) {
      const item = this._getHeaderActionItem(name);
      if (item) {
        item.disabled = true;
      }
      const process = () => {
        this.$headerActionsContainer.find(`[data-name="${name}"]`).addClass('disabled').attr('disabled');
      };
      if (this.isBeingRendered()) {
        this.whenRendered().then(() => process());
        return;
      }
      if (!this.isRendered()) {
        return;
      }
      process();
    }

    /**
     * Enable a menu item.
     *
     * @param {string} name A name.
     */
    enableMenuItem(name) {
      const item = this._getHeaderActionItem(name);
      if (item) {
        item.disabled = false;
      }
      const process = () => {
        this.$headerActionsContainer.find(`[data-name="${name}"]`).removeClass('disabled').removeAttr('disabled');
      };
      if (this.isBeingRendered()) {
        this.whenRendered().then(() => process());
        return;
      }
      if (!this.isRendered()) {
        return;
      }
      process();
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Action 'navigateToRoot'.
     *
     * @param {Object} data
     * @param {MouseEvent} event
     */
    actionNavigateToRoot(data, event) {
      event.stopPropagation();
      this.getRouter().checkConfirmLeaveOut(() => {
        const rootUrl = this.options.rootUrl || this.options.params.rootUrl || '#' + this.scope;
        this.getRouter().navigate(rootUrl, {
          trigger: true,
          isReturn: true
        });
      });
    }

    /**
     * @private
     * @param {string} name
     * @return {module:views/main~MenuItem|undefined}
     */
    _getHeaderActionItem(name) {
      for (const type of this.headerActionItemTypeList) {
        if (!this.menu[type]) {
          continue;
        }
        for (const item of this.menu[type]) {
          if (item && item.name === name) {
            return item;
          }
        }
      }
      return undefined;
    }

    /**
     * Hide a menu item.
     *
     * @param {string} name A name.
     */
    hideHeaderActionItem(name) {
      const item = this._getHeaderActionItem(name);
      if (item) {
        item.hidden = true;
      }
      if (!this.isRendered()) {
        return;
      }
      this.$headerActionsContainer.find(`li > .action[data-name="${name}"]`).parent().addClass('hidden');
      this.$headerActionsContainer.find(`a.action[data-name="${name}"]`).addClass('hidden');
      this.controlMenuDropdownVisibility();
      this.adjustButtons();
      if (this.getHeaderView()) {
        this.getHeaderView().trigger('action-item-update');
      }
    }

    /**
     * Show a hidden menu item.
     *
     * @param {string} name A name.
     */
    showHeaderActionItem(name) {
      const item = this._getHeaderActionItem(name);
      if (item) {
        item.hidden = false;
      }
      const processUi = () => {
        const $dropdownItem = this.$headerActionsContainer.find(`li > .action[data-name="${name}"]`).parent();
        const $button = this.$headerActionsContainer.find(`a.action[data-name="${name}"]`);

        // Item can be available but not rendered as it was skipped by access check in getMenu.
        if (item && !$dropdownItem.length && !$button.length) {
          if (this.getHeaderView()) {
            this.getHeaderView().reRender();
          }
          return;
        }
        $dropdownItem.removeClass('hidden');
        $button.removeClass('hidden');
        this.controlMenuDropdownVisibility();
        this.adjustButtons();
        if (this.getHeaderView()) {
          this.getHeaderView().trigger('action-item-update');
        }
      };
      if (!this.isRendered()) {
        if (this.isBeingRendered()) {
          this.whenRendered().then(() => processUi());
        }
        return;
      }
      processUi();
    }

    /**
     * Whether a menu has any non-hidden dropdown items.
     *
     * @private
     * @returns {boolean}
     */
    hasMenuVisibleDropdownItems() {
      let hasItems = false;
      (this.menu.dropdown || []).forEach(item => {
        if (!item.hidden) {
          hasItems = true;
        }
      });
      return hasItems;
    }

    /**
     * @private
     */
    controlMenuDropdownVisibility() {
      const $group = this.$headerActionsContainer.find('.dropdown-group');
      if (this.hasMenuVisibleDropdownItems()) {
        $group.removeClass('hidden');
        $group.find('> button').removeClass('hidden');
        return;
      }
      $group.addClass('hidden');
      $group.find('> button').addClass('hidden');
    }

    /**
     * @protected
     * @return {module:views/header}
     */
    getHeaderView() {
      return this.getView('header');
    }

    /**
     * @private
     */
    adjustButtons() {
      const $buttons = this.$headerActionsContainer.find('.btn');
      $buttons.removeClass('radius-left').removeClass('radius-right');
      const $buttonsVisible = $buttons.filter(':not(.hidden)');
      $buttonsVisible.first().addClass('radius-left');
      $buttonsVisible.last().addClass('radius-right');
    }

    /**
     * Called when a stored view is reused (by the controller).
     *
     * @public
     * @param {Object.<string, *>} params Routing params.
     */
    setupReuse(params) {}
  }
  var _default = _exports.default = MainView;
});

define("views/base", ["exports", "view"], function (_exports, _view) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/base */

  class BaseView extends _view.default {
    /**
     * @typedef {Object} module:views/base~options
     * @property {string} [template] A template.
     */

    /**
     * @param {module:views/base~options & Object.<string, *>} [options] Options.
     */
    constructor(options) {
      super(options);
    }
  }
  var _default = _exports.default = BaseView;
});

define("views/record/panels-container", ["exports", "view"], function (_exports, _view) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/record/panels-container */

  /**
   * A panel container view. For bottom and side views.
   */
  class PanelsContainerRecordView extends _view.default {
    /** @private */
    panelSoftLockedTypeList = ['default', 'acl', 'delimiter', 'dynamicLogic'];

    /**
     * A panel.
     *
     * @typedef {Object} module:views/record/panels-container~panel
     *
     * @property {string} name A name.
     * @property {boolean} [hidden] Hidden.
     * @property {string} [label] A label.
     * @property {'default'|'success'|'danger'|'warning'} [style] A style.
     * @property {string} [titleHtml] A title HTML.
     * @property {boolean} [notRefreshable] Not refreshable.
     * @property {boolean} [isForm] If for a form.
     * @property {module:views/record/panels-container~button[]} [buttonList] Buttons.
     * @property {module:views/record/panels-container~action[]} [actionList] Dropdown actions.
     * @property {string} [view] A view name.
     * @property {Object.<string, *>} [Options] A view options.
     * @property {boolean} [sticked] To stick to an upper panel.
     * @property {Number} [tabNumber] A tab number.
     * @property {string} [aclScope] A scope to check access to.
     * @property {Espo.Utils~AccessDefs[]} [accessDataList] Access control defs.
     */

    /**
     * A button. Handled by an `action{Action}` method or a click handler.
     *
     * @typedef {Object} module:views/record/panels-container~button
     *
     * @property {string} [action] An action.
     * @property {string} [name] A name. Required if a handler is used.
     * @property {boolean} [hidden] Hidden.
     * @property {string} [label] A label. Translatable.
     * @property {string} [html] A HTML.
     * @property {string} [text] A text.
     * @property {string} [title] A title (on hover). Translatable.
     * @property {Object.<string, (string|number|boolean)>} [data] Data attributes.
     * @property {string} [handler] A handler.
     * @property {string} [actionFunction] An action function.
     * @property {function()} [onClick] A click event.
     */

    /**
     * An action. Handled by an `action{Action}` method or a click handler.
     *
     * @typedef {Object} module:views/record/panels-container~action
     *
     * @property {string} [action] An action.
     * @property {string} [name] A name. Required if a handler is used.
     * @property {string} [link] A link URL.
     * @property {boolean} [hidden] Hidden.
     * @property {string} [label] A label. Translatable.
     * @property {string} [html] A HTML.
     * @property {string} [text] A text.
     * @property {Object.<string, (string|number|boolean)>} [data] Data attributes.
     * @property {string} [handler] A handler.
     * @property {string} [actionFunction] An action function.
     * @property {function()} [onClick] A click event.
     */

    /**
     * A panel list.
     *
     * @protected
     * @type {module:views/record/panels-container~panel[]}
     */
    panelList = null;

    /** @private */
    hasTabs = false;

    /**
     * @private
     * @type {Object.<string,*>[]|null}
     */
    tabDataList = null;

    /**
     * @protected
     */
    currentTab = 0;

    /**
     * @protected
     * @type {string}
     */
    scope = '';

    /**
     * @protected
     * @type {string}
     */
    entityType = '';

    /**
     * @protected
     * @type {string}
     */
    name = '';

    /**
     * A mode.
     *
     * @type 'detail'|'edit'
     */
    mode = 'detail';
    data() {
      const tabDataList = this.hasTabs ? this.getTabDataList() : [];
      return {
        panelList: this.panelList,
        scope: this.scope,
        entityType: this.entityType,
        tabDataList: tabDataList
      };
    }
    events = {
      'click .action': function (e) {
        const $target = $(e.currentTarget);
        const panel = $target.data('panel');
        if (!panel) {
          return;
        }
        const panelView = this.getView(panel);
        if (!panelView) {
          return;
        }
        let actionItems;
        if (typeof panelView.getButtonList === 'function' && typeof panelView.getActionList === 'function') {
          actionItems = [...panelView.getButtonList(), ...panelView.getActionList()];
        }
        Espo.Utils.handleAction(panelView, e.originalEvent, e.currentTarget, {
          actionItems: actionItems,
          className: 'panel-action'
        });

        // @todo Check data. Maybe pass cloned data with unset params.

        /*
        let action = $target.data('action');
        let data = $target.data();
          if (action && panel) {
            let method = 'action' + Espo.Utils.upperCaseFirst(action);
            let d = _.clone(data);
              delete d['action'];
            delete d['panel'];
              let view = this.getView(panel);
              if (view && typeof view[method] == 'function') {
                view[method].call(view, d, e);
            }
        }*/
      },
      'click .panels-show-more-delimiter [data-action="showMorePanels"]': 'actionShowMorePanels',
      /** @this module:views/record/panels-container */
      'click .tabs > button': function (e) {
        const tab = parseInt($(e.currentTarget).attr('data-tab'));
        this.selectTab(tab);
      }
    };
    afterRender() {
      this.adjustPanels();
    }
    adjustPanels() {
      if (!this.isRendered()) {
        return;
      }
      const $panels = this.$el.find('> .panel');
      $panels.removeClass('first').removeClass('last').removeClass('in-middle');
      const $visiblePanels = $panels.filter(`:not(.tab-hidden):not(.hidden)`);
      const groups = [];
      let currentGroup = [];
      let inTab = false;
      $visiblePanels.each((i, el) => {
        const $el = $(el);
        let breakGroup = false;
        if (!breakGroup && this.hasTabs && !inTab && $el.attr('data-tab') !== '-1') {
          inTab = true;
          breakGroup = true;
        }
        if (!breakGroup && !$el.hasClass('sticked')) {
          breakGroup = true;
        }
        if (breakGroup) {
          if (i !== 0) {
            groups.push(currentGroup);
          }
          currentGroup = [];
        }
        currentGroup.push($el);
        if (i === $visiblePanels.length - 1) {
          groups.push(currentGroup);
        }
      });
      groups.forEach(group => {
        group.forEach(($el, i) => {
          if (i === group.length - 1) {
            if (i === 0) {
              return;
            }
            $el.addClass('last');
            return;
          }
          if (i === 0 && group.length) {
            $el.addClass('first');
            return;
          }
          $el.addClass('in-middle');
        });
      });
    }

    /**
     * Set read-only.
     */
    setReadOnly() {
      this.readOnly = true;
    }

    /**
     * Set not read-only.
     */
    setNotReadOnly(onlyNotSetAsReadOnly) {
      this.readOnly = false;
      if (!onlyNotSetAsReadOnly) {
        return;
      }
      this.panelList.forEach(item => {
        this.applyAccessToActions(item.buttonList);
        this.applyAccessToActions(item.actionList);
        this.whenRendered().then(() => {
          if (this.getPanelActionsView(item.name)) {
            this.getPanelActionsView(item.name).reRender();
          }
        });
      });
    }

    /**
     * @private
     * @param {Object[]} actionList
     */
    applyAccessToActions(actionList) {
      if (!actionList) {
        return;
      }
      actionList.forEach(item => {
        if (!Espo.Utils.checkActionAvailability(this.getHelper(), item)) {
          item.hidden = true;
          return;
        }
        const access = Espo.Utils.checkActionAccess(this.getAcl(), this.model, item, true);
        if (access) {
          if (item.isHiddenByAcl) {
            item.isHiddenByAcl = false;
            item.hidden = false;
            delete item.hiddenByAclSoft;
          }
        } else if (!item.hidden) {
          item.isHiddenByAcl = true;
          item.hidden = true;
          delete item.hiddenByAclSoft;
          if (access === null) {
            item.hiddenByAclSoft = true;
          }
        }
      });
    }

    /**
     * Set up panel views.
     *
     * @protected
     */
    setupPanelViews() {
      this.panelList.forEach(p => {
        const name = p.name;
        let options = {
          model: this.model,
          panelName: name,
          selector: '.panel[data-name="' + name + '"] > .panel-body',
          defs: p,
          mode: this.mode,
          recordHelper: this.recordHelper,
          inlineEditDisabled: this.inlineEditDisabled,
          readOnly: this.readOnly,
          disabled: p.hidden || false,
          recordViewObject: this.recordViewObject,
          dataObject: this.options.dataObject
        };
        options = _.extend(options, p.options);
        this.createView(name, p.view, options, view => {
          let hasSoftHidden = false;
          if ('getActionList' in view) {
            p.actionList = view.getActionList();
            this.applyAccessToActions(p.actionList);

            // noinspection JSUnresolvedReference
            if (p.actionList.find(it => it.hiddenByAclSoft)) {
              hasSoftHidden = true;
            }
          }
          if ('getButtonList' in view) {
            p.buttonList = view.getButtonList();
            this.applyAccessToActions(p.buttonList);

            // noinspection JSUnresolvedReference
            if (p.buttonList.find(it => it.hiddenByAclSoft)) {
              hasSoftHidden = true;
            }
          }
          if (hasSoftHidden) {
            this.listenToOnce(this.model, 'sync', () => {
              this.applyAccessToActions(p.actionList);
              this.applyAccessToActions(p.buttonList);
              view.whenRendered().then(() => {
                if (this.getPanelActionsView(name)) {
                  this.getPanelActionsView(name).reRender();
                }
              });
            });
          }
          if (view.titleHtml) {
            p.titleHtml = view.titleHtml;
          } else {
            if (p.label) {
              p.title = this.translate(p.label, 'labels', this.scope);
            } else {
              p.title = view.title;
            }
          }

          // @todo Use name_Actions.
          this.createView(name + 'Actions', 'views/record/panel-actions', {
            selector: '.panel[data-name="' + p.name + '"] > .panel-heading > .panel-actions-container',
            model: this.model,
            defs: p,
            scope: this.scope,
            entityType: this.entityType
          });
        });
      });
    }

    /**
     * @param {string} name
     * @return {import('views/record/panel-actions')}
     */
    getPanelActionsView(name) {
      return this.getView(name + 'Actions');
    }

    /**
     * Set up panels.
     *
     * @protected
     */
    setupPanels() {}

    /**
     * Get field views.
     *
     * @param {boolean} [withHidden] With hidden.
     * @return {Object.<string, module:views/fields/base>}
     */
    getFieldViews(withHidden) {
      let fields = {};
      this.panelList.forEach(p => {
        const panelView = this.getView(p.name);
        if ((!panelView.disabled || withHidden) && 'getFieldViews' in panelView) {
          fields = _.extend(fields, panelView.getFieldViews());
        }
      });
      return fields;
    }

    /**
     * Fetch.
     *
     * @return {Object.<string, *>}
     */
    fetch() {
      let data = {};
      this.panelList.forEach(p => {
        const panelView = this.getView(p.name);
        if (!panelView.disabled && 'fetch' in panelView) {
          data = _.extend(data, panelView.fetch());
        }
      });
      return data;
    }

    /**
     * @param {string} name
     * @return {boolean}
     */
    hasPanel(name) {
      return !!this.panelList.find(item => item.name === name);
    }
    processShowPanel(name, callback, wasShown) {
      if (this.recordHelper.getPanelStateParam(name, 'hidden')) {
        return;
      }
      if (!this.hasPanel(name)) {
        return;
      }
      this.panelList.filter(item => item.name === name).forEach(item => {
        item.hidden = false;
        if (typeof item.tabNumber !== 'undefined') {
          this.controlTabVisibilityShow(item.tabNumber);
        }
      });
      this.showPanelFinalize(name, callback, wasShown);
    }
    processHidePanel(name, callback) {
      if (!this.recordHelper.getPanelStateParam(name, 'hidden')) {
        return;
      }
      if (!this.hasPanel(name)) {
        return;
      }
      this.panelList.filter(item => item.name === name).forEach(item => {
        item.hidden = true;
        if (typeof item.tabNumber !== 'undefined') {
          this.controlTabVisibilityHide(item.tabNumber);
        }
      });
      this.hidePanelFinalize(name, callback);
    }
    showPanelFinalize(name, callback, wasShown) {
      const process = wasRendered => {
        const view = this.getView(name);
        if (view) {
          view.$el.closest('.panel').removeClass('hidden');
          view.disabled = false;
          view.trigger('show');
          view.trigger('panel-show-propagated');
          if (wasRendered && !wasShown && view.getFieldViews) {
            const fields = view.getFieldViews();
            if (fields) {
              for (const i in fields) {
                fields[i].reRender();
              }
            }
          }
        }
        if (typeof callback === 'function') {
          callback.call(this);
        }
      };
      if (this.isRendered()) {
        process(true);
        this.adjustPanels();
        return;
      }
      this.once('after:render', () => {
        process();
      });
    }
    hidePanelFinalize(name, callback) {
      if (this.isRendered()) {
        const view = this.getView(name);
        if (view) {
          view.$el.closest('.panel').addClass('hidden');
          view.disabled = true;
          view.trigger('hide');
        }
        if (typeof callback === 'function') {
          callback.call(this);
        }
        this.adjustPanels();
        return;
      }
      if (typeof callback === 'function') {
        this.once('after:render', () => {
          callback.call(this);
        });
      }
    }

    /**
     * @param {string} name
     * @param {string} [softLockedType]
     * @param [callback] Not to be used.
     */
    showPanel(name, softLockedType, callback) {
      if (this.recordHelper.getPanelStateParam(name, 'hiddenLocked')) {
        return;
      }
      if (softLockedType) {
        const param = 'hidden' + Espo.Utils.upperCaseFirst(softLockedType) + 'Locked';
        this.recordHelper.setPanelStateParam(name, param, false);
        for (let i = 0; i < this.panelSoftLockedTypeList.length; i++) {
          const iType = this.panelSoftLockedTypeList[i];
          if (iType === softLockedType) {
            continue;
          }
          const iParam = 'hidden' + Espo.Utils.upperCaseFirst(iType) + 'Locked';
          if (this.recordHelper.getPanelStateParam(name, iParam)) {
            return;
          }
        }
      }
      const wasShown = this.recordHelper.getPanelStateParam(name, 'hidden') === false;
      this.recordHelper.setPanelStateParam(name, 'hidden', false);
      this.processShowPanel(name, callback, wasShown);
    }

    /**
     * @param {string} name
     * @param {boolean} [locked=false]
     * @param {string} [softLockedType]
     * @param [callback] Not to be used.
     */
    hidePanel(name, locked, softLockedType, callback) {
      this.recordHelper.setPanelStateParam(name, 'hidden', true);
      if (locked) {
        this.recordHelper.setPanelStateParam(name, 'hiddenLocked', true);
      }
      if (softLockedType) {
        const param = 'hidden' + Espo.Utils.upperCaseFirst(softLockedType) + 'Locked';
        this.recordHelper.setPanelStateParam(name, param, true);
      }
      this.processHidePanel(name, callback);
    }
    alterPanels(layoutData) {
      layoutData = layoutData || this.layoutData || {};
      const tabBreakIndexList = [];
      const tabDataList = [];
      for (const name in layoutData) {
        const item = layoutData[name];
        if (name === '_delimiter_') {
          this.panelList.push({
            name: name
          });
        }
        if (item.tabBreak) {
          tabBreakIndexList.push(item.index);
          tabDataList.push({
            index: item.index,
            label: item.tabLabel
          });
        }
      }

      /**
       * @private
       * @type {Object.<string,*>[]}
       */
      this.tabDataList = tabDataList.sort((v1, v2) => v1.index - v2.index);
      this.panelList = this.panelList.filter(item => {
        return !this.recordHelper.getPanelStateParam(item.name, 'hiddenLocked');
      });
      const newList = [];
      this.panelList.forEach((item, i) => {
        item.index = 'index' in item ? item.index : i;
        let allowedInLayout = false;
        if (item.name) {
          const itemData = layoutData[item.name] || {};
          if (itemData.disabled) {
            return;
          }
          if (layoutData[item.name]) {
            allowedInLayout = true;
          }
          for (const i in itemData) {
            item[i] = itemData[i];
          }
        }
        if (item.disabled && !allowedInLayout) {
          return;
        }
        item.tabNumber = tabBreakIndexList.length - tabBreakIndexList.slice().reverse().findIndex(index => item.index > index) - 1;
        if (item.tabNumber === tabBreakIndexList.length) {
          item.tabNumber = -1;
        }
        newList.push(item);
      });
      newList.sort((v1, v2) => v1.index - v2.index);
      const firstTabIndex = newList.findIndex(item => item.tabNumber !== -1);
      if (firstTabIndex !== -1) {
        newList[firstTabIndex].isTabsBeginning = true;
        this.hasTabs = true;
        this.currentTab = newList[firstTabIndex].tabNumber;
        this.panelList.filter(item => item.tabNumber !== -1 && item.tabNumber !== this.currentTab).forEach(item => {
          item.tabHidden = true;
        });
        this.panelList.forEach((item, i) => {
          if (item.tabNumber !== -1 && (i === 0 || this.panelList[i - 1].tabNumber !== item.tabNumber)) {
            item.sticked = false;
          }
        });
      }
      this.panelList = newList;
      if (this.recordViewObject && this.recordViewObject.dynamicLogic) {
        const dynamicLogic = this.recordViewObject.dynamicLogic;
        this.panelList.forEach(item => {
          if (item.dynamicLogicVisible) {
            dynamicLogic.addPanelVisibleCondition(item.name, item.dynamicLogicVisible);
            if (this.recordHelper.getPanelStateParam(item.name, 'hidden')) {
              item.hidden = true;
            }
          }
          if (item.style && item.style !== 'default' && item.dynamicLogicStyled) {
            dynamicLogic.addPanelStyledCondition(item.name, item.dynamicLogicStyled);
          }
        });
      }
      if (this.hasTabs && this.options.isReturn && this.isStoredTabForThisRecord()) {
        this.selectStoredTab();
      }
    }
    setupPanelsFinal() {
      let afterDelimiter = false;
      let rightAfterDelimiter = false;
      let index = -1;
      this.panelList.forEach((p, i) => {
        if (p.name === '_delimiter_') {
          afterDelimiter = true;
          rightAfterDelimiter = true;
          index = i;
          return;
        }
        if (afterDelimiter) {
          p.hidden = true;
          p.hiddenAfterDelimiter = true;
          this.recordHelper.setPanelStateParam(p.name, 'hidden', true);
          this.recordHelper.setPanelStateParam(p.name, 'hiddenDelimiterLocked', true);
        }
        if (rightAfterDelimiter) {
          p.isRightAfterDelimiter = true;
          rightAfterDelimiter = false;
        }
      });
      if (~index) {
        this.panelList.splice(index, 1);
      }
      this.panelsAreSet = true;
      this.trigger('panels-set');
    }
    actionShowMorePanels() {
      this.panelList.forEach(p => {
        if (!p.hiddenAfterDelimiter) {
          return;
        }
        delete p.isRightAfterDelimiter;
        this.showPanel(p.name, 'delimiter');
      });
      this.$el.find('.panels-show-more-delimiter').remove();
    }
    onPanelsReady(callback) {
      Promise.race([new Promise(resolve => {
        if (this.panelsAreSet) {
          resolve();
        }
      }), new Promise(resolve => {
        this.once('panels-set', resolve);
      })]).then(() => {
        callback.call(this);
      });
    }
    getTabDataList() {
      return this.tabDataList.map((item, i) => {
        let label = item.label;
        if (!label) {
          label = (i + 1).toString();
        } else if (label[0] === '$') {
          label = this.translate(label.substring(1), 'tabs', this.scope);
        }
        const hidden = this.panelList.filter(panel => panel.tabNumber === i).findIndex(panel => !this.recordHelper.getPanelStateParam(panel.name, 'hidden')) === -1;
        return {
          label: label,
          isActive: i === this.currentTab,
          hidden: hidden
        };
      });
    }
    selectTab(tab) {
      this.currentTab = tab;
      if (this.isRendered()) {
        $('body > .popover').remove();
        this.$el.find('.tabs > button').removeClass('active');
        this.$el.find(`.tabs > button[data-tab="${tab}"]`).addClass('active');
        this.$el.find('.panel[data-tab]:not([data-tab="-1"])').addClass('tab-hidden');
        this.$el.find(`.panel[data-tab="${tab}"]`).removeClass('tab-hidden');
      }
      this.adjustPanels();
      this.panelList.filter(item => item.tabNumber === tab && item.name).forEach(item => {
        const view = this.getView(item.name);
        if (view) {
          view.trigger('tab-show');
          view.propagateEvent('panel-show-propagated');
        }
        item.tabHidden = false;
      });
      this.panelList.filter(item => item.tabNumber !== tab && item.name).forEach(item => {
        const view = this.getView(item.name);
        if (view) {
          view.trigger('tab-hide');
        }
        if (item.tabNumber > -1) {
          item.tabHidden = true;
        }
      });
      this.storeTab();
    }

    /** @private */
    storeTab() {
      const key = 'tab_' + this.name;
      const keyRecord = 'tab_' + this.name + '_record';
      this.getSessionStorage().set(key, this.currentTab);
      this.getSessionStorage().set(keyRecord, this.entityType + '_' + this.model.id);
    }

    /** @private */
    isStoredTabForThisRecord() {
      const keyRecord = 'tab_' + this.name + '_record';
      return this.getSessionStorage().get(keyRecord) === this.entityType + '_' + this.model.id;
    }

    /** @private */
    selectStoredTab() {
      const key = 'tab_' + this.name;
      const tab = this.getSessionStorage().get(key);
      if (tab > 0) {
        this.selectTab(tab);
      }
    }

    /** @private */
    controlTabVisibilityShow(tab) {
      if (!this.hasTabs) {
        return;
      }
      if (this.isBeingRendered()) {
        this.once('after:render', () => this.controlTabVisibilityShow(tab));
        return;
      }
      this.$el.find(`.tabs > [data-tab="${tab.toString()}"]`).removeClass('hidden');
    }

    /** @private */
    controlTabVisibilityHide(tab) {
      if (!this.hasTabs) {
        return;
      }
      if (this.isBeingRendered()) {
        this.once('after:render', () => this.controlTabVisibilityHide(tab));
        return;
      }
      const panelList = this.panelList.filter(panel => panel.tabNumber === tab);
      const allIsHidden = panelList.findIndex(panel => !this.recordHelper.getPanelStateParam(panel.name, 'hidden')) === -1;
      if (!allIsHidden) {
        return;
      }
      const $tab = this.$el.find(`.tabs > [data-tab="${tab.toString()}"]`);
      $tab.addClass('hidden');
      if (this.currentTab === tab) {
        const firstVisiblePanel = this.panelList.find(panel => panel.tabNumber > -1 && !panel.hidden);
        const firstVisibleTab = firstVisiblePanel ? firstVisiblePanel.tabNumber : 0;
        this.selectTab(firstVisibleTab);
      }
    }
  }
  var _default = _exports.default = PanelsContainerRecordView;
});

define("views/record/list-expanded", ["exports", "views/record/list"], function (_exports, _list) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _list = _interopRequireDefault(_list);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module module:views/record/list-expanded */

  class ListExpandedRecordView extends _list.default {
    template = 'record/list-expanded';
    checkboxes = false;
    selectable = false;
    rowActionsView = false;
    _internalLayoutType = 'list-row-expanded';
    presentationType = 'expanded';
    paginationDisabled = true;
    header = false;
    _internalLayout = null;
    checkedList = null;
    listContainerEl = '.list > ul';
    columnResize = false;
    init() {
      if (this.options.forcePagination) {
        this.paginationDisabled = false;
      }
      super.init();
    }
    setup() {
      super.setup();
      this.on('after:save', model => {
        const view = this.getView(model.id);
        if (!view) {
          return;
        }
        view.reRender();
      });

      // Prevents displaying an empty buttons container.
      this.displayTotalCount = false;
    }
    _loadListLayout(callback) {
      const type = this.type + 'Expanded';
      this.layoutLoadCallbackList.push(callback);
      if (this.layoutIsBeingLoaded) {
        return;
      }
      this.layoutIsBeingLoaded = true;
      this._helper.layoutManager.get(this.collection.entityType, type, listLayout => {
        this.layoutLoadCallbackList.forEach(c => {
          c(listLayout);
          this.layoutLoadCallbackList = [];
          this.layoutIsBeingLoaded = false;
        });
      });
    }
    _convertLayout(listLayout, model) {
      model = model || this.collection.prepareModel();
      const layout = {
        rows: [],
        right: false
      };
      for (const i in listLayout.rows) {
        const row = listLayout.rows[i];
        const layoutRow = [];
        for (const j in row) {
          const rowItem = row[j];
          const type = rowItem.type || model.getFieldType(rowItem.name) || 'base';
          const item = {
            name: rowItem.name + 'Field',
            field: rowItem.name,
            view: rowItem.view || model.getFieldParam(rowItem.name, 'view') || this.getFieldManager().getViewName(type),
            options: {
              defs: {
                name: rowItem.name,
                params: rowItem.params || {}
              },
              mode: 'list'
            },
            align: rowItem.align,
            small: rowItem.small,
            soft: rowItem.soft
          };
          if (rowItem.options) {
            for (const optionName in rowItem.options) {
              if (typeof item.options[optionName] !== 'undefined') {
                continue;
              }
              item.options[optionName] = rowItem.options[optionName];
            }
          }
          if (rowItem.link) {
            item.options.mode = 'listLink';
          }
          layoutRow.push(item);
        }
        layout.rows.push(layoutRow);
      }
      if ('right' in listLayout) {
        if (listLayout.right) {
          const name = listLayout.right.name || 'right';
          layout.right = {
            field: name,
            name: name,
            view: listLayout.right.view,
            options: {
              defs: {
                params: {
                  width: listLayout.right.width || '7%'
                }
              }
            }
          };
        }
      } else if (this.rowActionsView) {
        layout.right = this.getRowActionsDefs();
      }
      return layout;
    }
    getRowSelector(id) {
      return 'li[data-id="' + id + '"]';
    }
    getCellSelector(model, item) {
      const name = item.field || item.columnName;
      return `${this.getSelector()} ${this.getRowSelector(model.id)} .cell[data-name="${name}"]`;
    }
    getRowContainerHtml(id) {
      return $('<li>').attr('data-id', id).addClass('list-group-item list-row').get(0).outerHTML;
    }
    prepareInternalLayout(internalLayout, model) {
      const rows = internalLayout.rows || [];
      rows.forEach(row => {
        row.forEach(col => {
          col.el = this.getCellSelector(model, col);
        });
      });
      if (internalLayout.right) {
        internalLayout.right.el = this.getCellSelector(model, internalLayout.right);
      }
    }
    fetchAttributeListFromLayout() {
      const list = [];
      if (this.listLayout.rows) {
        this.listLayout.rows.forEach(row => {
          row.forEach(item => {
            if (!item.name) {
              return;
            }
            const field = item.name;
            const fieldType = this.getMetadata().get(['entityDefs', this.scope, 'fields', field, 'type']);
            if (!fieldType) {
              return;
            }
            this.getFieldManager().getEntityTypeFieldAttributeList(this.scope, field).forEach(attribute => {
              list.push(attribute);
            });
          });
        });
      }
      return list;
    }
  }
  var _default = _exports.default = ListExpandedRecordView;
});

define("views/record/panels/bottom", ["exports", "view"], function (_exports, _view) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/record/panels/bottom */

  /**
   * @typedef {Object} module:views/record/panels/bottom~defs
   * @property [buttonList]
   * @property [actionList]
   * @property [tabNumber] For internal purposes.
   */

  /**
   * A bottom panel.
   */
  class BottomPanelView extends _view.default {
    template = 'record/panels/side';

    /**
     * A field list.
     *
     * @protected
     * @type {module:views/record/panels/side~field[]}
     */
    fieldList;

    /**
     * @protected
     * @type {Array<module:views/record/panels-container~action|false>}
     */
    actionList;

    /**
     * @protected
     * @type {module:views/record/panels-container~button[]}
     */
    buttonList;

    /**
     * @protected
     * @type {module:views/record/panels/bottom~defs|Object.<string, *>}
     */
    defs;

    /**
     * A mode.
     *
     * @protected
     * @type {'list'|'detail'|'edit'}
     */
    mode = 'detail';

    /**
     * Disable.
     *
     * @protected
     */
    disabled = false;
    events = {
      /** @this BottomPanelView */
      'click .action': function (e) {
        Espo.Utils.handleAction(this, e.originalEvent, e.currentTarget, {
          actionItems: [...this.buttonList, ...this.actionList],
          className: 'panel-action'
        });
      }
    };
    data() {
      return {
        scope: this.scope,
        entityType: this.entityType,
        name: this.panelName,
        hiddenFields: this.recordHelper.getHiddenFields(),
        fieldList: this.getFieldList()
      };
    }
    init() {
      this.panelName = this.options.panelName;
      this.defs = this.options.defs || {};
      this.recordHelper = this.options.recordHelper;
      if ('disabled' in this.options) {
        this.disabled = this.options.disabled;
      }
      this.mode = this.options.mode || this.mode;
      this.readOnlyLocked = this.options.readOnlyLocked || this.readOnly;
      this.readOnly = this.readOnly || this.options.readOnly;
      this.inlineEditDisabled = this.inlineEditDisabled || this.options.inlineEditDisabled;
      this.buttonList = Espo.Utils.clone(this.defs.buttonList || this.buttonList || []);
      this.actionList = Espo.Utils.clone(this.defs.actionList || this.actionList || []);
      this.actionList.forEach(it => {
        if (it.name) {
          it.action = it.name;
        }
      });
      this.fieldList = this.options.fieldList || this.fieldList || [];
      this.recordViewObject = this.options.recordViewObject;
    }
    setup() {
      this.setupFields();
      this.fieldList = this.fieldList.map(d => {
        let item = d;
        if (typeof item !== 'object') {
          item = {
            name: item,
            viewKey: item + 'Field'
          };
        }
        item = Espo.Utils.clone(item);
        item.viewKey = item.name + 'Field';
        item.label = item.label || item.name;
        if (this.recordHelper.getFieldStateParam(item.name, 'hidden') !== null) {
          item.hidden = this.recordHelper.getFieldStateParam(item.name, 'hidden');
        } else {
          this.recordHelper.setFieldStateParam(item.name, 'hidden', item.hidden || false);
        }
        return item;
      });
      this.fieldList = this.fieldList.filter(item => {
        if (!item.name) {
          return;
        }
        if (!(item.name in ((this.model.defs || {}).fields || {}))) {
          return;
        }
        return true;
      });
      this.createFields();
    }

    /**
     * Set up fields.
     *
     * @protected
     */
    setupFields() {}

    /**
     * @return {module:views/record/panels-container~button[]}
     */
    getButtonList() {
      return this.buttonList || [];
    }

    /**
     * @return {module:views/record/panels-container~action[]}
     */
    getActionList() {
      return this.actionList || [];
    }

    /**
     * Get field views.
     *
     * @return {Object.<string,module:views/fields/base>}
     */
    getFieldViews() {
      const fields = {};
      this.getFieldList().forEach(item => {
        if (this.hasView(item.viewKey)) {
          fields[item.name] = this.getView(item.viewKey);
        }
      });
      return fields;
    }

    /**
     * @deprecated Use `getFieldViews`.
     * @todo Remove in v10.0.
     */
    getFields() {
      return this.getFieldViews();
    }

    /**
     * Get a field list.
     *
     * @return {module:views/record/panels/side~field[]}
     */
    getFieldList() {
      return this.fieldList.map(item => {
        if (typeof item !== 'object') {
          return {
            name: item
          };
        }
        return item;
      });
    }

    /**
     * @private
     */
    createFields() {
      this.getFieldList().forEach(item => {
        let view = null;
        let field;
        let readOnly = null;
        if (typeof item === 'object') {
          field = item.name;
          view = item.view;
          if ('readOnly' in item) {
            readOnly = item.readOnly;
          }
        } else {
          field = item;
        }
        if (!(field in this.model.defs.fields)) {
          return;
        }
        this.createField(field, view, null, null, readOnly);
      });
    }

    /**
     * Create a field view.
     *
     * @protected
     * @param {string} field A field name.
     * @param {string|null} [viewName] A view name/path.
     * @param {Object<string,*>} [params] Field params.
     * @param {'detail'|'edit'|'list'|null} [mode='edit'] A mode.
     * @param {boolean} [readOnly] Read-only.
     * @param {Object<string,*>} [options] View options.
     */
    createField(field, viewName, params, mode, readOnly, options) {
      const type = this.model.getFieldType(field) || 'base';
      viewName = viewName || this.model.getFieldParam(field, 'view') || this.getFieldManager().getViewName(type);
      const o = {
        model: this.model,
        selector: '.field[data-name="' + field + '"]',
        defs: {
          name: field,
          params: params || {}
        },
        mode: mode || this.mode,
        dataObject: this.options.dataObject
      };
      if (options) {
        for (const param in options) {
          o[param] = options[param];
        }
      }
      let readOnlyLocked = this.readOnlyLocked;
      if (this.readOnly) {
        o.readOnly = true;
      } else {
        if (readOnly !== null) {
          o.readOnly = readOnly;
        }
      }
      if (readOnly) {
        readOnlyLocked = true;
      }
      if (this.inlineEditDisabled) {
        o.inlineEditDisabled = true;
      }
      if (this.recordHelper.getFieldStateParam(field, 'hidden')) {
        o.disabled = true;
      }
      if (this.recordHelper.getFieldStateParam(field, 'hiddenLocked')) {
        o.disabledLocked = true;
      }
      if (this.recordHelper.getFieldStateParam(field, 'readOnly')) {
        o.readOnly = true;
      }
      if (this.recordHelper.getFieldStateParam(field, 'required') !== null) {
        o.defs.params.required = this.recordHelper.getFieldStateParam(field, 'required');
      }
      if (!readOnlyLocked && this.recordHelper.getFieldStateParam(field, 'readOnlyLocked')) {
        readOnlyLocked = true;
      }
      if (readOnlyLocked) {
        o.readOnlyLocked = readOnlyLocked;
      }
      if (this.recordHelper.hasFieldOptionList(field)) {
        o.customOptionList = this.recordHelper.getFieldOptionList(field);
      }
      if (this.recordViewObject) {
        o.validateCallback = () => this.recordViewObject.validateField(field);
      }
      const viewKey = field + 'Field';
      this.createView(viewKey, viewName, o);
    }

    // noinspection JSUnusedGlobalSymbols
    /**
     * Is tab-hidden.
     *
     * @return {boolean}
     */
    isTabHidden() {
      if (this.defs.tabNumber === -1 || typeof this.defs.tabNumber === 'undefined') {
        return false;
      }
      const parentView = this.getParentView();
      if (!parentView) {
        return this.defs.tabNumber > 0;
      }

      // noinspection JSUnresolvedReference
      if (parentView && parentView.hasTabs) {
        return parentView.currentTab !== defs.tabNumber;
      }
      return false;
    }
  }
  var _default = _exports.default = BottomPanelView;
});

define("views/fields/wysiwyg", ["exports", "views/fields/text", "helpers/misc/summernote-custom"], function (_exports, _text, _summernoteCustom) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _text = _interopRequireDefault(_text);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/wysiwyg */

  /**
   * A wysiwyg field.
   *
   * @extends TextFieldView<module:views/fields/wysiwyg~params>
   */
  class WysiwygFieldView extends _text.default {
    /**
     * @typedef {Object} module:views/fields/wysiwyg~options
     * @property {
     *     module:views/fields/wysiwyg~params &
     *     module:views/fields/base~params &
     *     Record
     * } [params] Parameters.
     */

    /**
     * @typedef {Object} module:views/fields/wysiwyg~params
     * @property {boolean} [required] Required.
     * @property {number} [maxLength] A max length.
     * @property {number} [height] A height in pixels.
     * @property {number} [minHeight] A min height in pixels.
     * @property {boolean} [useIframe] Use iframe.
     * @property {Array} [toolbar] A custom toolbar.
     * @property {string} [attachmentField] An attachment field name.
     */

    /**
     * @param {
     *     module:views/fields/wysiwyg~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }
    type = 'wysiwyg';
    listTemplate = 'fields/wysiwyg/detail';
    detailTemplate = 'fields/wysiwyg/detail';
    editTemplate = 'fields/wysiwyg/edit';
    height = 250;
    rowsDefault = 10000;
    fallbackBodySideMargin = 5;
    fallbackBodyTopMargin = 4;
    seeMoreDisabled = true;
    fetchEmptyValueAsNull = true;
    validationElementSelector = '.note-editor';
    htmlPurificationDisabled = false;
    htmlPurificationForEditDisabled = false;
    tableClassName = 'table table-bordered';
    noStylesheet = false;
    useIframe = false;
    handlebars = false;

    /** @protected */
    toolbar;
    /** @protected */
    hasBodyPlainField = false;
    events = {
      /** @this WysiwygFieldView */
      'click .note-editable': function () {
        this.fixPopovers();
      },
      /** @this WysiwygFieldView */
      'focus .note-editable': function () {
        this.$noteEditor.addClass('in-focus');
      },
      /** @this WysiwygFieldView */
      'blur .note-editable': function () {
        this.$noteEditor.removeClass('in-focus');
      }
    };
    setup() {
      super.setup();
      this.loadSummernote();
      if ('height' in this.params) {
        this.height = this.params.height;
      }
      if ('minHeight' in this.params) {
        this.minHeight = this.params.minHeight;
      }
      this.useIframe = this.params.useIframe || this.useIframe;
      this.setupToolbar();
      this.setupIsHtml();
      this.once('remove', () => this.destroySummernote());
      this.on('inline-edit-off', () => this.destroySummernote());
      this.on('render', () => this.destroySummernote());
      this.once('remove', () => {
        $(window).off(`resize.${this.cid}`);
        if (this.$scrollable) {
          this.$scrollable.off(`scroll.${this.cid}-edit`);
        }
      });
    }

    /** @private */
    loadSummernote() {
      this.wait(Espo.loader.requirePromise('lib!summernote').then(() => {
        if (!$.summernote.options || 'espoImage' in $.summernote.options) {
          return;
        }
        this.initEspoPlugin();
      }));
    }

    /** @protected */
    setupIsHtml() {
      if (!this.hasBodyPlainField) {
        return;
      }
      this.listenTo(this.model, 'change:isHtml', (model, value, o) => {
        if (o.ui && this.isEditMode()) {
          if (!this.isRendered()) {
            return;
          }
          if (this.isHtml()) {
            let value = this.plainToHtml(this.model.get(this.name));
            if (this.lastHtmlValue && this.model.get(this.name) === this.htmlToPlain(this.lastHtmlValue)) {
              value = this.lastHtmlValue;
            }
            this.model.set(this.name, value, {
              skipReRender: true
            });
            this.enableWysiwygMode();
            return;
          }
          this.lastHtmlValue = this.model.get(this.name);
          const value = this.htmlToPlain(this.model.get(this.name));
          this.disableWysiwygMode();
          this.model.set(this.name, value);
          return;
        }
        if (this.isDetailMode() && this.isRendered()) {
          this.reRender();
        }
      });
    }
    data() {
      const data = super.data();
      data.useIframe = this.useIframe;
      data.isPlain = !this.isHtml();

      // noinspection JSValidateTypes
      return data;
    }
    setupToolbar() {
      this.buttons = {};
      const codeviewName = this.getConfig().get('wysiwygCodeEditorDisabled') ? 'codeview' : 'aceCodeview';
      this.toolbar = this.params.toolbar || this.toolbar || [['style', ['style']], ['style', ['bold', 'italic', 'underline', 'clear']], ['fontsize', ['fontsize']], ['color', ['color']], ['para', ['ul', 'ol', 'paragraph']], ['height', ['height']], ['table', ['espoTable', 'espoLink', 'espoImage', 'hr']], ['misc', [codeviewName, 'fullscreen']]];
      if (this.params.toolbar) {
        return;
      }
      if (!this.params.attachmentField) {
        return;
      }
      this.toolbar.push(['attachment', ['attachment']]);
      this.buttons['attachment'] = () => {
        const ui = $.summernote.ui;
        const button = ui.button({
          contents: '<i class="fas fa-paperclip"></i>',
          tooltip: this.translate('Attach File'),
          click: () => {
            this.attachFile();
          }
        });
        return button.render();
      };
    }

    /**
     * @protected
     * @return {boolean}
     */
    isHtml() {
      if (!this.hasBodyPlainField) {
        return true;
      }
      return !this.model.has('isHtml') || this.model.get('isHtml');
    }
    fixPopovers() {
      $('body > .note-popover').removeClass('hidden');
    }
    getValueForDisplay() {
      if (!this.isReadMode() && this.isHtml()) {
        return undefined;
      }
      const value = super.getValueForDisplay();
      if (!this.isHtml()) {
        return value;
      }
      return this.sanitizeHtml(value);
    }

    /**
     * @protected
     * @param {string} value
     * @return {string}
     */
    sanitizeHtml(value) {
      if (!value) {
        return '';
      }
      if (this.htmlPurificationDisabled) {
        return this.sanitizeHtmlLight(value);
      }
      value = this.getHelper().sanitizeHtml(value);
      if (this.isEditMode()) {
        // Trick to handle the issue that attributes are re-ordered.
        value = this.getHelper().sanitizeHtml(value);
      }
      return value;
    }
    sanitizeHtmlLight(value) {
      return this.getHelper().moderateSanitizeHtml(value);
    }
    getValueForEdit() {
      const value = this.model.get(this.name) || '';
      if (this.htmlPurificationForEditDisabled) {
        return this.sanitizeHtmlLight(value);
      }
      return this.sanitizeHtml(value);
    }
    afterRender() {
      super.afterRender();
      if (this.isEditMode()) {
        this.$summernote = this.$el.find('.summernote');
      }
      const language = this.getConfig().get('language');
      if (!(language in $.summernote.lang)) {
        $.summernote.lang[language] = this.getLanguage().translate('summernote', 'sets');
      }
      if (this.isEditMode()) {
        if (this.isHtml()) {
          this.enableWysiwygMode();
        } else {
          this.$element.removeClass('hidden');
        }
        if (this.params.attachmentField && this.isInlineEditMode()) {
          this.$el.find('.note-attachment').addClass('hidden');
        }
      }
      if (this.isReadMode()) {
        this.renderDetail();
      }
    }
    renderDetail() {
      if (!this.isHtml()) {
        this.$el.find('.plain').removeClass('hidden');
        return;
      }
      if (!this.useIframe) {
        this.$element = this.$el.find('.html-container');
        return;
      }
      this.$el.find('iframe').removeClass('hidden');
      const $iframe = this.$el.find('iframe');

      /** @type {HTMLIFrameElement} */
      const iframeElement = this.iframe = $iframe.get(0);
      if (!iframeElement || !iframeElement.contentWindow) {
        return;
      }
      $iframe.on('load', () => {
        $iframe.contents().find('a').attr('target', '_blank');
      });
      const documentElement = iframeElement.contentWindow.document;
      let bodyHtml = this.getValueForIframe();
      const useFallbackStylesheet = this.getThemeManager().getParam('isDark') && this.htmlHasColors(bodyHtml);
      const addFallbackClass = this.getThemeManager().getParam('isDark') && (this.htmlHasColors(bodyHtml) || this.noStylesheet);
      const $iframeContainer = $iframe.parent();
      addFallbackClass ? $iframeContainer.addClass('fallback') : $iframeContainer.removeClass('fallback');
      if (!this.noStylesheet) {
        const linkElement = iframeElement.contentWindow.document.createElement('link');
        linkElement.type = 'text/css';
        linkElement.rel = 'stylesheet';
        linkElement.href = this.getBasePath() + (useFallbackStylesheet ? this.getThemeManager().getIframeFallbackStylesheet() : this.getThemeManager().getIframeStylesheet());
        bodyHtml = linkElement.outerHTML + bodyHtml;
      }
      let headHtml = '';
      if (this.noStylesheet) {
        const styleElement = documentElement.createElement('style');
        styleElement.textContent = `\ntable.bordered, table.bordered td, table.bordered th {border: 1px solid;}\n`;
        headHtml = styleElement.outerHTML;
      }

      // noinspection HtmlRequiredTitleElement
      const documentHtml = `<head>${headHtml}</head><body>${bodyHtml}</body>`;
      documentElement.write(documentHtml);
      documentElement.close();
      const $body = $iframe.contents().find('html body');
      $body.find('img').each((i, img) => {
        const $img = $(img);
        if ($img.css('max-width') !== 'none') {
          return;
        }
        $img.css('max-width', '100%');
      });
      const $document = $(documentElement);

      // Make dropdowns closed.
      $document.on('click', () => {
        const event = new MouseEvent('click', {
          bubbles: true
        });
        $iframe[0].dispatchEvent(event);
      });

      // Make notifications & global-search popup closed.
      $document.on('mouseup', () => {
        const event = new MouseEvent('mouseup', {
          bubbles: true
        });
        $iframe[0].dispatchEvent(event);
      });

      // Make shortcuts working.
      $document.on('keydown', e => {
        const originalEvent = /** @type {KeyboardEvent} */e.originalEvent;
        const event = new KeyboardEvent('keydown', {
          bubbles: true,
          code: originalEvent.code,
          ctrlKey: originalEvent.ctrlKey,
          metaKey: originalEvent.metaKey,
          altKey: originalEvent.altKey
        });
        $iframe[0].dispatchEvent(event);
      });
      const processWidth = function () {
        const bodyElement = $body.get(0);
        if (bodyElement) {
          if (bodyElement.clientWidth !== iframeElement.scrollWidth) {
            iframeElement.style.height = iframeElement.scrollHeight + 20 + 'px';
          }
        }
      };
      if (useFallbackStylesheet) {
        $iframeContainer.css({
          paddingLeft: this.fallbackBodySideMargin + 'px',
          paddingRight: this.fallbackBodySideMargin + 'px',
          paddingTop: this.fallbackBodyTopMargin + 'px'
        });
      }
      const increaseHeightStep = 10;
      const processIncreaseHeight = function (iteration, previousDiff) {
        $body.css('height', '');
        iteration = iteration || 0;
        if (iteration > 200) {
          return;
        }
        iteration++;
        const diff = $document.height() - iframeElement.scrollHeight;
        if (typeof previousDiff !== 'undefined') {
          if (diff === previousDiff) {
            $body.css('height', iframeElement.clientHeight - increaseHeightStep + 'px');
            processWidth();
            return;
          }
        }
        if (diff) {
          const height = iframeElement.scrollHeight + increaseHeightStep;
          iframeElement.style.height = height + 'px';
          processIncreaseHeight(iteration, diff);
        } else {
          processWidth();
        }
      };
      const processBg = () => {
        const color = iframeElement.contentWindow.getComputedStyle($body.get(0)).backgroundColor;
        $iframeContainer.css({
          backgroundColor: color
        });
      };
      const processHeight = function (isOnLoad) {
        if (!isOnLoad) {
          $iframe.css({
            overflowY: 'hidden',
            overflowX: 'hidden'
          });
          iframeElement.style.height = '0px';
        } else {
          if (iframeElement.scrollHeight >= $document.height()) {
            return;
          }
        }
        const $body = $iframe.contents().find('html body');
        let height = $body.height();
        if (height === 0) {
          height = $body.children().height() + 100;
        }
        iframeElement.style.height = height + 'px';
        processIncreaseHeight();
        if (!isOnLoad) {
          $iframe.css({
            overflowY: 'hidden',
            overflowX: 'scroll'
          });
        }
      };
      $iframe.css({
        visibility: 'hidden'
      });
      setTimeout(() => {
        processHeight();
        $iframe.css({
          visibility: 'visible'
        });
        $iframe.on('load', () => {
          processHeight(true);
          if (useFallbackStylesheet && !this.noStylesheet) {
            processBg();
          }
        });
      }, 40);
      if (!this.model.get(this.name)) {
        $iframe.addClass('hidden');
      }
      let windowWidth = $(window).width();
      $(window).off('resize.' + this.cid);
      $(window).on('resize.' + this.cid, () => {
        if ($(window).width() !== windowWidth) {
          processHeight();
          windowWidth = $(window).width();
        }
      });
    }

    /**
     * @protected
     * @return {string}
     */
    getValueForIframe() {
      return this.sanitizeHtml(this.model.get(this.name) || '');
    }
    enableWysiwygMode() {
      if (!this.$element) {
        return;
      }
      this.$element.addClass('hidden');
      this.$summernote.removeClass('hidden');
      const contents = this.getValueForEdit();
      this.$summernote.html(contents);

      // The same sanitizing in the email body field.
      this.$summernote.find('style').remove();
      this.$summernote.find('link[ref="stylesheet"]').remove();
      const keyMap = Espo.Utils.cloneDeep($.summernote.options.keyMap);
      keyMap.pc['CTRL+K'] = 'espoLink.show';
      keyMap.mac['CMD+K'] = 'espoLink.show';
      keyMap.pc['CTRL+DELETE'] = 'removeFormat';
      keyMap.mac['CMD+DELETE'] = 'removeFormat';
      delete keyMap.pc['CTRL+ENTER'];
      delete keyMap.mac['CMD+ENTER'];
      delete keyMap.pc['CTRL+BACKSLASH'];
      delete keyMap.mac['CMD+BACKSLASH'];
      const toolbar = this.toolbar;
      let lastChangeKeydown = new Date();
      const changeKeydownInterval = this.changeInterval * 1000;

      // noinspection JSUnusedGlobalSymbols
      const options = {
        handlebars: this.handlebars,
        prettifyHtml: false,
        // should not be true
        disableResizeEditor: true,
        isDark: this.getThemeManager().getParam('isDark'),
        espoView: this,
        lang: this.getConfig().get('language'),
        keyMap: keyMap,
        callbacks: {
          onImageUpload: files => {
            const file = files[0];
            Espo.Ui.notify(this.translate('Uploading...'));
            this.uploadInlineAttachment(file).then(attachment => {
              const url = '?entryPoint=attachment&id=' + attachment.id;
              this.$summernote.summernote('insertImage', url);
              Espo.Ui.notify(false);
            });
          },
          onBlur: () => {
            this.trigger('change');
          },
          onKeydown: () => {
            if (Date.now() - lastChangeKeydown > changeKeydownInterval) {
              this.trigger('change');
              lastChangeKeydown = Date.now();
            }
          }
        },
        onCreateLink(link) {
          return link;
        },
        toolbar: toolbar,
        buttons: this.buttons,
        dialogsInBody: this.$el,
        codeviewFilter: true,
        tableClassName: this.tableClassName,
        // Dnd has issues.
        disableDragAndDrop: true,
        colorButton: {
          foreColor: '#000000',
          backColor: '#FFFFFF'
        }
      };
      if (this.height) {
        options.height = this.height;
      } else {
        let $scrollable = this.$el.closest('.modal-body');
        if (!$scrollable.length) {
          $scrollable = $(window);
        }
        this.$scrollable = $scrollable;
        $scrollable.off(`scroll.${this.cid}-edit`);
        $scrollable.on(`scroll.${this.cid}-edit`, e => this.onScrollEdit(e));
      }
      if (this.minHeight) {
        options.minHeight = this.minHeight;
      }
      this.destroySummernote();
      this.$summernote.summernote(options);
      this.summernoteIsInitialized = true;
      this.$toolbar = this.$el.find('.note-toolbar');
      this.$area = this.$el.find('.note-editing-area');
      this.$noteEditor = this.$el.find('> .note-editor');
    }
    focusOnInlineEdit() {
      if (this.$noteEditor) {
        this.$summernote.summernote('focus');
        return;
      }
      super.focusOnInlineEdit();
    }
    uploadInlineAttachment(file) {
      return new Promise((resolve, reject) => {
        this.getModelFactory().create('Attachment', attachment => {
          const fileReader = new FileReader();
          fileReader.onload = e => {
            attachment.set('name', file.name);
            attachment.set('type', file.type);
            attachment.set('role', 'Inline Attachment');
            attachment.set('global', true);
            attachment.set('size', file.size);
            if (this.model.id) {
              attachment.set('relatedId', this.model.id);
            }
            attachment.set('relatedType', this.model.entityType);
            attachment.set('file', e.target.result);
            attachment.set('field', this.name);
            attachment.save().then(() => resolve(attachment)).catch(() => reject());
          };
          fileReader.readAsDataURL(file);
        });
      });
    }
    destroySummernote() {
      if (this.summernoteIsInitialized && this.$summernote) {
        this.$summernote.summernote('destroyAceCodeview');
        this.$summernote.summernote('destroy');
        this.summernoteIsInitialized = false;
      }
    }
    plainToHtml(html) {
      html = html || '';
      return html.replace(/\n/g, '<br>');
    }

    /**
     * @protected
     * @param {string} html
     * @return {string}
     */
    htmlToPlain(html) {
      const div = document.createElement('div');
      div.innerHTML = html;

      /**
       * @param {Node|HTMLElement} node
       * @return {string}
       */
      function processNode(node) {
        if (node.nodeType === Node.TEXT_NODE) {
          return node.nodeValue;
        }
        if (node.nodeType === Node.ELEMENT_NODE) {
          if (node instanceof HTMLAnchorElement) {
            return `${node.textContent} (${node.href})`;
          }
          if (node instanceof HTMLQuoteElement) {
            return `> ${node.textContent.trim()}`;
          }
          switch (node.tagName.toLowerCase()) {
            case 'br':
            case 'p':
            case 'div':
              return `\n${Array.from(node.childNodes).map(processNode).join('')}\n`;
          }
          return Array.from(node.childNodes).map(processNode).join('');
        }
        return '';
      }
      return processNode(div).replace(/\n{2,}/g, '\n\n').trim();
    }
    disableWysiwygMode() {
      this.destroySummernote();
      this.$noteEditor = null;
      if (this.$summernote) {
        this.$summernote.addClass('hidden');
      }
      this.$element.removeClass('hidden');
      if (this.$scrollable) {
        this.$scrollable.off('scroll.' + this.cid + '-edit');
      }
    }
    fetch() {
      const data = {};
      if (this.isHtml()) {
        let code = this.$summernote.summernote('code');
        if (code === '<p><br></p>') {
          code = '';
        }
        const imageTagString = `<img src="${window.location.origin}${window.location.pathname}?entryPoint=attachment`;
        code = code.replace(new RegExp(imageTagString.replace(/([.*+?^=!:${}()|\[\]\/\\])/g, "\\$1"), 'g'), '<img src="?entryPoint=attachment');
        data[this.name] = code;
      } else {
        data[this.name] = this.$element.val();
      }
      if (this.fetchEmptyValueAsNull && !data[this.name]) {
        data[this.name] = null;
      }
      if (this.hasBodyPlainField && this.model.has('isHtml')) {
        const plainAttribute = this.name + 'Plain';
        if (data[this.name] === null) {
          data[plainAttribute] = null;
        } else {
          data[plainAttribute] = this.isHtml() ? this.htmlToPlain(data[this.name]) : data[this.name];
        }
      }
      return data;
    }
    onScrollEdit(e) {
      const $target = $(e.target);
      const toolbarHeight = this.$toolbar.height();
      const toolbarWidth = this.$toolbar.parent().width();
      let edgeTop, edgeTopAbsolute;

      // noinspection JSIncompatibleTypesComparison
      if ($target.get(0) === window.document) {
        const $buttonContainer = $target.find('.detail-button-container:not(.hidden)');
        const offset = $buttonContainer.offset();
        if (offset) {
          edgeTop = offset.top + $buttonContainer.outerHeight();
          edgeTopAbsolute = edgeTop - $(window).scrollTop();
        }
      } else {
        const offset = $target.offset();
        if (offset) {
          edgeTop = offset.top;
          edgeTopAbsolute = edgeTop - $(window).scrollTop();
        }
      }
      const top = this.$el.offset().top;
      const bottom = top + this.$el.height() - toolbarHeight;
      let toStick = false;
      if (edgeTop > top && bottom > edgeTop) {
        toStick = true;
      }
      if (toStick) {
        this.$toolbar.css({
          top: edgeTopAbsolute + 'px',
          width: toolbarWidth + 'px'
        });
        this.$toolbar.addClass('sticked');
        this.$area.css({
          marginTop: toolbarHeight + 'px',
          backgroundColor: ''
        });
        return;
      }
      this.$toolbar.css({
        top: '',
        width: ''
      });
      this.$toolbar.removeClass('sticked');
      this.$area.css({
        marginTop: ''
      });
    }
    attachFile() {
      const $form = this.$el.closest('.record');
      $form.find(`.field[data-name="${this.params.attachmentField}"] input.file`).click();
      this.stopListening(this.model, 'attachment-uploaded:attachments');
      this.listenToOnce(this.model, 'attachment-uploaded:attachments', /** module:model[] */attachments => {
        if (this.isEditMode()) {
          const msg = this.translate('Attached') + '\n' + attachments.map(m => m.attributes.name).join('\n');
          Espo.Ui.notify(msg, 'success', 3000);
        }
      });
    }
    initEspoPlugin() {
      const langSets = this.getLanguage().get('Global', 'sets', 'summernote') || {
        image: {},
        link: {},
        video: {}
      };
      (0, _summernoteCustom.init)(langSets);
    }
    htmlHasColors(string) {
      if (~string.indexOf('background-color:')) {
        return true;
      }
      if (~string.indexOf('color:')) {
        return true;
      }
      if (~string.indexOf('<font color="')) {
        return true;
      }
      return false;
    }

    /**
     * @param {string} text
     * @since 8.4.0
     */
    insertText(text) {
      if (this.isHtml()) {
        this.$summernote.summernote('insertText', text);
      }
    }
  }
  var _default = _exports.default = WysiwygFieldView;
});

define("views/fields/link", ["exports", "views/fields/base", "helpers/record-modal", "ui/autocomplete"], function (_exports, _base, _recordModal, _autocomplete) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _base = _interopRequireDefault(_base);
  _recordModal = _interopRequireDefault(_recordModal);
  _autocomplete = _interopRequireDefault(_autocomplete);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/link */

  /**
   * A link field (belongs-to relation).
   *
   * @extends BaseFieldView<module:views/fields/link~params>
   */
  class LinkFieldView extends _base.default {
    /**
     * @typedef {Object} module:views/fields/link~options
     * @property {
     *     module:views/fields/link~params &
     *     module:views/fields/base~params &
     *     Record
     * } [params] Parameters.
     */

    /**
     * @typedef {Object} module:views/fields/link~params
     * @property {boolean} [required] Required.
     * @property {boolean} [autocompleteOnEmpty] Autocomplete on empty input.
     * @property {boolean} [createButton] Show 'Create' button.
     */

    /**
     * @param {
     *     module:views/fields/link~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }

    /** @inheritDoc */
    type = 'link';

    /** @inheritDoc */
    listTemplate = 'fields/link/list';
    /** @inheritDoc */
    detailTemplate = 'fields/link/detail';
    /** @inheritDoc */
    editTemplate = 'fields/link/edit';
    /** @inheritDoc */
    searchTemplate = 'fields/link/search';

    /**
     * A name attribute name.
     *
     * @type {string}
     */
    nameName;

    /**
     * An ID attribute name.
     *
     * @type {string}
     */
    idName;

    /**
     * A foreign entity type.
     *
     * @type {string|null}
     */
    foreignScope = null;

    /**
     * A select-record view.
     *
     * @protected
     * @type {string}
     */
    selectRecordsView = 'views/modals/select-records';

    /**
     * Autocomplete disabled.
     *
     * @protected
     * @type {boolean}
     */
    autocompleteDisabled = false;

    /**
     * Create disabled.
     *
     * @protected
     * @type {boolean}
     */
    createDisabled = false;

    /**
     * To display the create button.
     *
     * @protected
     * @type {boolean}
     */
    createButton = false;

    /**
     * Force create button even is disabled in clientDefs > relationshipPanels.
     *
     * @protected
     * @type {boolean}
     */
    forceCreateButton = false;

    /**
     * A search type list.
     *
     * @protected
     * @type {string[]}
     */
    searchTypeList = ['is', 'isEmpty', 'isNotEmpty', 'isNot', 'isOneOf', 'isNotOneOf'];

    /**
     * A primary filter list that will be available when selecting a record.
     *
     * @protected
     * @type {string[]|null}
     */
    selectFilterList = null;

    /**
     * A select primary filter.
     *
     * @protected
     * @type {string|null}
     */
    selectPrimaryFilterName = null;

    /**
     * A select bool filter list.
     *
     * @protected
     * @type {string[]|null}
     */
    selectBoolFilterList = null;

    /**
     * An autocomplete max record number.
     *
     * @protected
     * @type {number|null}
     */
    autocompleteMaxCount = null;

    /**
     * Select all attributes.
     *
     * @protected
     * @type {boolean}
     */
    forceSelectAllAttributes = false;

    /**
     * @protected
     * @type {string[]|null}
     */
    mandatorySelectAttributeList = null;

    /**
     * Trigger autocomplete on empty input.
     *
     * @protected
     * @type {boolean}
     */
    autocompleteOnEmpty = false;

    /** @inheritDoc */
    events = {
      /** @this LinkFieldView */
      'auxclick a[href]:not([role="button"])': function (e) {
        if (!this.isReadMode()) {
          return;
        }
        const isCombination = e.button === 1 && (e.ctrlKey || e.metaKey);
        if (!isCombination) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        this.quickView();
      }
    };

    // noinspection JSCheckFunctionSignatures
    /** @inheritDoc */
    data() {
      let nameValue = this.model.has(this.nameName) ? this.model.get(this.nameName) : this.model.get(this.idName);
      if (nameValue === null) {
        nameValue = this.model.get(this.idName);
      }
      if (this.isReadMode() && !nameValue && this.model.get(this.idName)) {
        nameValue = this.translate(this.foreignScope, 'scopeNames');
      }
      let iconHtml = null;
      if (this.isDetailMode() || this.isListMode()) {
        iconHtml = this.getHelper().getScopeColorIconHtml(this.foreignScope);
      }
      const createButton = this.createButton && (!this.createDisabled || this.forceCreateButton);

      // noinspection JSValidateTypes
      return {
        ...super.data(),
        idName: this.idName,
        nameName: this.nameName,
        idValue: this.model.get(this.idName),
        nameValue: nameValue,
        foreignScope: this.foreignScope,
        valueIsSet: this.model.has(this.idName),
        iconHtml: iconHtml,
        url: this.getUrl(),
        createButton: createButton
      };
    }

    /**
     * @protected
     * @return {?string}
     */
    getUrl() {
      const id = this.model.get(this.idName);
      if (!id) {
        return null;
      }
      return '#' + this.foreignScope + '/view/' + id;
    }

    /**
     * Get advanced filters (field filters) to be applied when select a record.
     * Can be extended.
     *
     * @protected
     * @return {Object.<string,module:search-manager~advancedFilter>|null}
     */
    getSelectFilters() {
      return null;
    }

    /**
     * Get a select bool filter list. Applied when select a record.
     * Can be extended.
     *
     * @protected
     * @return {string[]|null}
     */
    getSelectBoolFilterList() {
      return this.selectBoolFilterList;
    }

    /**
     * Get a select primary filter. Applied when select a record.
     * Can be extended.
     *
     * @protected
     * @return {string|null}
     */
    getSelectPrimaryFilterName() {
      return this.selectPrimaryFilterName;
    }

    /**
     * Get a primary filter list that will be available when selecting a record.
     * Can be extended.
     *
     * @return {string[]|null}
     */
    getSelectFilterList() {
      return this.selectFilterList;
    }

    /**
     * Attributes to pass to a model when creating a new record.
     * Can be extended.
     *
     * @return {Object.<string,*>|null}
     */
    getCreateAttributes() {
      const attributeMap = this.getMetadata().get(['clientDefs', this.entityType, 'relationshipPanels', this.name, 'createAttributeMap']) || {};
      const attributes = {};
      Object.keys(attributeMap).forEach(attr => attributes[attributeMap[attr]] = this.model.get(attr));
      return attributes;
    }

    /** @inheritDoc */
    setup() {
      this.nameName = this.name + 'Name';
      this.idName = this.name + 'Id';
      this.foreignScope = this.options.foreignScope || this.foreignScope;
      this.foreignScope = this.foreignScope || this.model.getFieldParam(this.name, 'entity') || this.model.getLinkParam(this.name, 'entity');
      if ('createDisabled' in this.options) {
        this.createDisabled = this.options.createDisabled;
      }
      if (!this.isListMode()) {
        this.addActionHandler('selectLink', () => this.actionSelect());
        this.addActionHandler('clearLink', () => this.clearLink());
      }
      if (this.isSearchMode()) {
        this.addActionHandler('selectLinkOneOf', () => this.actionSelectOneOf());
        this.events['click a[data-action="clearLinkOneOf"]'] = e => {
          const id = $(e.currentTarget).data('id').toString();
          this.deleteLinkOneOf(id);
        };
      }
      this.autocompleteOnEmpty = this.params.autocompleteOnEmpty || this.autocompleteOnEmpty;
      this.createButton = this.params.createButton || this.createButton;
      if (this.createButton && !this.getAcl().checkScope(this.foreignScope, 'create')) {
        this.createButton = false;
      }
      if (this.createButton) {
        this.addActionHandler('createLink', () => this.actionCreateLink());
      }

      /** @type {Object.<string, *>} */
      this.panelDefs = this.getMetadata().get(['clientDefs', this.entityType, 'relationshipPanels', this.name]) || {};
      if (this.panelDefs.createDisabled) {
        this.createDisabled = true;
      }
    }

    /**
     * Select.
     *
     * @param {module:model} model A model.
     * @protected
     */
    select(model) {
      this.$elementName.val(model.get('name') || model.id);
      this.$elementId.val(model.get('id'));
      if (this.mode === this.MODE_SEARCH) {
        this.searchData.idValue = model.get('id');
        this.searchData.nameValue = model.get('name') || model.id;
      }
      this.trigger('change');
      this.controlCreateButtonVisibility();
      const attributes = {};
      for (const [foreign, field] of Object.entries(this.getDependantForeignMap())) {
        attributes[field] = model.get(foreign);
      }
      this.getSelectFieldHandler().then(async handler => {
        this.model.set({
          ...attributes,
          ...(await handler.getAttributes(model))
        });
      });
    }

    /**
     * Clear.
     */
    clearLink() {
      this.$elementName.val('');
      this.$elementId.val('');
      this.trigger('change');
      this.controlCreateButtonVisibility();
      for (const [, field] of Object.entries(this.getDependantForeignMap())) {
        this.model.unset(field, {
          fromField: this.name
        });
      }
      this.getSelectFieldHandler().then(handler => {
        handler.getClearAttributes().then(attributes => {
          this.model.set(attributes, {
            fromField: this.name
          });
        });
      });
    }

    /** @private */
    controlCreateButtonVisibility() {
      if (!this.createButton) {
        return;
      }
      const $btn = this.$el.find('[data-action="createLink"]');
      this.model.get(this.idName) ? $btn.addClass('hidden') : $btn.removeClass('hidden');
    }

    /**
     * @private
     * @return {Promise<{
     *     getAttributes: function (module:model): Promise<Object.<string, *>>,
     *     getClearAttributes: function(): Promise<Object.<string, *>>,
     * }>}
     */
    getSelectFieldHandler() {
      if (!this.panelDefs.selectFieldHandler) {
        return Promise.resolve({
          getClearAttributes: () => Promise.resolve({}),
          getAttributes: () => Promise.resolve({})
        });
      }
      return new Promise(resolve => {
        Espo.loader.requirePromise(this.panelDefs.selectFieldHandler).then(Handler => {
          // Model is passed as of v8.2.
          const handler = new Handler(this.getHelper(), this.model);
          resolve(handler);
        });
      });
    }

    /** @inheritDoc */
    setupSearch() {
      this.searchData.oneOfIdList = this.getSearchParamsData().oneOfIdList || this.searchParams.oneOfIdList || [];
      this.searchData.oneOfNameHash = this.getSearchParamsData().oneOfNameHash || this.searchParams.oneOfNameHash || {};
      if (~['is', 'isNot', 'equals'].indexOf(this.getSearchType())) {
        this.searchData.idValue = this.getSearchParamsData().idValue || this.searchParams.idValue || this.searchParams.value;
        this.searchData.nameValue = this.getSearchParamsData().nameValue || this.searchParams.nameValue || this.searchParams.valueName;
      }
      this.events['change select.search-type'] = e => {
        const type = $(e.currentTarget).val();
        this.handleSearchType(type);
      };
    }

    /**
     * Handle a search type.
     *
     * @protected
     * @param {string} type A type.
     */
    handleSearchType(type) {
      if (~['is', 'isNot', 'isNotAndIsNotEmpty'].indexOf(type)) {
        this.$el.find('div.primary').removeClass('hidden');
      } else {
        this.$el.find('div.primary').addClass('hidden');
      }
      if (~['isOneOf', 'isNotOneOf', 'isNotOneOfAndIsNotEmpty'].indexOf(type)) {
        this.$el.find('div.one-of-container').removeClass('hidden');
      } else {
        this.$el.find('div.one-of-container').addClass('hidden');
      }
    }

    /**
     * Get an autocomplete max record number. Can be extended.
     *
     * @protected
     * @return {number}
     */
    getAutocompleteMaxCount() {
      if (this.autocompleteMaxCount) {
        return this.autocompleteMaxCount;
      }
      return this.getConfig().get('recordsPerPage');
    }

    /**
     * @private
     * @return {string[]}
     */
    getMandatorySelectAttributeList() {
      const list = this.mandatorySelectAttributeList || this.panelDefs.selectMandatoryAttributeList || [];
      const map = this.getDependantForeignMap();
      return [...list, ...Object.keys(map)];
    }

    /**
     * @private
     * @return {Record<string>}
     */
    getDependantForeignMap() {
      if (this._dependantForeignMap) {
        return this._dependantForeignMap;
      }
      const map = {};
      this.model.getFieldList().filter(it => {
        return this.model.getFieldType(it) === 'foreign' && this.model.getFieldParam(it, 'link') === this.name;
      }).forEach(field => {
        const foreign = this.model.getFieldParam(field, 'field');
        if (!foreign) {
          return;
        }
        map[foreign] = field;
      });
      this._dependantForeignMap = map;
      return map;
    }

    // noinspection JSUnusedLocalSymbols
    /**
     * Compose an autocomplete URL. Can be extended.
     *
     * @protected
     * @param {string} [q] A query.
     * @return {string|Promise<string>}
     */
    getAutocompleteUrl(q) {
      let url = this.foreignScope + '?maxSize=' + this.getAutocompleteMaxCount();
      if (!this.forceSelectAllAttributes) {
        const mandatorySelectAttributeList = this.getMandatorySelectAttributeList();
        let select = ['id', 'name'];
        if (mandatorySelectAttributeList) {
          select = select.concat(mandatorySelectAttributeList);
        }
        url += '&select=' + select.join(',');
      }
      if (this.panelDefs.selectHandler) {
        return new Promise(resolve => {
          this._getSelectFilters().then(filters => {
            if (filters.bool) {
              url += '&' + $.param({
                'boolFilterList': filters.bool
              });
            }
            if (filters.primary) {
              url += '&' + $.param({
                'primaryFilter': filters.primary
              });
            }
            if (filters.advanced && Object.keys(filters.advanced).length) {
              url += '&' + $.param({
                'where': filters.advanced
              });
            }
            const orderBy = filters.orderBy || this.panelDefs.selectOrderBy;
            const orderDirection = filters.orderBy ? filters.order : this.panelDefs.selectOrderDirection;
            if (orderBy) {
              url += '&' + $.param({
                orderBy: orderBy,
                order: orderDirection || 'asc'
              });
            }
            resolve(url);
          });
        });
      }
      const boolList = [...(this.getSelectBoolFilterList() || []), ...(this.panelDefs.selectBoolFilterList || [])];
      const primary = this.getSelectPrimaryFilterName() || this.panelDefs.selectPrimaryFilterName;
      if (boolList.length) {
        url += '&' + $.param({
          'boolFilterList': boolList
        });
      }
      if (primary) {
        url += '&' + $.param({
          'primaryFilter': primary
        });
      }
      if (this.panelDefs.selectOrderBy) {
        const direction = this.panelDefs.selectOrderDirection || 'asc';
        url += '&' + $.param({
          orderBy: this.panelDefs.selectOrderBy,
          order: direction
        });
      }
      return url;
    }

    /** @inheritDoc */
    afterRender() {
      if (this.isEditMode() || this.isSearchMode()) {
        this.$elementId = this.$el.find('input[data-name="' + this.idName + '"]');
        this.$elementName = this.$el.find('input[data-name="' + this.nameName + '"]');
        this.$elementName.on('change', () => {
          if (this.$elementName.val() === '') {
            this.clearLink();
          }
        });
        this.$elementName.on('blur', e => {
          setTimeout(() => {
            if (this.mode === this.MODE_EDIT && this.model.has(this.nameName)) {
              e.currentTarget.value = this.model.get(this.nameName) || this.model.get(this.idName);
            }
          }, 100);
        });
        const $elementName = this.$elementName;
        if (!this.autocompleteDisabled) {
          /** @type {module:ajax.AjaxPromise & Promise<any>} */
          let lastAjaxPromise;
          const autocomplete = new _autocomplete.default(this.$elementName.get(0), {
            name: this.name,
            handleFocusMode: 2,
            autoSelectFirst: true,
            forceHide: true,
            triggerSelectOnValidInput: false,
            onSelect: item => {
              this.getModelFactory().create(this.foreignScope, model => {
                model.set(item.attributes);
                this.select(model);
                this.$elementName.focus();
              });
            },
            lookupFunction: query => {
              if (!this.autocompleteOnEmpty && query.length === 0) {
                const onEmptyPromise = this.getOnEmptyAutocomplete();
                if (onEmptyPromise) {
                  return onEmptyPromise.then(list => this._transformAutocompleteResult({
                    list: list
                  }));
                }
                return Promise.resolve([]);
              }
              return Promise.resolve(this.getAutocompleteUrl(query)).then(url => {
                if (lastAjaxPromise && lastAjaxPromise.getReadyState() < 4) {
                  lastAjaxPromise.abort();
                }
                lastAjaxPromise = Espo.Ajax.getRequest(url, {
                  q: query
                });
                return lastAjaxPromise;
              }).then(response => this._transformAutocompleteResult(response));
            }
          });
          this.once('render remove', () => autocomplete.dispose());
          if (this.isSearchMode()) {
            const $elementOneOf = this.$el.find('input.element-one-of');

            /** @type {module:ajax.AjaxPromise & Promise<any>} */
            let lastAjaxPromise;
            const autocomplete = new _autocomplete.default($elementOneOf.get(0), {
              minChars: 1,
              focusOnSelect: true,
              handleFocusMode: 3,
              autoSelectFirst: true,
              triggerSelectOnValidInput: false,
              forceHide: true,
              onSelect: item => {
                this.getModelFactory().create(this.foreignScope, model => {
                  model.set(item.attributes);
                  this.selectOneOf([model]);
                  $elementOneOf.val('');
                  setTimeout(() => $elementOneOf.focus(), 50);
                });
              },
              lookupFunction: query => {
                return Promise.resolve(this.getAutocompleteUrl(query)).then(url => {
                  if (lastAjaxPromise && lastAjaxPromise.getReadyState() < 4) {
                    lastAjaxPromise.abort();
                  }
                  lastAjaxPromise = Espo.Ajax.getRequest(url, {
                    q: query
                  });
                  return lastAjaxPromise;
                }).then(/** {list: Record[]} */response => {
                  return response.list.map(item => ({
                    value: item.name,
                    attributes: item
                  }));
                });
              }
            });
            this.once('render remove', () => autocomplete.dispose());
            this.$el.find('select.search-type').on('change', () => {
              this.trigger('change');
            });
          }
        }
        $elementName.on('change', () => {
          if (!this.isSearchMode() && !this.model.get(this.idName)) {
            $elementName.val(this.model.get(this.nameName));
          }
        });
      }
      if (this.isSearchMode()) {
        const type = this.$el.find('select.search-type').val();
        this.handleSearchType(type);
        if (~['isOneOf', 'isNotOneOf', 'isNotOneOfAndIsNotEmpty'].indexOf(type)) {
          this.searchData.oneOfIdList.forEach(id => {
            this.addLinkOneOfHtml(id, this.searchData.oneOfNameHash[id]);
          });
        }
      }
    }

    /**
     * @private
     */
    _transformAutocompleteResult(response) {
      const list = [];
      response.list.forEach(item => {
        list.push({
          id: item.id,
          name: item.name || item.id,
          data: item.id,
          value: item.name || item.id,
          attributes: item
        });
      });
      return list;
    }

    /** @inheritDoc */
    getValueForDisplay() {
      return this.model.get(this.nameName);
    }

    /** @inheritDoc */
    validateRequired() {
      if (this.isRequired()) {
        if (this.model.get(this.idName) == null) {
          const msg = this.translate('fieldIsRequired', 'messages').replace('{field}', this.getLabelText());
          this.showValidationMessage(msg);
          return true;
        }
      }
    }

    /**
     * Delete a one-of item. For search mode.
     *
     * @param {string} id An ID.
     */
    deleteLinkOneOf(id) {
      this.deleteLinkOneOfHtml(id);
      const index = this.searchData.oneOfIdList.indexOf(id);
      if (index > -1) {
        this.searchData.oneOfIdList.splice(index, 1);
      }
      delete this.searchData.oneOfNameHash[id];
      this.trigger('change');
    }

    /**
     * Add a one-of item. For search mode.
     *
     * @param {string} id An ID.
     * @param {string} name A name.
     */
    addLinkOneOf(id, name) {
      if (!~this.searchData.oneOfIdList.indexOf(id)) {
        this.searchData.oneOfIdList.push(id);
        this.searchData.oneOfNameHash[id] = name;
        this.addLinkOneOfHtml(id, name);
        this.trigger('change');
      }
    }

    /**
     * @protected
     * @param {string} id An ID.
     */
    deleteLinkOneOfHtml(id) {
      this.$el.find('.link-one-of-container .link-' + id).remove();
    }

    /**
     * @protected
     * @param {string} id An ID.
     * @param {string} name A name.
     * @return {JQuery}
     */
    addLinkOneOfHtml(id, name) {
      const $container = this.$el.find('.link-one-of-container');
      const $el = $('<div>').addClass('link-' + id).addClass('list-group-item');
      $el.append($('<a>').attr('role', 'button').addClass('pull-right').attr('data-id', id).attr('data-action', 'clearLinkOneOf').append($('<span>').addClass('fas fa-times')), $('<span>').text(name), ' ');
      $container.append($el);
      return $el;
    }

    /** @inheritDoc */
    fetch() {
      const data = {};
      data[this.nameName] = this.$el.find('[data-name="' + this.nameName + '"]').val() || null;
      data[this.idName] = this.$el.find('[data-name="' + this.idName + '"]').val() || null;
      return data;
    }

    /** @inheritDoc */
    fetchSearch() {
      const type = this.$el.find('select.search-type').val();
      const value = this.$el.find('[data-name="' + this.idName + '"]').val();
      if (~['isOneOf', 'isNotOneOf'].indexOf(type) && !this.searchData.oneOfIdList.length) {
        return {
          type: 'isNotNull',
          attribute: 'id',
          data: {
            type: type
          }
        };
      }
      if (type === 'isEmpty') {
        return {
          type: 'isNull',
          attribute: this.idName,
          data: {
            type: type
          }
        };
      }
      if (type === 'isNotEmpty') {
        return {
          type: 'isNotNull',
          attribute: this.idName,
          data: {
            type: type
          }
        };
      }
      if (type === 'isOneOf') {
        return {
          type: 'in',
          attribute: this.idName,
          value: this.searchData.oneOfIdList,
          data: {
            type: type,
            oneOfIdList: this.searchData.oneOfIdList,
            oneOfNameHash: this.searchData.oneOfNameHash
          }
        };
      }
      if (type === 'isNotOneOf') {
        return {
          type: 'or',
          value: [{
            type: 'notIn',
            attribute: this.idName,
            value: this.searchData.oneOfIdList
          }, {
            type: 'isNull',
            attribute: this.idName
          }],
          data: {
            type: type,
            oneOfIdList: this.searchData.oneOfIdList,
            oneOfNameHash: this.searchData.oneOfNameHash
          }
        };
      }
      if (type === 'isNotOneOfAndIsNotEmpty') {
        return {
          type: 'notIn',
          attribute: this.idName,
          value: this.searchData.oneOfIdList,
          data: {
            type: type,
            oneOfIdList: this.searchData.oneOfIdList,
            oneOfNameHash: this.searchData.oneOfNameHash
          }
        };
      }
      if (type === 'isNot') {
        if (!value) {
          return false;
        }
        const nameValue = this.$el.find('[data-name="' + this.nameName + '"]').val();
        return {
          type: 'or',
          value: [{
            type: 'notEquals',
            attribute: this.idName,
            value: value
          }, {
            type: 'isNull',
            attribute: this.idName
          }],
          data: {
            type: type,
            idValue: value,
            nameValue: nameValue
          }
        };
      }
      if (type === 'isNotAndIsNotEmpty') {
        if (!value) {
          return false;
        }
        const nameValue = this.$el.find('[data-name="' + this.nameName + '"]').val();
        return {
          type: 'notEquals',
          attribute: this.idName,
          value: value,
          data: {
            type: type,
            idValue: value,
            nameValue: nameValue
          }
        };
      }
      if (!value) {
        return false;
      }
      const nameValue = this.$el.find('[data-name="' + this.nameName + '"]').val();
      return {
        type: 'equals',
        attribute: this.idName,
        value: value,
        data: {
          type: type,
          idValue: value,
          nameValue: nameValue
        }
      };
    }

    /** @inheritDoc */
    getSearchType() {
      return this.getSearchParamsData().type || this.searchParams.typeFront || this.searchParams.type;
    }

    /**
     * @protected
     */
    quickView() {
      const id = this.model.get(this.idName);
      if (!id) {
        return;
      }
      const entityType = this.foreignScope;
      const helper = new _recordModal.default();
      helper.showDetail(this, {
        id: id,
        scope: entityType
      });
    }

    /**
     * @protected
     * @return {function(): Promise<Object.<string, *>>}
     */
    getCreateAttributesProvider() {
      return () => {
        const attributes = this.getCreateAttributes() || {};
        if (!this.panelDefs.createHandler) {
          return Promise.resolve(attributes);
        }
        return new Promise(resolve => {
          Espo.loader.requirePromise(this.panelDefs.createHandler).then(Handler => new Handler(this.getHelper())).then(handler => {
            handler.getAttributes(this.model).then(additionalAttributes => {
              resolve({
                ...attributes,
                ...additionalAttributes
              });
            });
          });
        });
      };
    }

    /**
     * @protected
     */
    actionSelect() {
      Espo.Ui.notify(' ... ');
      const panelDefs = this.panelDefs;
      const viewName = panelDefs.selectModalView || this.getMetadata().get(['clientDefs', this.foreignScope, 'modalViews', 'select']) || this.selectRecordsView;
      const mandatorySelectAttributeList = this.getMandatorySelectAttributeList();
      const createButton = this.isEditMode() && (!this.createDisabled || this.forceCreateButton);
      const createAttributesProvider = createButton ? this.getCreateAttributesProvider() : null;
      this._getSelectFilters().then(filters => {
        const orderBy = filters.orderBy || this.panelDefs.selectOrderBy;
        const orderDirection = filters.orderBy ? filters.order : this.panelDefs.selectOrderDirection;
        this.createView('dialog', viewName, {
          scope: this.foreignScope,
          createButton: createButton,
          filters: filters.advanced,
          boolFilterList: filters.bool,
          primaryFilterName: filters.primary,
          mandatorySelectAttributeList: mandatorySelectAttributeList,
          forceSelectAllAttributes: this.forceSelectAllAttributes,
          filterList: this.getSelectFilterList(),
          createAttributesProvider: createAttributesProvider,
          layoutName: this.panelDefs.selectLayout,
          orderBy: orderBy,
          orderDirection: orderDirection
        }, view => {
          view.render();
          Espo.Ui.notify(false);
          this.listenToOnce(view, 'select', model => {
            this.clearView('dialog');
            this.select(model);
          });
        });
      });
    }

    /**
     * @param {Object} advanced
     * @private
     */
    _applyAdditionalFilter(advanced) {
      const foreignLink = this.model.getLinkParam(this.name, 'foreign');
      if (!foreignLink) {
        return;
      }
      if (advanced[foreignLink]) {
        return;
      }
      const linkType = this.model.getLinkParam(this.name, 'type');
      const foreignLinkType = this.getMetadata().get(['entityDefs', this.foreignScope, 'links', foreignLink, 'type']);
      const foreignFieldType = this.getMetadata().get(['entityDefs', this.foreignScope, 'fields', foreignLink, 'type']);
      if (!foreignFieldType) {
        return;
      }
      const isOneToOne = (linkType === 'hasOne' || foreignLinkType === 'hasOne') && ['link', 'linkOne'].includes(foreignFieldType);
      if (!isOneToOne) {
        return;
      }
      advanced[foreignLink] = {
        type: 'isNull',
        attribute: foreignLink + 'Id',
        data: {
          type: 'isEmpty'
        }
      };
    }

    /**
     * @private
     * @return {Promise<{
     *     bool?: string[],
     *     advanced?: Object,
     *     primary?: string,
     *     orderBy?: string,
     *     order?: 'asc'|'desc',
     * }>}
     */
    _getSelectFilters() {
      const handler = this.panelDefs.selectHandler;
      const localBoolFilterList = this.getSelectBoolFilterList();
      if (!handler || this.isSearchMode()) {
        const boolFilterList = localBoolFilterList || this.panelDefs.selectBoolFilterList ? [...(localBoolFilterList || []), ...(this.panelDefs.selectBoolFilterList || [])] : undefined;
        const advanced = this.getSelectFilters() || {};
        this._applyAdditionalFilter(advanced);
        return Promise.resolve({
          primary: this.getSelectPrimaryFilterName() || this.panelDefs.selectPrimaryFilterName,
          bool: boolFilterList,
          advanced: advanced
        });
      }
      return new Promise(resolve => {
        Espo.loader.requirePromise(handler).then(Handler => new Handler(this.getHelper())).then(/** module:handlers/select-related */handler => {
          return handler.getFilters(this.model);
        }).then(filters => {
          const advanced = {
            ...(this.getSelectFilters() || {}),
            ...(filters.advanced || {})
          };
          const primaryFilter = this.getSelectPrimaryFilterName() || filters.primary || this.panelDefs.selectPrimaryFilterName;
          const boolFilterList = localBoolFilterList || filters.bool || this.panelDefs.selectBoolFilterList ? [...(localBoolFilterList || []), ...(filters.bool || []), ...(this.panelDefs.selectBoolFilterList || [])] : undefined;
          this._applyAdditionalFilter(advanced);
          const orderBy = filters.orderBy;
          const order = orderBy ? filters.order : undefined;
          resolve({
            bool: boolFilterList,
            primary: primaryFilter,
            advanced: advanced,
            orderBy: orderBy,
            order: order
          });
        });
      });
    }
    actionSelectOneOf() {
      Espo.Ui.notify(' ... ');
      const viewName = this.getMetadata().get(['clientDefs', this.foreignScope, 'modalViews', 'select']) || this.selectRecordsView;
      this.createView('dialog', viewName, {
        scope: this.foreignScope,
        createButton: false,
        filters: this.getSelectFilters(),
        boolFilterList: this.getSelectBoolFilterList(),
        primaryFilterName: this.getSelectPrimaryFilterName(),
        multiple: true,
        layoutName: this.panelDefs.selectLayout
      }, view => {
        view.render();
        Espo.Ui.notify(false);
        this.listenToOnce(view, 'select', models => {
          this.clearView('dialog');
          if (Object.prototype.toString.call(models) !== '[object Array]') {
            models = [models];
          }
          this.selectOneOf(models);
        });
      });
    }

    /**
     * Get an empty autocomplete result.
     *
     * @protected
     * @return {Promise<[{name: ?string, id: string} & Record]>}
     */
    getOnEmptyAutocomplete() {
      return undefined;
    }
    actionCreateLink() {
      const viewName = this.getMetadata().get(['clientDefs', this.foreignScope, 'modalViews', 'edit']) || 'views/modals/edit';
      Espo.Ui.notify(' ... ');
      this.getCreateAttributesProvider()().then(attributes => {
        this.createView('dialog', viewName, {
          scope: this.foreignScope,
          fullFormDisabled: true,
          attributes: attributes
        }, view => {
          view.render().then(() => Espo.Ui.notify(false));
          this.listenToOnce(view, 'after:save', model => {
            view.close();
            this.clearView('dialog');
            this.select(model);
          });
        });
      });
    }

    /**
     * @protected
     * @param {module:model[]} models
     * @since 8.0.4
     */
    selectOneOf(models) {
      models.forEach(model => {
        this.addLinkOneOf(model.id, model.get('name'));
      });
    }
  }
  var _default = _exports.default = LinkFieldView;
});

define("views/fields/float", ["exports", "views/fields/int"], function (_exports, _int) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _int = _interopRequireDefault(_int);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/float */

  /**
   * A float field.
   *
   * @extends IntFieldView<module:views/fields/float~params>
   */
  class FloatFieldView extends _int.default {
    /**
     * @typedef {Object} module:views/fields/float~options
     * @property {
     *     module:views/fields/float~params &
     *     module:views/fields/base~params &
     *     Record
     * } [params] Parameters.
     */

    /**
     * @typedef {Object} module:views/fields/float~params
     * @property {number} [min] A max value.
     * @property {number} [max] A max value.
     * @property {boolean} [required] Required.
     * @property {boolean} [disableFormatting] Disable formatting.
     * @property {number|null} [decimalPlaces] A number of decimal places.
     */

    /**
     * @param {
     *     module:views/fields/float~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }
    type = 'float';
    editTemplate = 'fields/float/edit';
    decimalMark = '.';
    decimalPlacesRawValue = 10;

    /**
     * @inheritDoc
     * @type {Array<(function (): boolean)|string>}
     */
    validations = ['required', 'float', 'range'];

    /** @inheritDoc */
    setup() {
      super.setup();
      if (this.getPreferences().has('decimalMark')) {
        this.decimalMark = this.getPreferences().get('decimalMark');
      } else if (this.getConfig().has('decimalMark')) {
        this.decimalMark = this.getConfig().get('decimalMark');
      }
      if (!this.decimalMark) {
        this.decimalMark = '.';
      }
      if (this.decimalMark === this.thousandSeparator) {
        this.thousandSeparator = '';
      }
    }

    /** @inheritDoc */
    setupAutoNumericOptions() {
      // noinspection JSValidateTypes
      this.autoNumericOptions = {
        digitGroupSeparator: this.thousandSeparator || '',
        decimalCharacter: this.decimalMark,
        modifyValueOnWheel: false,
        selectOnFocus: false,
        decimalPlaces: this.decimalPlacesRawValue,
        decimalPlacesRawValue: this.decimalPlacesRawValue,
        allowDecimalPadding: false,
        showWarnings: false,
        formulaMode: true
      };
    }
    getValueForDisplay() {
      const value = isNaN(this.model.get(this.name)) ? null : this.model.get(this.name);
      return this.formatNumber(value);
    }
    formatNumber(value) {
      if (this.disableFormatting) {
        return value;
      }
      return this.formatNumberDetail(value);
    }
    formatNumberDetail(value) {
      if (value === null) {
        return '';
      }
      const decimalPlaces = this.params.decimalPlaces;
      if (decimalPlaces === 0) {
        value = Math.round(value);
      } else if (decimalPlaces) {
        value = Math.round(value * Math.pow(10, decimalPlaces)) / Math.pow(10, decimalPlaces);
      }
      const parts = value.toString().split(".");
      parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, this.thousandSeparator);
      if (decimalPlaces === 0) {
        return parts[0];
      } else if (decimalPlaces) {
        let decimalPartLength = 0;
        if (parts.length > 1) {
          decimalPartLength = parts[1].length;
        } else {
          parts[1] = '';
        }
        if (decimalPlaces && decimalPartLength < decimalPlaces) {
          const limit = decimalPlaces - decimalPartLength;
          for (let i = 0; i < limit; i++) {
            parts[1] += '0';
          }
        }
      }
      return parts.join(this.decimalMark);
    }
    setupMaxLength() {}
    validateFloat() {
      const value = this.model.get(this.name);
      if (isNaN(value)) {
        const msg = this.translate('fieldShouldBeFloat', 'messages').replace('{field}', this.getLabelText());
        this.showValidationMessage(msg);
        return true;
      }
    }
    parse(value) {
      value = value !== '' ? value : null;
      if (value === null) {
        return null;
      }
      value = value.split(this.thousandSeparator).join('').split(this.decimalMark).join('.');
      return parseFloat(value);
    }
    fetch() {
      let value = this.$element.val();
      value = this.parse(value);
      const data = {};
      data[this.name] = value;
      return data;
    }
  }
  var _default = _exports.default = FloatFieldView;
});

define("views/fields/date", ["exports", "views/fields/base", "moment", "ui/datepicker"], function (_exports, _base, _moment, _datepicker) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _base = _interopRequireDefault(_base);
  _moment = _interopRequireDefault(_moment);
  _datepicker = _interopRequireDefault(_datepicker);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module views/fields/date */

  /**
   * A date field.
   *
   * @extends BaseFieldView<module:views/fields/date~params>
   */
  class DateFieldView extends _base.default {
    /**
     * @typedef {Object} module:views/fields/date~options
     * @property {
     *     module:views/fields/date~params &
     *     module:views/fields/base~params &
     *     Record
     * } [params] Parameters.
     * @property {string} [otherFieldLabelText] A label text of other field. Used in before/after validations.
     */

    /**
     * @typedef {Object} module:views/fields/date~params
     * @property {boolean} [required] Required.
     * @property {boolean} [useNumericFormat] Use numeric format.
     * @property {string} [after] Validate to be after another date field.
     * @property {string} [before] Validate to be before another date field.
     * @property {boolean} [afterOrEqual] Allow an equal date for 'after' validation.
     */

    /**
     * @param {
     *     module:views/fields/date~options &
     *     module:views/fields/base~options
     * } options Options.
     */
    constructor(options) {
      super(options);
    }
    type = 'date';
    listTemplate = 'fields/date/list';
    listLinkTemplate = 'fields/date/list-link';
    detailTemplate = 'fields/date/detail';
    editTemplate = 'fields/date/edit';
    searchTemplate = 'fields/date/search';

    /**
     * @inheritDoc
     * @type {Array<(function (): boolean)|string>}
     */
    validations = ['required', 'date', 'after', 'before'];
    searchTypeList = ['lastSevenDays', 'ever', 'isEmpty', 'currentMonth', 'lastMonth', 'nextMonth', 'currentQuarter', 'lastQuarter', 'currentYear', 'lastYear', 'today', 'past', 'future', 'lastXDays', 'nextXDays', 'olderThanXDays', 'afterXDays', 'on', 'after', 'before', 'between'];
    initialSearchIsNotIdle = true;

    /**
     * @private
     * @type {import('ui/datepicker').default}
     */
    datepicker;
    setup() {
      super.setup();
      if (this.getConfig().get('fiscalYearShift')) {
        this.searchTypeList = Espo.Utils.clone(this.searchTypeList);
        if (this.getConfig().get('fiscalYearShift') % 3 !== 0) {
          this.searchTypeList.push('currentFiscalQuarter');
          this.searchTypeList.push('lastFiscalQuarter');
        }
        this.searchTypeList.push('currentFiscalYear');
        this.searchTypeList.push('lastFiscalYear');
      }
      if (this.params.after) {
        this.listenTo(this.model, `change:${this.params.after}`, async () => {
          if (!this.isEditMode()) {
            return;
          }
          await this.whenRendered();

          // Timeout prevents the picker popping one when the duration field adjusts the date end.
          setTimeout(() => {
            this.datepicker.setStartDate(this.getStartDateForDatePicker());
          }, 100);
        });
      }

      /** @protected */
      this.useNumericFormat = this.getConfig().get('readableDateFormatDisabled') || this.params.useNumericFormat;
    }

    // noinspection JSCheckFunctionSignatures
    data() {
      const data = super.data();
      data.dateValue = this.getDateStringValue();
      data.isNone = data.dateValue === null;
      if (data.dateValue === -1) {
        data.dateValue = null;
        data.isLoading = true;
      }
      if (this.isSearchMode()) {
        const value = this.getSearchParamsData().value || this.searchParams.dateValue;
        const valueTo = this.getSearchParamsData().valueTo || this.searchParams.dateValueTo;
        data.dateValue = this.getDateTime().toDisplayDate(value);
        data.dateValueTo = this.getDateTime().toDisplayDate(valueTo);
        if (['lastXDays', 'nextXDays', 'olderThanXDays', 'afterXDays'].includes(this.getSearchType())) {
          data.number = this.searchParams.value;
        }
      }
      if (this.isListMode()) {
        data.titleDateValue = data.dateValue;
      }
      if (this.useNumericFormat) {
        data.useNumericFormat = true;
      }

      // noinspection JSValidateTypes
      return data;
    }
    setupSearch() {
      this.addHandler('change', 'select.search-type', (e, /** HTMLSelectElement */target) => {
        this.handleSearchType(target.value);
      });
    }
    stringifyDateValue(value) {
      if (!value) {
        if (this.mode === this.MODE_EDIT || this.mode === this.MODE_SEARCH || this.mode === this.MODE_LIST || this.mode === this.MODE_LIST_LINK) {
          return '';
        }
        return null;
      }
      if (this.mode === this.MODE_LIST || this.mode === this.MODE_DETAIL || this.mode === this.MODE_LIST_LINK) {
        return this.convertDateValueForDetail(value);
      }
      return this.getDateTime().toDisplayDate(value);
    }
    convertDateValueForDetail(value) {
      if (this.useNumericFormat) {
        return this.getDateTime().toDisplayDate(value);
      }
      const timezone = this.getDateTime().getTimeZone();
      const internalDateTimeFormat = this.getDateTime().internalDateTimeFormat;
      const readableFormat = this.getDateTime().getReadableDateFormat();
      const valueWithTime = value + ' 00:00:00';
      const today = _moment.default.tz(timezone).startOf('day');
      let dateTime = _moment.default.tz(valueWithTime, internalDateTimeFormat, timezone);
      const temp = today.clone();
      const ranges = {
        'today': [temp.unix(), temp.add(1, 'days').unix()],
        'tomorrow': [temp.unix(), temp.add(1, 'days').unix()],
        'yesterday': [temp.add(-3, 'days').unix(), temp.add(1, 'days').unix()]
      };
      if (dateTime.unix() >= ranges['today'][0] && dateTime.unix() < ranges['today'][1]) {
        return this.translate('Today');
      }
      if (dateTime.unix() >= ranges['tomorrow'][0] && dateTime.unix() < ranges['tomorrow'][1]) {
        return this.translate('Tomorrow');
      }
      if (dateTime.unix() >= ranges['yesterday'][0] && dateTime.unix() < ranges['yesterday'][1]) {
        return this.translate('Yesterday');
      }

      // Need to use UTC, otherwise there's a DST issue with old dates.
      dateTime = _moment.default.utc(valueWithTime, internalDateTimeFormat);
      if (dateTime.format('YYYY') === today.format('YYYY')) {
        return dateTime.format(readableFormat);
      }
      return dateTime.format(readableFormat + ', YYYY');
    }
    getDateStringValue() {
      if (this.mode === this.MODE_DETAIL && !this.model.has(this.name)) {
        return -1;
      }
      const value = this.model.get(this.name);
      return this.stringifyDateValue(value);
    }

    /**
     * @protected
     * @return {string|undefined}
     */
    getStartDateForDatePicker() {
      if (!this.isEditMode() || !this.params.after) {
        return undefined;
      }

      /** @type {string} */
      let date = this.model.attributes[this.params.after];
      if (date == null) {
        return undefined;
      }
      if (date.length > 10) {
        date = this.getDateTime().toDisplay(date);
        [date] = date.split(' ');
        return date;
      }
      return this.getDateTime().toDisplayDate(date);
    }
    afterRender() {
      if (this.mode === this.MODE_EDIT || this.mode === this.MODE_SEARCH) {
        this.$element = this.$el.find(`[data-name="${this.name}"]`);

        /** @type {HTMLElement} */
        const element = this.$element.get(0);
        const options = {
          format: this.getDateTime().dateFormat,
          weekStart: this.getDateTime().weekStart,
          startDate: this.getStartDateForDatePicker(),
          todayButton: this.getConfig().get('datepickerTodayButton') || false
        };
        this.datepicker = undefined;
        if (element) {
          this.datepicker = new _datepicker.default(element, {
            ...options,
            onChange: () => this.trigger('change')
          });
        }
        if (this.mode === this.MODE_SEARCH) {
          this.$el.find('select.search-type').on('change', () => this.trigger('change'));
          this.$el.find('input.number').on('change', () => this.trigger('change'));
          const element = this.$el.find('.input-group.additional').get(0);
          if (element) {
            new _datepicker.default(element, options);
            this.initDatePickerEventHandlers('input.filter-from');
            this.initDatePickerEventHandlers('input.filter-to');
          }
        }
        this.$element.parent().find('button.date-picker-btn').on('click', () => {
          this.datepicker.show();
        });
        if (this.mode === this.MODE_SEARCH) {
          const $searchType = this.$el.find('select.search-type');
          this.handleSearchType($searchType.val());
        }
      }
    }

    /**
     * @private
     * @param {string} selector
     */
    initDatePickerEventHandlers(selector) {
      const $input = this.$el.find(selector);
      $input.on('change', /** Record */e => {
        this.trigger('change');
        if (e.isTrigger) {
          if (document.activeElement !== $input.get(0)) {
            $input.focus();
          }
        }
      });
    }
    handleSearchType(type) {
      this.$el.find('div.primary').addClass('hidden');
      this.$el.find('div.additional').addClass('hidden');
      this.$el.find('div.additional-number').addClass('hidden');
      if (['on', 'notOn', 'after', 'before'].includes(type)) {
        this.$el.find('div.primary').removeClass('hidden');
      } else if (['lastXDays', 'nextXDays', 'olderThanXDays', 'afterXDays'].includes(type)) {
        this.$el.find('div.additional-number').removeClass('hidden');
      } else if (type === 'between') {
        this.$el.find('div.primary').addClass('hidden');
        this.$el.find('div.additional').removeClass('hidden');
      }
    }
    parseDate(string) {
      return this.getDateTime().fromDisplayDate(string);
    }

    /**
     * @param {string} string
     * @return {string|-1|null}
     */
    parse(string) {
      if (!string) {
        return null;
      }
      return this.parseDate(string);
    }

    /** @inheritDoc */
    fetch() {
      const data = {};
      data[this.name] = this.parse(this.$element.val());
      return data;
    }

    /** @inheritDoc */
    fetchSearch() {
      const type = this.fetchSearchType();
      if (type === 'between') {
        const valueFrom = this.parseDate(this.$el.find('input.filter-from').val());
        const valueTo = this.parseDate(this.$el.find('input.filter-to').val());
        if (!valueFrom || !valueTo) {
          return null;
        }
        return {
          type: type,
          value: [valueFrom, valueTo],
          data: {
            value: valueFrom,
            valueTo: valueTo
          }
        };
      }
      let data;
      const value = this.parseDate(this.$element.val());
      if (['lastXDays', 'nextXDays', 'olderThanXDays', 'afterXDays'].includes(type)) {
        const number = this.$el.find('input.number').val();
        data = {
          type: type,
          value: number,
          date: true
        };
      } else if (['on', 'notOn', 'after', 'before'].includes(type)) {
        if (!value) {
          return null;
        }
        data = {
          type: type,
          value: value,
          data: {
            value: value
          }
        };
      } else if (type === 'isEmpty') {
        data = {
          type: 'isNull',
          data: {
            type: type
          }
        };
      } else {
        data = {
          type: type,
          date: true
        };
      }
      return data;
    }
    getSearchType() {
      return this.getSearchParamsData().type || this.searchParams.typeFront || this.searchParams.type;
    }
    validateRequired() {
      if (!this.isRequired()) {
        return;
      }
      if (this.model.get(this.name) === null) {
        const msg = this.translate('fieldIsRequired', 'messages').replace('{field}', this.getLabelText());
        this.showValidationMessage(msg);
        return true;
      }
    }

    // noinspection JSUnusedGlobalSymbols
    validateDate() {
      if (this.model.get(this.name) === -1) {
        const msg = this.translate('fieldShouldBeDate', 'messages').replace('{field}', this.getLabelText());
        this.showValidationMessage(msg);
        return true;
      }
    }

    // noinspection JSUnusedGlobalSymbols
    validateAfter() {
      const field = this.params.after;
      if (!field) {
        return false;
      }
      const value = this.model.get(this.name);
      const otherValue = this.model.get(field);
      if (!(value && otherValue)) {
        return false;
      }
      const unix = (0, _moment.default)(value).unix();
      const otherUnix = (0, _moment.default)(otherValue).unix();
      if (this.params.afterOrEqual && unix === otherUnix) {
        return false;
      }
      if (unix <= otherUnix) {
        const otherFieldLabelText = this.options.otherFieldLabelText || this.translate(field, 'fields', this.entityType);
        const msg = this.translate('fieldShouldAfter', 'messages').replace('{field}', this.getLabelText()).replace('{otherField}', otherFieldLabelText);
        this.showValidationMessage(msg);
        return true;
      }
      return false;
    }

    // noinspection JSUnusedGlobalSymbols
    validateBefore() {
      const field = this.params.before;
      if (!field) {
        return false;
      }
      const value = this.model.get(this.name);
      const otherValue = this.model.get(field);
      if (!(value && otherValue)) {
        return;
      }
      if ((0, _moment.default)(value).unix() >= (0, _moment.default)(otherValue).unix()) {
        const msg = this.translate('fieldShouldBefore', 'messages').replace('{field}', this.getLabelText()).replace('{otherField}', this.translate(field, 'fields', this.entityType));
        this.showValidationMessage(msg);
        return true;
      }
    }
  }
  var _default = _exports.default = DateFieldView;
});

define("views/detail/modes", ["exports", "view"], function (_exports, _view) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _view = _interopRequireDefault(_view);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  class DetailModesView extends _view.default {
    // language=Handlebars
    templateContent = `
        <div class="button-container clearfix">
            <div class="btn-group">
                {{#each modeDataList}}
                    <button
                        class="btn btn-text btn-xs-wide{{#if active}} active{{/if}}"
                        data-action="switchMode"
                        data-value="{{name}}"
                        {{#if ../disabled}}disabled="disabled"{{/if}}
                    >{{label}}</button>
                {{/each}}
            </div>
        </div>
    `;

    /** @private */
    disabled = false;

    /**
     * @param {{
     *     modeList: string[],
     *     mode: string,
     *     scope: string.
     * }} options
     */
    constructor(options) {
      super(options);

      /** @private */
      this.modeList = options.modeList;
      /** @private */
      this.mode = options.mode;
      /** @private */
      this.scope = options.scope;

      /**
       * @private
       * @type {Object.<string, boolean>}
       */
      this.hiddenMap = {};
    }
    data() {
      return {
        disabled: this.disabled,
        modeDataList: this.modeList.filter(mode => !this.hiddenMap[mode] || mode === this.mode).map(mode => ({
          name: mode,
          active: mode === this.mode,
          label: this.translate(mode, 'detailViewModes', this.scope)
        }))
      };
    }

    /**
     * Change mode.
     *
     * @param {string} mode
     * @return {Promise}
     */
    changeMode(mode) {
      this.mode = mode;
      return this.reRender();
    }

    /**
     * Hide a mode.
     *
     * @param {string} mode
     */
    async hideMode(mode) {
      this.hiddenMap[mode] = true;
      await this.reRender();
    }

    /**
     * Show a mode.
     *
     * @param {string} mode
     */
    async showMode(mode) {
      delete this.hiddenMap[mode];
      await this.reRender();
    }

    /**
     * Disable.
     *
     * @return {Promise}
     */
    disable() {
      this.disabled = true;
      return this.reRender();
    }

    /**
     * Enable.
     *
     * @return {Promise}
     */
    enable() {
      this.disabled = false;
      return this.reRender();
    }
  }
  var _default = _exports.default = DetailModesView;
});

define("helpers/file-upload", ["exports", "di", "models/settings"], function (_exports, _di, _settings) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _settings = _interopRequireDefault(_settings);
  let _init_config, _init_extra_config;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  /** @module helpers/file-upload */
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * A file-upload helper.
   */
  class FileUploadExport {
    static {
      [_init_config, _init_extra_config] = _applyDecs(this, [], [[(0, _di.inject)(_settings.default), 0, "config"]]).e;
    }
    constructor() {
      _init_extra_config(this);
    }
    /**
     * @private
     * @type {Settings}
     */
    config = _init_config(this);

    /**
     * @typedef {Object} module:helpers/file-upload~Options
     *
     * @property {function(number):void} [afterChunkUpload] After every chunk is uploaded.
     * @property {function(module:model):void} [afterAttachmentSave] After an attachment is saved.
     * @property {{isCanceled?: boolean}} [mediator] A mediator.
     */

    /**
     * Upload.
     *
     * @param {File} file A file.
     * @param {module:model} attachment An attachment model.
     * @param {module:helpers/file-upload~Options} [options] Options.
     * @returns {Promise}
     */
    upload(file, attachment, options) {
      options = options || {};
      options.afterChunkUpload = options.afterChunkUpload || (() => {});
      options.afterAttachmentSave = options.afterAttachmentSave || (() => {});
      options.mediator = options.mediator || {};
      attachment.set('name', file.name);
      attachment.set('type', file.type || 'text/plain');
      attachment.set('size', file.size);
      if (this._useChunks(file)) {
        return this._uploadByChunks(file, attachment, options);
      }
      return new Promise((resolve, reject) => {
        const fileReader = new FileReader();
        fileReader.onload = e => {
          attachment.set('file', e.target.result);
          attachment.save({}, {
            timeout: 0
          }).then(() => resolve()).catch(() => reject());
        };
        fileReader.readAsDataURL(file);
      });
    }

    /**
     * @private
     */
    _uploadByChunks(file, attachment, options) {
      return new Promise((resolve, reject) => {
        attachment.set('isBeingUploaded', true);
        attachment.save().then(() => {
          options.afterAttachmentSave(attachment);
          return this._uploadChunks(file, attachment, resolve, reject, options);
        }).catch(() => reject());
      });
    }

    /**
     * @private
     */
    _uploadChunks(file, attachment, resolve, reject, options, start) {
      start = start || 0;
      let end = start + this._getChunkSize() + 1;
      if (end > file.size) {
        end = file.size;
      }
      if (options.mediator.isCanceled) {
        reject();
        return;
      }
      const blob = file.slice(start, end);
      const fileReader = new FileReader();
      fileReader.onloadend = e => {
        if (e.target.readyState !== FileReader.DONE) {
          return;
        }
        Espo.Ajax.postRequest('Attachment/chunk/' + attachment.id, e.target.result, {
          headers: {
            contentType: 'multipart/form-data'
          }
        }).then(() => {
          options.afterChunkUpload(end);
          if (end === file.size) {
            resolve();
            return;
          }
          this._uploadChunks(file, attachment, resolve, reject, options, end);
        }).catch(() => reject());
      };
      fileReader.readAsDataURL(blob);
    }

    /**
     * @private
     */
    _useChunks(file) {
      const chunkSize = this._getChunkSize();
      if (!chunkSize) {
        return false;
      }
      if (file.size > chunkSize) {
        return true;
      }
      return false;
    }

    /**
     * @private
     */
    _getChunkSize() {
      return (this.config.get('attachmentUploadChunkSize') || 0) * 1024 * 1024;
    }
  }
  var _default = _exports.default = FileUploadExport;
});

define("helpers/record/select-related", ["exports", "di", "metadata"], function (_exports, _di, _metadata) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _metadata = _interopRequireDefault(_metadata);
  let _init_metadata, _init_extra_metadata;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * @internal
   */
  class SelectRelatedHelper {
    static {
      [_init_metadata, _init_extra_metadata] = _applyDecs(this, [], [[(0, _di.inject)(_metadata.default), 0, "metadata"]]).e;
    }
    /**
     * @private
     * @type {Metadata}
     */
    metadata = _init_metadata(this);

    /**
     * @param {import('view').default} view
     */
    constructor(view) {
      _init_extra_metadata(this);
      /** @private */
      this.view = view;
    }

    /**
     * @param {import('model').default} model
     * @param {string} link
     * @param {{
     *     foreignEntityType?: string,
     *     massSelect?: boolean,
     *     primaryFilterName?: string,
     *     boolFilterList?: string[]|string,
     *     viewKey?: string,
     *     hasCreate?: boolean,
     *     onCreate?: function(): void,
     * }} options
     */
    process(model, link, options = {}) {
      if (!options.foreignEntityType && !model.defs['links'][link]) {
        throw new Error(`Link ${link} does not exist.`);
      }
      const scope = options.foreignEntityType || model.defs['links'][link].entity;

      /** @var {Object.<string, *>} */
      const panelDefs = this.metadata.get(['clientDefs', model.entityType, 'relationshipPanels', link]) || {};
      const massRelateEnabled = options.massSelect || panelDefs.massSelect;
      let advanced = {};
      const foreignLink = model.getLinkParam(link, 'foreign');
      if (foreignLink && scope) {
        // Select only records not related with any.
        const foreignLinkType = this.metadata.get(['entityDefs', scope, 'links', foreignLink, 'type']);
        const foreignLinkFieldType = this.metadata.get(['entityDefs', scope, 'fields', foreignLink, 'type']);
        if (['belongsTo', 'belongsToParent'].includes(foreignLinkType) && foreignLinkFieldType && !advanced[foreignLink] && ['link', 'linkParent'].includes(foreignLinkFieldType)) {
          advanced[foreignLink] = {
            type: 'isNull',
            attribute: foreignLink + 'Id',
            data: {
              type: 'isEmpty'
            }
          };
        }
      }
      let primaryFilterName = options.primaryFilterName || null;
      if (typeof primaryFilterName === 'function') {
        primaryFilterName = primaryFilterName.call(this);
      }
      let dataBoolFilterList = options.boolFilterList;
      if (typeof options.boolFilterList === 'string') {
        dataBoolFilterList = options.boolFilterList.split(',');
      }
      let boolFilterList = dataBoolFilterList || panelDefs.selectBoolFilterList;
      if (typeof boolFilterList === 'function') {
        boolFilterList = boolFilterList.call(this);
      }
      boolFilterList = Espo.Utils.clone(boolFilterList);
      primaryFilterName = primaryFilterName || panelDefs.selectPrimaryFilterName || null;
      const viewKey = options.viewKey || 'select';
      const viewName = panelDefs.selectModalView || this.metadata.get(['clientDefs', scope, 'modalViews', viewKey]) || 'views/modals/select-records';
      Espo.Ui.notify(' ... ');
      const handler = panelDefs.selectHandler || null;
      new Promise(resolve => {
        if (!handler) {
          resolve({});
          return;
        }
        Espo.loader.requirePromise(handler).then(Handler => new Handler(this.view.getHelper())).then(/** module:handlers/select-related */handler => {
          handler.getFilters(model).then(filters => resolve(filters));
        });
      }).then(filters => {
        advanced = {
          ...advanced,
          ...(filters.advanced || {})
        };
        if (boolFilterList || filters.bool) {
          boolFilterList = [...(boolFilterList || []), ...(filters.bool || [])];
        }
        if (filters.primary && !primaryFilterName) {
          primaryFilterName = filters.primary;
        }
        const orderBy = filters.orderBy || panelDefs.selectOrderBy;
        const orderDirection = filters.orderBy ? filters.order : panelDefs.selectOrderDirection;
        const createButton = options.hasCreate === true && options.onCreate !== undefined;

        /** @type {import('views/modals/select-records').default} */
        let modalView;
        this.view.createView('dialogSelectRelated', viewName, {
          scope: scope,
          multiple: true,
          filters: advanced,
          massRelateEnabled: massRelateEnabled,
          primaryFilterName: primaryFilterName,
          boolFilterList: boolFilterList,
          mandatorySelectAttributeList: panelDefs.selectMandatoryAttributeList,
          layoutName: panelDefs.selectLayout,
          orderBy: orderBy,
          orderDirection: orderDirection,
          createButton: createButton,
          onCreate: () => {
            modalView.close();
            if (options.onCreate) {
              options.onCreate();
            }
          }
        }, view => {
          modalView = view;
          view.render();
          Espo.Ui.notify(false);
          this.view.listenToOnce(view, 'select', selectObj => {
            const data = {};
            if (Object.prototype.toString.call(selectObj) === '[object Array]') {
              const ids = [];
              selectObj.forEach(model => ids.push(model.id));
              data.ids = ids;
            } else if (selectObj.massRelate) {
              data.massRelate = true;
              data.where = selectObj.where;
              data.searchParams = selectObj.searchParams;
            } else {
              data.id = selectObj.id;
            }
            const url = `${model.entityType}/${model.id}/${link}`;
            Espo.Ajax.postRequest(url, data).then(() => {
              Espo.Ui.success(this.view.translate('Linked'));
              model.trigger(`update-related:${link}`);
              model.trigger('after:relate');
              model.trigger(`after:relate:${link}`);
            });
          });
        });
      });
    }
  }
  var _default = _exports.default = SelectRelatedHelper;
});

define("helpers/record/create-related", ["exports", "di", "metadata"], function (_exports, _di, _metadata) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _metadata = _interopRequireDefault(_metadata);
  let _init_metadata, _init_extra_metadata;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  function _applyDecs(e, t, n, r, o, i) { var a, c, u, s, f, l, p, d = Symbol.metadata || Symbol.for("Symbol.metadata"), m = Object.defineProperty, h = Object.create, y = [h(null), h(null)], v = t.length; function g(t, n, r) { return function (o, i) { n && (i = o, o = e); for (var a = 0; a < t.length; a++) i = t[a].apply(o, r ? [i] : []); return r ? i : o; }; } function b(e, t, n, r) { if ("function" != typeof e && (r || void 0 !== e)) throw new TypeError(t + " must " + (n || "be") + " a function" + (r ? "" : " or undefined")); return e; } function applyDec(e, t, n, r, o, i, u, s, f, l, p) { function d(e) { if (!p(e)) throw new TypeError("Attempted to access private element on non-instance"); } var h = [].concat(t[0]), v = t[3], w = !u, D = 1 === o, S = 3 === o, j = 4 === o, E = 2 === o; function I(t, n, r) { return function (o, i) { return n && (i = o, o = e), r && r(o), P[t].call(o, i); }; } if (!w) { var P = {}, k = [], F = S ? "get" : j || D ? "set" : "value"; if (f ? (l || D ? P = { get: _setFunctionName(function () { return v(this); }, r, "get"), set: function (e) { t[4](this, e); } } : P[F] = v, l || _setFunctionName(P[F], r, E ? "" : F)) : l || (P = Object.getOwnPropertyDescriptor(e, r)), !l && !f) { if ((c = y[+s][r]) && 7 != (c ^ o)) throw Error("Decorating two elements with the same name (" + P[F].name + ") is not supported yet"); y[+s][r] = o < 3 ? 1 : o; } } for (var N = e, O = h.length - 1; O >= 0; O -= n ? 2 : 1) { var T = b(h[O], "A decorator", "be", !0), z = n ? h[O - 1] : void 0, A = {}, H = { kind: ["field", "accessor", "method", "getter", "setter", "class"][o], name: r, metadata: a, addInitializer: function (e, t) { if (e.v) throw new TypeError("attempted to call addInitializer after decoration was finished"); b(t, "An initializer", "be", !0), i.push(t); }.bind(null, A) }; if (w) c = T.call(z, N, H), A.v = 1, b(c, "class decorators", "return") && (N = c);else if (H.static = s, H.private = f, c = H.access = { has: f ? p.bind() : function (e) { return r in e; } }, j || (c.get = f ? E ? function (e) { return d(e), P.value; } : I("get", 0, d) : function (e) { return e[r]; }), E || S || (c.set = f ? I("set", 0, d) : function (e, t) { e[r] = t; }), N = T.call(z, D ? { get: P.get, set: P.set } : P[F], H), A.v = 1, D) { if ("object" == typeof N && N) (c = b(N.get, "accessor.get")) && (P.get = c), (c = b(N.set, "accessor.set")) && (P.set = c), (c = b(N.init, "accessor.init")) && k.unshift(c);else if (void 0 !== N) throw new TypeError("accessor decorators must return an object with get, set, or init properties or undefined"); } else b(N, (l ? "field" : "method") + " decorators", "return") && (l ? k.unshift(N) : P[F] = N); } return o < 2 && u.push(g(k, s, 1), g(i, s, 0)), l || w || (f ? D ? u.splice(-1, 0, I("get", s), I("set", s)) : u.push(E ? P[F] : b.call.bind(P[F])) : m(e, r, P)), N; } function w(e) { return m(e, d, { configurable: !0, enumerable: !0, value: a }); } return void 0 !== i && (a = i[d]), a = h(null == a ? null : a), f = [], l = function (e) { e && f.push(g(e)); }, p = function (t, r) { for (var i = 0; i < n.length; i++) { var a = n[i], c = a[1], l = 7 & c; if ((8 & c) == t && !l == r) { var p = a[2], d = !!a[3], m = 16 & c; applyDec(t ? e : e.prototype, a, m, d ? "#" + p : _toPropertyKey(p), l, l < 2 ? [] : t ? s = s || [] : u = u || [], f, !!t, d, r, t && d ? function (t) { return _checkInRHS(t) === e; } : o); } } }, p(8, 0), p(0, 0), p(8, 1), p(0, 1), l(u), l(s), c = f, v || w(e), { e: c, get c() { var n = []; return v && [w(e = applyDec(e, [t], r, e.name, 5, n)), g(n, 1)]; } }; }
  function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
  function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
  function _setFunctionName(e, t, n) { "symbol" == typeof t && (t = (t = t.description) ? "[" + t + "]" : ""); try { Object.defineProperty(e, "name", { configurable: !0, value: n ? n + " " + t : t }); } catch (e) {} return e; }
  function _checkInRHS(e) { if (Object(e) !== e) throw TypeError("right-hand side of 'in' should be an object, got " + (null !== e ? typeof e : "null")); return e; }
  /**
   * @internal
   */
  class CreateRelatedHelper {
    static {
      [_init_metadata, _init_extra_metadata] = _applyDecs(this, [], [[(0, _di.inject)(_metadata.default), 0, "metadata"]]).e;
    }
    /**
     * @private
     * @type {Metadata}
     */
    metadata = _init_metadata(this);

    /**
     * @param {import('view').default} view
     */
    constructor(view) {
      _init_extra_metadata(this);
      /** @private */
      this.view = view;
    }

    /**
     * @param {import('model').default} model
     * @param {string} link
     * @param {Record} options
     */
    process(model, link, options = {}) {
      const scope = model.defs['links'][link].entity;
      const foreignLink = model.defs['links'][link].foreign;
      let attributes = {};
      const attributeMap = this.metadata.get(['clientDefs', model.entityType, 'relationshipPanels', link, 'createAttributeMap']) || {};
      Object.keys(attributeMap).forEach(attr => attributes[attributeMap[attr]] = model.get(attr));
      Espo.Ui.notify(' ... ');
      const handler = this.metadata.get(['clientDefs', model.entityType, 'relationshipPanels', link, 'createHandler']);
      new Promise(resolve => {
        if (!handler) {
          resolve({});
          return;
        }
        Espo.loader.requirePromise(handler).then(Handler => new Handler(this.view.getHelper())).then(handler => {
          handler.getAttributes(model).then(attributes => resolve(attributes));
        });
      }).then(additionalAttributes => {
        attributes = {
          ...attributes,
          ...additionalAttributes
        };
        const viewName = this.metadata.get(['clientDefs', scope, 'modalViews', 'edit']) || 'views/modals/edit';
        this.view.createView('quickCreate', viewName, {
          scope: scope,
          relate: {
            model: model,
            link: foreignLink
          },
          attributes: attributes
        }, view => {
          view.render();
          view.notify(false);
          this.view.listenToOnce(view, 'after:save', () => {
            if (options.fromSelectRelated) {
              setTimeout(() => this.view.clearView('dialogSelectRelated'), 25);
            }
            model.trigger(`update-related:${link}`);
            model.trigger('after:relate');
            model.trigger(`after:relate:${link}`);
          });
        });
      });
    }
  }
  var _default = _exports.default = CreateRelatedHelper;
});

define("helpers/misc/stored-text-search", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module helpers/misc/stored-text-search */

  class _default {
    /**
     * @param {module:storage} storage
     * @param {string} scope
     * @param {Number} [maxCount]
     */
    constructor(scope, storage, maxCount) {
      this.scope = scope;
      this.storage = storage;
      this.key = 'textSearches';
      this.maxCount = maxCount || 100;
      /** @type {string[]|null} */
      this.list = null;
    }

    /**
     * Match.
     *
     * @param {string} text
     * @param {Number} [limit]
     * @return {string[]}
     */
    match(text, limit) {
      text = text.toLowerCase().trim();
      const list = this.get();
      const matchedList = [];
      for (const item of list) {
        if (item.toLowerCase().startsWith(text)) {
          matchedList.push(item);
        }
        if (limit !== undefined && matchedList.length === limit) {
          break;
        }
      }
      return matchedList;
    }

    /**
     * Get stored text filters.
     *
     * @private
     * @return {string[]}
     */
    get() {
      if (this.list === null) {
        this.list = this.getFromStorage();
      }
      return this.list;
    }

    /**
     * @private
     * @return {string[]}
     */
    getFromStorage() {
      /** @var {string[]} */
      return this.storage.get(this.key, this.scope) || [];
    }

    /**
     * Store a text filter.
     *
     * @param {string} text
     */
    store(text) {
      text = text.trim();
      let list = this.getFromStorage();
      const index = list.indexOf(text);
      if (index !== -1) {
        list.splice(index, 1);
      }
      list.unshift(text);
      if (list.length > this.maxCount) {
        list = list.slice(0, this.maxCount);
      }
      this.list = list;
      this.storage.set(this.key, this.scope, list);
    }

    /**
     * Remove a text filter.
     *
     * @param {string} text
     */
    remove(text) {
      text = text.trim();
      const list = this.getFromStorage();
      const index = list.indexOf(text);
      if (index === -1) {
        return;
      }
      list.splice(index, 1);
      this.list = list;
      this.storage.set(this.key, this.scope, list);
    }
  }
  _exports.default = _default;
});

define("web-socket-manager", ["exports", "js-base64"], function (_exports, _jsBase) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _jsBase = _interopRequireDefault(_jsBase);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module web-socket-manager */

  /**
   * A web-socket manager.
   */
  class WebSocketManager {
    /**
     * @private
     * @type {number}
     */
    pingInterval = 60;

    /**
     * @private
     * @type {number}
     */
    reconnectInterval = 3;

    /**
     * @private
     */
    pingTimeout;

    /**
     * @param {module:models/settings} config A config.
     */
    constructor(config) {
      /**
       * @private
       * @type {module:models/settings}
       */
      this.config = config;

      /**
       * @private
       * @type {{category: string, callback: Function}[]}
       */
      this.subscribeQueue = [];

      /**
       * @private
       * @type {{category: string, callback: Function}[]}
       */
      this.subscriptions = [];

      /**
       * @private
       * @type {boolean}
       */
      this.isConnected = false;

      /**
       * @private
       */
      this.connection = null;

      /**
       * @private
       * @type {string}
       */
      this.url = '';

      /**
       * @private
       * @type {string}
       */
      this.protocolPart = '';
      const url = this.config.get('webSocketUrl');
      if (url) {
        if (url.indexOf('wss://') === 0) {
          this.url = url.substring(6);
          this.protocolPart = 'wss://';
        } else {
          this.url = url.substring(5);
          this.protocolPart = 'ws://';
        }
      } else {
        const siteUrl = this.config.get('siteUrl') || '';
        if (siteUrl.indexOf('https://') === 0) {
          this.url = siteUrl.substring(8);
          this.protocolPart = 'wss://';
        } else {
          this.url = siteUrl.substring(7);
          this.protocolPart = 'ws://';
        }
        if (~this.url.indexOf('/')) {
          this.url = this.url.replace(/\/$/, '');
        }
        const port = this.protocolPart === 'wss://' ? 443 : 8080;
        const si = this.url.indexOf('/');
        if (~si) {
          this.url = this.url.substring(0, si) + ':' + port;
        } else {
          this.url += ':' + port;
        }
        if (this.protocolPart === 'wss://') {
          this.url += '/wss';
        }
      }
    }

    /**
     * Connect.
     *
     * @param {string} auth An auth string.
     * @param {string} userId A user ID.
     */
    connect(auth, userId) {
      const authArray = _jsBase.default.decode(auth).split(':');
      const authToken = authArray[1];
      const url = `${this.protocolPart + this.url}?authToken=${authToken}&userId=${userId}`;
      try {
        this.connectInternal(auth, userId, url);
      } catch (e) {
        console.error(e.message);
        this.connection = null;
      }
    }

    /**
     * @private
     * @param {string} auth
     * @param {string} userId
     * @param {string} url
     */
    connectInternal(auth, userId, url) {
      this.connection = new ab.Session(url, () => {
        this.isConnected = true;
        this.subscribeQueue.forEach(item => {
          this.subscribe(item.category, item.callback);
        });
        this.subscribeQueue = [];
        this.schedulePing();
      }, code => {
        if (code === ab.CONNECTION_CLOSED) {
          this.subscribeQueue = [];
        }
        if (code === ab.CONNECTION_LOST || code === ab.CONNECTION_UNREACHABLE) {
          if (this.isConnected) {
            this.subscribeQueue = this.subscriptions;
            this.subscriptions = [];
          }
          setTimeout(() => this.connect(auth, userId), this.reconnectInterval * 1000);
        }
        this.isConnected = false;
      }, {
        skipSubprotocolCheck: true
      });
    }

    /**
     * Subscribe to a topic.
     *
     * @param {string} category A topic.
     * @param {function(string, *): void} callback A callback.
     */
    subscribe(category, callback) {
      if (!this.connection) {
        return;
      }
      if (!this.isConnected) {
        this.subscribeQueue.push({
          category: category,
          callback: callback
        });
        return;
      }
      try {
        this.connection.subscribe(category, callback);
        this.subscriptions.push({
          category: category,
          callback: callback
        });
      } catch (e) {
        if (e.message) {
          console.error(e.message);
        } else {
          console.error("WebSocket: Could not subscribe to " + category + ".");
        }
      }
    }

    /**
     * Unsubscribe.
     *
     * @param {string} category A topic.
     * @param {Function} [callback] A callback.
     */
    unsubscribe(category, callback) {
      if (!this.connection) {
        return;
      }
      this.subscribeQueue = this.subscribeQueue.filter(item => {
        return item.category !== category && item.callback !== callback;
      });
      this.subscriptions = this.subscriptions.filter(item => {
        return item.category !== category && item.callback !== callback;
      });
      try {
        this.connection.unsubscribe(category, callback);
      } catch (e) {
        if (e.message) {
          console.error(e.message);
        } else {
          console.error("WebSocket: Could not unsubscribe from " + category + ".");
        }
      }
    }

    /**
     * Close a connection.
     */
    close() {
      this.stopPing();
      if (!this.connection) {
        return;
      }
      this.subscribeQueue = [];
      this.subscriptions = [];
      try {
        this.connection.close();
      } catch (e) {
        console.error(e.message);
      }
      this.isConnected = false;
    }

    /**
     * @private
     */
    stopPing() {
      this.pingTimeout = undefined;
    }

    /**
     * @private
     */
    schedulePing() {
      //ab._debugws = true;

      if (!this.connection) {
        this.stopPing();
        return;
      }
      this.pingTimeout = setTimeout(() => {
        if (!this.connection) {
          return;
        }
        this.connection.publish('', '');
        this.schedulePing();
      }, this.pingInterval * 1000);
    }
  }
  var _default = _exports.default = WebSocketManager;
});

define("ui", ["exports", "marked", "dompurify", "jquery"], function (_exports, _marked, _dompurify, _jquery) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _dompurify = _interopRequireDefault(_dompurify);
  _jquery = _interopRequireDefault(_jquery);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module ui */

  /**
   * Dialog parameters.
   *
   * @typedef {Object} module:ui.Dialog~Params
   *
   * @property {string} [className='dialog'] A class-name or multiple space separated.
   * @property {'static'|true|false} [backdrop='static'] A backdrop.
   * @property {boolean} [closeButton=true] A close button.
   * @property {boolean} [collapseButton=false] A collapse button.
   * @property {string|null} [header] A header HTML.
   * @property {string} [body] A body HTML.
   * @property {number|null} [width] A width.
   * @property {boolean} [removeOnClose=true] To remove on close.
   * @property {boolean} [draggable=false] Is draggable.
   * @property {function(): void} [onRemove] An on-remove callback.
   * @property {function(): void} [onClose] An on-close callback.
   * @property {function(): void} [onBackdropClick] An on-backdrop-click callback.
   * @property {string} [container='body'] A container selector.
   * @property {boolean} [keyboard=true] Enable a keyboard control. The `Esc` key closes a dialog.
   * @property {boolean} [footerAtTheTop=false] To display a footer at the top.
   * @property {module:ui.Dialog~Button[]} [buttonList] Buttons.
   * @property {Array<module:ui.Dialog~Button|false>} [dropdownItemList] Dropdown action items.
   * @property {boolean} [fullHeight] Deprecated.
   * @property {Number} [bodyDiffHeight]
   * @property {Number} [screenWidthXs]
   */

  /**
   * A button or dropdown action item.
   *
   * @typedef {Object} module:ui.Dialog~Button
   *
   * @property {string} name A name.
   * @property {boolean} [pullLeft=false] Deprecated. Use the `position` property.
   * @property {'left'|'right'} [position='left'] A position.
   * @property {string} [html] HTML.
   * @property {string} [text] A text.
   * @property {boolean} [disabled=false] Disabled.
   * @property {boolean} [hidden=false] Hidden.
   * @property {'default'|'danger'|'success'|'warning'} [style='default'] A style.
   * @property {function(Espo.Ui.Dialog, JQueryEventObject): void} [onClick] An on-click callback.
   * @property {string} [className] An additional class name.
   * @property {string} [title] A title.
   */

  /**
   * @alias Espo.Ui.Dialog
   */
  class Dialog {
    height;
    fitHeight;
    onRemove;
    onClose;
    onBackdropClick;
    buttons;
    screenWidthXs;

    /**
     * @param {module:ui.Dialog~Params} options Options.
     */
    constructor(options) {
      options = options || {};

      /** @private */
      this.className = 'dialog-confirm';
      /** @private */
      this.backdrop = 'static';
      /** @private */
      this.closeButton = true;
      /** @private */
      this.collapseButton = false;
      /** @private */
      this.header = null;
      /** @private */
      this.body = '';
      /** @private */
      this.width = null;
      /**
       * @private
       * @type {module:ui.Dialog~Button[]}
       */
      this.buttonList = [];
      /**
       * @private
       * @type {Array<module:ui.Dialog~Button|false>}
       */
      this.dropdownItemList = [];
      /** @private */
      this.removeOnClose = true;
      /** @private */
      this.draggable = false;
      /** @private */
      this.container = 'body';
      /** @private */
      this.options = options;
      /** @private */
      this.keyboard = true;
      this.activeElement = document.activeElement;
      const params = ['className', 'backdrop', 'keyboard', 'closeButton', 'collapseButton', 'header', 'body', 'width', 'height', 'fitHeight', 'buttons', 'buttonList', 'dropdownItemList', 'removeOnClose', 'draggable', 'container', 'onRemove', 'onClose', 'onBackdropClick'];
      params.forEach(param => {
        if (param in options) {
          this[param] = options[param];
        }
      });

      /** @private */
      this.onCloseIsCalled = false;
      if (this.buttons && this.buttons.length) {
        /**
         * @private
         * @type {module:ui.Dialog~Button[]}
         */
        this.buttonList = this.buttons;
      }
      this.id = 'dialog-' + Math.floor(Math.random() * 100000);
      if (typeof this.backdrop === 'undefined') {
        /** @private */
        this.backdrop = 'static';
      }
      const $header = this.getHeader();
      const $footer = this.getFooter();
      const $body = (0, _jquery.default)('<div>').addClass('modal-body body').html(this.body);
      const $content = (0, _jquery.default)('<div>').addClass('modal-content');
      if ($header) {
        $content.append($header);
      }
      if ($footer && this.options.footerAtTheTop) {
        $content.append($footer);
      }
      $content.append($body);
      if ($footer && !this.options.footerAtTheTop) {
        $content.append($footer);
      }
      const $dialog = (0, _jquery.default)('<div>').addClass('modal-dialog').append($content);
      const $container = (0, _jquery.default)(this.container);
      (0, _jquery.default)('<div>').attr('id', this.id).attr('class', this.className + ' modal').attr('role', 'dialog').attr('tabindex', '-1').append($dialog).appendTo($container);

      /**
       * An element.
       *
       * @type {JQuery}
       */
      this.$el = (0, _jquery.default)('#' + this.id);

      /**
       * @private
       * @type {Element}
       */
      this.el = this.$el.get(0);
      this.$el.find('header a.close').on('click', () => {
        //this.close();
      });
      this.initButtonEvents();
      if (this.draggable) {
        this.$el.find('header').css('cursor', 'pointer');

        // noinspection JSUnresolvedReference
        this.$el.draggable({
          handle: 'header'
        });
      }
      const modalContentEl = this.$el.find('.modal-content');
      if (this.width) {
        modalContentEl.css('width', this.width);
        modalContentEl.css('margin-left', '-' + parseInt(this.width.replace('px', '')) / 5 + 'px');
      }
      if (this.removeOnClose) {
        this.$el.on('hidden.bs.modal', e => {
          if (this.$el.get(0) === e.target) {
            if (!this.onCloseIsCalled) {
              this.close();
            }
            if (this.skipRemove) {
              return;
            }
            this.remove();
          }
        });
      }
      const $window = (0, _jquery.default)(window);
      this.$el.on('shown.bs.modal', () => {
        (0, _jquery.default)('.modal-backdrop').not('.stacked').addClass('stacked');
        const headerHeight = this.$el.find('.modal-header').outerHeight() || 0;
        const footerHeight = this.$el.find('.modal-footer').outerHeight() || 0;
        let diffHeight = headerHeight + footerHeight;
        if (!options.fullHeight) {
          diffHeight = diffHeight + options.bodyDiffHeight;
        }
        if (this.fitHeight || options.fullHeight) {
          const processResize = () => {
            const windowHeight = window.innerHeight;
            const windowWidth = $window.width();
            if (!options.fullHeight && windowHeight < 512) {
              this.$el.find('div.modal-body').css({
                maxHeight: 'none',
                overflow: 'auto',
                height: 'none'
              });
              return;
            }
            const cssParams = {
              overflow: 'auto'
            };
            if (options.fullHeight) {
              cssParams.height = windowHeight - diffHeight + 'px';
              this.$el.css('paddingRight', 0);
            } else {
              if (windowWidth <= options.screenWidthXs) {
                cssParams.maxHeight = 'none';
              } else {
                cssParams.maxHeight = windowHeight - diffHeight + 'px';
              }
            }
            this.$el.find('div.modal-body').css(cssParams);
          };
          $window.off('resize.modal-height');
          $window.on('resize.modal-height', processResize);
          processResize();
        }
      });
      const $documentBody = (0, _jquery.default)(document.body);
      this.$el.on('hidden.bs.modal', () => {
        if ((0, _jquery.default)('.modal:visible').length > 0) {
          $documentBody.addClass('modal-open');
        }
      });
    }

    /** @private */
    callOnClose() {
      if (this.onClose) {
        this.onClose();
      }
    }

    /** @private */
    callOnBackdropClick() {
      if (this.onBackdropClick) {
        this.onBackdropClick();
      }
    }

    /** @private */
    callOnRemove() {
      if (this.onRemove) {
        this.onRemove();
      }
    }

    /**
     * Set action items.
     *
     * @param {module:ui.Dialog~Button[]} buttonList
     * @param {Array<module:ui.Dialog~Button|false>} dropdownItemList
     */
    setActionItems(buttonList, dropdownItemList) {
      this.buttonList = buttonList;
      this.dropdownItemList = dropdownItemList;
    }

    /**
     * Init button events.
     */
    initButtonEvents() {
      this.buttonList.forEach(o => {
        if (typeof o.onClick === 'function') {
          const $button = (0, _jquery.default)('#' + this.id + ' .modal-footer button[data-name="' + o.name + '"]');
          $button.on('click', e => o.onClick(this, e));
        }
      });
      this.dropdownItemList.forEach(o => {
        if (o === false) {
          return;
        }
        if (typeof o.onClick === 'function') {
          const $button = (0, _jquery.default)('#' + this.id + ' .modal-footer a[data-name="' + o.name + '"]');
          $button.on('click', e => o.onClick(this, e));
        }
      });
    }

    /**
     * @private
     * @return {JQuery|null}
     */
    getHeader() {
      if (!this.header) {
        return null;
      }
      const $header = (0, _jquery.default)('<header />').addClass('modal-header').addClass(this.options.fixedHeaderHeight ? 'fixed-height' : '').append((0, _jquery.default)('<h4 />').addClass('modal-title').append((0, _jquery.default)('<span />').addClass('modal-title-text').html(this.header)));
      if (this.collapseButton) {
        $header.prepend((0, _jquery.default)('<a>').addClass('collapse-button').attr('role', 'button').attr('tabindex', '-1').attr('data-action', 'collapseModal').append((0, _jquery.default)('<span />').addClass('fas fa-minus')));
      }
      if (this.closeButton) {
        $header.prepend((0, _jquery.default)('<a>').addClass('close').attr('data-dismiss', 'modal').attr('role', 'button').attr('tabindex', '-1').append((0, _jquery.default)('<span />').attr('aria-hidden', 'true').html('&times;')));
      }
      return $header;
    }

    /**
     * Get a footer.
     *
     * @return {JQuery|null}
     */
    getFooter() {
      if (!this.buttonList.length && !this.dropdownItemList.length) {
        return null;
      }
      const $footer = (0, _jquery.default)('<footer>').addClass('modal-footer');
      const $main = (0, _jquery.default)('<div>').addClass('btn-group').addClass('main-btn-group');
      const $additional = (0, _jquery.default)('<div>').addClass('btn-group').addClass('additional-btn-group');
      this.buttonList.forEach(/** module:ui.Dialog~Button */o => {
        const style = o.style || 'default';
        const $button = (0, _jquery.default)('<button>').attr('type', 'button').attr('data-name', o.name).addClass('btn').addClass('btn-' + style).addClass(o.className || 'btn-xs-wide');
        if (o.disabled) {
          $button.attr('disabled', 'disabled');
          $button.addClass('disabled');
        }
        if (o.hidden) {
          $button.addClass('hidden');
        }
        if (o.title) {
          $button.attr('title', o.title);
        }
        if (o.text) {
          $button.text(o.text);
        }
        if (o.html) {
          $button.html(o.html);
        }
        if (o.pullLeft || o.position === 'right') {
          $additional.append($button);
          return;
        }
        $main.append($button);
      });
      const allDdItemsHidden = this.dropdownItemList.filter(o => o && !o.hidden).length === 0;
      const $dropdown = (0, _jquery.default)('<div>').addClass('btn-group').addClass(allDdItemsHidden ? 'hidden' : '').append((0, _jquery.default)('<button>').attr('type', 'button').addClass('btn btn-default dropdown-toggle').addClass(allDdItemsHidden ? 'hidden' : '').attr('data-toggle', 'dropdown').append((0, _jquery.default)('<span>').addClass('fas fa-ellipsis-h')));
      const $ul = (0, _jquery.default)('<ul>').addClass('dropdown-menu pull-right');
      $dropdown.append($ul);
      this.dropdownItemList.forEach((o, i) => {
        if (o === false) {
          if (i === this.dropdownItemList.length - 1) {
            return;
          }
          $ul.append(`<li class="divider"></li>`);
          return;
        }
        const $a = (0, _jquery.default)('<a>').attr('role', 'button').attr('tabindex', '0').attr('data-name', o.name);
        if (o.text) {
          $a.text(o.text);
        }
        if (o.title) {
          $a.attr('title', o.title);
        }
        if (o.html) {
          $a.html(o.html);
        }
        const $li = (0, _jquery.default)('<li>').addClass(o.hidden ? ' hidden' : '').append($a);
        $ul.append($li);
      });
      if ($ul.children().length) {
        $main.append($dropdown);
      }
      if ($additional.children().length) {
        $footer.append($additional);
      }
      $footer.append($main);
      return $footer;
    }

    /**
     * Show.
     */
    show() {
      // noinspection JSUnresolvedReference
      this.$el.modal({
        backdrop: this.backdrop,
        keyboard: this.keyboard
      });
      this.$el.find('.modal-content').removeClass('hidden');
      const $modalBackdrop = (0, _jquery.default)('.modal-backdrop');
      $modalBackdrop.each((i, el) => {
        if (i < $modalBackdrop.length - 1) {
          (0, _jquery.default)(el).addClass('hidden');
        }
      });
      const $modalContainer = (0, _jquery.default)('.modal-container');
      $modalContainer.each((i, el) => {
        if (i < $modalContainer.length - 1) {
          (0, _jquery.default)(el).addClass('overlaid');
        }
      });
      this.$el.off('click.dismiss.bs.modal');
      this.$el.on('click.dismiss.bs.modal', '> div.modal-dialog > div.modal-content > header [data-dismiss="modal"]', () => this.close());
      this.$el.on('mousedown', e => {
        this.$mouseDownTarget = (0, _jquery.default)(e.target);
      });
      this.$el.on('click.dismiss.bs.modal', e => {
        if (e.target !== e.currentTarget) {
          return;
        }
        if (this.$mouseDownTarget && this.$mouseDownTarget.closest('.modal-content').length) {
          return;
        }
        this.callOnBackdropClick();
        if (this.backdrop === 'static') {
          return;
        }
        this.close();
      });
      (0, _jquery.default)('body > .popover').addClass('hidden');
    }

    /**
     * Hide.
     */
    hide() {
      this.$el.find('.modal-content').addClass('hidden');
    }

    /**
     * Hide with a backdrop.
     */
    hideWithBackdrop() {
      const $modalBackdrop = (0, _jquery.default)('.modal-backdrop');
      $modalBackdrop.last().addClass('hidden');
      (0, _jquery.default)($modalBackdrop.get($modalBackdrop.length - 2)).removeClass('hidden');
      const $modalContainer = (0, _jquery.default)('.modal-container');
      (0, _jquery.default)($modalContainer.get($modalContainer.length - 2)).removeClass('overlaid');
      this.skipRemove = true;
      setTimeout(() => {
        this.skipRemove = false;
      }, 50);

      // noinspection JSUnresolvedReference
      this.$el.modal('hide');
      this.$el.find('.modal-content').addClass('hidden');
    }

    /**
     * @private
     */
    _close() {
      const $modalBackdrop = (0, _jquery.default)('.modal-backdrop');
      $modalBackdrop.last().removeClass('hidden');
      const $modalContainer = (0, _jquery.default)('.modal-container');
      (0, _jquery.default)($modalContainer.get($modalContainer.length - 2)).removeClass('overlaid');
    }

    /**
     * @private
     * @param {Element} element
     * @return {Element|null}
     */
    _findClosestFocusableElement(element) {
      // noinspection JSUnresolvedReference
      const isVisible = !!(element.offsetWidth || element.offsetHeight || element.getClientRects().length);
      if (isVisible) {
        // noinspection JSUnresolvedReference
        element.focus({
          preventScroll: true
        });
        return element;
      }
      const $element = (0, _jquery.default)(element);
      if ($element.closest('.dropdown-menu').length) {
        const $button = $element.closest('.btn-group').find(`[data-toggle="dropdown"]`);
        if ($button.length) {
          // noinspection JSUnresolvedReference
          $button.get(0).focus({
            preventScroll: true
          });
          return $button.get(0);
        }
      }
      return null;
    }

    /**
     * Close.
     */
    close() {
      if (!this.onCloseIsCalled) {
        this.callOnClose();
        this.onCloseIsCalled = true;
        if (this.activeElement) {
          setTimeout(() => {
            const element = this._findClosestFocusableElement(this.activeElement);
            if (element) {
              // noinspection JSUnresolvedReference
              element.focus({
                preventScroll: true
              });
            }
          }, 50);
        }
      }
      this._close();
      // noinspection JSUnresolvedReference
      this.$el.modal('hide');
      (0, _jquery.default)(this).trigger('dialog:close');
    }

    /**
     * Remove.
     */
    remove() {
      this.callOnRemove();

      // Hack allowing multiple backdrops.
      // `close` function may be called twice.
      this._close();
      this.$el.remove();
      (0, _jquery.default)(this).off();
      (0, _jquery.default)(window).off('resize.modal-height');
    }
  }

  /**
   * UI utils.
   */
  Espo.Ui = {
    Dialog: Dialog,
    /**
     * @typedef {Object} Espo.Ui~ConfirmOptions
     *
     * @property {string} confirmText A confirm-button text.
     * @property {string} cancelText A cancel-button text.
     * @property {'danger'|'success'|'warning'|'default'} [confirmStyle='danger']
     *   A confirm-button style.
     * @property {'static'|boolean} [backdrop=false] A backdrop.
     * @property {function():void} [cancelCallback] A cancel-callback.
     * @property {boolean} [isHtml=false] Whether the message is HTML.
     */

    /**
     * Show a confirmation dialog.
     *
     * @param {string} message A message.
     * @param {Espo.Ui~ConfirmOptions|{}} o Options.
     * @param {function} [callback] Deprecated. Use a promise.
     * @param {Object} [context] Deprecated.
     * @returns {Promise} Resolves if confirmed.
     */
    confirm: function (message, o, callback, context) {
      o = o || {};
      const confirmText = o.confirmText;
      const cancelText = o.cancelText;
      const confirmStyle = o.confirmStyle || 'danger';
      let backdrop = o.backdrop;
      if (typeof backdrop === 'undefined') {
        backdrop = false;
      }
      let isResolved = false;
      const processCancel = () => {
        if (!o.cancelCallback) {
          return;
        }
        if (context) {
          o.cancelCallback.call(context);
          return;
        }
        o.cancelCallback();
      };
      if (!o.isHtml) {
        message = Handlebars.Utils.escapeExpression(message);
      }
      return new Promise(resolve => {
        const dialog = new Dialog({
          backdrop: backdrop,
          header: null,
          className: 'dialog-confirm',
          body: '<span class="confirm-message">' + message + '</a>',
          buttonList: [{
            text: ' ' + confirmText + ' ',
            name: 'confirm',
            className: 'btn-s-wide',
            onClick: () => {
              isResolved = true;
              if (callback) {
                if (context) {
                  callback.call(context);
                } else {
                  callback();
                }
              }
              resolve();
              dialog.close();
            },
            style: confirmStyle,
            position: 'right'
          }, {
            text: cancelText,
            name: 'cancel',
            className: 'btn-s-wide',
            onClick: () => {
              isResolved = true;
              dialog.close();
              processCancel();
            },
            position: 'left'
          }],
          onClose: () => {
            if (isResolved) {
              return;
            }
            processCancel();
          }
        });
        dialog.show();
        dialog.$el.find('button[data-name="confirm"]').focus();
      });
    },
    /**
     * Create a dialog.
     *
     * @param {module:ui.Dialog~Params} options Options.
     * @returns {Dialog}
     */
    dialog: function (options) {
      return new Dialog(options);
    },
    /**
     * Popover options.
     *
     * @typedef {Object} Espo.Ui~PopoverOptions
     *
     * @property {'bottom'|'top'|'left'|'right'} [placement='bottom'] A placement.
     * @property {string|JQuery} [container] A container selector.
     * @property {string} [content] An HTML content.
     * @property {string} [text] A text.
     * @property {'manual'|'click'|'hover'|'focus'} [trigger='manual'] A trigger type.
     * @property {boolean} [noToggleInit=false] Skip init toggle on click.
     * @property {boolean} [preventDestroyOnRender=false] Don't destroy on re-render.
     * @property {boolean} [noHideOnOutsideClick=false] Don't hide on clicking outside.
     * @property {function(): void} [onShow] On-show callback.
     * @property {function(): void} [onHide] On-hide callback.
     * @property {string|function(): string} [title] A title text.
     * @property {boolean} [keepElementTitle] Keep an original element's title.
     */

    /**
     * Init a popover.
     *
     * @param {Element|JQuery} element An element.
     * @param {Espo.Ui~PopoverOptions} o Options.
     * @param {module:view} [view] A view.
     * @return {{
     *     hide: function(),
     *     destroy: function(),
     *     show: function(): string,
     *     detach: function(),
     * }}
     */
    popover: function (element, o, view) {
      const $el = (0, _jquery.default)(element);
      const $body = (0, _jquery.default)('body');
      const content = o.content || Handlebars.Utils.escapeExpression(o.text || '');
      let isShown = false;
      let container = o.container;
      if (!container) {
        const $modalBody = $el.closest('.modal-body');
        container = $modalBody.length ? $modalBody : 'body';
      }

      // noinspection JSUnresolvedReference
      $el.popover({
        placement: o.placement || 'bottom',
        container: container,
        viewport: container,
        html: true,
        content: content,
        trigger: o.trigger || 'manual',
        title: o.title,
        keepElementTitle: o.keepElementTitle
      }).on('shown.bs.popover', () => {
        isShown = true;
        if (!view) {
          return;
        }
        if (view && !o.noHideOnOutsideClick) {
          $body.off(`click.popover-${view.cid}`);
          $body.on(`click.popover-${view.cid}`, e => {
            if ((0, _jquery.default)(e.target).closest('.popover').get(0)) {
              return;
            }
            if (_jquery.default.contains($el.get(0), e.target)) {
              return;
            }
            if ($el.get(0) === e.target) {
              return;
            }
            $body.off(`click.popover-${view.cid}`);
            // noinspection JSUnresolvedReference
            $el.popover('hide');
          });
        }
        if (o.onShow) {
          o.onShow();
        }
      }).on('hidden.bs.popover', () => {
        isShown = false;
        if (o.onHide) {
          o.onHide();
        }
      });
      if (!o.noToggleInit) {
        $el.on('click', () => {
          // noinspection JSUnresolvedReference
          $el.popover('toggle');
        });
      }
      let isDetached = false;
      const detach = () => {
        if (view) {
          $body.off(`click.popover-${view.cid}`);
          view.off('remove', destroy);
          view.off('render', destroy);
          view.off('render', hide);
        }
        isDetached = true;
      };
      const destroy = () => {
        if (isDetached) {
          return;
        }

        // noinspection JSUnresolvedReference
        $el.popover('destroy');
        detach();
      };
      const hide = () => {
        if (!isShown) {
          return;
        }

        // noinspection JSUnresolvedReference
        $el.popover('hide');
      };
      const show = () => {
        // noinspection JSUnresolvedReference
        $el.popover('show');
        return $el.attr('aria-describedby');
      };
      if (view) {
        view.once('remove', destroy);
        if (!o.preventDestroyOnRender) {
          view.once('render', destroy);
        }
        if (o.preventDestroyOnRender) {
          view.on('render', hide);
        }
      }
      return {
        hide: () => hide(),
        destroy: () => destroy(),
        show: () => show(),
        detach: () => detach()
      };
    },
    /**
     * Notify options.
     *
     * @typedef {Object} Espo.Ui~NotifyOptions
     * @property {boolean} [closeButton] A close button.
     * @property {boolean} [suppress] Suppress other warning alerts while this is displayed.
     */

    /**
     * Show a notify-message.
     *
     * @param {string|false} [message=false] A message. False removes an already displayed message.
     * @param {'warning'|'danger'|'success'|'info'} [type='warning'] A type.
     * @param {number} [timeout] Microseconds. If empty, then won't be hidden.
     *   Should be hidden manually or by displaying another message.
     * @param {Espo.Ui~NotifyOptions} [options] Options.
     */
    notify: function (message, type, timeout, options) {
      type = type || 'warning';
      options = {
        ...options
      };
      if (type === 'warning' && notifySuppressed) {
        return;
      }
      (0, _jquery.default)('#notification').remove();
      if (!message) {
        return;
      }
      if (options.suppress && timeout) {
        notifySuppressed = true;
        setTimeout(() => notifySuppressed = false, timeout);
      }
      const parsedMessage = message.indexOf('\n') !== -1 ? _marked.marked.parse(message) : _marked.marked.parseInline(message);
      let sanitizedMessage = _dompurify.default.sanitize(parsedMessage, {}).toString();
      const closeButton = options.closeButton || false;
      if (type === 'error') {
        // For bc.
        type = 'danger';
      }
      if (sanitizedMessage === ' ... ') {
        sanitizedMessage = ' <span class="fas fa-spinner fa-spin"> ';
      }
      const additionalClassName = closeButton ? ' alert-closable' : '';
      const $el = (0, _jquery.default)('<div>').addClass('alert alert-' + type + additionalClassName + ' fade in').attr('id', 'notification').css({
        'position': 'fixed',
        'top': '0',
        'left': '50vw',
        'transform': 'translate(-50%, 0)',
        'z-index': 2000
      }).append((0, _jquery.default)('<div>').addClass('message').html(sanitizedMessage));
      if (closeButton) {
        const $close = (0, _jquery.default)('<button>').attr('type', 'button').attr('data-dismiss', 'modal').attr('aria-hidden', 'true').addClass('close').html(`<span class="fas fa-times"></span>`);
        $el.append((0, _jquery.default)('<div>').addClass('close-container').append($close));
        $close.on('click', () => $el.alert('close'));
      }
      if (timeout) {
        setTimeout(() => $el.alert('close'), timeout);
      }
      $el.appendTo('body');
    },
    /**
     * Show a warning message.
     *
     * @param {string} message A message.
     * @param {Espo.Ui~NotifyOptions} [options] Options.
     */
    warning: function (message, options) {
      Espo.Ui.notify(message, 'warning', 2000, options);
    },
    /**
     * Show a success message.
     *
     * @param {string} message A message.
     * @param {Espo.Ui~NotifyOptions} [options] Options.
     */
    success: function (message, options) {
      Espo.Ui.notify(message, 'success', 2000, options);
    },
    /**
     * Show an error message.
     *
     * @param {string} message A message.
     * @param {Espo.Ui~NotifyOptions|true} [options] Options. If true, then only closeButton option will be applied.
     */
    error: function (message, options) {
      options = typeof options === 'boolean' ? {
        closeButton: options
      } : {
        ...options
      };
      const timeout = options.closeButton ? 0 : 4000;
      Espo.Ui.notify(message, 'danger', timeout, options);
    },
    /**
     * Show an info message.
     *
     * @param {string} message A message.
     * @param {Espo.Ui~NotifyOptions} [options] Options.
     */
    info: function (message, options) {
      Espo.Ui.notify(message, 'info', 2000, options);
    }
  };
  let notifySuppressed = false;

  /**
   * @deprecated Use `Espo.Ui`.
   */
  Espo.ui = Espo.Ui;
  var _default = _exports.default = Espo.Ui;
});

define("theme-manager", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module theme-manager */

  /**
   * A theme manager.
   */
  class ThemeManager {
    /**
     * @param {module:models/settings} config A config.
     * @param {module:models/preferences} preferences Preferences.
     * @param {module:metadata} metadata Metadata.
     * @param {?string} [name] A name. If not set, then will be obtained from config and preferences.
     */
    constructor(config, preferences, metadata, name) {
      /**
       * @private
       * @type {module:models/settings}
       */
      this.config = config;

      /**
       * @private
       * @type {module:models/preferences}
       */
      this.preferences = preferences;

      /**
       * @private
       * @type {module:metadata}
       */
      this.metadata = metadata;

      /**
       * @private
       * @type {?string}
       */
      this.name = name || null;
    }

    /**
     * @private
     */
    defaultParams = {
      screenWidthXs: 768,
      dashboardCellHeight: 40,
      dashboardCellMargin: 16
    };

    /**
     * Get a theme name for the current user.
     *
     * @returns {string}
     */
    getName() {
      if (this.name) {
        return this.name;
      }
      if (!this.config.get('userThemesDisabled')) {
        const name = this.preferences.get('theme');
        if (name && name !== '') {
          return name;
        }
      }
      return this.config.get('theme');
    }

    /**
     * Get a theme name currently applied to the DOM.
     *
     * @returns {string|null} Null if not applied.
     */
    getAppliedName() {
      const name = window.getComputedStyle(document.body).getPropertyValue('--theme-name');
      if (!name) {
        return null;
      }
      return name.trim();
    }

    /**
     * Whether a current theme is applied to the DOM.
     *
     * @returns {boolean}
     */
    isApplied() {
      const appliedName = this.getAppliedName();
      if (!appliedName) {
        return true;
      }
      return this.getName() === appliedName;
    }

    /**
     * Get a stylesheet path for a current theme.
     *
     * @returns {string}
     */
    getStylesheet() {
      let link = this.getParam('stylesheet') || 'client/css/espo/espo.css';
      if (this.config.get('cacheTimestamp')) {
        link += '?r=' + this.config.get('cacheTimestamp').toString();
      }
      return link;
    }

    /**
     * Get an iframe stylesheet path for a current theme.
     *
     * @returns {string}
     */
    getIframeStylesheet() {
      let link = this.getParam('stylesheetIframe') || 'client/css/espo/espo-iframe.css';
      if (this.config.get('cacheTimestamp')) {
        link += '?r=' + this.config.get('cacheTimestamp').toString();
      }
      return link;
    }

    /**
     * Get an iframe-fallback stylesheet path for a current theme.
     *
     * @returns {string}
     */
    getIframeFallbackStylesheet() {
      let link = this.getParam('stylesheetIframeFallback') || 'client/css/espo/espo-iframe.css';
      if (this.config.get('cacheTimestamp')) {
        link += '?r=' + this.config.get('cacheTimestamp').toString();
      }
      return link;
    }

    /**
     * Get a theme parameter.
     *
     * @param {string} name A parameter name.
     * @returns {*} Null if not set.
     */
    getParam(name) {
      if (name !== 'params' && name !== 'mappedParams') {
        const varValue = this.getVarParam(name);
        if (varValue !== null) {
          return varValue;
        }
        const mappedValue = this.getMappedParam(name);
        if (mappedValue !== null) {
          return mappedValue;
        }
      }
      let value = this.metadata.get(['themes', this.getName(), name]);
      if (value !== null) {
        return value;
      }
      value = this.metadata.get(['themes', this.getParentName(), name]);
      if (value !== null) {
        return value;
      }
      return this.defaultParams[name] || null;
    }

    /**
     * @private
     * @param {string} name
     * @returns {*}
     */
    getVarParam(name) {
      const params = this.getParam('params') || {};
      if (!(name in params)) {
        return null;
      }
      let values = null;
      if (!this.config.get('userThemesDisabled') && this.preferences.get('theme')) {
        values = this.preferences.get('themeParams');
      }
      if (!values) {
        values = this.config.get('themeParams');
      }
      if (values && name in values) {
        return values[name];
      }
      if ('default' in params[name]) {
        return params[name].default;
      }
      return null;
    }

    /**
     * @private
     * @param {string} name
     * @returns {*}
     */
    getMappedParam(name) {
      const mappedParams = this.getParam('mappedParams') || {};
      if (!(name in mappedParams)) {
        return null;
      }
      const mapped = mappedParams[name].param;
      const valueMap = mappedParams[name].valueMap;
      if (mapped && valueMap) {
        const key = this.getParam(mapped);
        return valueMap[key];
      }
      return null;
    }

    /**
     * @private
     * @returns {string}
     */
    getParentName() {
      return this.metadata.get(['themes', this.getName(), 'parent']) || 'Espo';
    }

    /**
     * Whether a current theme is different from a system default theme.
     *
     * @returns {boolean}
     */
    isUserTheme() {
      if (this.config.get('userThemesDisabled')) {
        return false;
      }
      const name = this.preferences.get('theme');
      if (!name || name === '') {
        return false;
      }
      return name !== this.config.get('theme');
    }

    /**
     * Get a font-size factor. To adjust px sizes based on font-size.
     *
     * @return {number}
     * @since 9.0.0
     * @internal Experimental.
     */
    getFontSizeFactor() {
      const paramFontSize = this.getParam('fontSize') || 14;
      const fontSize = parseInt(getComputedStyle(document.body).fontSize);
      return Math.round(fontSize / paramFontSize * 10000) / 10000;
    }
  }
  var _default = _exports.default = ThemeManager;
});

define("storage", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module storage */

  /**
   * A storage. Data is saved across browser sessions, has no expiration time.
   */
  class Storage {
    constructor() {}

    /** @protected */
    prefix = 'espo';

    /** @protected */
    storageObject = localStorage;

    /**
     * @private
     * @param {string} type
     * @returns {string}
     */
    composeFullPrefix(type) {
      return this.prefix + '-' + type;
    }

    /**
     * @private
     * @param {string} type
     * @param {string} name
     * @returns {string}
     */
    composeKey(type, name) {
      return this.composeFullPrefix(type) + '-' + name;
    }

    /**
     * @private
     * @param {string} type
     */
    checkType(type) {
      if (typeof type === 'undefined' && toString.call(type) !== '[object String]' || type === 'cache') {
        throw new TypeError("Bad type \"" + type + "\" passed to Espo.Storage.");
      }
    }

    /**
     * Has a value.
     *
     * @param {string} type A type (category).
     * @param {string} name A name.
     * @returns {boolean}
     */
    has(type, name) {
      this.checkType(type);
      const key = this.composeKey(type, name);
      return this.storageObject.getItem(key) !== null;
    }

    /**
     * Get a value.
     *
     * @param {string} type A type (category).
     * @param {string} name A name.
     * @returns {*} Null if not stored.
     */
    get(type, name) {
      this.checkType(type);
      const key = this.composeKey(type, name);
      let stored;
      try {
        stored = this.storageObject.getItem(key);
      } catch (error) {
        console.error(error);
        return null;
      }
      if (stored) {
        let result = stored;
        if (stored.length > 9 && stored.substring(0, 9) === '__JSON__:') {
          const jsonString = stored.slice(9);
          try {
            result = JSON.parse(jsonString);
          } catch (error) {
            result = stored;
          }
        } else if (stored[0] === "{" || stored[0] === "[") {
          // for backward compatibility
          try {
            result = JSON.parse(stored);
          } catch (error) {
            result = stored;
          }
        }
        return result;
      }
      return null;
    }

    /**
     * Set (store) a value.
     *
     * @param {string} type A type (category).
     * @param {string} name A name.
     * @param {*} value A value.
     */
    set(type, name, value) {
      this.checkType(type);
      if (value === null) {
        this.clear(type, name);
        return;
      }
      const key = this.composeKey(type, name);
      if (value instanceof Object || Array.isArray(value) || value === true || value === false || typeof value === 'number') {
        value = '__JSON__:' + JSON.stringify(value);
      }
      try {
        this.storageObject.setItem(key, value);
      } catch (error) {
        console.error(error);
        return null;
      }
    }

    /**
     * Clear a value.
     *
     * @param {string} type A type (category).
     * @param {string} name A name.
     */
    clear(type, name) {
      let reText;
      if (typeof type !== 'undefined') {
        if (typeof name === 'undefined') {
          reText = '^' + this.composeFullPrefix(type);
        } else {
          reText = '^' + this.composeKey(type, name);
        }
      } else {
        reText = '^' + this.prefix + '-';
      }
      const re = new RegExp(reText);
      for (const i in this.storageObject) {
        if (re.test(i)) {
          delete this.storageObject[i];
        }
      }
    }
  }
  var _default = _exports.default = Storage;
});

define("session-storage", ["exports"], function (_exports) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module session-storage */

  /**
   * A session storage. Cleared when a page session ends.
   */
  class SessionStorage {
    /** @private */
    storageObject = sessionStorage;

    /**
     * Get a value.
     *
     * @param {string} name A name.
     * @returns {*} Null if not set.
     */
    get(name) {
      let stored;
      try {
        stored = this.storageObject.getItem(name);
      } catch (error) {
        console.error(error);
        return null;
      }
      if (stored) {
        let result = stored;
        if (stored.length > 9 && stored.substring(0, 9) === '__JSON__:') {
          const jsonString = stored.slice(9);
          try {
            result = JSON.parse(jsonString);
          } catch (error) {
            result = stored;
          }
        }
        return result;
      }
      return null;
    }

    /**
     * Set (store) a value.
     *
     * @param {string} name A name.
     * @param {*} value A value.
     */
    set(name, value) {
      if (value === null) {
        this.clear(name);
        return;
      }
      if (value instanceof Object || Array.isArray(value) || value === true || value === false || typeof value === 'number') {
        value = '__JSON__:' + JSON.stringify(value);
      }
      try {
        this.storageObject.setItem(name, value);
      } catch (error) {
        console.error(error);
      }
    }

    /**
     * Has a value.
     *
     * @param {string} name A name.
     * @returns {boolean}
     */
    has(name) {
      return this.storageObject.getItem(name) !== null;
    }

    /**
     * Clear a value.
     *
     * @param {string} name A name.
     */
    clear(name) {
      for (const i in this.storageObject) {
        if (i === name) {
          delete this.storageObject[i];
        }
      }
    }
  }
  var _default = _exports.default = SessionStorage;
});

define("router", ["exports", "backbone"], function (_exports, _backbone) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _backbone = _interopRequireDefault(_backbone);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it will be useful,
   * but WITHOUT ANY WARRANTY; without even the implied warranty of
   * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   * GNU Affero General Public License for more details.
   *
   * You should have received a copy of the GNU Affero General Public License
   * along with this program. If not, see <https://www.gnu.org/licenses/>.
   *
   * The interactive user interfaces in modified source and object code versions
   * of this program must display Appropriate Legal Notices, as required under
   * Section 5 of the GNU Affero General Public License version 3.
   *
   * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
   * these Appropriate Legal Notices must retain the display of the "EspoCRM" word.
   ************************************************************************/

  /** @module router */

  /**
   * On route.
   *
   * @event Backbone.Router#route
   * @param {string} name A route name.
   * @param {any[]} args Arguments.
   */

  /**
   * After dispatch.
   *
   * @event module:router#routed
   * @param {{
   *   controller: string,
   *   action:string,
   *   options: Object.<string,*>,
   * }} data A route data.
   */

  /**
   * Subscribe.
   *
   * @function on
   * @memberof module:router#
   * @param {string} event An event.
   * @param {function(*): void} callback A callback.
   */

  /**
   * Subscribe once.
   *
   * @function once
   * @memberof module:router#
   * @param {string} event An event.
   * @param {function(): void} callback A callback.
   */

  /**
   * Unsubscribe.
   *
   * @function off
   * @memberof module:router#
   * @param {string} event An event.
   */

  /**
   * Trigger an event.
   *
   * @function trigger
   * @memberof module:router#
   * @param {string} event An event.
   */

  // noinspection JSUnusedGlobalSymbols
  /**
   * A router.
   *
   * @class
   * @mixes Espo.Events
   */
  const Router = _backbone.default.Router.extend(/** @lends Router# */{
    /**
     * @private
     */
    routeList: [{
      route: "clearCache",
      resolution: "clearCache"
    }, {
      route: ":controller/view/:id/:options",
      resolution: "view"
    }, {
      route: ":controller/view/:id",
      resolution: "view"
    }, {
      route: ":controller/edit/:id/:options",
      resolution: "edit"
    }, {
      route: ":controller/edit/:id",
      resolution: "edit"
    }, {
      route: ":controller/create",
      resolution: "create"
    }, {
      route: ":controller/related/:id/:link",
      resolution: "related"
    }, {
      route: ":controller/:action/:options",
      resolution: "action",
      order: 100
    }, {
      route: ":controller/:action",
      resolution: "action",
      order: 200
    }, {
      route: ":controller",
      resolution: "defaultAction",
      order: 300
    }, {
      route: "*actions",
      resolution: "home",
      order: 500
    }],
    /**
     * @private
     */
    _bindRoutes: function () {},
    /**
     * @private
     */
    setupRoutes: function () {
      this.routeParams = {};
      if (this.options.routes) {
        const routeList = [];
        Object.keys(this.options.routes).forEach(route => {
          const item = this.options.routes[route];
          routeList.push({
            route: route,
            resolution: item.resolution || 'defaultRoute',
            order: item.order || 0
          });
          this.routeParams[route] = item.params || {};
        });
        this.routeList = Espo.Utils.clone(this.routeList);
        routeList.forEach(item => {
          this.routeList.push(item);
        });
        this.routeList = this.routeList.sort((v1, v2) => {
          return (v1.order || 0) - (v2.order || 0);
        });
      }
      this.routeList.reverse().forEach(item => {
        this.route(item.route, item.resolution);
      });
    },
    /**
     * @private
     */
    _last: null,
    /**
     * Whether a confirm-leave-out was set.
     *
     * @public
     * @type {boolean}
     */
    confirmLeaveOut: false,
    /**
     * Whether back has been processed.
     *
     * @public
     * @type {boolean}
     */
    backProcessed: false,
    /**
     * @type {string}
     * @internal
     */
    confirmLeaveOutMessage: 'Are you sure?',
    /**
     * @type {string}
     * @internal
     */
    confirmLeaveOutConfirmText: 'Yes',
    /**
     * @type {string}
     * @internal
     */
    confirmLeaveOutCancelText: 'No',
    /**
     * @private
     */
    initialize: function (options) {
      this.options = options || {};
      this.setupRoutes();
      this._isReturn = false;
      this.history = [];
      let hashHistory = [window.location.hash];
      window.addEventListener('hashchange', () => {
        const hash = window.location.hash;
        if (hashHistory.length > 1 && hashHistory[hashHistory.length - 2] === hash) {
          hashHistory = hashHistory.slice(0, -1);
          this.backProcessed = true;
          setTimeout(() => this.backProcessed = false, 50);
          return;
        }
        hashHistory.push(hash);
      });
      this.on('route', () => {
        this.history.push(_backbone.default.history.fragment);
      });
      window.addEventListener('beforeunload', e => {
        e = e || window.event;
        if (this.confirmLeaveOut) {
          e.preventDefault();
          e.returnValue = this.confirmLeaveOutMessage;
          return this.confirmLeaveOutMessage;
        }
      });
    },
    /**
     * Get a current URL.
     *
     * @returns {string}
     */
    getCurrentUrl: function () {
      return '#' + _backbone.default.history.fragment;
    },
    /**
     * @callback module:router~checkConfirmLeaveOutCallback
     */

    /**
     * Process confirm-leave-out.
     *
     * @param {module:router~checkConfirmLeaveOutCallback} callback Proceed if confirmed.
     * @param {Object|null} [context] A context.
     * @param {boolean} [navigateBack] To navigate back if not confirmed.
     */
    checkConfirmLeaveOut: function (callback, context, navigateBack) {
      if (this.confirmLeaveOutDisplayed) {
        this.navigateBack({
          trigger: false
        });
        this.confirmLeaveOutCanceled = true;
        return;
      }
      context = context || this;
      if (this.confirmLeaveOut) {
        this.confirmLeaveOutDisplayed = true;
        this.confirmLeaveOutCanceled = false;
        Espo.Ui.confirm(this.confirmLeaveOutMessage, {
          confirmText: this.confirmLeaveOutConfirmText,
          cancelText: this.confirmLeaveOutCancelText,
          backdrop: true,
          cancelCallback: () => {
            this.confirmLeaveOutDisplayed = false;
            if (navigateBack) {
              this.navigateBack({
                trigger: false
              });
            }
          }
        }, () => {
          this.confirmLeaveOutDisplayed = false;
          this.confirmLeaveOut = false;
          if (!this.confirmLeaveOutCanceled) {
            callback.call(context);
          }
        });
        return;
      }
      callback.call(context);
    },
    /**
     * @private
     */
    route: function (route, name /*, callback*/) {
      const routeOriginal = route;
      if (!_.isRegExp(route)) {
        route = this._routeToRegExp(route);
      }
      let callback;

      // @todo Revise.
      /*if (_.isFunction(name)) {
          callback = name;
          name = '';
      }*/

      /*if (!callback) {
          callback = this['_' + name];
      }*/
      callback = this['_' + name];
      const router = this;
      _backbone.default.history.route(route, function (fragment) {
        const args = router._extractParameters(route, fragment);
        const options = {};
        if (name === 'defaultRoute') {
          const keyList = [];
          routeOriginal.split('/').forEach(key => {
            if (key && key.indexOf(':') === 0) {
              keyList.push(key.substr(1));
            }
          });
          keyList.forEach((key, i) => {
            options[key] = args[i];
          });
        }

        // @todo Revise.
        router.execute(callback, args, name, routeOriginal, options);
        //if (router.execute(callback, args, name, routeOriginal, options) !== false) {
        router.trigger.apply(router, ['route:' + name].concat(args));
        router.trigger('route', name, args);
        _backbone.default.history.trigger('route', router, name, args);
        //}
      });
      return this;
    },
    /**
     * @private
     */
    execute: function (callback, args, name, routeOriginal, options) {
      this.checkConfirmLeaveOut(() => {
        if (name === 'defaultRoute') {
          this._defaultRoute(this.routeParams[routeOriginal], options);
          return;
        }
        _backbone.default.Router.prototype.execute.call(this, callback, args, name);
      }, null, true);
    },
    /**
     * Navigate.
     *
     * @param {string} fragment An URL fragment.
     * @param {{
     *     trigger?: boolean,
     *     replace?: boolean,
     *     isReturn?: boolean,
     * }} [options] Options.
     */
    navigate: function (fragment, options = {}) {
      if (!options.trigger) {
        this.history.push(fragment);
      }
      if (options.isReturn) {
        this._isReturn = true;
      }
      return _backbone.default.Router.prototype.navigate.call(this, fragment, options);
    },
    /**
     * Navigate back.
     *
     * @param {Object} [options] Options: trigger, replace.
     */
    navigateBack: function (options) {
      let url;
      url = this.history.length > 1 ? this.history[this.history.length - 2] : this.history[0];
      this.navigate(url, options);
    },
    /**
     * @private
     */
    _parseOptionsParams: function (string) {
      if (!string) {
        return {};
      }
      if (string.indexOf('&') === -1 && string.indexOf('=') === -1) {
        return string;
      }
      const options = {};
      if (typeof string !== 'undefined') {
        string.split('&').forEach(item => {
          const p = item.split('=');
          options[p[0]] = true;
          if (p.length > 1) {
            options[p[0]] = p[1];
          }
        });
      }
      return options;
    },
    /**
     * @private
     */
    _defaultRoute: function (params, options) {
      const controller = params.controller || options.controller;
      const action = params.action || options.action;
      this.dispatch(controller, action, options);
    },
    /**
     * @private
     */
    _record: function (controller, action, id, options) {
      options = this._parseOptionsParams(options);
      options.id = id;
      this.dispatch(controller, action, options);
    },
    /**
     * @private
     */
    _view: function (controller, id, options) {
      this._record(controller, 'view', id, options);
    },
    /**
     * @private
     */
    _edit: function (controller, id, options) {
      this._record(controller, 'edit', id, options);
    },
    /**
     * @private
     */
    _related: function (controller, id, link, options) {
      options = this._parseOptionsParams(options);
      options.id = id;
      options.link = link;
      this.dispatch(controller, 'related', options);
    },
    /**
     * @private
     */
    _create: function (controller, options) {
      this._record(controller, 'create', null, options);
    },
    /**
     * @private
     */
    _action: function (controller, action, options) {
      this.dispatch(controller, action, this._parseOptionsParams(options));
    },
    /**
     * @private
     */
    _defaultAction: function (controller) {
      this.dispatch(controller, null);
    },
    /**
     * @private
     */
    _home: function () {
      this.dispatch('Home', null);
    },
    /**
     * @private
     */
    _clearCache: function () {
      this.dispatch(null, 'clearCache');
    },
    /**
     * Process `logout` route.
     */
    logout: function () {
      this.dispatch(null, 'logout');
      this.navigate('', {
        trigger: false
      });
    },
    /**
     * Dispatch a controller action.
     *
     * @param {string|null} [controller] A controller.
     * @param {string|null} [action] An action.
     * @param {Object} [options] Options.
     * @fires module:router#routed
     */
    dispatch: function (controller, action, options) {
      if (this._isReturn) {
        options = {
          ...options
        };
        options.isReturn = true;
        this._isReturn = false;
      }
      const o = {
        controller: controller,
        action: action,
        options: options
      };
      if (controller && /[a-z]/.test(controller[0])) {
        o.controllerClassName = controller;
        delete o.controller;
      }
      this._last = o;
      this.trigger('routed', o);
    },
    /**
     * Get the last route data.
     *
     * @returns {Object}
     */
    getLast: function () {
      return this._last;
    }
  });
  var _default = _exports.default = Router;
  function isIOS9UIWebView() {
    const userAgent = window.navigator.userAgent;
    return /(iPhone|iPad|iPod).* OS 9_\d/.test(userAgent) && !/Version\/9\./.test(userAgent);
  }

  // Fixes issue that navigate with {trigger: false} fired
  // route change if there's a whitespace character.
  _backbone.default.history.getHash = function (window) {
    const match = (window || this).location.href.match(/#(.*)$/);
    return match ? this.decodeFragment(match[1]) : '';
  };

  // Override `backbone.history.loadUrl()` and `backbone.history.navigate()`
  // to fix the navigation issue (`location.hash` not changed immediately) on iOS9.
  if (isIOS9UIWebView()) {
    _backbone.default.history.loadUrl = function (fragment, oldHash) {
      fragment = this.fragment = this.getFragment(fragment);
      return _.any(this.handlers, function (handler) {
        if (handler.route.test(fragment)) {
          function runCallback() {
            handler.callback(fragment);
          }
          function wait() {
            if (oldHash === location.hash) {
              window.setTimeout(wait, 50);
            } else {
              runCallback();
            }
          }
          wait();
          return true;
        }
      });
    };
    _backbone.default.history.navigate = function (fragment, options) {
      const pathStripper = /#.*$/;
      if (!_backbone.default.History.started) {
        return false;
      }
      if (!options || options === true) {
        options = {
          trigger: !!options
        };
      }
      let url = this.root + '#' + (fragment = this.getFragment(fragment || ''));
      fragment = fragment.replace(pathStripper, '');
      if (this.fragment === fragment) {
        return;
      }
      this.fragment = fragment;
      if (fragment === '' && url !== '/') {
        url = url.slice(0, -1);
      }
      const oldHash = location.hash;
      if (this._hasPushState) {
        this.history[options.replace ? 'replaceState' : 'pushState']({}, document.title, url);
      } else if (this._wantsHashChange) {
        this._updateHash(this.location, fragment, options.replace);
        if (this.iframe && fragment !== this.getFragment(this.getHash(this.iframe))) {
          if (!options.replace) {
            this.iframe.document.open().close();
          }
          this._updateHash(this.iframe.location, fragment, options.replace);
        }
      } else {
        return this.location.assign(url);
      }
      if (options.trigger) {
        return this.loadUrl(fragment, oldHash);
      }
    };
  }
});

define("page-title", ["exports", "jquery"], function (_exports, _jquery) {
  "use strict";

  Object.defineProperty(_exports, "__esModule", {
    value: true
  });
  _exports.default = void 0;
  _jquery = _interopRequireDefault(_jquery);
  function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
  /************************************************************************
   * This file is part of EspoCRM.
   *
   * EspoCRM – Open Source CRM application.
   * Copyright (C) 2014-2025 Yurii Kuznietsov, Taras Machyshyn, Oleksii Avramenko
   * Website: https://www.espocrm.com
   *
   * This program is free software: you can redistribute it and/or modify
   * it under the terms of the GNU Affero General Public License as published by
   * the Free Software Foundation, either version 3 of the License, or
   * (at your option) any later version.
   *
   * This program is distributed in the hope that it wi